/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Mar 98                                                   *
*  Last Update : Mar 98                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : memoryK.m4                                               *
*                                                                         *
*  Function    : memory transfer functions (vectorizable)                 *
*                                                                         *
*  ATTENTION: vectorization can assume no overlap in memory for pointers  *
*                                                                         *
*  dalib_gather1dK (target, source, size, stride1, n1)                    *
*  dalib_gather2dK (target, source, size, stride1, n1, stride2, n2)       *
*  ....                                                                   *
*  dalib_gather7dK (target, source, size, stride1, n1, ..., stride7, n7)  *
*                                                                         *
*  dalib_scatter1dK (target, source, size, stride1, n1)                   *
*  dalib_scatter2dK (target, source, size, stride1, n1, stride2, n2)      *
*  ....                                                                   *
*  dalib_scatter7dK (target, source, size, stride1, n1, ..., stride7, n7) *
*                                                                         *
*  dalib_memcopyK  (target, source, size)                                 *
*                                                                         *
*  dalib_move1dK (target, source, size, stride1, n1)                      *
*  dalib_move2dK (target, source, size, stride1, n1, stride2, n2)         *
*  ....                                                                   *
*  dalib_move7dK (target, source, size, stride1, n1, ..., stride7, n7)    *
*                                                                         *
*  dalib_memgetK  (target, source, ind_val, n)                            *
*  dalib_memgetpK (target, source, ind_val, n)                            *
*  dalib_memgetmK (target, source, ind_val, mask, n)                      *
*                                                                         *
*  dalib_memsetK  (target, ind_val, source, n)                            *
*  dalib_memsetpK (target, ind_val, source, n)                            *
*  dalib_memsetmK (target, ind_val, source, mask, n)                      *
*                                                                         *
*  TYPE    *target, *source;                                              *
*  int size, stride1, n1, ...., stridek, nk                               *
*                                                                         *
**************************************************************************/

/**************************************************************************
*                                                                         *
*   Indirect Addressing of Local Memory for different data sizes          *
*                                                                         *
*   - note: source, target, ind_val must not overlap  (vectorization)     *
*                                                                         *
*   dalib_memgetK (TYPE target[], TYPE source[], int ind_val[], int n)    *
*                                                                         *
**************************************************************************/

           /*********************************************
           *                                            *
           *   dalib_memgetK (global get)               *
           *                                            *
           *********************************************/

void dalib_memgetK (target, source, ind_val, n)
TYPE *target, *source;
int *ind_val, n;

{ int i;

  for (i=0; i<n; i++)
     target[i] = source[ind_val[i]];

} /* dalib_memgetK */

           /********************************************
           *                                           *
           *   dalib_memgetpK (get only for positive)  *
           *                                           *
           ********************************************/

void dalib_memgetpK (target, source, ind_val, n)
TYPE *target, *source;
int *ind_val, n;

{ int i;
  for (i=0; i<n; i++)
    if (ind_val[i] >= 0)
       target[i] = source[ind_val[i]];
}

           /********************************************
           *                                           *
           *   dalib_memgetmK    (get with mask)       *
           *                                           *
           ********************************************/

void dalib_memgetmK (target, source, ind_val, mask, n)
TYPE *target, *source;
int *ind_val, *mask, n;

{ int i;
  for (i=0; i<n; i++)
   if (mask[i])
     target[i] = source[ind_val[i]];
} /* memgetmK */

          /*********************************************
           *                                           *
           *   dalib_memsetK (global set)              *
           *                                           *
           ********************************************/

void dalib_memsetK (target, ind_val, source, n)
TYPE *target, *source;
int *ind_val, n;

{ int i;
  for (i=0; i<n; i++)
     target[ind_val[i]] = source[i];
} /* dalib_memsetK */

          /*********************************************
           *                                           *
           *   dalib_memsetpK (set only for positive)  *
           *                                           *
           ********************************************/

void dalib_memsetpK (target, ind_val, source, n)
TYPE *target, *source;
int *ind_val, n;

{ int i;
  for (i=0; i<n; i++)
   if (ind_val[i] >= 0)
     target[ind_val[i]] = source[i];
} /* dalib_memsetpK */

          /*********************************************
           *                                           *
           *   dalib_memsetmK    (set with mask)       *
           *                                           *
           ********************************************/

void dalib_memsetmK (target, ind_val, source, mask, n)
TYPE *target, *source;
int *ind_val, *mask, n;

{ int i;
  for (i=0; i<n; i++)
   if (mask[i])
     target[ind_val[i]] = source[i];
} /* memsetmK */

          /*********************************************
           *                                           *
           *   dalib_memcopyK                          *
           *                                           *
           ********************************************/

void dalib_memcopyK (target, source, size)
TYPE *target, *source;
int size;

{ int j;

#ifdef DEBUG
  printf ("%d calls dalib_memcopyK, size = %d\n", pcb.i, size);
#endif

  if (size == 1)
    *target = *source;
  else
   { for (j=0; j<size; j++)
        *target++ = *source++;
   }

} /* dalib_memcopyK */

/**************************************************************************
*                                                                         *
*     GGGGG       A    TTTTTTT  H     H    EEEEEE   RRRR                  *
*    G           A A      T     H     H    E       R    R                 *
*    G           A A      T     H     H    E       R    R                 *
*    G   GGG    A   A     T     HHHHHHH    EEEEEE  RRRRR                  *
*    G     G    AAAAA     T     H     H    E       R  R                   *
*    G     G    A   A     T     H     H    E       R   R                  *
*     GGGGGG    A   A     T     H     H    EEEEEE  R    R                 *
*                                                                         *
**************************************************************************/

void dalib_gather1dK (target, source, size, stride1, n1)
TYPE *target, *source;
int size, stride1, n1;

{ int i, j, ofs;

#ifdef DEBUG
  printf ("%d calls dalib_gather1dK, size = %d, stride1 = %d, n1 = %d\n",
           pcb.i, size, stride1, n1);
#endif

  if (size == 1)
   { for (i=n1; i--; source += stride1)
        *target++ = *source;
   }
  else
   { for (i=n1; i--; source += stride1)
       for (j=0; j<size; j++)
          *target++ = source[j];
   }

} /* dalib_gather1dK */

          /*********************************************
           *                                           *
           *   dalib_gather2dK                         *
           *                                           *
           ********************************************/

void dalib_gather2dK (target, source, size, stride1, n1, stride2, n2)
TYPE *target, *source;
int size, stride1, n1, stride2, n2;

{ int i1, i2, j, ofs;

#ifdef DEBUG
  printf (
   "%d: dalib_gather2dK, size = %d, str1 = %d, n1 = %d, str2 = %d, n2 = %d\n",
        pcb.i, size, stride1, n1, stride2, n2);
#endif

  if (size == 1)
   { for (i2=n2; i2--; source += stride2 - n1*stride1)
       for (i1=n1; i1--; source += stride1)
          *target++ = *source;
   }
  else
   { for (i2=n2; i2--; source += stride2 - n1*stride1)
       for (i1=n1; i1--; source += stride1)
          for (j=0; j<size; j++)
             *target++ = source[j];
   }

} /* dalib_gather2dK */

          /*********************************************
           *                                           *
           *   dalib_gather3dK                         *
           *                                           *
           ********************************************/

void dalib_gather3dK (target, source, size, stride1, n1, stride2, n2,
                                            stride3, n3             )
TYPE *target, *source;
int size, stride1, n1, stride2, n2, stride3, n3;

{ int i1, i2, i3, j, ofs;

#ifdef DEBUG
  printf (
   "%d: dalib_gather3dK, size = %d, str - n : %d:%d %d:%d %d:%d\n",
        pcb.i, size, stride1, n1, stride2, n2, stride3, n3);
#endif

  if (size == 1)
   { for (i3=n3; i3--; source += stride3 - n2*stride2)
        for (i2=n2; i2--; source += stride2 - n1*stride1)
          for (i1=n1; i1--; source += stride1)
             *target++ = *source;
   }
  else
   { for (i3=n3; i3--; source += stride3 - n2*stride2)
        for (i2=n2; i2--; source += stride2 - n1*stride1)
          for (i1=n1; i1--; source += stride1)
             for (j=0; j<size; j++)
                *target++ = source[j];
   }


} /* dalib_gather3dK */

          /*********************************************
           *                                           *
           *   dalib_gather4dK                         *
           *                                           *
           ********************************************/

void dalib_gather4dK (target, source, size, stride1, n1, stride2, n2,
                                            stride3, n3, stride4, n4)
TYPE *target, *source;
int size, stride1, n1, stride2, n2, stride3, n3;

{ int i1, i2, i3, i4, j, ofs;

#ifdef DEBUG
  printf (
   "%d: dalib_gather4dK, size = %d, str - n : %d:%d %d:%d %d:%d\n",
        pcb.i, size, stride1, n1, stride2, n2, stride3, n3, stride4, n4);
#endif

  if (size == 1)
   { for (i4=n4; i4--; source += stride4 - n3*stride3)
        for (i3=n3; i3--; source += stride3 - n2*stride2)
           for (i2=n2; i2--; source += stride2 - n1*stride1)
             for (i1=n1; i1--; source += stride1)
                *target++ = *source;
   }
  else
   { for (i4=n4; i4--; source += stride4 - n3*stride3)
        for (i3=n3; i3--; source += stride3 - n2*stride2)
           for (i2=n2; i2--; source += stride2 - n1*stride1)
              for (i1=n1; i1--; source += stride1)
                 for (j=0; j<size; j++)
                    *target++ = source[j];
   }

} /* dalib_gather4dK */

          /*********************************************
           *                                           *
           *   dalib_gather5dK                         *
           *                                           *
           ********************************************/

void dalib_gather5dK (target, source, size, stride1, n1, stride2, n2,
                               stride3, n3, stride4, n4, stride5, n5)
TYPE *target, *source;
int size, stride1, n1, stride2, n2, stride3, n3, stride4, n4, stride5, n5;

{ int i1, i2, i3, i4, i5, j, ofs;

#ifdef DEBUG
  printf (
   "%d: dalib_gather5dK, size = %d, str - n : %d:%d %d:%d %d:%d %d:%d\n",
        pcb.i, size, stride1, n1, stride2, n2, stride3, n3, stride4, n4,
                     stride5, n5);
#endif

  if (size == 1)
   { for (i5=n5; i5--; source += stride5 - n4*stride4)
       for (i4=n4; i4--; source += stride4 - n3*stride3)
          for (i3=n3; i3--; source += stride3 - n2*stride2)
             for (i2=n2; i2--; source += stride2 - n1*stride1)
               for (i1=n1; i1--; source += stride1)
                  *target++ = *source;
   }
  else
   { for (i5=n5; i5--; source += stride5 - n4*stride4)
       for (i4=n4; i4--; source += stride4 - n3*stride3)
          for (i3=n3; i3--; source += stride3 - n2*stride2)
             for (i2=n2; i2--; source += stride2 - n1*stride1)
                for (i1=n1; i1--; source += stride1)
                   for (j=0; j<size; j++)
                      *target++ = source[j];
   }

} /* dalib_gather5dK */

          /*********************************************
           *                                           *
           *   dalib_gather6dK                         *
           *                                           *
           ********************************************/

void dalib_gather6dK (target, source, size, stride1, n1, stride2, n2,
                                            stride3, n3, stride4, n4, 
                                            stride5, n5, stride6, n6)
TYPE *target, *source;
int size, stride1, n1, stride2, n2, stride3, n3, stride4, n4, 
                                    stride5, n5, stride6, n6;

{ int i1, i2, i3, i4, i5, i6, j, ofs;

#ifdef DEBUG
  printf (
   "%d: dalib_gather6dK, size = %d, str - n : %d:%d %d:%d %d:%d %d:%d %d:%d\n",
        pcb.i, size, stride1, n1, stride2, n2, stride3, n3, stride4, n4,
                     stride5, n5, stride6, n6);
#endif

  if (size == 1)
   { for (i6=n6; i6--; source += stride6 - n5*stride5)
        for (i5=n5; i5--; source += stride5 - n4*stride4)
          for (i4=n4; i4--; source += stride4 - n3*stride3)
             for (i3=n3; i3--; source += stride3 - n2*stride2)
                for (i2=n2; i2--; source += stride2 - n1*stride1)
                  for (i1=n1; i1--; source += stride1)
                     *target++ = *source;
   }
  else
   { for (i6=n6; i6--; source += stride6 - n5*stride5)
        for (i5=n5; i5--; source += stride5 - n4*stride4)
          for (i4=n4; i4--; source += stride4 - n3*stride3)
             for (i3=n3; i3--; source += stride3 - n2*stride2)
                for (i2=n2; i2--; source += stride2 - n1*stride1)
                   for (i1=n1; i1--; source += stride1)
                      for (j=0; j<size; j++)
                         *target++ = source[j];
   }

} /* dalib_gather6dK */

          /*********************************************
           *                                           *
           *   dalib_gather7dK                         *
           *                                           *
           ********************************************/

void dalib_gather7dK (target, source, size, stride1, n1, stride2, n2,
                                            stride3, n3, stride4, n4, 
                               stride5, n5, stride6, n6, stride7, n7)

TYPE *target, *source;
int size, stride1, n1, stride2, n2, stride3, n3, stride4, n4, 
                       stride5, n5, stride6, n6, stride7, n7;

{ int i1, i2, i3, i4, i5, i6, i7, j, ofs;

#ifdef DEBUG
  printf (
   "%d: dalib_gather7dK, size = %d, str - n : %d:%d %d:%d %d:%d %d:%d %d:%d %d:%d\n",
        pcb.i, size, stride1, n1, stride2, n2, stride3, n3, stride4, n4,
                     stride5, n5, stride6, n6, stride7, n7);
#endif

  if (size == 1)
   { for (i7=n7; i7--; source += stride7 - n6*stride6)
        for (i6=n6; i6--; source += stride6 - n5*stride5)
           for (i5=n5; i5--; source += stride5 - n4*stride4)
             for (i4=n4; i4--; source += stride4 - n3*stride3)
                for (i3=n3; i3--; source += stride3 - n2*stride2)
                   for (i2=n2; i2--; source += stride2 - n1*stride1)
                     for (i1=n1; i1--; source += stride1)
                        *target++ = *source;
   }
  else
   { for (i7=n7; i7--; source += stride7 - n6*stride6)
        for (i6=n6; i6--; source += stride6 - n5*stride5)
           for (i5=n5; i5--; source += stride5 - n4*stride4)
             for (i4=n4; i4--; source += stride4 - n3*stride3)
                for (i3=n3; i3--; source += stride3 - n2*stride2)
                   for (i2=n2; i2--; source += stride2 - n1*stride1)
                      for (i1=n1; i1--; source += stride1)
                         for (j=0; j<size; j++)
                            *target++ = source[j];
   }

} /* dalib_gather7dK */

/**************************************************************************
*                                                                         *
*     SSSSSS     CCCCC    A    TTTTTTT  TTTTTTT  EEEEEE   RRRR            *
*    S          C        A A      T        T     E       R    R           *
*    S          C        A A      T        T     E       R    R           *
*     SSSSS     C       A   A     T        T     EEEEEE  RRRRR            *
*          S    C       AAAAA     T        T     E       R  R             *
*          S    C       A   A     T        T     E       R   R            *
*     SSSSS      CCCCC  A   A     T        T     EEEEEE  R    R           *
*                                                                         *
**************************************************************************/

          /*********************************************
           *                                           *
           *   dalib_scatter1dK                         *
           *                                           *
           ********************************************/

void dalib_scatter1dK (target, source, size, stride1, n1)
TYPE *target, *source;
int size, stride1, n1;

{ int i, j, ofs;

#ifdef DEBUG
  printf ("%d: dalib_scatter1dK, size = %d, str1 = %d, n1 = %d\n",
          pcb.i, size, stride1, n1);
#endif

  if (size == 1)
   { for (i=n1; i--; target += stride1)
        *target = *source++;
   }
  else
   { for (i=n1; i--; target += stride1)
       for (j=0; j<size; j++)
          target[j] = *source++;
   }


} /* dalib_scatter1dK */

          /*********************************************
           *                                           *
           *   dalib_scatter2dK                        *
           *                                           *
           ********************************************/

void dalib_scatter2dK (target, source, size, stride1, n1, stride2, n2)
TYPE *target, *source;
int size, stride1, n1, stride2, n2;

{ int i1, i2, j, ofs;

#ifdef DEBUG
  printf (
   "%d: dalib_scatter2dK, size = %d, str1 = %d, n1 = %d, str2 = %d, n2 = %d\n",
        pcb.i, size, stride1, n1, stride2, n2);
#endif

  if (size == 1)
   { for (i2=n2; i2--; target += stride2 - n1*stride1)
        for (i1=n1; i1--; target += stride1)
           *target = *source++;
   }
  else
   { for (i2=n2; i2--; target += stride2 - n1*stride1)
        for (i1=n1; i1--; target += stride1)
           for (j=0; j<size; j++)
              target[j] = *source++;
   }

} /* dalib_scatter2dK */

          /*********************************************
           *                                           *
           *   dalib_scatter3dK                        *
           *                                           *
           ********************************************/

void dalib_scatter3dK (target, source, size, stride1, n1, stride2, n2,
                                             stride3, n3              )
TYPE *target, *source;
int size, stride1, n1, stride2, n2, stride3, n3;

{ int i1, i2, i3, j, ofs;

#ifdef DEBUG
  printf (
   "%d: dalib_scatter3dK, size = %d, str-n : %d-%d, %d-%d, %d-%d\n",
        pcb.i, size, stride1, n1, stride2, n2, stride3, n3);
#endif

  if (size == 1)
   { for (i3=n3; i3--; target += stride3 - n2*stride2)
        for (i2=n2; i2--; target += stride2 - n1*stride1)
           for (i1=n1; i1--; target += stride1)
              *target = *source++;
   }
  else
   { for (i3=n3; i3--; target += stride3 - n2*stride2)
        for (i2=n2; i2--; target += stride2 - n1*stride1)
           for (i1=n1; i1--; target += stride1)
              for (j=0; j<size; j++)
                 target[j] = *source++;
   }


} /* dalib_scatter3dK */

          /*********************************************
           *                                           *
           *   dalib_scatter4dK                        *
           *                                           *
           ********************************************/

void dalib_scatter4dK (target, source, size, stride1, n1, stride2, n2,
                                             stride3, n3, stride4, n4 )

TYPE *target, *source;
int size, stride1, n1, stride2, n2, stride3, n3, stride4, n4;

{ int i1, i2, i3, i4, j, ofs;

#ifdef DEBUG
  printf (
   "%d: dalib_scatter4dK, size = %d, str-n : %d-%d, %d-%d, %d-%d\n",
        pcb.i, size, stride1, n1, stride2, n2, stride3, n3, stride4, n4);
#endif

  if (size == 1)
   { for (i4=n4; i4--; target += stride4 - n3*stride3)
        for (i3=n3; i3--; target += stride3 - n2*stride2)
           for (i2=n2; i2--; target += stride2 - n1*stride1)
              for (i1=n1; i1--; target += stride1)
                 *target = *source++;
   }
  else
   { for (i4=n4; i4--; target += stride4 - n3*stride3)
        for (i3=n3; i3--; target += stride3 - n2*stride2)
           for (i2=n2; i2--; target += stride2 - n1*stride1)
              for (i1=n1; i1--; target += stride1)
                 for (j=0; j<size; j++)
                    target[j] = *source++;
   }

} /* dalib_scatter4dK */

          /*********************************************
           *                                           *
           *   dalib_scatter5dK                        *
           *                                           *
           ********************************************/

void dalib_scatter5dK (target, source, size, stride1, n1, stride2, n2,
                                stride3, n3, stride4, n4, stride5, n5 )

TYPE *target, *source;
int size, stride1, n1, stride2, n2, stride3, n3, stride4, n4, stride5, n5;

{ int i1, i2, i3, i4, i5, j, ofs;

#ifdef DEBUG
  printf (
   "%d: dalib_scatter5dK, size = %d, str-n : %d-%d, %d-%d, %d-%d, %d-%d\n",
        pcb.i, size, stride1, n1, stride2, n2, stride3, n3, stride4, n4,
        stride5, n5);
#endif

  if (size == 1)
   { for (i5=n5; i5--; target += stride5 - n4*stride4)
        for (i4=n4; i4--; target += stride4 - n3*stride3)
           for (i3=n3; i3--; target += stride3 - n2*stride2)
              for (i2=n2; i2--; target += stride2 - n1*stride1)
                 for (i1=n1; i1--; target += stride1)
                    *target = *source++;
   }
  else
   { for (i5=n5; i5--; target += stride5 - n4*stride4)
        for (i4=n4; i4--; target += stride4 - n3*stride3)
           for (i3=n3; i3--; target += stride3 - n2*stride2)
              for (i2=n2; i2--; target += stride2 - n1*stride1)
                 for (i1=n1; i1--; target += stride1)
                    for (j=0; j<size; j++)
                       target[j] = *source++;
   }

} /* dalib_scatter5dK */

          /*********************************************
           *                                           *
           *   dalib_scatter6dK                        *
           *                                           *
           ********************************************/

void dalib_scatter6dK (target, source, size, stride1, n1, stride2, n2,
                   stride3, n3, stride4, n4, stride5, n5, stride6, n6 )

TYPE *target, *source;
int size, stride1, n1, stride2, n2, stride3, n3, stride4, n4, 
                                    stride5, n5, stride6, n6;

{ int i1, i2, i3, i4, i5, i6, j, ofs;

#ifdef DEBUG
  printf (
   "%d: dalib_scatter6dK, size = %d, str-n : %d-%d, %d-%d, %d-%d, %d-%d, %d-%d\n",
        pcb.i, size, stride1, n1, stride2, n2, stride3, n3, stride4, n4,
        stride5, n5, stride6, n6);
#endif

  if (size == 1)
   { for (i6=n6; i6--; target += stride6 - n5*stride5)
        for (i5=n5; i5--; target += stride5 - n4*stride4)
           for (i4=n4; i4--; target += stride4 - n3*stride3)
              for (i3=n3; i3--; target += stride3 - n2*stride2)
                 for (i2=n2; i2--; target += stride2 - n1*stride1)
                    for (i1=n1; i1--; target += stride1)
                       *target = *source++;
   }
  else
   { for (i6=n6; i6--; target += stride6 - n5*stride5)
        for (i5=n5; i5--; target += stride5 - n4*stride4)
           for (i4=n4; i4--; target += stride4 - n3*stride3)
              for (i3=n3; i3--; target += stride3 - n2*stride2)
                 for (i2=n2; i2--; target += stride2 - n1*stride1)
                    for (i1=n1; i1--; target += stride1)
                       for (j=0; j<size; j++)
                          target[j] = *source++;
   }

} /* dalib_scatter6dK */

          /*********************************************
           *                                           *
           *   dalib_scatter7dK                         *
           *                                           *
           ********************************************/

void dalib_scatter7dK (target, source, size, stride1, n1, stride2, n2,
                                             stride3, n3, stride4, n4, 
                                stride5, n5, stride6, n6, stride7, n7)

TYPE *target, *source;
int size, stride1, n1, stride2, n2, stride3, n3, stride4, n4, 
                       stride5, n5, stride6, n6, stride7, n7;

{ int i1, i2, i3, i4, i5, i6, i7, j, ofs;


#ifdef DEBUG
  printf (
   "%d: dalib_scatter7dK, size = %d, str - n : %d:%d %d:%d %d:%d %d:%d %d:%d %d:%d\n",
        pcb.i, size, stride1, n1, stride2, n2, stride3, n3, stride4, n4,
                     stride5, n5, stride6, n6, stride7, n7);
#endif

  if (size == 1)
   { for (i7=n7; i7--; target += stride7 - n6*stride6)
        for (i6=n6; i6--; target += stride6 - n5*stride5)
           for (i5=n5; i5--; target += stride5 - n4*stride4)
             for (i4=n4; i4--; target += stride4 - n3*stride3)
                for (i3=n3; i3--; target += stride3 - n2*stride2)
                   for (i2=n2; i2--; target += stride2 - n1*stride1)
                     for (i1=n1; i1--; target += stride1)
                        *target = *source++;
   }
  else
   { for (i7=n7; i7--; target += stride7 - n6*stride6)
        for (i6=n6; i6--; target += stride6 - n5*stride5)
           for (i5=n5; i5--; target += stride5 - n4*stride4)
             for (i4=n4; i4--; target += stride4 - n3*stride3)
                for (i3=n3; i3--; target += stride3 - n2*stride2)
                   for (i2=n2; i2--; target += stride2 - n1*stride1)
                      for (i1=n1; i1--; target += stride1)
                         for (j=0; j<size; j++)
                            target[j] = *source++;
   }

} /* dalib_scatter7dK */

/**************************************************************************
*                                                                         *
*             M     M    OOOO   V     V   EEEEEE                          *
*             MM   MM   O    O   V   V    E                               *
*             M M M M   O    O   V   V    E                               *
*             M  M  M   O    O    V V     EEEEEE                          *
*             M     M   O    O    V V     E                               *
*             M     M   O    O     V      E                               *
*             M     M    OOOO      V      EEEEEE                          *
*                                                                         *
**************************************************************************/

          /*********************************************
           *                                           *
           *   dalib_move1dK                           *
           *                                           *
           ********************************************/

void dalib_move1dK (target, source, size, stride1, n1)
TYPE *target, *source;
int size, stride1, n1;

{ int i, j;

#ifdef DEBUG
  printf (
   "%d: dalib_move1dK, size = %d,  inc/n = %d/%d \n",
        pcb.i, size, stride1, n1);
#endif

  if (size == 1)
   { for (i=n1; i--; target += stride1, source += stride1)
        *target = *source;
   }
  else
   { for (i=n1; i--; target += stride1, source += stride1)
       for (j=0; j<size; j++)
          target[j] = source[j];
   }

} /* dalib_move1dK */

          /*********************************************
           *                                           *
           *   dalib_move2dK                           *
           *                                           *
           ********************************************/

void dalib_move2dK (target, source, size, stride1, n1, stride2, n2)
TYPE *target, *source;
int size, stride1, n1, stride2, n2;

{ int i1, i2, j;

#ifdef DEBUG
  printf (
   "%d: dalib_move2dK, size = %d,  inc/n = %d/%d %d/%d\n",
        pcb.i, size, stride1, n1, stride2, n2);
#endif

  if (size == 1)
   { for (i2=n2; i2--; target += stride2 - n1*stride1,
                       source += stride2 - n1*stride1)
        for (i1=n1; i1--; target += stride1, source += stride1)
           *target = *source;
   }
  else
   { for (i2=n2; i2--; target += stride2 - n1*stride1,
                       source += stride2 - n1*stride1)
        for (i1=n1; i1--; target += stride1, source += stride1)
          for (j=0; j<size; j++)
             target[j] = source[j];
   }

} /* dalib_move2dK */

          /*********************************************
           *                                           *
           *   dalib_move3dK                           *
           *                                           *
           ********************************************/

void dalib_move3dK (target, source, size, stride1, n1, stride2, n2, stride3, n3)
TYPE *target, *source;
int size, stride1, n1, stride2, n2, stride3, n3;
{ int i1, i2, i3, j;

#ifdef DEBUG
  printf (
   "%d: dalib_move3dK, size = %d,  inc/n = %d/%d %d/%d %d/%d\n",
        pcb.i, size, stride1, n1, stride2, n2, stride3, n3);
#endif

  if (size == 1)
   { for (i3=n3; i3--; target += stride3 - n2*stride2,
                       source += stride3 - n2*stride2)
        for (i2=n2; i2--; target += stride2 - n1*stride1,
                          source += stride2 - n1*stride1)
           for (i1=n1; i1--; target += stride1, source += stride1)
              *target = *source;
   }
  else
   { for (i3=n3; i3--; target += stride3 - n2*stride2,
                       source += stride3 - n2*stride2)
        for (i2=n2; i2--; target += stride2 - n1*stride1,
                          source += stride2 - n1*stride1)
           for (i1=n1; i1--; target += stride1, source += stride1)
             for (j=0; j<size; j++)
                target[j] = source[j];
   }

} /* dalib_move3dK */
