/***************************************************************************
 *                                                                         *
 *  Author      : Thomas Brandes GMD, SCAI.LAB                             *
 *  Copyright   : Public Domain                                            *
 *  Date        : April 93                                                 *
 *  Last Update : October 93                                               *
 *                                                                         *
 *  This Module is part of the DALIB                                       *
 *                                                                         *
 *  Module      : memcopy.c                                                *
 *                                                                         *
 *  Function    : copy of sections                                         *
 *                                                                         *
 *  Export :    ONLY INTERNAL USE IN DALIB                                 *
 *                                                                         *
 *   void dalib_memcopy (target, source, size)                             *
 *                                                                         *
 *   void dalib_move1d (target, source, size, stride1, n1)                 *
 *   void dalib_move2d (target, source, size, stride1, n1, stride2, n2)    *
 *                                                                         *
 *   ....                                                                  *
 *                                                                         *
 *   unsigned char *target, *source;                                       *
 *   int size, stride1, n1, ...., stridek, nk                              *
 *                                                                         *
 **************************************************************************/

# undef DEBUG

#include <stdio.h>
#include "dalib.h"

/**************************************************************************
*                                                                         *
*  dalib_memcopy (target, source, size)                                   *
*                                                                         *
*    -> dalib_memcopy8                                                    *
*    -> dalib_memcopy4                                                    *
*    -> dalib_memcopy1                                                    *
*                                                                         *
**************************************************************************/

          /*********************************************
           *                                           *
           *   dalib_memcopy                           *
           *                                           *
           ********************************************/

void dalib_memcopy (target, source, size)
unsigned char * target, * source;
int size;
 
{   int code = ((int) target) | ((int) source) | size ;

    if ((code & 0x07) == 0)

    {  /* Optimise cases which are multiples of doubles, double aligned */

       dalib_memcopy8 (target, source, size >> 3);
    }

    else if ((code & 0x03) == 0)

    {  /* Optimise cases which are multiples of ints, int aligned */

       dalib_memcopy4 (target, source, size >> 2);
    }

    else

       dalib_memcopy1 (target, source, size);

} /* dalib_memcopy */

/**************************************************************************
*                                                                         *
*             M     M    OOOO   V     V   EEEEEE                          *
*             MM   MM   O    O   V   V    E                               *
*             M M M M   O    O   V   V    E                               *
*             M  M  M   O    O    V V     EEEEEE                          *
*             M     M   O    O    V V     E                               *
*             M     M   O    O     V      E                               *
*             M     M    OOOO      V      EEEEEE                          *
*                                                                         *
**************************************************************************/

/**************************************************************************
*                                                                         *
*  dalib_move1d (target, source, size, stride1, n1)                       *
*                                                                         *
*    -> dalib_move1d8                                                     *
*    -> dalib_move1d4                                                     *
*    -> dalib_move1d1                                                     *
*                                                                         *
**************************************************************************/

void dalib_move1d (target, source, size, stride1, n1)
unsigned char * target, * source;
int size, stride1, n1;
 
{   int code = ((int) target) | ((int) source) | size | stride1 ;

#ifdef DEBUG
  printf (
   "%d: dalib_move1d, %d <- %d, size = %d, inc/n = %d/%d\n",
        pcb.i, target, source, size, stride1, n1);
#endif

    if ((code & 0x07) == 0)

    {  /* Optimise cases which are multiples of doubles, double aligned */

       dalib_move1d8 (target, source, size >> 3, stride1 >> 3, n1);
    }

    else if ((code & 0x03) == 0)

    {  /* Optimise cases which are multiples of ints, int aligned */

       dalib_move1d4 (target, source, size >> 2, stride1 >> 2, n1);
    }

    else

       dalib_move1d1 (target, source, size, stride1, n1);

} /* dalib_move1d */

/**************************************************************************
*                                                                         *
*  dalib_move2d (target, source, size, stride1, n1, stride2, n2)          *
*                                                                         *
*    -> dalib_move2d8                                                     *
*    -> dalib_move2d4                                                     *
*    -> dalib_move2d1                                                     *
*                                                                         *
**************************************************************************/

void dalib_move2d (target, source, size, stride1, n1, stride2, n2)
unsigned char *target, *source;
int size, stride1, n1, stride2, n2;

{   int code;

#ifdef DEBUG
  printf (
   "%d: dalib_move2d, %d <- %d, size = %d, inc/n = %d/%d, %d/%d\n",
        pcb.i, target, source, size, stride1, n1, stride2, n2);
#endif
 
    if (stride1*n1 == stride2)
       { dalib_move1d (target, source, size, stride1, n1*n2);
         return;
       }

    code = ((int) target) | ((int) source) | size | stride1 | stride2 ;

    if ((code & 0x07) == 0)

    {  /* Optimise cases which are multiples of doubles, double aligned */

       dalib_move2d8 (target, source, size >> 3, stride1 >> 3, n1,
                                                 stride2 >> 3, n2);
    }

    else if ((code & 0x03) == 0)

    {  /* Optimise cases which are multiples of ints, int aligned */

       dalib_move2d4 (target, source, size >> 2, stride1 >> 2, n1,
                                                 stride2 >> 2, n2);
    }

    else

       dalib_move2d1 (target, source, size, stride1, n1, stride2, n2);

} /* dalib_move2d */

/**************************************************************************
*                                                                         *
*  dalib_move3d (target, source, size, stride1, n1, ..., stride3, n3)     *
*                                                                         *
*    -> dalib_move3d8                                                     *
*    -> dalib_move3d4                                                     *
*    -> dalib_move3d1                                                     *
*                                                                         *
**************************************************************************/

void dalib_move3d (target, source, size, stride1, n1, stride2, n2,
                                                      stride3, n3)

unsigned char *target, *source;
int size, stride1, n1, stride2, n2, stride3, n3;

{   int code;
 
#ifdef DEBUG
  printf (
   "%d: dalib_move3d, size = %d, str-n : %d-%d %d-%d %d-%d\n",
        pcb.i, size, stride1, n1, stride2, n2, stride3, n3);
#endif

    if (stride2*n2 == stride3)
       { dalib_move2d (target, source, size, stride1, n1, stride2, n2*n3);
         return;
       }

    if (stride1*n1 == stride2)
       { dalib_move2d (target, source, size, stride1, n1*n2, stride3, n3);
         return;
       }

    code = ((int) target) | ((int) source) | size | stride1 | stride2 
                                           | stride3 ;

    if ((code & 0x07) == 0)

    {  /* Optimise cases which are multiples of doubles, double aligned */

       dalib_move3d8 (target, source, size >> 3, stride1 >> 3, n1,
                                  stride2 >> 3, n2, stride3 >> 3, n3);
    }

    else if ((code & 0x03) == 0)

    {  /* Optimise cases which are multiples of ints, int aligned */

       dalib_move3d4 (target, source, size >> 2, stride1 >> 2, n1,
                                  stride2 >> 2, n2, stride3 >> 2, n3);
    }

    else

       dalib_move3d1 (target, source, size, stride1, n1, stride2, n2,
                                               stride3, n3              );

} /* dalib_move3d */

