/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.WR                         *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Jul 97                                                   *
*  Last Update : Aug 98                                                   *
*                                                                         *
*  This Module is part of the UNILIB / DALIB                              *
*                                                                         *
*  Module      : match                                                    *
*                                                                         *
*  Function    : comparison of descriptors                                *
*                                                                         *
*                used for matchin actual descriptors with the dummies     *
*                                                                         *
*  Changes:                                                               *
*                                                                         *
**************************************************************************/

#undef DEBUG

#include "dalib.h"

static char *common_str_ptr;
static int  common_str_len;

/**************************************************************************
*                                                                         *
*  int dalib_match_global_shape (array_info dsp1, array_info dsp2)        *
*                                                                         *
*     0  : dsp1 and dsp2 do not match at all                              *
*     1  : same shape and same lower bounds                               *
*     2  : same shape, but different dimensions                           *
*                                                                         *
**************************************************************************/
 
int dalib_match_global_shape (dsp1, dsp2)

array_info dsp1, dsp2;

{ DimInfo *dim1, *dim2;
  int i, rank, size;

  int result;

#ifdef DEBUG
  printf ("%d: dalib_match_global_shape of %p (rank=%d) and %p (rank=%d)\n", 
           pcb.i, dsp1, dsp1->rank, dsp2, dsp2->rank);
#endif

  /* Step 1 : verification of the same rank  */

  rank = dsp1->rank;
  if (rank != dsp2->rank) return (0);

  size = dsp1->size;
  if (size != dsp2->size) return (0);

  dim1 = dsp1->dimensions;
  dim2 = dsp2->dimensions;

  result = 1;  /* default : same boundaries in all dimensions */

  for (i=0; i<rank; i++, dim1++, dim2++)   /* comparison of every dimension */

       { int lb1, ub1, lb2, ub2;

         lb1 = dim1->global_size[0];
         ub1 = dim1->global_size[1];
         lb2 = dim2->global_size[0];
         ub2 = dim2->global_size[1];

         if ((ub1 - lb1) != (ub2 - lb2))

            { result = 0; break; }

         if (lb1 != lb2) result = 2;

       }

  return (result);

} /* dalib_match_global_shape */

/**************************************************************************
*                                                                         *
*  int dalib_match_local_shape (array_info dsp1, array_info dsp2)         *
*                                                                         *
*     0  : dsp1 and dsp2 do not match at all                              *
*     1  : same local shape                                               *
*                                                                         *
*  Note: this routine is only called if arrays have same global shape     *
*                                                                         *
**************************************************************************/
 
int dalib_match_local_shape (dsp1, dsp2)

array_info dsp1, dsp2;

{ DimInfo *dim1, *dim2;
  int rank;
  int i, result;

#ifdef DEBUG
  printf ("%d: dalib_match_local_shape of %d (rank=%d) and %d (rank=%d)\n", 
           pcb.i, dsp1, dsp1->rank, dsp2, dsp2->rank);
#endif

  rank = dsp1->rank;
  if (rank != dsp2->rank) return (0);

  dim1 = dsp1->dimensions;
  dim2 = dsp2->dimensions;

  result = 1;  /* default : same local shape in all dimensions */

  for (i=0; i<rank; i++, dim1++, dim2++)   /* comparison of every dimension */

       { int ext1, ext2, str1, str2, ofs1, ofs2;

         ext1 = dim1->local_size[1] - dim1->local_size[0];
         ext2 = dim2->local_size[1] - dim2->local_size[0];

         if (ext1 != ext2) { result = 0; break; }

         str1 = dim1->local_size[2];
         str2 = dim2->local_size[2];

         if (str1 != str2) { result = 0; break; }

         /* offset between global and local size must be the same */

         ofs1 = dim1->global_size[0] - dim2->global_size[0];
         ofs2 = dim1->local_size [0] - dim2->local_size [0];

         if (ofs1 != ofs2) { result = 0; break; }

         /* if it is a mapped dimension, they must 
            have the same distribution                   */

         if (dim2->map_flag != dim1->map_flag) { result = 0; break; }
  
         /* both dimensions are mapped, so they should have same distrib. */

         if (dim1->map_flag) { result = 0; break; }

       }

#ifdef DEBUG
  printf ("%d: match local shape, result = %d\n", pcb.i, result);
#endif

  return (result);

} /* dalib_match_local_shape */

/**************************************************************************
*                                                                         *
*  int dalib_sufficient_shadow (array_info dsp1, array_info dsp2)         *
*                                                                         *
*    - returns 1 if shadow (dsp2) >= shadow (dsp1)                        *
*                                                                         *
*     1  : dsp2 has more or equal shadow than expected by dsp1            *
*     0  : dsp2 has less shadow than expected here                        *
*                                                                         *
**************************************************************************/
 
int dalib_sufficient_shadow (dsp1, dsp2)

array_info dsp1, dsp2;

{ DimInfo *dim1, *dim2;
  int i, rank;

  int result;

#ifdef DEBUG
  printf ("%d: dalib_sufficient_shadow of %p (rank=%d) and %p (rank=%d)\n", 
           pcb.i, dsp1, dsp1->rank, dsp2, dsp2->rank);
#endif

  /* Step 1 : verification of the same rank, otherwise problems  */

  rank = dsp1->rank;
  if (rank != dsp2->rank) return (0);

  dim1 = dsp1->dimensions;
  dim2 = dsp2->dimensions;

  result = 0;

  for (i=0; i<rank; i++, dim1++, dim2++)   /* comparison of every dimension */

    { if (dim1->shadow[0] > dim2->shadow[0]) return (0);
      if (dim1->shadow[1] > dim2->shadow[1]) return (0);
    }

  return (1);

} /* dalib_sufficient_shadow */

/**************************************************************************
*                                                                         *
*   int dalib_use_actual_data (dummy_id, actual_id)                       *
*                                                                         *
*    - returns true if data of actual can be used directly                *
*    - condition 1 : actual_id has same local shape as dummy_id           *
*    - condition 2 : if dummy_id is shared, actual_id must be shared too  *
*                                                                         *
**************************************************************************/
 
int dalib_use_actual_data (dummy_id, actual_id)

array_info dummy_id, actual_id;

{ /* Step 1: match the global shape (should be != 0 in all cases ?)  */

  if (!dalib_match_global_shape (dummy_id, actual_id)) return (0);

  /* Step 2: can never use actual data if dummy needs more shadow    */

  if (!dalib_sufficient_shadow (dummy_id, actual_id)) return (0);

  /* Step 3: compare the shared information                          */

  if (dummy_id->SharedInfo != NO_SHARED)

    { /* we cannot use the data of the actual if it was not shared */

      if (actual_id->SharedInfo == NO_SHARED)  return (0);

      /* for shared arrays we are ready now, we can use data directly */

      return (1);
    }

  /* really distributed arrays, so match the local shapes            */

  if (!dalib_match_local_shape (dummy_id, actual_id))

      return (0);

  /* if dummy needs remote access but array should be shared */

  if (dummy_id->RemoteInfo != NO_REMOTE)

     { if (actual_id->RemoteInfo == NO_REMOTE)
          return (0);
     }

  return (1);

}  /* dalib_use_actual_data */

/**************************************************************************
*                                                                         *
*   int dalib_is_dummy_data (array_id, dummy_id)                          *
*                                                                         *
*    - returns true if array_id has still same data as dummy              *
*                                                                         *
**************************************************************************/
 
int dalib_is_dummy_data (array_id, dummy_id)

array_info array_id, dummy_id;

{ if (array_id->f_data != dummy_id->f_data) return (0);  /* cannot be used */

  if (array_id->dsp_status_flag == DSP_OWN_DATA)

     return (0);  /* has new data allocated */

  return (1);   /* now we are sure that it is the same data */

} /* dalib_is_dummy_data */

/*******************************************************************
*                                                                  *
*  void dalib_get_actual_info (array_info dummy_dsp, actual_dsp)   *
*                                                                  *
*  - dummy inherits the data directly from the actual argument     *
*  - dummy inherits also the shadow size of the actual argument    *
*                                                                  *
*******************************************************************/

void dalib_get_actual_info (dummy_dsp, actual_dsp)

array_info dummy_dsp, actual_dsp;

{ DimInfo *local_dims, *dummy_dims;
  int i, rank;
 
  local_dims = dummy_dsp->dimensions;
  dummy_dims = actual_dsp->dimensions;
 
  rank = dummy_dsp->rank;   /* == actual_dsp->rank */

  /* inherit data pointer */

  dummy_dsp->f_data          = actual_dsp->f_data;
  dummy_dsp->dsp_status_flag = actual_dsp->dsp_status_flag;

  if (actual_dsp->dsp_status_flag == DSP_OWN_DATA)
    dummy_dsp->dsp_status_flag = DSP_PTR_DATA;  /* is default, but make sure */

  /* inherit size of shadow area */

  for (i=0; i<rank; i++, local_dims++, dummy_dims++)

     { local_dims->shadow[0] = dummy_dims->shadow[0];
       local_dims->shadow[1] = dummy_dims->shadow[1];
     }

  /* inherit the addressing attribute in case of static arrays */

  dummy_dsp->global_addressing = actual_dsp->global_addressing;

} /* dalib_get_actual_info */

/**************************************************************************
*                                                                         *
*  void dalib_copy_info (array_info dsp1, array_info dsp2)                *
*                                                                         *
*  - copy the valid info from descriptor dsp2 to dsp1                     *
*                                                                         *
**************************************************************************/

void dalib_copy_valid_info (dsp1, dsp2, destroy_flag)

array_info dsp1, dsp2;

int destroy_flag;

{ array_info a1, a2;

  if (dalib_is_section_info (dsp1))
     a1 = ((section_info) dsp1)->array_id;
    else
     a1 = dsp1;
    
  if (dalib_is_section_info (dsp2))
     a2 = ((section_info) dsp2)->array_id;
    else
     a2 = dsp2;
    
  if (a2->trace)

     { if (a1->trace)

         a1->reuse_bits = a2->reuse_bits;
   
        else if (destroy_flag)

         FUNCTION(dalib_array_set_dirty) (&a2);

     }

} /* dalib_copy_valid_info */

/*******************************************************************
*                                                                  *
*  void dalib_do_redistribution (array_info *new_dsp,              *
*                                array_info *old_dsp,              *
*                                char msg[])                       *
*                                                                  *
*   - do remapping of an array (old descriptor contains existing   *
*     information + pointer to data, new dsp specs new mapping)    *
*                                                                  *
*******************************************************************/

void dalib_do_redistribution (new_dsp, old_dsp, msg)

array_info *new_dsp, *old_dsp;
char msg[];

{ unsigned char *f_ptr;

  f_ptr = (*old_dsp)->f_data;

  dalib_array_full_allocate (*new_dsp, f_ptr);

  FUNCTION(dalib_assign) (new_dsp, old_dsp);

  if (pcb.redist_flag)

     { printf ("%d: remapping of %s - %s\n",
                pcb.i, (*new_dsp)->name, msg);
     } 

  dalib_free_descriptor (*old_dsp);

  *old_dsp = *new_dsp;

} /* dalib_do_redistribution */

/**********************************************************************
*                                                                     *
*  bool dalib_are_dynamic (array_info actual_id, array_info dummy_id) *
*                                                                     *
**********************************************************************/

int dalib_are_dynamic (actual_id, dummy_id)

array_info actual_id, dummy_id;

{ if (!dalib_is_array_info (actual_id)) return (0);

  if (!actual_id->dynamic) return (0);
  if (!dummy_id->dynamic) return (0);

  return (1);

} /* dalib_are_dynamic */
