/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*                                                                         *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Mar 98                                                   *
*  Last Update : Mar 98                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : linear.m4                                                *
*                                                                         *
***************************************************************************/

#include <stdio.h>
#include "dalib.h"

#undef DEBUG
#define CHECK

/**************************************************************************
*                                                                         *
*  void dalib_linear1_relpos (int pos, int rank, int shape[rank],         *
*                             int relpos[rank]                   )        *
*                                                                         *
*            1        2        3         rank = 2,  shape = [2, 3]        *
*          |----------------------|                                       *
*       1  | 1        3        5  |      1 ->  [1, 1]                     *
*          |                      |      2 ->  [2, 1]                     *
*       2  | 2        4        6  |      3 ->  [2, 1]                     *
*          ------------------------      6 ->  [2, 3]                     *
*                                                                         *
**************************************************************************/

void dalib_linear1_relpos (pos, rank, shape, relpos)

int pos;           /* 1 <= pos <= shape[0] * ... * shape[rank-1] */
int rank;          /* rank of the processor array                */
int shape[];       /* shape of the processor array               */
int relpos[];      /* out: relative position of pos in topology  */

{ int pos0;
  int dims[MAX_RANK];
  int i;
  int index;

  pos0 = pos - 1;

  /* now 0 <= pos0 < N = shape[0] * ... * shape[rank-1] */

  dims[0] = 1;

  for (i=1; i<rank; i++)
     dims [i] = dims[i-1] * shape[i-1];

  /* dims[0] = 1, dims[1] = n1, dims[2] = n1 * n2, ... */

  for (i = rank - 1; i >= 0; i--)

     { index     = pos0 / dims[i];
       relpos[i] = index + 1;
       pos0      -= index * dims[i];
     }

#ifdef DEBUG
  printf ("dalib_linear1_relpos computed, %d:%d:%d of %d:%d%:%d (rank=%d)\n",
          relpos[0], relpos[1], relpos[2], shape[0], shape[1], shape[2], rank);
#endif

} /* dalib_linear1_relpos */

/**************************************************************************
*                                                                         *
*  int dalib_linear1_abspos (int rank, int shape[], int relpos[])         *
*                                                                         *
*   - shape  = [n_1, n_2, ..., n_rank]                                    *
*   - relpos = [p_1, p_2, ..., p_rank],  1<=p_k<=n_k                      *
*                                                                         *
*   - returs linearized pos 1 <= pos <= n_1 * ... * n_rank                *
*                                                                         *
***************************************************************************/

int dalib_linear1_abspos (rank, shape, relpos)

int rank; 
int shape[];
int relpos [];

{ int i;        /* counter variable  */
  int pos;      /* used for position */
  int ofs;      /* used for offsets  */

  /* ofs = 1, ofs = n1, ofs = n1 * n2, ... */

  pos = 1;
  ofs = 1;

  for (i=0; i<rank; i++)  
    { pos += (relpos[i]-1) * ofs;
      ofs *= shape[i];
    }

  return pos;

} /* dalib_linear1_abspos */

int dalib_linear1_section (rank, lb, ub, is_range, r_lb, r_ub, r_str, 
                           new_rank, first, n, inc)

int rank;
int lb[];
int ub[];
int is_range[];
int r_lb[];
int r_ub[];
int r_str[];
int *new_rank;
int *first;
int n[];
int inc[];

{ int i;
  int sn;
  int dist;

  *new_rank = 0;
  *first    = 1;

  dist = 1;

  for (i=0; i<rank; i++)

    { sn = dalib_range_size (r_lb[i], r_ub[i], r_str[i]);

      *first += (r_lb[i]-lb[i]) * dist;

      if (is_range[i])

        { inc [*new_rank] = dist * r_str[i];
          n   [*new_rank] = sn;

          (*new_rank)++;
        }

       else

        {
        }

      if (lb[i] <= ub[i])
         dist *= (ub[i] - lb[i] + 1);
       else
         dist  = 0;
    }

} /* dalib_linear1_section */

int dalib_linear1_restrict (rank, shape, relpos, is_full,
                            new_rank, first, n, inc)

int rank;
int shape[];
int relpos[];
int is_full[];
int *new_rank;
int *first;
int n[];
int inc[];

{ int i;
  int sn;
  int dist;

  *new_rank = 0;
  *first    = 1;

  dist = 1;

  for (i=0; i<rank; i++)

    { if (is_full[i])

        { inc [*new_rank] = dist;
          n   [*new_rank] = shape[i];

          (*new_rank)++;
        }

       else

         *first += (relpos[i]-1) * dist;

       dist *= shape[i];
    }

} /* dalib_linear1_restrict */

/***************************************************************************
*                                                                          *
*  void dalib_linear_compress (int rank, int n[], int inc[],               *
*                              int *new_rank, int n1[], int inc1[])        *
*                                                                          *
*   elems = inc[0]*i0 + ... + inc[k]*ik     k = rank-1                     *
*                                                                          *
*            0 <= ij < n[j]                                                *
*                                                                          *
***************************************************************************/

void dalib_linear_compress (rank, n, inc, rank1, n1, inc1)

int rank;
int n[];
int inc[];
int *rank1;
int n1[];
int inc1[];

{ int i;
  int new_rank;

  new_rank = 0;

  for (i=0; i<rank; i++)

    { int doit = 1;

      if (n[i] == 1) doit = 0;

      if (doit && (new_rank > 0))

         { if (n1[new_rank] * inc1[new_rank] == inc[i])

              { n1[new_rank] *= n[i]; doit = 0; }
         }

      if (doit)

         { n1[new_rank] = n[i]; inc1[new_rank] = inc[i]; new_rank++; }

    } /* for i */

#ifdef DEBUG
    printf ("%d: compressed (", pcb.i);
    for (i=0; i<rank; i++) printf ("n=%d,inc=%d ",n[i],inc[i]);
    printf (") to (");
    for (i=0; i<new_rank; i++) printf ("n=%d,inc=%d ",n1[i],inc1[i]);
    printf (")\n");
#endif

  *rank1 = new_rank;

} /* dalib_linear_compress */

/**************************************************************************
*                                                                         *
*  void dalib_linear_set (int first, int rank, int n[], int inc[],        *
*                        int *size, int vals[])                           *
*                                                                         *
*  - first, n[0], ..., n[rank-1], inc[0], ..., inc[rank-1]                *
*    specifies processor set first + k0*inc[0] +  ...                     *
*  - lists the size = n[0] * ... * n[rank-1] processors in vals           *
*                                                                         *
**************************************************************************/

void dalib_linear_set (first, rank, n, inc, size, vals)

int first, rank;
int n[], inc[];
int *size;
int vals[];

{ int i0, i1, i2, i3;
  int N;

  N = 0;

  if ((rank < 1) || (rank > 4))

    { dalib_internal_error ("dalib_linear_set, rank error");
      dalib_stop ();
    }

  if (rank == 1)

    { for (i1=0; i1<n[1]; i1++)
        vals[N++] = first + i0 * inc[0];
    }

  if (rank == 2)

    { for (i1=0; i1<n[1]; i1++)
        for (i0=0; i0<n[0]; i0++)
           vals[N++] = first + i0 * inc[0] + i1 * inc[1];
    }

  if (rank == 3)

    { for (i2=0; i2<n[2]; i2++)
       for (i1=0; i1<n[1]; i1++)
         for (i0=0; i0<n[0]; i0++)
           vals[N++] = first + i0 * inc[0] + i1 * inc[1] + i2 * inc[2];
    }

  if (rank == 4)

    { for (i3=0; i3<n[3]; i3++)
       for (i2=0; i2<n[2]; i2++)
        for (i1=0; i1<n[1]; i1++)
         for (i0=0; i0<n[0]; i0++)
          vals[N++] = first + i0 * inc[0] + i1 * inc[1]
                             + i2 * inc[2] + i3 * inc[3];
    }

  *size = N;

} /* dalib_linear_set */

