/**************************************************************************
*                                                                         *
*  Author      : Thomas Brandes, GMD, SCAI.LAB                            *
*                Cecile Germain, Orsay, LRI                               *
*                                                                         *
*  Copyright   : LRI, Orsay, France                                       *
*  Date        : Mar 97                                                   *
*  Last Update : Sep 97                                                   *
*                                                                         *
*  Module      : insp_build.m4                                            *
*                                                                         *
*                                                                         *
*  Changes/Updates                                                        *
*  ===============                                                        *
*                                                                         *
*   09/97   :  builds also communication schedules for shadow updates     *
*                                                                         *
**************************************************************************/

#include "dalib.h"
#include "inspector.h"


#undef DEBUG
#define CHECK

  /* REMOTE : global offsets become local offsets before asking
              for non-local data                                 */

#undef REMOTE   

/*******************************************************************
*                                                                  *
*   void dalib_dummy_align (new_dsp, array_dsp)                    *
*                                                                  *
*    !hpf$ align NEW (I) with ARRAY (*,*,*,...,I)                  *
*                                                                  *
*******************************************************************/

static void dalib_dummy_align (new_dsp, array_dsp)

array_info *new_dsp, *array_dsp;

{ int rank;

  int type, is_rep;
  int sdim;
  int base;
  int stride;
  
  rank = (*array_dsp)->rank;

  /* !HPF$ ALIGN ARRAY1(I) WITH ARRAY(*,*,...,I) */

  type   = kALIGNED_DIM;
  sdim   = rank;
  base   = 0;
  stride = 1;

  FUNCTION(dalib_align_source) (new_dsp, array_dsp,
                                &type, &sdim, &base, &stride);

  type   = kSOURCE_DIM;
  sdim   = 1;

  is_rep  = kREPLICATED_DIM;

  switch (rank) {

  case 1 : FUNCTION(dalib_align_target) (new_dsp, array_dsp, &type, &sdim);
           break;

  case 2 : FUNCTION(dalib_align_target) (new_dsp, array_dsp,
                                         &is_rep, &sdim, &type, &sdim);
           break;

  case 3 : FUNCTION(dalib_align_target) (new_dsp, array_dsp, &is_rep, &sdim,
                                         &is_rep, &sdim, &type, &sdim);
           break;

  case 4 : FUNCTION(dalib_align_target) (new_dsp, array_dsp,
                                         &is_rep, &sdim, &is_rep, &sdim,
                                         &is_rep, &sdim, &type, &sdim);
           break;

  case 5 : FUNCTION(dalib_align_target) (new_dsp, array_dsp, &is_rep, &sdim,
                                         &is_rep, &sdim, &is_rep, &sdim,
                                         &is_rep, &sdim, &type, &sdim);
           break;

  case 6 : FUNCTION(dalib_align_target) (new_dsp, array_dsp,
                                         &is_rep, &sdim, &is_rep, &sdim,
                                         &is_rep, &sdim, &is_rep, &sdim,
                                         &is_rep, &sdim, &type, &sdim);
           break;

  case 7 : FUNCTION(dalib_align_target) (new_dsp, array_dsp, &is_rep, &sdim,
                                         &is_rep, &sdim, &is_rep, &sdim,
                                         &is_rep, &sdim, &is_rep, &sdim,
                                         &is_rep, &sdim, &type, &sdim);
           break;

  } /* switch */

} /* dalib_dummy_align */

/*******************************************************************
*                                                                  *
*  array_info dalib_dummy_template (array_info array_dsp)          *
*                                                                  *
*    - creates one-dimensional template distributed like array     *
*      along the last dimension                                    *
*                                                                  *
*******************************************************************/

array_info dalib_dummy_template (array_dsp)

array_info array_dsp;

{ DimInfo *dim, *dim1;
  array_info temp_dsp, dalib_array_new_dsp ();
  int rank;

  rank = array_dsp->rank;

  if (rank == 1) return (array_dsp);

  temp_dsp = dalib_array_new_dsp (1, 0, kIS_DYNAMIC);  /* rank = 1, size = 0 */

  dim1 = temp_dsp->dimensions;             /* first dimension */
  dim  = array_dsp->dimensions + rank -1;  /* last  dimension */

  dim1->global_size[0] = dim->global_size[0];
  dim1->global_size[1] = dim->global_size[1];

  /* dummy template gets right overlap of array for size verification */

  dim1->shadow[1]     = dim->shadow[1];

  /* !HPF$ ALIGN ARRAY1(I) WITH ARRAY(*,*,...,I) */

  dalib_dummy_align (&temp_dsp, &array_dsp);

  /* IMPORTANT : now compute local sizes for addressing  */

  dalib_align_local_sizes (temp_dsp);

  return (temp_dsp);

} /* dalib_dummy_template */

/**********************************************************************
*                                                                     *
*  void dalib_make_base_dsp (array_info base_array,                   *
*                            base_dsp   *base_info)                   *
*                                                                     *
**********************************************************************/

void dalib_make_base_dsp (base_array, base_info)

array_info base_array;
insp_base_dsp   *base_info;

{ int i, rank;

  array_info template_id;
  int        top_id;

  int        top_dim, base, stride, lb, ub, kind;
  DistDim    mapping;

  rank = base_array->rank;

  base_info->rank = rank;

  dalib_array_info (base_array, &template_id, &top_id);

  base_info->top_id = top_id;

  for (i=0; i<rank; i++)

    { base_info->lb[i] = base_array->dimensions[i].global_size[0];
      base_info->ub[i] = base_array->dimensions[i].global_size[1];

      base_info->left_ov[i]  = base_array->dimensions[i].shadow[0];
      base_info->right_ov[i] = base_array->dimensions[i].shadow[1];

      dalib_array_dim_mapping (base_array, i+1, &base, &stride,
                               &lb, &ub, &top_id, &mapping);
 
      dalib_dim_mapping_info (mapping, &kind, &top_dim);

      base_info->t_base[i]    = base;
      base_info->t_stride[i]  = stride;
      base_info->t_kind[i]    = kind;
      base_info->t_lb[i]      = lb;
      base_info->t_ub[i]      = ub;

      base_info->top_dim[i]   = top_dim;

    }

} /* dalib_make_base_dsp */

/**********************************************************************
*                                                                     *
*                                                                     *
**********************************************************************/

void dalib_make_index_dsp (dsp, data, ind_dsp)

array_info dsp;
int        *data;

insp_index_dsp *ind_dsp;

{ int i, rank;

  if (dsp == (array_info) 0)

    { ind_dsp->rank = 0;
      ind_dsp->scalar_val = *data;
    }

   else if (dalib_is_array_info (dsp))

    { int i;

      ind_dsp->rank = dsp->rank;

      for (i=0; i < ind_dsp->rank; i++)

        { ind_dsp->section_bounds[i][0] = dsp->dimensions[i].global_size[0];
          ind_dsp->section_bounds[i][1] = dsp->dimensions[i].global_size[1];
          ind_dsp->section_bounds[i][2] = 1;
        }

      ind_dsp->data_ptr = (int *) dsp->f_data;

    }

   else if (dalib_is_section_info (dsp))
 
    { section_info sdsp;

      int i;

      sdsp = (section_info) dsp;
          
      ind_dsp->rank = sdsp->array_id->rank;

      for (i=0; i < ind_dsp->rank; i++)

        { ind_dsp->section_bounds[i][0] = sdsp->dimensions[i].global_range[0];
          ind_dsp->section_bounds[i][1] = sdsp->dimensions[i].global_range[1];
          ind_dsp->section_bounds[i][2] = sdsp->dimensions[i].global_range[2];
        }

      ind_dsp->data_ptr = (int *) sdsp->array_id->f_data;
    }


} /* dalib_make_index_dsp */

/**********************************************************************
*                                                                     *
*                                                                     *
**********************************************************************/

static int dalib_is_traced (dsp)

array_info dsp;

{ if (dsp == (array_info) 0) return (1);

  if (dalib_is_array_info (dsp)) 

      return (dsp->trace);

  if (dalib_is_section_info (dsp))

      return (((section_info) dsp)->array_id->trace);

  return (0);       /* should not happen */

}  /* dalib_is_traced */

/**********************************************************************
*                                                                     *
*  void dalib_inspector_fill_data (inspector_data *I_Data,            *
*                                  inspector_info *I_Info)            *
*                                                                     *
*   in :  I_Info with actual descriptors                              *
*   out : I_Data with comparable inspector data                       *
*                                                                     *
*  Note : schedule_id is not computed here                            *
*                                                                     *
**********************************************************************/

void dalib_inspector_fill_data (I_Data, I_Info)

inspector_data *I_Data;
inspector_info *I_Info;

{ int idim, rank;

  I_Data->number_of_indexes = I_Info->number_of_indexes;

  dalib_make_base_dsp (I_Info->base_array, &(I_Data->base_array));

  I_Data->is_traced = 1;  /* assume tracing as default */

  I_Data->is_shadow = I_Info->is_shadow;

  if (I_Info->is_shadow)  rank = 1;
    else                  rank = I_Info->base_rank;

  for (idim=0; idim < rank; idim++)

   { dalib_make_index_dsp (I_Info->ind_array[idim], I_Info->ind_data[idim],
                           &(I_Data->ind_array[idim]));

     if (!dalib_is_traced (I_Info->ind_array[idim]))
     
        I_Data->is_traced = 0;

   }

  if (I_Info->is_masked)

      dalib_make_index_dsp (I_Info->mask_array, I_Info->mask_data,
                            &(I_Data->mask_array));

     else

      { I_Data->mask_array.rank = 0;
        I_Data->mask_array.scalar_val = 1;  /* everything is TRUE */
      }

  if (!dalib_is_traced (I_Info->mask_array))

     I_Data->is_traced = 0;

} /* dalib_inspector_fill_data */

/**********************************************************************
*                                                                     *
*  void dalib_inspector_build (inspector_data *I_Data,                *
*                              inspector_info *I_Info)                *
*                                                                     *
*    - computes the schedule (inspector phase)                        *
*    - schedule identification is put into inspector_data             *
*                                                                     *
*  Note: I_Info is needed to have access to actual descriptors        *
*                                                                     *
**********************************************************************/

void dalib_inspector_build (I_Data, I_Info)

inspector_data *I_Data;
inspector_info *I_Info;

{ int *indexes  [MAX_DIMENSIONS];  /* all indexes (will be contiguous)   */
  int index_new [MAX_DIMENSIONS];  /* memory must be freed               */

  int *mask_data;
  int mask_new;     /* memory for mask must be freed */

  int no, size;     /* dummy arguments               */

  int *owners;      /* array for ownership, dynamically allocated       */
  int *offsets;     /* array for local addresses, dynamically allocated */

  array_info template_dummy;

  array_info base_array;    /* array that is indirectly addressed */

  SchedulePtr schedule_id;

  /* get basic information out of inspector_info                 */

  no         = I_Info->number_of_indexes;
  base_array = I_Info->base_array;

  if (I_Info->is_shadow) 

       base_array = dalib_dummy_template (base_array);

  dalib_insp_info_get (I_Info, indexes, index_new, &mask_data, &mask_new);

  /* now calculate ownerships, local addresses */

  owners  = (int *) dalib_int_malloc (no, "indexes_owners");
  offsets = (int *) dalib_int_malloc (no, "indexes_local_offsets");

  /* Attention: calculation of addresses might have side effects
                on indexes (permutations), so they can be new       */

  dalib_find_owners (owners, no, base_array, mask_data,
                     indexes, index_new);

#ifdef REMOTE
  dalib_make_remote_offsets (offsets, no, base_array,
                              owners, mask_data, indexes);
#else
  dalib_make_global_offsets (offsets, no, base_array,
                              mask_data, indexes);
#endif

  /* build the schedule data */

  if (I_Info->is_shadow)

     { /******************************************
       *   compute shadow schedule               *
       ******************************************/

       int *index_changes;

       dalib_shadow_calc (&schedule_id, base_array,
                          indexes[0], no, owners, offsets, &index_changes);

       I_Data->index_changes   = index_changes;
       I_Data->index_temp_size = 0;

     }

   else

     { /******************************************
       *   compute normal schedule               *
       ******************************************/

       int source_topology;
       int base_topology;

       dalib_array_info (I_Info->source_array, 
                         &template_dummy, &source_topology);

       dalib_array_info (base_array, 
                         &template_dummy, &base_topology);

       dalib_new_indirect (&schedule_id, 
                           source_topology, base_topology, 
                           no, owners, offsets);

      }

       /******************************************
       *   localize global addresses             *
       *   (only when global addresses used)     *
       ******************************************/

#ifndef REMOTE
   dalib_indirect_localize_addresses (base_array, schedule_id);
#endif

  I_Data->schedule_id = schedule_id;

#ifdef DEBUG
  printf ("%d: dalib_inspector_build, new inspector data (s=%d) computed\n",
           pcb.i, schedule_id);
#endif

  dalib_int_free (offsets, no);
  dalib_int_free (owners, no);

  /* free new allocated memory for indexes and mask */

  dalib_insp_info_free (I_Info, indexes, index_new, mask_data, mask_new);

  /* free descriptor of dummy template if necessary */

  if (I_Info->base_array != base_array) dalib_free_descriptor (base_array);

} /* dalib_inspector_build */

