/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Dec 98                                                   *
*  Last Update : Dec 98                                                   *
*                                                                         *
*  This Module is part of the DALIB and UNILIB                            *
*                                                                         *
*  Module      : in_out                                                   *
*                                                                         *
*  Function    : copy in and copy out for dummy arguments                 *
*                                                                         *
**************************************************************************/

#undef DEBUG

#include "dalib.h"

/***************************************************************************
*                                                                          *
*  dalib_no_actual_dsp_copy_in (array_info dummy_dsp, char *actual_data)   *
*                                                                          *
*   -> actual argument has no descriptor                                   *
*                                                                          *
***************************************************************************/

static void dalib_no_actual_dsp_copy_in (dummy_dsp, actual_data)

array_info *dummy_dsp;
char       *actual_data;

{ /* Possibilty 1 : actual is optional argument that is not available */

   if (!FUNCTION(dalib_present) (actual_data))

      /* make sure that dummy_dsp-> dsp_status_flag == DSP_UNDEFINED */

      return;   /* dummy descriptor for actual will not be changed at all */


  /* Possiblity 2 : actual argument has been passed as a scalar pointer 
                    e.g. call SUB (...,A(K),...)                          */

  /* actual argument is a pointer */

  if (   ((*dummy_dsp)->DistributeInfo == NO_DISTRIBUTION)
      && ((*dummy_dsp)->AlignInfo      == NO_ALIGNMENT)    )

    { /* dummy will be descriptor for replicated data */
            
      FUNCTION(dalib_array_setdata) (dummy_dsp, actual_data);

      return;  /* that works fine */
    }

  dalib_internal_error ("copy_in : actual argument has no descriptor");
  dalib_stop ();

} /* dalib_no_actual_dsp_copy_in */

/***************************************************************************
*                                                                          *
*  FUNCTION(dalib_copy_in) (dummy_dsp, actual_dsp,                         *
*                           actual_data, copy_flag)                        *
*                                                                          *
*  FUNCTION(dalib_copy_out) (local_dsp, actual_dsp, copy_flag)             *
*                                                                          *
*  FORTRAN Interface                                                       *
*                                                                          *
*     subroutine SUB (A, N)                                                *
*     real A(N)                                                            *
*                                                                          *
*   ->                                                                     *
*                                                                          *
*     subroutine SUB (A, N, A_DDSP, N_DDSP)                                *
*     ...                                                                  *
*     call dalib_array_make_dsp (A_DSP, ...)                               *
*     ...                                                                  *
*     call dalib_copy_in (A_DSP, A_DDSP, A, 1)                             *
*     ...                                                                  *
*     call dalib_copy_out (A_DSP, A_DDSP, 1)                               *
*     end                                                                  *
*                                                                          *
*   IMPORTANT: A_DSP (dummy_dsp) will be the descriptor used within        *
*              the subprogram for all operations                           *
*                                                                          *
***************************************************************************/

void FUNCTION(dalib_copy_in) (dummy_dsp, actual_dsp, actual_data, copy_flag) 

array_info *dummy_dsp;
array_info *actual_dsp;
char       *actual_data;
int        *copy_flag;

{ if (!FUNCTION(dalib_present) (actual_dsp))

     dalib_no_actual_dsp_copy_in (dummy_dsp, actual_data);

  dalib_copy_valid_info (*dummy_dsp, *actual_dsp, 0);

  /* destroy flag is false as actual descriptor will not be destroyed */

  if (*dummy_dsp == *actual_dsp)

     return;                       /* actual descriptor is just dummy */

  /* for underspecified mappings we have to inherit the mapping and 
     to compute the local sizes of the dummy descriptor              */

  if (dalib_array_map_underspecified (*dummy_dsp))

     { int done;

       done = dalib_inherit_mapping (*dummy_dsp, *actual_dsp);

       if (done) return;
     }

  /* now dummy has in any case a full specified mapping */

  if (dalib_use_actual_data (*dummy_dsp, *actual_dsp))

     {  dalib_get_actual_info (*dummy_dsp, *actual_dsp);

#ifdef DEBUG
        printf ("%d: copy in, local = %d, dummy = %d, not necessary\n",
                 pcb.i, *dummy_dsp, *actual_dsp);
#endif

        /* assign might be necessary if other processors want us
           to be involved in their assignment                     */

        if (*copy_flag != 0)

           { if (dalib_has_replication (*dummy_dsp, *actual_dsp))

                 FUNCTION(dalib_assign) (dummy_dsp, actual_dsp); 
           }

        return;
     }

  /* new descriptor, so new data is needed, dsp_status_flag == OWN_DATA */

#ifdef DEBUG
  printf ("%d: copy in, local = %d, dummy = %d, new data necessary\n",
               pcb.i, *dummy_dsp, *actual_dsp);
#endif

  /* allocate new local data, aligned to the actual data */

  dalib_array_full_allocate (*dummy_dsp, actual_data);

  FUNCTION(dalib_array_set_dirty) (dummy_dsp);

  /* so copy in might be necessary */

  if (*copy_flag != 0)

     { char msg[150];

       sprintf (msg, "copy in redistribution for dummy %s\n", 
                     (*dummy_dsp)->name);

       if (pcb.redist_flag) dalib_attention (msg);

       FUNCTION(dalib_assign) (dummy_dsp, actual_dsp); 
     }

#ifdef DEBUG
  printf ("copy in finished\n");
#endif 

} /* dalib_copy_in */

/***************************************************************************
*                                                                          *
*  FUNCTION(dalib_copy_in) (dummy_dsp, actual_dsp,                         *
*                           actual_data, copy_flag)                        *
*                                                                          *
*   - own routine needed for STRING arguments                              *
*                                                                          *
***************************************************************************/

void FUNCTION(dalib_ccopy_in) 

  ARGS(`dummy_dsp, actual_dsp, STRING_ARG(actual_data), copy_flag') 

array_info *dummy_dsp;
array_info *actual_dsp;
STRING_ARG_DECL(actual_data);
int        *copy_flag;

{ FUNCTION(dalib_copy_in) (dummy_dsp, actual_dsp,
                           STRING_PTR(actual_data), copy_flag);
}

/*******************************************************************
*                                                                  *
*  FORTRAN Interface                                               *
*                                                                  *
*  FUNCTION(dalib_copy_out) (dummy_dsp, actual_dsp, copy_flag)      *
*                                                                  *
*    - copy local array back to dummy array                        *
*    - free the new descriptor                                     *
*                                                                  *
*******************************************************************/

void FUNCTION(dalib_copy_out) (dummy_dsp, actual_dsp, copy_flag) 

array_info *dummy_dsp;
array_info *actual_dsp;
int        *copy_flag;

{ if (!FUNCTION(dalib_present) (actual_dsp))

     { /* dummy is a pointer, checked within copy_in */

       return;
     }

  dalib_copy_valid_info (*actual_dsp, *dummy_dsp, 1);

  /* destroy flag is true as dummy descriptor will be freed afterwards */

  if (*dummy_dsp == *actual_dsp)
     return;                       /* local descriptor is just dummy */

  if (dalib_is_dummy_data (*dummy_dsp, *actual_dsp))

     { /* copy not necessary, same data, but delete descriptor */

#ifdef DEBUG
        printf ("%d: copy out, local = %p, dummy = %p, not necessary\n",
                 pcb.i, *dummy_dsp, *actual_dsp);
#endif

        if (*copy_flag != 0)
 
           { if (dalib_has_replication (*actual_dsp, *dummy_dsp))
 
                 FUNCTION(dalib_assign) (actual_dsp, dummy_dsp);
           }
 
       dalib_free_descriptor (*dummy_dsp);  

       return;

     }

  if (dalib_are_dynamic (*actual_dsp, *dummy_dsp))

     { /* acutal descpritor becomes dummy descriptor */

#ifdef DEBUG
        printf ("%d: copy out, local = %p, dummy = %p, dynamic out\n",
                 pcb.i, *dummy_dsp, *actual_dsp);
#endif

       /* actual descriptor is no longer needed */

       dalib_free_descriptor (*actual_dsp);

       *actual_dsp = *dummy_dsp;

       return;

     }

  /* different descriptors, so copy out be necessary */

#ifdef DEBUG
  printf ("%d: copy out, local = %d, dummy = %d, copy out necessary\n",
               pcb.i, *dummy_dsp, *actual_dsp);
#endif

  FUNCTION(dalib_array_set_dirty) (dummy_dsp);

  if (*copy_flag != 0)

     { char msg[150];

       sprintf (msg, "copy out redistribution for dummy %s\n", 
                     (*dummy_dsp)->name);

       if (pcb.redist_flag) dalib_attention (msg); 

       FUNCTION(dalib_assign) (actual_dsp, dummy_dsp);
     }

#ifdef DEBUG
   printf ("%d: copy out, local = %d, dummy = %d, done\n",
            pcb.i, *dummy_dsp, *actual_dsp);
#endif
 

  dalib_free_descriptor (*dummy_dsp);  /* do also delete data */

} /* dalib_copy_out */

