/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Jan 98                                                   *
*  Last Update : Nov 98                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : hpf_tasks.m4                                             *
*                                                                         *
*  Function    : HPF interaction routines for task parallelism            *
*                (HPF_TASK_LIBRARY)                                       *
*                                                                         *
*  Export :  FORTRAN Interface                                            *
*                                                                         *
*   void dalib_start_tasking ()                                           *
*                                                                         *
*     - increase task nesting and accept new task identifiers             *
*                                                                         *
*   void dalib_stop_tasking ()                                            *
*                                                                         *
*     - stop task environment                                             *
*                                                                         *
*  Export :  DALIB Interface                                              *
*                                                                         *
**************************************************************************/

#define CHECK
#undef  DEBUG

#include "dalib.h"

/**************************************************************************
*                                                                         *
*  CONSTANTS                                                              *
*                                                                         *
*    MAX_TASKS       : maximal nesting of tasking (my involvments)        *
*    MAX_TASK_GROUPS : maximal groups of tasks I am involved with         *
*    MAX_TASK_COMM   : maximal number of communication handles            *
*                                                                         *
**************************************************************************/

#define MAX_TASK_GROUPS  100
#define MAX_TASK_COMM    10

/**************************************************************************
*                                                                         *
*  GLOBAL VARIABLES for TASKING                                           *
*                                                                         *
*   Note: a task identifier corresponds to a group identifier             *
*                                                                         *
**************************************************************************/

typedef struct
 
   { int task_init;   /* 0 if not initialized                      */
     int task_size;   /* number of tasks, corresponds task_size    */
     int task_rank;   /* my own task_position                      */
     int task_group;  /* group of all processors in any task       */
     int task_local;  /* all tasks are single processors          
                         implies HPF_LOCAL                         */
     int *task_ids;   /* pointer into all_task_ids                 */

   } task_control_block;
   
static task_control_block task_stack [MAX_TASKS];

static int task_nest = 0;

static int all_task_ids [MAX_TASK_GROUPS];

static int task_offset = 0;  /* offset in task_ids for current task environ */

/**************************************************************************
*                                                                         *
*  some simple routines to handle the stack                               *
*                                                                         *
**************************************************************************/

     /***********************************************************
     *   task_control_block *new_task_entry ()                  *
     ***********************************************************/

static task_control_block *new_task_entry ()

{ task_control_block *tcb;

  if (task_nest >= MAX_TASKS)

     { char msg[128];

       sprintf (msg, "hpf_tasks : too deep task nesting (max = %d)",
                     MAX_TASKS);
       dalib_internal_error (msg);
       dalib_stop ();
     }

  tcb = task_stack + task_nest;

  task_nest++;   /* increments the nesting of tasks */

  return tcb;

} /* new_task_entry */

     /***********************************************************
     *   task_control_block *current_task_entry ()              *
     ***********************************************************/

static task_control_block *current_task_entry (check_init)

int check_init;

{ task_control_block *tcb;

  if (task_nest <= 0)

     { dalib_internal_error ("hpf_tasks : no tasking defined");
       dalib_stop ();
     }

  tcb = task_stack + task_nest - 1;

  if (check_init)

   { if (!tcb->task_init)

      { dalib_internal_error ("HPF_TASK_LIBRARY: not HPF_TASK_INIT called");
        dalib_stop ();
      }
   }
 
  return tcb;

}  /* current_task_entry */

/**************************************************************************
*                                                                         *
*  GLOBAL VARIABLES for COMMUNCIATION REQUESTS                            *
*                                                                         *
**************************************************************************/

static int task_request_top = 0;

typedef struct

   { int kind;
     array_info local_dsp, remote_dsp;

   } comm_request;

static comm_request task_request_stack [MAX_TASK_COMM];

/**************************************************************************
*                                                                         *
**************************************************************************/

static int dalib_new_request (kind, local_dsp, remote_dsp)

int kind;
array_info local_dsp, remote_dsp;

{ comm_request *comm_handle;

  if (task_request_top >= MAX_TASK_COMM)

     { dalib_internal_error ("too many requests for task communication");
       dalib_stop();
     }

  comm_handle = task_request_stack + task_request_top;
  task_request_top++;

  comm_handle->kind = kind;
  comm_handle->local_dsp  = local_dsp;
  comm_handle->remote_dsp = remote_dsp;

  return task_request_top;

} /* dalib_new_request */

static void dalib_get_request (request, kind, local_dsp, remote_dsp)

int request;
int *kind;
array_info *local_dsp, *remote_dsp;

{ comm_request *comm_handle;

  if ((request < 1) || (request > task_request_top))

    { char msg[100];

      sprintf (msg, "illegal task request %d, must be in 1 - %d\n",
                     request, task_request_top);
 
      dalib_internal_error (msg);
      dalib_stop ();
    }

  comm_handle = task_request_stack + request - 1;

  *kind       = comm_handle->kind;
  *local_dsp  = comm_handle->local_dsp;
  *remote_dsp = comm_handle->remote_dsp;

} /* dalib_get_request */

/**************************************************************************
*                                                                         *
*   void dalib_start_local_tasking (int group)                            *
*   void dalib_stop_local_tasking  ()                                     *
*                                                                         *
*     - enables/disables tasking for the specified group                  *
*                                                                         *
**************************************************************************/

     /***********************************************************
     *   dalib_start_local_tasking (int group)                  *
     ***********************************************************/

void dalib_start_local_tasking (group)

int group;

{ task_control_block *tcb;

  tcb = new_task_entry ();

  tcb->task_init  = 0;
  tcb->task_size  = dalib_group_size (group);
  tcb->task_rank  = dalib_group_position (group, pcb.i);
  tcb->task_group = group;
  tcb->task_local = 1;      /* every hpf_task is trivial now */

  tcb->task_ids   = (int *) 0;

#ifdef DEBUG
   printf ("%d: start_local_tasking, group = %d\n", pcb.i, group);
#endif 

} /* dalib_start_local_tasking */

     /***********************************************************
     *   dalib_stop_local_tasking ()                            *
     ***********************************************************/

void dalib_stop_local_tasking ()

{ task_control_block *tcb;

  tcb = current_task_entry (0);

  if (!tcb->task_local)

   { dalib_internal_error ("dalib_stop_local_tasking: local_tasking not set");
     dalib_stop ();
   }

#ifdef DEBUG
   printf ("%d: stop_local_tasking, group = %d\n", pcb.i, tcb->task_group);
#endif 

  task_nest --;

} /* dalib_stop_local_tasking */

/**************************************************************************
*                                                                         *
*   void FUNCTION(dalib_start_tasking)  ()                                *
*                                                                         *
*     - increase task nesting and accept new task identifiers             *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_start_tasking) ()

{ task_control_block *tcb;

  tcb = new_task_entry ();

  tcb->task_init  = 0;
  tcb->task_size  = 0;
  tcb->task_rank  = -1;
  tcb->task_group = dalib_context_group ();
  tcb->task_local = 0;                      /* not LOCAL environment  */

  tcb->task_ids   = all_task_ids + task_offset;

} /* dalib_start_tasking */

/**************************************************************************
*                                                                         *
*   void FUNCTION(dalib_stop_tasking)  ()                                 *
*                                                                         *
*     - stop task environment, decrease the stack                         *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_stop_tasking) ()

{ task_control_block *tcb;

  tcb = current_task_entry (0);

  if (tcb->task_local != 0)

   { dalib_internal_error ("dalib_stop_tasking: local_tasking was set");
     dalib_stop ();
   }

  task_nest--;
  task_offset -= tcb->task_size;

} /* dalib_stop_tasking */

/**************************************************************************
*                                                                         *
*  void dalib_new_task (int t_id)                                         *
*                                                                         *
*    - defines a new task executed by processors in group t_id            *
*                                                                         *
**************************************************************************/

static void dalib_new_task (t_id)

int t_id;

{ task_control_block *tcb;

  int i;
  char msg[128];

  tcb = current_task_entry (0);

  if (task_offset + tcb->task_size >= MAX_TASK_GROUPS)

     { sprintf (msg, "hpf_tasks : too many task (max = %d)", MAX_TASK_GROUPS);
       dalib_internal_error (msg);
       dalib_stop ();
     }

  for (i=0; i< tcb->task_size; i++)

   if (!dalib_group_distinct (t_id, tcb->task_ids[i]))

    { sprintf (msg, "Tasks %d (group=%d) and %d (group=%d) use same proc", 
               i+1, tcb->task_ids[i], i + 1, t_id);
      dalib_internal_error (msg);
      dalib_group_print ();
      dalib_stop ();
    }

  tcb->task_ids [tcb->task_size] = t_id;
  tcb->task_size += 1;
   
  task_offset++;

} /* dalib_new_task */

/**************************************************************************
*                                                                         *
*  void dalib_set_group_tasks (int no_tasks, int group_ids)               *
*                                                                         *
*    - start tasking for no_tasks disjoint groups                         *
*                                                                         *
**************************************************************************/

void dalib_set_group_tasks (no_tasks, group_ids)

int no_tasks;
int group_ids[];

{ task_control_block *tcb;

  int i;
  char msg[128];

  tcb = current_task_entry (0);

  if (tcb->task_size != 0)

     { sprintf (msg, "set_group_tasks: already tasks defined");
       dalib_internal_error (msg);
       dalib_stop ();
     }

  if (task_offset+no_tasks >= MAX_TASK_GROUPS)

     { sprintf (msg, "hpf_tasks : too many tasks (max = %d)", MAX_TASK_GROUPS);
       dalib_internal_error (msg);
       dalib_stop ();
     }

  tcb->task_size  = no_tasks;

  for (i=0; i<no_tasks; i++) tcb->task_ids[i] = group_ids[i];
   
  task_offset += no_tasks;

} /* dalib_set_group_tasks */

/**************************************************************************
*                                                                         *
*  void FUNCTION(dalib_set_task_home) (array_info *array_id)              *
*  void FUNCTION(dalib_set_procs_home) (int *topology)                    *
*                                                                         *
*    - defines a new task by the group of processors defined by           *
*      an array or by the topology                                        *
*                                                                         *
**************************************************************************/

     /***********************************************************
     *   dalib_set_task_home  (array_info *array_id)            *
     ***********************************************************/

void FUNCTION(dalib_set_task_home) (array_id)

array_info *array_id;

{ int task_id;

  if (dalib_is_array_info (*array_id))

     { array_info template;
       int        topology;

       dalib_array_info (*array_id, &template, &topology);

       task_id = dalib_top_group (topology);

       dalib_new_task (task_id);
     }

   else

     { dalib_internal_error ("dalib_set_task_home: not an array");
       dalib_stop ();
     }

} /* dalib_set_task_home */

     /***********************************************************
     *   dalib_set_task_procs  (int *topology)                  *
     ***********************************************************/

void FUNCTION(dalib_set_task_procs) (topology)

int *topology;

{ int task_id;

  task_id = dalib_top_group (*topology);

  dalib_new_task (task_id);

} /* dalib_set_task_procs */

/**************************************************************************
*                                                                         *
*  void dalib_check_task_id (task_control_block *tcb, int task_id)        *
*                                                                         *
*   - verifies that task_id is a legal task id in task_environment tcb    *
*                                                                         *
**************************************************************************/

void dalib_check_task_id (tcb, task_id, routine)

task_control_block *tcb;
int task_id;
char *routine;

{ if ((task_id < 1) || (task_id > tcb->task_size))
     
    { char msg[128];

      sprintf (msg, "HPF_TASK_LIBRARY.%s task_id %d out of range (1:%d)",
                    routine, task_id, tcb->task_size);

      dalib_internal_error (msg);
      dalib_stop ();
    }

} /* dalib_check_task_id */

/**************************************************************************
*                                                                         *
*   void FUNCTION(dalib_hpf_task_init)  ()                                *
*                                                                         *
*   void FUNCTION(dalib_hpf_task_exit)  ()                                *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_hpf_task_init) ()

{ task_control_block *tcb;
  int pids [MAXP];
  int my_task_id;
  int i;

  if (task_nest <= 0)

    { dalib_internal_error (
         "HPF_TASK_LIBRARY.HPF_TASK_INIT : no tasks created");
      dalib_stop ();
     }

  tcb = current_task_entry (0);

  tcb->task_init = 1;

  /* in a local task environment HPF_LOCAL everything is already done */

  if (tcb->task_local) return;

  /* we have to define a new group containing the roots */

  my_task_id = dalib_context_group ();

  for (i=0; i < tcb->task_size; i++)

     { int task_id;

       task_id = tcb->task_ids[i];

       if (my_task_id == task_id) tcb->task_rank = i + 1;

       pids[i] = dalib_group_element (task_id, 1);

     }

  tcb->task_group = dalib_group_create (tcb->task_size, pids);

} /* dalib_hpf_task_init */

void FUNCTION(dalib_hpf_task_exit) ()

{ 
} /* dalib_hpf_task_exit */

/**************************************************************************
*                                                                         *
*  void FUNCTION(dalib_hpf_task_rank) (int *rank)                         *
*  void FUNCTION(dalib_hpf_task_size) (int *size)                         *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_hpf_task_rank) (rank)

int *rank;

{ task_control_block *tcb;

  int i;
  int task_id;

  tcb = current_task_entry (1);

  *rank = tcb->task_rank;

#ifdef DEBUG
   printf ("%d: dalib_hpf_task_rank, group = %d, rank = %d\n",
            pcb.i, tcb->task_group, *rank);
#endif

} /* dalib_hpf_task_rank */

void FUNCTION(dalib_hpf_task_size) (size)

int *size;

{ task_control_block *tcb;

  tcb = current_task_entry (1);

  *size = tcb->task_size;

#ifdef DEBUG
       printf ("%d: dalib_hpf_task_rank, local group = %d, size = %d\n",
                pcb.i, local_group, *size);
#endif

} /* dalib_hpf_task_size */

/**************************************************************************
*                                                                         *
*  void dalib_hpf_send_scalar (void *data, int *size, int *task_id)       *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_hpf_send_scalar) (data, size, task_id)

int *task_id, *size;
void *data;

{ task_control_block *tcb;

  int pid;
  int group;

  tcb = current_task_entry (1);

  if (tcb->task_local)

     {  pid = dalib_group_element (tcb->task_group, *task_id);

        dalib_send (pid, data, *size, 1);

        return;
     }

  dalib_check_task_id (tcb, *task_id, "HPF_SEND");

  if (dalib_context_pid() == 1)

    { /* I am responsible for sending */

      group = tcb->task_ids[*task_id-1];
      pid   = dalib_group_first (group);

#ifdef DEBUG
      printf ("%d: hpf_send_scalar %d bytes to hpf task %d (=> pid = %d)\n",
              pcb.i, *size, *task_id, pid);
#endif

      dalib_send (pid, data, *size, 1);

    }

} /* dalib_hpf_send_scalar */

/**************************************************************************
*                                                                         *
*  void dalib_hpf_recv_scalar (void *data, int *size, int *task_id)       *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_hpf_recv_scalar) (data, size, task_id)

int *task_id, *size;
void *data;

{ task_control_block *tcb;
  int pid;
  int group;

  tcb = current_task_entry (1);

  if (tcb->task_local)

     {  if (FUNCTION(dalib_present)(task_id))

          { pid = dalib_group_element (tcb->task_group, *task_id);
            dalib_receive (pid, data, *size);
          }

         else

          machine_mpi_receive (-1, data, *size);

        return;
     }

  if (dalib_context_pid() == 1)

    { /* I am responsible for receiving at first */

      if (FUNCTION(dalib_present)(task_id))

        { dalib_check_task_id (tcb, *task_id, "HPF_RECV");

          group = tcb->task_ids[*task_id-1];
          pid   = dalib_group_first (group);

#ifdef DEBUG
      printf ("%d: hpf_recv_scalar %d bytes from hpf task %d (=> pid = %d)\n",
              pcb.i, *size, *task_id, pid);
#endif

          dalib_receive (pid, data, *size);

        }

      else

        { /* receive from any task */

#ifdef DEBUG
      printf ("%d: hpf_recv_scalar %d bytes from any task\n",
              pcb.i, *size);
#endif

          machine_mpi_receive (-1, data, *size);

#ifdef DEBUG
          printf ("%d: received data from unknown task\n", pcb.i);
#endif

        }

    }

  dalib_group_bcast (data, *size, 1, dalib_context_group()); 

} /* dalib_hpf_recv_scalar */

/**************************************************************************
*                                                                         *
*  void dalib_hpf_bcast_scalar (void *data, int *size, int *root_id)      *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_hpf_bcast_scalar) (data, size, root_id)

int *root_id, *size;
void *data;

{ task_control_block *tcb;

  int pid;
  int group;
  int i;

  tcb = current_task_entry (1);

  dalib_check_task_id (tcb, *root_id, "HPF_BCAST");

  dalib_group_bcast (data, *size, *root_id, tcb->task_group);

  if (tcb->task_local) return;

  /* otherwise we have to broadcast the value within our own task */

  group = tcb->task_ids[tcb->task_rank-1];

  dalib_group_bcast (data, *size, 1, group);

} /* dalib_hpf_bcast_scalar */

/**************************************************************************
*                                                                         *
*                                                                         *
*                                                                         *
**************************************************************************/

#define MAX_DSP_LENGTH 512

/**************************************************************************
*                                                                         *
*  void dalib_pack_dsp (char *buffer, array_info array_id,                *
*                       array_info *align_array)                          *
*                                                                         *
*  - packs all the descriptor data into the buffer (to send it)           *
*  - in case of alignment it returns also descriptor of align target      *
*                                                                         *
**************************************************************************/

void dalib_pack_dsp (buffer, descriptor, align_array)

char       *buffer;
array_info descriptor;
array_info *align_array;

{ char *ptr;        /* points into buffer to fill it up      */
  int  rank;
  int  length;

  array_info   array_id;
  Distribution dist;
  Alignment align;

  *align_array = NO_ARRAY;

  ptr    = buffer;

  if (dalib_is_array_info (descriptor))

     { /* this will work fine from now */

       array_id = descriptor;
  
     }

   else if (dalib_is_section_info (descriptor))

     { section_info section_id;

       section_id = (section_info) descriptor;
       array_id = section_id->array_id;
     }

   else 

     { dalib_internal_error ("dalib_pack_dsp: not array/section info");
       dalib_stop ();
     }

  /* put the rank of section/array descriptor into the buffer */

  rank   = array_id->rank;
  length = sizeof(rank);
  dalib_memcopy (ptr, &rank, length);
  ptr   += length;

  /* in case of section we put section descriptor at first into buffer */

  if (dalib_is_section_info (descriptor))

    { dalib_pack_section (ptr, rank, descriptor, &length);
      ptr += length;

#ifdef DEBUG
      printf ("%d: dalib_pack_dsp, section descriptor has length = %d\n",
              pcb.i, length);
#endif

    }

  length = dalib_array_dsp_size (rank);
  dalib_memcopy (ptr, array_id, length);
  ptr += length;

#ifdef DEBUG
  printf ("%d: dalib_pack_dsp, dsp has length = %d\n",
          pcb.i, length);
#endif

  dist = array_id->DistributeInfo;

  if (dist != NO_DISTRIBUTION)

    { /* pack the distribution of the descriptor into buffer */

      dalib_pack_distribution (ptr, rank, dist, &length);
      ptr += length;

#ifdef DEBUG
      printf ("%d: dalib_pack_dsp, dist has length = %d\n",
               pcb.i, length);
#endif

    }

  align = array_id->AlignInfo;

  if (align != NO_ALIGNMENT)

    { /* pack the alignment of the descriptor into buffer */

      dalib_pack_alignment (ptr, rank, align, &length, align_array);
      ptr += length;

#ifdef DEBUG
      printf ("%d: dalib_pack_alignment, align has length = %d\n",
               pcb.i, length);
#endif

      /* note the descriptor of the array align_array has also be be sent 
         because array can be aligned to a remote created template/array */

    }

} /* dalib_pack_dsp */

/**************************************************************************
*                                                                         *
*   static array_info dalib_unpack_dsp (char buffer[])                    *
*                                                                         *
*    - allocates memory for a new descriptor and unpacks buffer into it   *
*                                                                         *
**************************************************************************/

static array_info dalib_unpack_dsp (buffer)

char *buffer;

{ int rank;
  int length;
  char *ptr;

  section_info section_id;

  array_info array_id, dalib_array_new_dsp ();

  Distribution dist;
  Alignment    align, dalib_unpack_alignment ();

  ptr = buffer; 

  length = sizeof (rank);
  dalib_memcopy (&rank, ptr, length);
  ptr += length;

#ifdef DEBUG
  printf ("%d: unpack_dsp for array/section descriptor, rank = %d\n", 
           pcb.i, rank);
#endif

  section_id = NO_SECTION;
  array_id   = dalib_array_new_dsp (rank, 0, kIS_DYNAMIC);

  length = dalib_array_dsp_size (array_id->rank);
  dalib_memcopy (array_id, ptr, length);

  if (dalib_is_section_info (array_id))

     { int sec_length;

       /* descriptor is one of section */

       dalib_unpack_section (ptr, rank, &section_id, &sec_length);
       ptr += sec_length;

       section_id->array_id = array_id;   /* take local pointer now */

#ifdef DEBUG
       printf ("%d: unpack_dsp for section descriptor, length = %d\n",
                pcb.i, sec_length);
#endif

       /* now we have the array descriptor in the buffer */

       dalib_memcopy (array_id, ptr, length);
     }

  ptr += length;

  /* set status and data information correctly (not valid here) */

  array_id->dsp_status_flag = DSP_DEFINED;

  array_id->n_data          = 0;
  array_id->f_data          = NO_DATA;
  array_id->c_data          = NO_DATA;

#ifdef DEBUG
  printf ("%d: unpack_dsp for array descriptor, length = %d bytes\n",
           pcb.i, length);
  dalib_print_array_info (array_id);
#endif

  dist = array_id->DistributeInfo; 

  if (dist != NO_DISTRIBUTION)

    { /* attention: dist is pointer that might not be valid here */

      dalib_unpack_distribution (ptr, rank, &dist, &length);
      ptr += length;

      array_id->DistributeInfo = dist;

#ifdef DEBUG
      printf ("%d: unpack descriptor, now unpack distribution (length = %d)\n",
               pcb.i, length);
#endif

    }

  align = array_id->AlignInfo; 

  if (align != NO_ALIGNMENT)

    { align = dalib_unpack_alignment (ptr, rank, &length);
      ptr  += length;

      array_id->AlignInfo = align;

#ifdef DEBUG
      printf ("%d: unpack_dsp for alignment descriptor, length = %d\n",
               pcb.i, length);
#endif

      /* note: we cannot compute local sizes without knowing target */

    }

#ifdef DEBUG
   dalib_print_array_info (array_id);
#endif

   if (section_id == NO_SECTION)
      return array_id;
    else
      return (array_info) section_id;

} /* dalib_unpack_dsp */

/**************************************************************************
*                                                                         *
*  void dalib_send_dsp (int pid, array_info array_id)                     *
*                                                                         *
*   - sends descriptor of array_id to processor pid                       *
*                                                                         *
**************************************************************************/

void dalib_send_dsp (pid, array_id)

int pid;
array_info array_id;

{ char buffer [MAX_DSP_LENGTH];

  array_info align_array;

#ifdef DEBUG
  printf ("%d: pack descriptor for pid = %d\n", pcb.i, pid);
#endif

  dalib_pack_dsp (buffer, array_id, &align_array);

#ifdef DEBUG
  printf ("%d: sends descriptor to pid = %d\n", pcb.i, pid);
#endif

  dalib_send (pid, buffer, MAX_DSP_LENGTH, 1);

  /* there might be a recursive call if array is aligned */

  if (align_array != NO_ARRAY) dalib_send_dsp (pid, align_array);

} /* dalib_send_dsp */

/**************************************************************************
*                                                                         *
*                                                                         *
*                                                                         *
**************************************************************************/

void dalib_recv_dsp (pid, descriptor)

int pid;
array_info *descriptor;   /* descriptor can be section/array dsp */

{ char buffer [MAX_DSP_LENGTH];

  array_info array_id;

#ifdef DEBUG
  printf ("%d: receives descriptor from pid = %d\n", pcb.i, pid);
#endif

  if (dalib_context_pid() == 1)
    dalib_receive (pid, buffer, MAX_DSP_LENGTH);

  dalib_group_bcast (buffer, MAX_DSP_LENGTH, 1, dalib_context_group()); 

  *descriptor = dalib_unpack_dsp (buffer);

#ifdef DEBUG
  printf ("%d: recv_dsp, unpacked descriptor\n", pcb.i);
#endif

  /* now we have descriptor, but might be we need aligned arrays,
     in any case: compute local sizes (that should be 0 here       */

  if (dalib_is_section_info (*descriptor))
     array_id = ((section_info) (*descriptor))->array_id;
   else
     array_id = *descriptor;

  if (array_id->AlignInfo != NO_ALIGNMENT)

     { array_info align_target;

#ifdef DEBUG
  printf ("%d: recv_dsp, receive descriptor of align target\n", pcb.i);
#endif

       dalib_recv_dsp (pid, &align_target);

#ifdef DEBUG
  printf ("%d: recv_dsp, received descriptor of align target\n", pcb.i);
#endif

       dalib_align_set_target (array_id, align_target);

       dalib_align_local_sizes (array_id);

     }

   else if (array_id->DistributeInfo != NO_DISTRIBUTION)

     dalib_dist_local_sizes (array_id);

   else

     dalib_full_local_sizes (array_id);

#ifdef DEBUG
     printf ("%d: recv_dsp, got this array :\n", pcb.i);
     dalib_print_array_info (array_id);
#endif

  /* now we can also compute local sizes of section descriptor */

  if (dalib_is_section_info (*descriptor))
   
   { dalib_section_reset (*descriptor);

#ifdef DEBUG
     printf ("%d: recv_dsp, got this section :\n", pcb.i);
     dalib_print_section_info (*descriptor);
#endif
   }

} /* dalib_recv_dsp */

/**************************************************************************
*                                                                         *
*  void dalib_free_remote_dsp (array_info array_id)                       *
*                                                                         *
*    - free memory needed for a remote array/section descriptor           *
*    - recursively called for align targets of the descriptor             *
*                                                                         *
**************************************************************************/

static void dalib_free_remote_dsp (array_id)

array_info array_id;

{
  if (dalib_is_section_info (array_id))

     { section_info section_id;

       section_id = (section_info) array_id;

       dalib_free_remote_dsp (section_id->array_id);

       FUNCTION(dalib_section_free) (&section_id);

     }

   else if (dalib_is_array_info (array_id))

    { if (array_id->AlignInfo != NO_ALIGNMENT)

        { array_info align_target;

          dalib_align_get_target (array_id, &align_target);

          dalib_free_remote_dsp (align_target);

        }

       dalib_free_descriptor (array_id);

    }

  else

    { dalib_internal_error ("dalib_free_remote_dsp: not array/section info");
      dalib_stop ();
    }

} /* dalib_free_remote_dsp */

/**************************************************************************
*                                                                         *
*                                                                         *
**************************************************************************/

static void dalib_hpf_send_start (array_id, task_id, remote_dsp)

array_info array_id;
int        task_id;
array_info *remote_dsp;

{ task_control_block *tcb;

  int pid;
  int group;

  tcb = current_task_entry (1);

  dalib_check_task_id (tcb, task_id, "HPF_SEND");

  group = tcb->task_ids[task_id-1];
  pid   = dalib_group_first (group);

#ifdef DEBUG
      printf ("%d: hpf_send_start (rank=%d) to hpf task %d (=> pid = %d)\n",
              pcb.i, array_id->rank, task_id, pid);
#endif

  if (dalib_context_pid() == 1)

    { /* I am responsible for sending */

      dalib_send_dsp (pid, array_id);

    }

  dalib_recv_dsp (pid, remote_dsp);

#ifdef DEBUG
  printf ("%d: hpf_send_start, have received dsp\n", pcb.i);
#endif

} /* dalib_hpf_send_start */

/**************************************************************************
*                                                                         *
*                                                                         *
**************************************************************************/

static void dalib_hpf_recv_start (array_id, task_id, remote_dsp)

array_info array_id;
int        task_id;
array_info *remote_dsp;

{ task_control_block *tcb;

  int pid;
  int group;

  tcb = current_task_entry (1);

  group = tcb->task_ids[task_id-1];
  pid   = dalib_group_first (group);

  dalib_recv_dsp (pid, remote_dsp);

  if (dalib_context_pid() == 1)

    { /* I am responsible for sending our descriptor of the target
         so that the other task knows where the data has to be sent to  */

      dalib_send_dsp (pid, array_id);

    }

#ifdef DEBUG
  printf ("%d: hpf_recv_start, have received dsp, now assign\n", pcb.i);
#endif

} /* dalib_hpf_recv_start */

/**************************************************************************
*                                                                         *
*  void dalib_hpf_send_array (array_info *array_dsp, int *task_id)        *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_hpf_send_array) (array_id, task_id)

int *task_id;
array_info *array_id;

{ task_control_block *tcb;
  array_info remote_dsp;

  tcb = current_task_entry (1);

  if (tcb->task_local)

     {  int pid;

        pid = dalib_group_element (tcb->task_group, *task_id);

        dalib_secarray_send (pid, *array_id);

        return;
     }

  dalib_hpf_send_start (*array_id, *task_id, &remote_dsp);

#ifdef DEBUG
  printf ("%d: hpf_send_array, have received dsp, now assign\n", pcb.i);
#endif

  FUNCTION(dalib_assign) (&remote_dsp, array_id);

#ifdef DEBUG
  printf ("%d: hpf_send_array, have send data\n", pcb.i);
#endif

  dalib_free_remote_dsp (remote_dsp);

} /* dalib_hpf_send_array */

/**************************************************************************
*                                                                         *
*  void dalib_hpf_send_init (array_info *array_dsp, int *task_id,         *
*                            int request)                                 *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_hpf_send_init) (array_id, task_id, request)

int *task_id;
array_info *array_id;
int *request;

{ task_control_block *tcb;
  array_info remote_dsp;

  tcb = current_task_entry (1);

  if (tcb->task_local)

     { dalib_internal_error ("dalib_hpf_send_init: not for local tasking");
       dalib_stop ();
       return; 
     }

  dalib_hpf_send_start (*array_id, *task_id, &remote_dsp);

  *request = dalib_new_request (0, *array_id, remote_dsp);

#ifdef DEBUG
  printf ("%d: hpf_send_init, request = %d\n", pcb.i, *request);
#endif

} /* dalib_hpf_send_init */

/**************************************************************************
*                                                                         *
*  void dalib_hpf_recv_array (array_info *array_id, int *task_id)         *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_hpf_recv_array) (array_id, task_id)

int *task_id;
array_info *array_id;

{ task_control_block *tcb;
  array_info remote_dsp;

  int pid;
  int group;

  tcb = current_task_entry (1);

  if (!FUNCTION(dalib_present)(task_id))

     { dalib_internal_error ("hpf_recv_array now allowed from any task");
       dalib_stop ();
     }

  if (tcb->task_local)

     {  pid = dalib_group_element (tcb->task_group, *task_id);

        dalib_secarray_recv (pid, *array_id);

        return;
     }

  dalib_hpf_recv_start (*array_id, *task_id, &remote_dsp);

#ifdef DEBUG
  printf ("%d: hpf_recv_array, have received dsp, now assign\n", pcb.i);
#endif

  FUNCTION(dalib_assign) (array_id, &remote_dsp);

#ifdef DEBUG
  printf ("%d: hpf_recv_array, have received data\n", pcb.i);
#endif

  dalib_free_remote_dsp (remote_dsp);

#ifdef DEBUG
  printf ("%d: hpf_recv_array, have freed remoete descriptors\n", pcb.i);
#endif

} /* dalib_hpf_recv_array */

/**************************************************************************
*                                                                         *
*  void dalib_hpf_recv_init (array_info *array_dsp, int *task_id,         *
*                            int request)                                 *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_hpf_recv_init) (array_id, task_id, request)

int *task_id;
array_info *array_id;
int *request;

{ task_control_block *tcb;
  array_info remote_dsp;

  tcb = current_task_entry (1);

  if (tcb->task_local)

     { dalib_internal_error ("dalib_hpf_recv_init: not for local tasking");
       dalib_stop ();
       return; 
     }

  dalib_hpf_recv_start (*array_id, *task_id, &remote_dsp);

  *request = dalib_new_request (1, *array_id, remote_dsp);

} /* dalib_hpf_recv_init */

/**************************************************************************
*                                                                         *
*  void dalib_hpf_task_comm (int *request)                                *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_hpf_task_comm) (request)

int *request;

{ task_control_block *tcb;

  int kind;

  array_info array_id, remote_id;

  tcb = current_task_entry (1);

#ifdef DEBUG
  printf ("%d: handle task communication request %d\n", pcb.i, *request);
#endif

  if (tcb->task_local)

     { dalib_internal_error ("dalib_hpf_task_comm: not for local tasking");
       dalib_stop ();
       return; 
     }

  dalib_get_request (*request, &kind, &array_id, &remote_id);

  if (kind == 0)

     {  /* send array_id to remote_id */

        FUNCTION(dalib_assign) (&remote_id, &array_id);

     }

   else

     {  /* recv array_id to remote_id */

        FUNCTION(dalib_assign) (&array_id, &remote_id);

     }

} /* dalib_hpf_task_comm */
