/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : May 97                                                   *
*  Last Update : May 97                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : hpf_local.m4                                             *
*                                                                         *
*  Function    : HPF LOCAL LIBRARY                                        *
*                                                                         *
*  Export :  FORTRAN Interface                                            *
*                                                                         *
*   Note : these routines are called from (local) HPF programs            *
*                                                                         *
**************************************************************************/

#define CHECK
#undef  DEBUG

#include "dalib.h"

/**************************************************************************
*                                                                         *
*                                                                         *
**************************************************************************/

static array_info dalib_get_global_dsp (dsp)

section_info *dsp;

{ array_info array_dsp;
  array_info global_dsp;

  if (dalib_is_section_info (*dsp))

      { array_dsp = (*dsp)->array_id; }

   else if (dalib_is_array_info (*dsp))

      { array_dsp = (array_info) (*dsp); }

   else 

      { dalib_internal_error ("dalib_get_global_dsp : not array/section");
        dalib_stop ();
      }

  global_dsp = array_dsp->GlobalInfo;

  if (global_dsp == NO_ARRAY)

      { dalib_internal_error ("dalib_get_global_dsp : no global array");
        dalib_stop ();
      }

  return global_dsp;

} /* dalib_get_global_dsp */

/**************************************************************************
*                                                                         *
*   GLOBAL_SIZE (ARRAY [,DIM])  -> returns default integer scalar         *
*                                                                         *
*   GLOBAL_SIZE (SIZE, ARRAY, DIM)                                        *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_global_size) (size, array, dim,
                                  size_dsp, array_dsp, dim_dsp)

INTEGER    *size;
char       *array;    /* values not used, only descriptor */
INTEGER    *dim;

array_info *size_dsp;    /* size is assumed to be replicated */
array_info *array_dsp;
array_info *dim_dsp;

{ int idim;

  if (FUNCTION(dalib_present) (dim))
     idim = *dim;
   else
     idim = -1;

  FUNCTION(f77_global_size) (size, array_dsp, &idim);

} /* dalib_global_size */

/**************************************************************************
*                                                                         *
*   GLOBAL_SHAPE (ARRAY)                                                  *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_global_shape) (shape, array, shape_dsp, array_dsp)

INTEGER    *shape;
char       *array;

array_info *shape_dsp;
array_info *array_dsp;

{ 

  FUNCTION(f77_global_shape) (shape, array_dsp);

} /* dalib_global_shape */

/**************************************************************************
*                                                                         *
*   GLOBAL_TEMPLATE (ARRAY, ...)                                          *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_global_template)

  ARGS(`alignee, template_rank, lb, ub, STRING_ARG(axis_type), axis_info,
        number_aligned, dynamic,
        alignee_dsp, rank_dsp, lb_dsp, ub_dsp,
        at_dsp, ai_dsp, na_dsp, dyn_dsp')

char *alignee;
int  *lb, *ub;
STRING_ARG_DECL(axis_type);
int  *axis_info, *number_aligned, *dynamic;
int  *template_rank;

section_info *alignee_dsp;
array_info *rank_dsp, *lb_dsp, *ub_dsp, *at_dsp, *ai_dsp, *na_dsp, *dyn_dsp;

{ array_info global_dsp;

  global_dsp = dalib_get_global_dsp (alignee_dsp);

  FUNCTION(dalib_hpf_template)

  ARGS(`alignee, template_rank, lb, ub, STRING_ARG(axis_type), axis_info,
        number_aligned, dynamic,
        &global_dsp, rank_dsp, lb_dsp, ub_dsp,
        at_dsp, ai_dsp, na_dsp, dyn_dsp');

} /* dalib_global_template */

/**************************************************************************
*                                                                         *
*   GLOBAL_DISTRIBUTION (ARRAY, ...)                                      *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_global_distribution)

 ARGS(`distributee, STRING_ARG(axis_type),
       axis_info,   processors_rank, processors_shape,
       plb,         pub,       pstride,     low_shadow,      high_shadow,
       dist_dsp,    type_dsp,  info_dsp,    rank_dsp,        shape_dsp,
       plb_dsp,     pub_dsp,   pstride_dsp, low_dsp,         high_dsp')

char *distributee;
STRING_ARG_DECL(axis_type);
int  *axis_info;
int  *processors_rank, *processors_shape;
int  *plb, *pub, *pstride, *low_shadow, *high_shadow;

section_info *dist_dsp, *type_dsp, *info_dsp, *rank_dsp, *shape_dsp,
             *plb_dsp,  *pub_dsp,  *pstride_dsp, *low_dsp, *high_dsp;

{ array_info global_dsp;

  global_dsp = dalib_get_global_dsp (dist_dsp);

  FUNCTION(dalib_hpf_distribution)

   ARGS(`distributee, STRING_ARG(axis_type),
         axis_info,   processors_rank, processors_shape,
         plb,         pub,       pstride,     low_shadow,      high_shadow,
         &global_dsp, type_dsp,  info_dsp,    rank_dsp,        shape_dsp,
         plb_dsp,     pub_dsp,   pstride_dsp, low_dsp,         high_dsp');

} /* dalib_global_template */

/**************************************************************************
*                                                                         *
*   GLOBAL_ALIGNMENT (ARRAY, ...)                                         *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_global_alignment)

  (alignee, lb, ub, stride, axis_map, identity_map, dynamic, ncopies,
   alignee_dsp, lb_dsp, ub_dsp, stride_dsp, map_dsp, id_dsp, ncopies_dsp)

char *alignee;
int  *lb, *ub, *stride, *identity_map, *dynamic, *ncopies;
int  *axis_map;

section_info *alignee_dsp;

{ array_info global_dsp;

  global_dsp = dalib_get_global_dsp (alignee_dsp);

  FUNCTION(dalib_hpf_alignment)

  (alignee, lb, ub, stride, axis_map, identity_map, dynamic, ncopies,
   &global_dsp, lb_dsp, ub_dsp, stride_dsp, map_dsp, id_dsp, ncopies_dsp);

} /* dalib_global_alignment */

/**************************************************************************
*                                                                         *
*   ABSTRACT_TO_PHYSICAL (ARRAY, INDEX, PROC)                             *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_abstract_to_physical)

  (array    , index    , proc,
   array_dsp, index_dsp, proc_dsp)

unsigned char *array;
INTEGER  *index, *proc;                          /* data pointers */

section_info *array_dsp, *index_dsp, *proc_dsp;   /* descriptors */

{ array_info global_dsp;

  array_info template;
  int        top_id;

  global_dsp = dalib_get_global_dsp (array_dsp);

  dalib_array_info (global_dsp, &template, &top_id);

  /* top_id is now the topology to which the global array is mapped */

  *proc = dalib_top_abspos (top_id, index) - 1;

  /* attention: HPF_LOCAL numbers physical processors from 0 to P-1 */

} /* dalib_abstract_to_physical */

/**************************************************************************
*                                                                         *
*   PHYSICAL_TO_ABSTRACT (ARRAY, PROC, INDEX)                             *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_physical_to_abstract)

  (array    , proc    , index,
   array_dsp, proc_dsp, index_dsp)

unsigned char *array;
INTEGER  *index, *proc;                          /* data pointers */

section_info *array_dsp, *index_dsp, *proc_dsp;   /* descriptors */

{ array_info global_dsp;
  int        is_in_top;

  array_info template;
  int        top_id;

  global_dsp = dalib_get_global_dsp (array_dsp);

  dalib_array_info (global_dsp, &template, &top_id);

  /* top_id is now the topology to which the global array is mapped */

  dalib_top_position (top_id, *proc + 1, &is_in_top, index);

  /* attention: HPF_LOCAL numbers physical processors from 0 to P-1 */

} /* dalib_physical_to_abstract */

/****************************************************************************
*                                                                           *
* GLOBAL_TO_LOCAL (ARRAY, G_INDEX [,L_INDEX] [,LOCAL] [,NCOPIES], [,PROCS]) *
*                                                                           *
****************************************************************************/

void FUNCTION(dalib_global_to_local)

  (array    , g_index, l_index, local, ncopies, procs,
   array_dsp, g_dsp,   l_dsp,   dsp1,  dsp2,    dsp3  )

unsigned char *array;
INTEGER  *g_index, *l_index;                     /* data pointers */
INTEGER  *local, *ncopies, *procs;               /* data pointers */

section_info *array_dsp, *g_dsp, *l_dsp;   
section_info *dsp1, *dsp2, *dsp3;                /* descriptors   */

{ array_info global_dsp;
  array_info local_dsp;

  global_dsp = dalib_get_global_dsp (array_dsp);

  if (! FUNCTION(dalib_present)(l_index))

     { dalib_internal_error ("GLOBAL_TO_LOCAL: needs L_INDEX");
       dalib_stop ();
     }

  dalib_local_addr (global_dsp, g_index, l_index);

  if (FUNCTION(dalib_present)(local))

    { /* check whether the global index is local */

      int is_local;
      int i, rank;
      int *local_size;

      DimInfo *dims;

      is_local = F_TRUE;    /* by default it is local */

      local_dsp = (array_info) *array_dsp;

      rank = local_dsp->rank;
      dims = local_dsp->dimensions;

      for (i=0; i<rank; i++)

        { local_size = dims->local_size;

          if (l_index[i] > local_size[1]) is_local = 0;
          if (l_index[i] < local_size[0]) is_local = 0;
          dims++;
        }

      *local = is_local;
    }

  if (FUNCTION(dalib_present)(ncopies))

    { dalib_internal_error ("GLOBAL_TO_LOCAL: NCOPIES not supported");
      dalib_stop ();
    }

  if (FUNCTION(dalib_present)(procs))

    { dalib_internal_error ("GLOBAL_TO_LOCAL: PROCS not supported");
      dalib_stop ();
    }

} /* dalib_global_to_local */

/************************************************************************
*                                                                       *
* LOCAL_TO_GLOBAL (ARRAY, L_INDEX, G_INDEX)                             *
*                                                                       *
************************************************************************/

void FUNCTION(dalib_local_to_global)

  (array    , l_index, g_index, 
   array_dsp, l_dsp,   g_dsp   )

unsigned char *array;
INTEGER  *g_index, *l_index;                     /* data pointers */

section_info *array_dsp, *g_dsp, *l_dsp;   

{ array_info global_dsp;

  global_dsp = dalib_get_global_dsp (array_dsp);

  dalib_global_addr (global_dsp, l_index, g_index);

} /* dalib_local_to_global */

/**************************************************************************
*                                                                         *
*   LOCAL_BLKCNT (ARRAY [,DIM] [,PROC])                                   *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_local_blkcnt)

  (result,     array,     dim,     proc,
   result_dsp, array_dsp, dim_dsp, proc_dsp)

unsigned char *array;
INTEGER  *result, *dim, *proc;                   /* data pointers */

section_info *array_dsp, *result_dsp, *dim_dsp, *proc_dsp;

{ array_info global_dsp;
  int pid;

  global_dsp = dalib_get_global_dsp (array_dsp);

  if (FUNCTION(dalib_present)(proc))

     pid = *proc + 1; 

   else

     pid = pcb.i;

  if (FUNCTION(dalib_present)(dim))

     result[0] = 1;

   else

     { int dim, rank;

       rank = global_dsp->rank;

       for (dim=0; dim<rank; dim++)

           result[dim] = 1;
     }

} /* dalib_local_blkcnt */

/**************************************************************************
*                                                                         *
*   LOCAL_LINDEX (ARRAY, DIM [,PROC])                                     *
*   LOCAL_UINDEX (ARRAY, DIM [,PROC])                                     *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_local_lindex)

  (result,     array,     dim,     proc,
   result_dsp, array_dsp, dim_dsp, proc_dsp)

unsigned char *array;
INTEGER  *result, *dim, *proc;                   /* data pointers */

section_info *array_dsp, *result_dsp, *dim_dsp, *proc_dsp;

{ array_info global_dsp;
  int pid;

  global_dsp = dalib_get_global_dsp (array_dsp);

  pid = pcb.i;

  if (FUNCTION(dalib_present)(proc))  pid = *proc + 1; 

  result[0] = global_dsp->dimensions[*dim-1].local_size[0];

} /* dalib_local_lindex */

void FUNCTION(dalib_local_uindex)

  (result,     array,     dim,     proc,
   result_dsp, array_dsp, dim_dsp, proc_dsp)

unsigned char *array;
INTEGER  *result, *dim, *proc;                   /* data pointers */

section_info *array_dsp, *result_dsp, *dim_dsp, *proc_dsp;

{ array_info global_dsp;
  int pid;

  global_dsp = dalib_get_global_dsp (array_dsp);

  pid = pcb.i;

  if (FUNCTION(dalib_present)(proc))  pid = *proc + 1; 

  result[0] = global_dsp->dimensions[*dim-1].local_size[1];

} /* dalib_local_uindex */
