/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Mar 96                                                   *
*  Last Update : Mar 96                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : hpf.m4                                                   *
*                                                                         *
*  Function    : HPF LOCAL LIBRARY                                        *
*                                                                         *
*  Export :  FORTRAN Interface                                            *
*                                                                         *
**************************************************************************/

#define CHECK
#undef DEBUG

#include "dalib.h"

/**************************************************************************
*                                                                         *
*  HPF_DISTRIBUTION (distributee, axis_type, axis_info,                   *
*                    processors_rank, processors_shape,                   *
*                    plb, pub, pstride, low_shadow, high_shadow)          *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_hpf_distribution)

 ARGS(`distributee, STRING_ARG(axis_type),
       axis_info,   processors_rank, processors_shape,
       plb,         pub,       pstride,     low_shadow,      high_shadow,
       dist_dsp,    type_dsp,  info_dsp,    rank_dsp,        shape_dsp,
       plb_dsp,     pub_dsp,   pstride_dsp, low_dsp,         high_dsp')

char *distributee;
STRING_ARG_DECL(axis_type);
int  *axis_info;
int  *processors_rank, *processors_shape;
int  *plb, *pub, *pstride, *low_shadow, *high_shadow;

section_info *dist_dsp, *type_dsp, *info_dsp, *rank_dsp, *shape_dsp,
             *plb_dsp,  *pub_dsp,  *pstride_dsp, *low_dsp, *high_dsp;

{ array_info template_id;

  int top_id;
  int top_rank, top_dim;
  int top_shape [MAX_DIMENSIONS];

  int i;
  int temp_dim, temp_rank;
  int NId;                    /* dummy integer variable */

  int kind  [MAX_DIMENSIONS];
  int *info [MAX_DIMENSIONS];
  int info1 [MAX_DIMENSIONS];  /* computed by info      */
  int map   [MAX_DIMENSIONS];
  int *size_ptr;              /* pointer to size array for irr. dist. */

  int lb, ub, base, stride;

#ifdef DEBUG
  printf ("%d: call of HPF_DISTRIBUTION\n", pcb.i);
#endif

       /******************************************************
       *                                                     *
       *  Step 1 : find template_id, top_id                  *
       *                                                     *
       ******************************************************/

  if (dist_dsp == (section_info *) 0)

     { /* distributee is a scalar */

       temp_rank = 0;
       top_rank  = 0;

       goto out_data;

     }

   else if (dalib_is_array_info (*dist_dsp))

      dalib_array_info (*dist_dsp, &template_id, &top_id);

   else if (dalib_is_section_info (*dist_dsp))

      dalib_array_info ((*dist_dsp)->array_id, &template_id, &top_id);

   else

     { dalib_internal_error ("HPF_DISTRIBUTION : no array/section/scalar");
       dalib_stop ();
     }

       /******************************************************
       *                                                     *
       *  Step 2 : get all data                              *
       *                                                     *
       ******************************************************/

#ifdef DEBUG
  printf ("%d: template_id = %d, top_id = %d\n", pcb.i, template_id, top_id);
#endif

  /* get the necessary data of topology in any case */

  top_rank = dalib_top_rank (top_id);

  dalib_top_shape (top_id, top_shape);

  /* get the data of the template */

  if (template_id == NO_ARRAY)

     { /* not distributed */

       temp_rank = 0;

     }

   else

     { temp_rank = template_id->rank;

       dalib_distribution_data (template_id, &top_id, kind, info, map);
     }

       /******************************************************
       *                                                     *
       *  Step 3 : ouptut data for present arguments         *
       *                                                     *
       ******************************************************/

 out_data :

  if (FUNCTION(dalib_present) (STRING_PTR(axis_type)))

   { char *ptr;
     int  length;
 
     ptr    = STRING_PTR(axis_type);
     length = dalib_secarray_size (type_dsp);
 
#ifdef DEBUG
     printf ("%d: HPF_DISTRIBUTION, AXIS_TYPE present, rank = %d\n",
             pcb.i, temp_rank);
#endif

     for (i=0; i < temp_rank; i++)

      { switch (kind[i]) {

          case kBLOCK_DIM     : dalib_set_fstring (ptr, 5, "BLOCK"); break;
          case kCYCLIC_DIM    : dalib_set_fstring (ptr, 6, "CYCLIC"); break;
          case kGEN_BLOCK_DIM : dalib_set_fstring (ptr, 9, "GEN_BLOCK"); break;
          case kINDIRECT_DIM  : dalib_set_fstring (ptr, 8, "INDIRECT"); break;
          case kARBITRARY_DIM : dalib_set_fstring (ptr, 9, "ARBITRARY"); break;
          case kSERIAL_DIM    : dalib_set_fstring (ptr, 9, "COLLAPSED"); break;
          default : dalib_internal_error ("illegal kind in HPF_DISTRIBUTION");
         }

         ptr += length;
      }

    } /* output of axis_type */

  if (FUNCTION(dalib_present)(axis_info))

    { /* compute info1 by info */

      for (i=0; i < temp_rank; i++)

      { int *offsets, NP, NId;

        switch (kind[i]) {

          case kARBITRARY_DIM :
          case kINDIRECT_DIM  : 
          case kGEN_BLOCK_DIM : 

             offsets  = info[i];

             dalib_top_info (top_id, map[i], &NP, &NId);
    
             /* block size via the offsets */

             if (NId == 1)
                info1[i] = offsets[0];
               else
                info1[i] = offsets[NId-1] - offsets[NId-2];

             break;

          default : info1[i] = (int) info[i];

         } /* switch */

      }  /* for i */

       dalib_replicate_out (info1, sizeof(int) * temp_rank,
                            axis_info, info_dsp);
    }

  if (FUNCTION(dalib_present)(processors_rank))

      *processors_rank = top_rank;

  if (FUNCTION(dalib_present)(processors_shape))

      dalib_replicate_out (top_shape, sizeof(int) * top_rank,
                           processors_shape, shape_dsp);

  if (FUNCTION(dalib_present)(plb))

      dalib_internal_error ("HPF_DISTRIBUTION: plb not supported");

  if (FUNCTION(dalib_present)(pub))

      dalib_internal_error ("HPF_DISTRIBUTION: pub not supported");

  if (FUNCTION(dalib_present)(pstride))

      dalib_internal_error ("HPF_DISTRIBUTION: pstride not supported");

  if (FUNCTION(dalib_present)(low_shadow))

      dalib_internal_error ("HPF_DISTRIBUTION: low_shadow not supported");

  if (FUNCTION(dalib_present)(high_shadow))

      dalib_internal_error ("HPF_DISTRIBUTION: high_shadow not supported");

} /* dalib_hpf_distribution */

/**************************************************************************
*                                                                         *
*                                                                         *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_hpf_template)

  ARGS(`alignee, template_rank, lb, ub, STRING_ARG(axis_type), axis_info,
        number_aligned, dynamic,
        alignee_dsp, rank_dsp, lb_dsp, ub_dsp,
        at_dsp, ai_dsp, na_dsp, dyn_dsp')

char *alignee;
int  *lb, *ub;
STRING_ARG_DECL(axis_type);
int  *axis_info, *number_aligned, *dynamic;
int  *template_rank;

section_info *alignee_dsp;
array_info *rank_dsp, *lb_dsp, *ub_dsp, *at_dsp, *ai_dsp, *na_dsp, *dyn_dsp;

{ array_info template_id;
  array_info array_id;
  array_info *alignees;

  int my_lb   [MAX_DIMENSIONS];
  int my_ub   [MAX_DIMENSIONS];
  int my_type [MAX_DIMENSIONS];
  int my_info [MAX_DIMENSIONS];
  int my_dynamic;
  int my_number_aligned;

  int i;
  int temp_rank;
  int top_id;        /* only used as dummy */

       /******************************************************
       *                                                     *
       *  Step 1 : find template_id, top_id                  *
       *                                                     *
       ******************************************************/
 
  if (alignee_dsp == (section_info *) 0)
 
     { /* alignee is a scalar */
 
       temp_rank         = 0;
       my_number_aligned = 0;
 
       goto out_data;
 
     }
 
   else if (dalib_is_array_info (*alignee_dsp))
 
      array_id = (array_info) *alignee_dsp;
 
   else if (dalib_is_section_info (*alignee_dsp))
 
      array_id = (*alignee_dsp)->array_id;
 
   else
 
     { dalib_internal_error ("HPF_TEMPLATE : no array/section/scalar");
       dalib_stop ();
     }

       /******************************************************
       *                                                     *
       *  Step 2 : get all data                              *
       *                                                     *
       ******************************************************/

  my_dynamic = array_id->dynamic;

  dalib_array_info (array_id, &template_id, &top_id);
 
  /* get the rank and the extensions of the template */

  dalib_array_global_shape (template_id, &temp_rank, my_lb, my_ub);

  if (template_id->DistributeInfo != NO_DISTRIBUTION)

     dalib_get_aligned_arrays (template_id, &my_number_aligned, &alignees);
 
   else
  
     my_number_aligned = 0;

  if (array_id->AlignInfo != NO_ALIGNMENT)

     dalib_align_target_data (array_id, my_type, my_info);

    else

     { /* no alignment, so take default values */

       for (i=0; i<temp_rank; i++)

          { my_type[i] = kSOURCE_DIM;
            my_info[i] = i+1;
          }
     }

       /******************************************************
       *                                                     *
       *  Step 3 : ouptut data for present arguments         *
       *                                                     *
       ******************************************************/

out_data:

  /* copy out the necessary data */

  if (FUNCTION(dalib_present)(template_rank))
      *template_rank = temp_rank;

  if (FUNCTION(dalib_present)(lb))
      { for (i=0; i<temp_rank; i++) lb[i] = my_lb[i]; }

  if (FUNCTION(dalib_present)(ub))
      { for (i=0; i<temp_rank; i++) ub[i] = my_ub[i]; }

  if (FUNCTION(dalib_present) (STRING_PTR(axis_type)))

   { char *ptr;
     int  length;

     ptr = STRING_PTR(axis_type);
     length = (*at_dsp)->size;

     for (i=0; i < temp_rank; i++)

      { switch (my_type[i]) {

         case kSOURCE_DIM     : dalib_set_fstring (ptr, 6, "NORMAL"); 
                                break;
         case kEMBEDDED_DIM   : dalib_set_fstring (ptr, 6, "SINGLE"); 
                                break;
         case kREPLICATED_DIM : dalib_set_fstring (ptr, 10, "REPLICATED"); 
                                break;
        } /* switch */

        ptr += length;
      }

    } /* out of axis_type */

  if (FUNCTION(dalib_present)(axis_info))
      { for (i=0; i<temp_rank; i++) axis_info[i] = my_info[i]; }

  if (FUNCTION(dalib_present)(number_aligned))
      *number_aligned = my_number_aligned;

  if (FUNCTION(dalib_present)(dynamic))
      *dynamic = my_dynamic;

} /* dalib_hpf_template */

/**************************************************************************
*                                                                         *
*  HPF_ALIGNMENT (ALIGNEE, LB, UB, STRIDE, AXIS_MAP, IDENTITY_MAP,        *
*                 DYNAMIC, NCOPIES)                                       *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_hpf_alignment)

  (alignee, lb, ub, stride, axis_map, identity_map, dynamic, ncopies,
   alignee_dsp, lb_dsp, ub_dsp, stride_dsp, map_dsp, id_dsp, ncopies_dsp)

char *alignee;
int  *lb, *ub, *stride, *identity_map, *dynamic, *ncopies;
int  *axis_map;

section_info *alignee_dsp;

{ array_info   template_id;
  array_info   var_id;
  section_info sec_id;

  int i, alignee_rank;

  int my_lb     [MAX_DIMENSIONS];
  int my_ub     [MAX_DIMENSIONS];
  int my_stride [MAX_DIMENSIONS];
  int my_map    [MAX_DIMENSIONS];
  int base      [MAX_DIMENSIONS];
  int step      [MAX_DIMENSIONS];
  int my_identity_map;
  int my_ncopies;
  int my_dynamic;

  if (alignee_dsp == (section_info *) 0)
 
     { /* alignneee is a scalar */
 
       alignee_rank = 0;
       my_identity_map = 1;
       my_ncopies      = pcb.p;
       my_dynamic      = 0;

       goto out_data;
 
     }

  sec_id = *alignee_dsp;

  if (dalib_is_array_info (sec_id))
 
     { var_id = (array_info) sec_id;

       dalib_array_global_shape (var_id, &alignee_rank, my_lb, my_ub);
       for (i=0; i<alignee_rank; i++) my_stride[i] = 1;
 
     }

   else if (dalib_is_section_info (sec_id))
 
     { var_id  = sec_id->array_id;

       dalib_section_global_shape (sec_id, &alignee_rank, 
                                   my_lb, my_ub, my_stride);
 
     }
 
   else
 
    { dalib_internal_error ("HPF_ALIGNMENT: not a section/array");
      dalib_stop ();
    }
 
  my_identity_map = 1;
  my_ncopies      = 1;
  my_dynamic      = var_id->dynamic;

  if (var_id->AlignInfo == NO_ALIGNMENT)

     { my_identity_map = 1;
       for (i=0; i<alignee_rank; i++) my_map[i] = i+1;
     }

    else

     { dalib_align_source_data (var_id, my_map, base, step);
 
       /* update my_lb, my_ub, my_stride with the data */

       for (i=0; i<alignee_rank; i++)

          { int new_lb, new_ub;

            new_lb = step[i] * my_lb[i] + base[i];
            new_ub = step[i] * my_ub[i] + base[i];

            if (my_map[i] != i+1)   my_identity_map = 0;
            if (new_lb != my_lb[i]) my_identity_map = 0;
            if (new_ub != my_ub[i]) my_identity_map = 0;

            my_lb[i]     = new_lb;
            my_ub[i]      = new_ub;
            my_stride[i] *= step[i];
 
            if (my_stride[i] != 1) my_identity_map = 0;

            /* stride of a collapsed axis must be set to zero */

            if (my_map[i] == 0) my_stride[i] = 0;
          }
     }

out_data :

  /* copy out the necessary data */

  if (FUNCTION(dalib_present)(lb))
      dalib_replicate_out (my_lb, sizeof(int)*alignee_rank, lb, lb_dsp);

  if (FUNCTION(dalib_present)(ub))
      dalib_replicate_out (my_ub, sizeof(int)*alignee_rank, ub, ub_dsp);

  if (FUNCTION(dalib_present)(stride))

      dalib_replicate_out (my_stride, sizeof(int)*alignee_rank, 
                           stride, stride_dsp);

  if (FUNCTION(dalib_present)(axis_map))

      dalib_replicate_out (my_map, sizeof(int)*alignee_rank, 
                           axis_map, map_dsp);

  if (FUNCTION(dalib_present)(identity_map))
      *identity_map = my_identity_map;

  if (FUNCTION(dalib_present)(dynamic))
      *dynamic = my_dynamic;

  if (FUNCTION(dalib_present)(ncopies))
      *ncopies = my_ncopies;

} /* dalib_hpf_alignment */

/**************************************************************************
*                                                                         *
*  HPF_SUBGRID_INFO (ARRAY, IERR,  DIM, LB, UB, STRIDE,                   *
*                    LB_EMBED, UB_EMBED, AXIS_MAP)                        *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_hpf_subgrid_info)

  (array, ierr, dim, lb, ub, stride, lb_embed, ub_embed, axis_map,
   array_dsp, ierr_dsp, dim_dsp, lb_dsp, ub_dsp, 
   lb_embed_dsp, ub_embed_dsp, axis_map_dsp)

int *ierr;
int *dim;
char *array;

section_info *array_dsp, *lb_dsp, *ub_dsp, *ierr_dsp;

{ int rank;
  int ierr1, ierr2;

  int my_lb       [7];
  int my_ub       [7];
  int my_str      [7];
  int my_lb_embed [7];
  int my_ub_embed [7];
  int my_axis_map [7];

  rank = dalib_secarray_rank (array_dsp);

  /* call the corressponding F77 version of it */

  FUNCTION(f77_subgrid_info) 

     (array_dsp, &ierr1, &ierr2, dim, my_lb, my_ub, my_str, 
                 my_lb_embed, my_ub_embed, my_axis_map);

  /* now copy the F77 arrays (?) into the HPF arrays */

  *ierr = 0;

  dalib_memcopy (lb, my_lb, rank*sizeof(INTEGER));
  dalib_memcopy (ub, my_ub, rank*sizeof(INTEGER));

} /* dalib_hpf_subgrid_info */
