/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Jun 94                                                   *
*  Last Update : Aug 98                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : groups.m4                                                *
*                                                                         *
*  Function: Definition of Processor Groups                               *
*                                                                         *
*  Export :  internal Interface                                           *
*  ============================                                           *
*                                                                         *
*    void dalib_groups_init (NP)                                          *
*                                                                         *
*       - initializes group 0 with all participating processors           *
*                                                                         *
*    int dalib_group1_create (int size, int first, int step)              *
*                                                                         *
*       - create group with size members: first, first+step, ....         *
*                                                                         *
*    int dalib_group_all ()                                               *
*                                                                         *
*       - special groups for all nodes with host and only nodes           *
*                                                                         *
*    UPDATES:                                                             *
*    ========                                                             *
*                                                                         *
*    03/98  :  groups now central data structur for processor subsets     *
*    08/98  :  irregular groups                                           *
*                                                                         *
**************************************************************************/

#include <stdio.h> 
#include "dalib.h"

#undef DEBUG
#define CHECK

     /*********************************************************
     *                                                        *
     *  Definition of the Data Structures                     *
     *                                                        *
     *********************************************************/

typedef struct

  { int size;     /* size of the group                                 */

    int kind;     /* 0 for regular, 1 for irregular                    */

    /* regular group:  first, first+step, first+2*step, ...            */

    int first;    /* first proc in group, its absolute processor id    */
    int step;     /* first + k * step, k < size, are other procs       */

    /* irregular group: elem[0], elem[1], ..., elem[size-1]            */

    int *elem;    /* identifiers are absoulte proc ids from 1..P       */

  } group_entry;

static group_entry *groups [MAX_GROUPS];
static int         group_top = 0;

static int         all_group;            /* special group of all processors */

/**************************************************************************
*                                                                         *
*  void dalib_group_check_valid (int group_id)                            *
*                                                                         *
*    ASSERT: 0 <= group_id < group_top                                    *
*                                                                         *
**************************************************************************/

#ifdef CHECK

static void dalib_group_check_valid (group_id)

int group_id;

{  char msg[80];

   if ((group_id < 0) || (group_id >= group_top))

     { sprintf (msg,
                "dalib_group_check_valid, %d is illegal (must be in 0:%d)\n",
                group_id, group_top-1);
  
       dalib_internal_error (msg);
     }

} /* dalib_group_check_valid */

#endif 

/**************************************************************************
*                                                                         *
*   void dalib_group_print ()                                             *
*                                                                         *
*    - prints all groups (used also for error messages in other modules)  *
*                                                                         *
**************************************************************************/

     /*********************************************************
     *  void dalib_print_reg_group                            *
     *********************************************************/

static void dalib_print_reg_group (nr, entry)

int nr;
group_entry *entry;

{ int size, first, step;                /* regular group    */
  int j;                                /* count variable   */

  if (entry->kind != 0) return;

  size   = entry->size;
  first  = entry->first;
  step   = entry->step;

  printf ("%d: group %d of size %d, regular: ",
           pcb.i, nr, size);
  for (j=0; j<size; j++) printf ("%d, ", first+j*step);
  printf ("\n");

} /* dalib_print_reg_group */

     /*********************************************************
     *  void dalib_print_irr_group                            *
     *********************************************************/

static void dalib_print_irr_group (nr, entry)

int nr;
group_entry *entry;

{ int *elem; int size;                  /* irregular group  */
  int j;                                /* count variable   */

  if (entry->kind != 1) return;

  size = entry->size;
  elem = entry->elem;

  printf ("%d: group %d of size %d, irregular: ", pcb.i, nr, size);
  for (j=0; j<size; j++) printf ("%d, ", elem[j]);
  printf ("\n");
       
} /* dalib_print_irr_group */

     /*********************************************************
     *  void dalib_group_print ()                             *
     *********************************************************/

void dalib_group_print ()

{ group_entry *entry;

  int k;   /* count variables */

  printf ("%d: summary of the %d groups\n", pcb.i, group_top);

  for (k=0; k<group_top; k++)

   { entry  = groups[k];

     if (entry->kind == 0)
        dalib_print_reg_group (k, entry);
       else if (entry->kind == 1)
        dalib_print_irr_group (k, entry);
       else
        { dalib_internal_error ("illegal group kind"); dalib_stop (); }

   }  /* for all groups */

} /* dalib_group_print */

/**************************************************************************
*                                                                         *
*  void dalib_group_define (group_entry *definition)                      *
*                                                                         *
*   - make a new group entry with same information as in definition       *
*   - note that for irregular groups new memory will be allocated         *
*                                                                         *
**************************************************************************/

static void dalib_group_define (definition)

group_entry *definition;

{ group_entry *entry;

  /* make sure that the stack is big enough */

  if (group_top >= MAX_GROUPS)

     { char msg[80];

       sprintf (msg, "dalib_group_define, too many groups (MAX_GROUPS = %d)",
                MAX_GROUPS);

       dalib_internal_error (msg);
       dalib_stop ();
     }

  /* now create the new entry, note that definition cannot be used    */

  entry = (group_entry *) dalib_malloc (sizeof (group_entry),
                                       "dalib_group_define (entry)");

  *entry = *definition;

  if (entry->kind == 1)

     { /* copy the elements from the definiton */

       int *new_elem, *old_elem;
       int i, size;

       size = definition->size;

       new_elem = dalib_int_malloc (size, "dalib_group_define (elems)");
       old_elem = definition->elem;

       for (i=0; i<size; i++) new_elem[i] = old_elem[i];

       entry->elem = new_elem;

     }

  groups[group_top++] = entry;

#ifdef DEBUG
  printf ("%d: new group (id=%d) defined, size=%d, kind=%d\n",
          pcb.i, group_top-1, entry->size, entry->kind);
  dalib_group_print ();
#endif

} /* dalib_group_define */

/**************************************************************************
*                                                                         *
*   int dalib_find_group (group_entry *entry)                             *
*                                                                         *
*    - searching a group entry in the stack with same information         *
*    - returns group_top if entry not found                               *
*                                                                         *
**************************************************************************/

static int dalib_find_group (entry)

group_entry *entry;

{ int pos;

  group_entry *entry1;

  int size, kind;

  pos = 0;

  size = entry->size;
  kind = entry->kind;

  while (pos < group_top)

   { entry1 = groups[pos];

     if (entry1->size != size) goto next;
     if (entry1->kind != kind) goto next;

     /* now we are sure that entry1 has same size and kind */

     if (kind == 0)

        { if (entry1->first != entry->first) goto next;
          if (entry1->step  != entry->step ) goto next;

          break;    /* same entry found at this position */

        }

      else

        { int *elems, *elems1;
          int same;
          int i;

          elems  = entry->elem;
          elems1 = entry1->elem;
          same   = 1;

          for (i=0; i < entry->size; i++)
            if (elems[i] != elems1[i]) { same = 0; break; }

          if (same) break;
       }

next:  pos++;

   }

   return (pos);

} /* dalib_find_group */

/**************************************************************************
*                                                                         *
*  int dalib_group1_create (int size, int first, int step)                *
*                                                                         *
*    - make a new group entry for processors first, first+step, ...       *
*    - new entry only created if processor group does not exist           *
*    - returns interal identification of the new group                    *
*                                                                         *
**************************************************************************/

int dalib_group1_create (size, first, step)

{ group_entry new_group;

  int group_id;

  new_group.size  = size;
  new_group.kind  = 0;
  new_group.first = first;
  new_group.step  = step;

  group_id = dalib_find_group (&new_group);

  if (group_id == group_top) dalib_group_define (&new_group);

#ifdef DEBUG
  printf ("%d: group1_create (Pn=%d,P1,=%d,Ps=%d) is group id %d\n",
           pcb.i, size, first, step, group_id);
#endif

  return (group_id);

} /* dalib_group1_create */

/**************************************************************************
*                                                                         *
*  int dalib_group_create (int size, int elements[size])                  *
*                                                                         *
*    - create a new group of processors specified in elements             *
*    - returns the group identifier (0, 1, ...)                           *
*                                                                         *
**************************************************************************/

int dalib_group_create (size, elements)

int size; 
int elements[];

{ group_entry new_group;

  int i;
  int group_id;

  new_group.size  = size;
  new_group.kind  = 0;             /* assume regular group  */
  new_group.first = elements[0];  

  if (size < 2) 
       new_group.step = 1;
    else
       new_group.step = elements[1] - elements[0];

  if (size > 2)

     { /* check for a regular step */

       for (i=2; i<size; i++)

          { int diff;
            diff = elements[i] - elements[i-1];
            if (diff != new_group.step)
              { new_group.kind = 1;
                new_group.elem = elements;
                break;
              }

          } /* for all elements */
     }

  group_id = dalib_find_group (&new_group);
 
  if (group_id == group_top) dalib_group_define (&new_group);

#ifdef DEBUG
  printf ("%d: group_create is group id %d\n", pcb.i, group_id);
#endif

  return (group_id);

} /* dalib_group_create */

/**************************************************************************
***************************************************************************
*                                                                         *
*  Query Routines for groups                                              *
*                                                                         *
*   int dalib_group_all ()                                                *
*   int dalib_group_size (int group_id)                                   *
*   int dalib_group_first (int group_id)                                  *
*                                                                         *
***************************************************************************
**************************************************************************/

/**************************************************************************
*                                                                         *
*   int dalib_group_all ()                                                *
*                                                                         *
*     - returns group identifier for group with all processors (global)   *
*                                                                         *
**************************************************************************/

int dalib_group_all ()
{ return (all_group);
} /* dalib_group_all */

/****************************************************************
*                                                               *
*   int dalib_group_size (int group_id)                         *
*                                                               *
****************************************************************/

int dalib_group_size (group_id)

int group_id;

{ return groups[group_id]->size;

} /* dalib_group_size */

/****************************************************************
*                                                               *
*   int dalib_group_first (int group_id)                        *
*                                                               *
*    - returns absolute processor id of first processor         *
*                                                               *
****************************************************************/

int dalib_group_first (group_id)

int group_id;

{ group_entry *group;

  group = groups[group_id];
 
  if (group->kind == 0) return groups[group_id]->first;
  if (group->kind == 1) return groups[group_id]->elem[0];

  dalib_internal_error ("illegal kind in dalib_group_first");
  dalib_stop ();

} /* dalib_group_first */

/**************************************************************************
*                                                                         *
*  void dalib_group_info (int group_id, int *size, int *kind              *
*                         int *first, int *step, int **elem)              *
*                                                                         *
*   - returns kind, first, size, step, elem for processor group           *
*   - use carefully (as pointer of own data for elem is returned)         *
*                                                                         *
**************************************************************************/

void dalib_group_info (group_id, size, kind, first, step, elem)

int group_id;
int *kind;
int *first;
int *size;
int *step;
int **elem;

{ group_entry *group;

#ifdef CHECK
  dalib_group_check_valid (group_id);
#endif

  group = groups[group_id];

  *kind  = group->kind;
  *size  = group->size;

  if (group->kind == 0)

     { *first = group->first;
       *step  = group->step;
       *elem  = (int *) 0;
     }

    else if (group->kind == 1)

     { *first = group->elem[0];
       *step  = 0;
       *elem  = group->elem;
     }

    else 

     { dalib_internal_error ("illegal kind in dalib_group_info");
       dalib_stop ();
     }

} /* dalib_group_info */

/*******************************************************************
*                                                                  *
*  int dalib_group_position (int group_id, int global_pid)         *
*                                                                  *
*   - returns relative position of global pid in group             *
*     with 1 <= pos <= dalib_group_size(group_id)                  *
*   - 0 if PId does not belong to the group                        *
*   - corresponds to MPI_Group_rank                                *
*                                                                  *
*******************************************************************/

int dalib_group_position (group_id, global_pid)

int group_id;
int global_pid;

{ group_entry *group;

  int size;
  int NId;

#ifdef CHECK
  dalib_group_check_valid (group_id);
#endif

  group = groups[group_id];

  size  = group->size;

  if (group->kind == 0)

    { int first, step;

      first = group->first;
      step  = group->step;

      NId   = (global_pid - first) / step + 1;

      if (NId <= 0)    return 0;
      if (NId >  size) return 0;

      if (first + (NId-1) * step != global_pid) return 0;

      return NId;
    }

   else

    { int i, *elem;

      elem = group->elem;

      for (i=0; i<size; i++) if (elem[i] == global_pid) return i+1;

      return 0;
    }

} /* dalib_group_position */

/**************************************************************************
*                                                                         *
*   int dalib_group_element (int group_id, int rel_pid)                   *
*                                                                         *
*   - 1 <= rel_pid <= dalib_group_size (group_id) specified relative pos  *
*   - returns global processor id for the corresponding element in group  *
*                                                                         *
**************************************************************************/

int dalib_group_element (group_id, rel_pid)

{ group_entry *group;

  group = groups[group_id];

  if (group->kind == 0)
     return group->first + (rel_pid-1) * group->step;
    else
     return group->elem[rel_pid-1];

} /* dalib_group_element */

/**************************************************************************
*                                                                         *
*  void dalib_group_all_elements (int group_id,                           *
*                                 int *group_size, int group_elem[])      *
*                                                                         *
*   - returns size of group and all identifiers (ordered)                 *
*   - note that group_elem must have been allocated before call           *
*                                                                         *
**************************************************************************/

void dalib_group_all_elements (group_id, group_size, group_elem)

int group_id;
int *group_size;
int group_elem[];

{ group_entry *group;

  int i, size;

#ifdef CHECK
  dalib_group_check_valid (group_id);
#endif

  group = groups[group_id];

  size = group->size;
  *group_size = size;

  if (group->kind == 0)

     { int first, step;

       first = group->first;
       step  = group->step ;

       for (i=0; i<size; i++) group_elem[i] = first + i * step;

     }

    else

       for (i=0; i<size; i++) group_elem[i] = group->elem[i];

} /* dalib_group_all_elements */

/**************************************************************************
*                                                                         *
*                                                                         *
*                                                                         *
**************************************************************************/

void dalib_group_get_elements (group_id, size, first, step, group_elem)

int group_id;
int size, first, step;
int group_elem[];

{ group_entry *group;

  int i;

#ifdef CHECK
  dalib_group_check_valid (group_id);
#endif

  group = groups[group_id];

  if (group->kind == 0)

     { int first1, step1;

       first1 = group->first;
       step1  = group->step ;

       for (i=0; i<size; i++) 

          group_elem[i] = first1 + (first + step * i - 1) * step1;

     }

    else

      for (i=0; i<size; i++) 

         group_elem[i] = group->elem[first + step * i - 1];

} /* dalib_group_get_elements */

/**************************************************************************
***************************************************************************
*                                                                         *
*   SUBGROUPS:  definition of subgroups of existing groups                *
*                                                                         *
***************************************************************************
**************************************************************************/

/**************************************************************************
*                                                                         *
*  int dalib_subgroup_create (int group_id, int size,                     *
*                             int elements[size]       )                  *
*                                                                         *
*  - make a subgroup within group_id by the specified processors          *
*    elements[0], elements[1], ..., elements[size-1]                      *
*                                                                         *
**************************************************************************/

int dalib_subgroup_create (group_id, size, elements)

int group_id;
int size;
int elements[];

{ int i;

  int group_size;

  int group_elements     [MAXP];
  int sub_group_elements [MAXP];

  int sub_group_id;

  /* get global identifiers of the existing group */

  dalib_group_all_elements (group_id, &group_size, group_elements);

  /* now select the specified processors for the subgroup */

  for (i=0; i<size; i++)
      sub_group_elements [i] = group_elements[elements[i]-1];

#ifdef DEBUG
  printf ("%d: dalib_subgroup_create of group %d (", pcb.i, group_id);
  for (i=0; i<group_size; i++) printf ("%d ", group_elements[i]);
  printf (") for elements (");
  for (i=0; i<size; i++) printf ("%d ", elements[i]);
  printf (") : ");
  for (i=0; i<size; i++) printf ("%d ", sub_group_elements[i]);
  printf ("\n");
#endif

  sub_group_id = dalib_group_create (size, sub_group_elements);

  return (sub_group_id);

} /* dalib_subgroup_create */

/**************************************************************************
*                                                                         *
*  int dalib_subgroup1_create (int group_id,                              *
*                              int size, int first, int step)             *
*                                                                         *
*   - creates a subprocessor group for a given processor group            *
*                                                                         *
**************************************************************************/

int dalib_subgroup1_create (group_id, size, first, step)       

int group_id;
int first;
int size;
int step;

{ group_entry *group;

  int g_first;
  int g_size;
  int g_step;

#ifdef CHECK
  dalib_group_check_valid (group_id);
#endif

  group = groups[group_id];

  if (group->kind != 0)

     { int sub_group_elements [MAXP];

       dalib_group_get_elements (group_id, size, first, step, 
                                 sub_group_elements);

       return dalib_group_create (size, sub_group_elements);

     }

  g_size  = group->size;
  g_first = group->first;
  g_step  = group->step;

  /* check that subgroup fits within group */

  g_first = g_first + (first-1) * step;
  g_step  = g_step  *  step;

#ifdef DEBUG
  printf ("%d: define subgroup of group %d: P1=%d, Ps=%d, Pn=%d\n",
           pcb.i, group_id, g_first, g_step, size);
#endif

  return dalib_group1_create (size, g_first, g_step);

} /* dalib_subgroup1_create */

/*******************************************************************
*                                                                  *
*  void dalib_groups_init (int NP)                                 *
*                                                                  *
*    - creates one group (group 0) for all processors              *
*                                                                  *
*******************************************************************/

void dalib_groups_init (NP)

int NP;  /* number of node processes */

{ if (group_top == 0)

   { /* create a group for all processes */

     all_group   = dalib_group1_create (NP, 1, 1);
   }
   
  else
  
   { /* groups already exist, update NP */

     groups[all_group]->size = NP;
   }

#ifdef DEBUG
  dalib_group_print ();
#endif

} /* dalib_groups_init */

/**************************************************************************
*                                                                         *
*   int dalib_group_distinct (int g_id1, int g_id2)                       *
*                                                                         *
*   - returns 1 if the two groups have not one common processor           *
*                                                                         *
**************************************************************************/

int dalib_group_distinct (g_id1, g_id2)

{ int kind1,  kind2;
  int *elem1, *elem2;

  int P1[3];
  int P2[3];
  int P3[3];

  dalib_group_info (g_id1, P1+1, &kind1, P1, P1+2, &elem1);
  dalib_group_info (g_id2, P2+1, &kind2, P2, P2+2, &elem2);

  if ((kind1 != 0) || (kind2 != 0))

     { int N1, N2;
       int k1, k2;

       int elements1 [MAXP];
       int elements2 [MAXP];

       dalib_group_all_elements (g_id1, &N1, elements1);
       dalib_group_all_elements (g_id2, &N2, elements2);

       for (k1=0; k1<N1; k1++)
          for (k2=0; k2<N2; k2++)
            if (elements1[k1] == elements2[k2]) return 0;

       return 1;
     }

  P1[1] = P1[0] + (P1[1]-1)*P1[2];
  P2[1] = P2[0] + (P2[1]-1)*P2[2];

  dalib_intersect_sections (P1, P2, P3);
  
#ifdef DEBUG
  printf ("%d: group distinct %d:%d:%d and %d:%d:%d is %d:%d:%d\n",
         pcb.i, P1[0], P1[1], P1[2], P2[0], P2[1], P2[2], P3[0], P3[1], P3[2]);
#endif

  if (dalib_range_size (P3[0], P3[1], P3[2]) == 0) return 1;

  return 0;

} /* dalib_group_distinct */

/*******************************************************************
*                                                                  *
*  void dalib_groups_exit ()                                       *
*                                                                  *
*   - free all groups and allocated memory                         *
*                                                                  *
*******************************************************************/

void dalib_groups_exit ()

{ int i;

  for (i=group_top-1; i>=0; i--)

    { group_entry *group;

      group = groups[i];

      if (group->kind == 1) 
         dalib_int_free (group->elem, group->size);

      dalib_free (group, sizeof(group_entry));

    }

  group_top = 0;

} /* dalib_groups_exit */

/**************************************************************************
*                                                                         *
*   void  dalib_pack_group (char buffer[], int group_id, int *length)     *
*                                                                         *
*    - packs the group descriptor of group_id into buffer                 *
*    - length returns the number of bytes needed within buffer            *
*                                                                         *
**************************************************************************/

void dalib_pack_group (buffer, group_id, length)

char buffer[];
int  group_id;
int  *length;

{ group_entry *entry;

  dalib_group_check_valid (group_id);

  entry = groups[group_id];

  *length = sizeof (group_entry);

  dalib_memcopy (buffer, entry, *length);

  if (entry->kind == 1)

     { /* pack also the group elements into the buffer */

       int elem_size;

       elem_size = (entry->size) * sizeof (int);

       dalib_memcopy (buffer + (*length), entry->elem, 
                      (entry->size) * sizeof (int));

       *length = *length + elem_size;

     }

#ifdef DEBUG
  printf ("%d: packed group id = %d\n", pcb.i, group_id);
#endif

} /* dalib_pack_group */

     /*************************************************************
     *   dalib_unpack_group (buffer => group_id, length)          *
     *************************************************************/

void dalib_unpack_group (buffer, group_id, length)

char buffer[];
int  *length;
int  *group_id;

{ group_entry entry;

  dalib_memcopy (&entry, buffer, sizeof(entry));

  *length = sizeof(entry);

  if (entry.kind != 0)

     { int elems [MAXP];
       int elem_size;

       elem_size = (entry.size) * sizeof (int);

       dalib_memcopy (elems, buffer + (*length), elem_size);

       *group_id = dalib_group_create (entry.size, elems);

       *length = *length + elem_size;
     }

    else

     *group_id = dalib_group1_create (entry.size, entry.first, entry.step);

#ifdef DEBUG
  printf ("%d: unpacked group %d (size=%d, first=%d, step=%d)\n",
          pcb.i, *group_id, entry.size, entry.first, entry.step);
#endif

} /* dalib_unpack_group */

