/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Apr 95                                                   *
*  Last Update : Apr 95                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : grade.m4                                                  *
*                                                                         *
*  Function    : HPF intrinsics grade_up, grade_down                      *
*                                                                         *
*  Export :  FORTRAN Interface                                            *
*                                                                         *
* void FUNCTION(dalib_grade_up) (array, dim)                              *
*                                                                         *
* void FUNCTION(dalib_grade_down) (array, dim)                            *
*                                                                         *
**************************************************************************/

#define CHECK
#undef  DEBUG

#include "dalib.h"

static int ascending;   /* + 1 for grade_up, - 1 for grade_down */

/**************************************************************************
*                                                                         *
*  dalib_get_ibits (val, npos, start, end)                                *
*                                                                         *
**************************************************************************/

int dalib_get_ibits (val, npos, start, end)

int npos, start, end;
unsigned int val;

{ int mask;

  /*******  1   2   3   ....  start .... end  .....   npos  *******/

  /* set bits from start:end in mask */

  mask = (1  << (end - start + 1)) - 1;
  mask = mask << (npos - end);
  val = val & mask;
  val =  val >> (npos - end);
  return (val);

} /* dalib_get_ibits */

/**************************************************************************
*                                                                         *
*   new ranking with some bits                                            *
*                                                                         *
**************************************************************************/

static void dalib_build_new_ranks (ranks, tmp_ranks, vals, n,
                                   nbits, start, end, ascending)

int ranks[];
int tmp_ranks[];
int vals[];
int n;
int nbits, start, end;
int ascending;

{ int count[16];
  int new_no, old_no;
  int i;
  int rsize;
  int radix;

  rsize = 1 << (end-start+1);

  if (rsize > 16)

     { dalib_internal_error ("illegal radix, too many bits ");
       dalib_stop ();
     }

  for (i=0; i<rsize; i++) count[i] = 0;

  for (i=0; i<n; i++)

    { radix = dalib_get_ibits (vals[ranks[i]-1], nbits, start, end);
      count[radix] ++;
#ifdef DEBUG
      printf ("%d has radix val %d\n", vals[ranks[i]-1], radix);
#endif
    };

 /* step 2 : count the beginning position of every radix */

  if (ascending > 0)

    { /* values will be sorted by ascending radix */

      old_no = count[0]; 
      count[0] = 1;

      for (i=1; i<rsize; i++)
        { new_no = count[i];
          count [i] = count[i-1] + old_no;
#ifdef DEBUG
          printf ("vals with radix %d start at rank pos %d\n", i, count[i]);
#endif
          old_no = new_no;
        };
    }

   else

    { /* values will be sorted by descending radix */

      old_no = count[rsize-1]; 
      count[rsize-1] = 1;

      for (i=rsize-2; i>=0; i--)
        { new_no = count[i];
          count [i] = count[i+1] + old_no;
#ifdef DEBUG
          printf ("vals with radix %d start at rank pos %d\n", i, count[i]);
#endif
          old_no = new_no;
        };
    }

  /* now we can make a new rank of the values */

  for (i=0; i<n; i++)

    { radix = dalib_get_ibits (vals[ranks[i]-1], nbits, start, end);
      tmp_ranks [count[radix]-1] = ranks[i];
#ifdef DEBUG
      printf ("val at pos %d has new rank %d\n", ranks[i], count[radix]);
#endif
      count[radix]++;
    }

  for (i=0; i<n; i++) ranks[i] = tmp_ranks[i];

} /* dalib_build_new_ranks */

/**************************************************************************
*                                                                         *
*   void dalib_grade_real4 (ranks, vals, n)                               *
*                                                                         *
**************************************************************************/

static void dalib_grade_real4 (ranks, vals, n)

int   ranks[];
float vals[];
int n;

{ int *tmp_ranks;    /* size must be n */

  int i;

#ifdef DEBUG
  printf ("grade up of %d values\n", n);
#endif

  tmp_ranks = (int *) dalib_int_malloc (n, "grade_real4");

  /* make a default ranking */

  for (i=0; i<n; i++) ranks[i] = i+1;

  dalib_build_new_ranks (ranks, tmp_ranks, vals, n, 32, 29, 32, ascending);
  dalib_build_new_ranks (ranks, tmp_ranks, vals, n, 32, 25, 28, ascending);
  dalib_build_new_ranks (ranks, tmp_ranks, vals, n, 32, 21, 24, ascending);
  dalib_build_new_ranks (ranks, tmp_ranks, vals, n, 32, 17, 20, ascending);
  dalib_build_new_ranks (ranks, tmp_ranks, vals, n, 32, 13, 16, ascending);
  dalib_build_new_ranks (ranks, tmp_ranks, vals, n, 32,  9, 12, ascending);
  dalib_build_new_ranks (ranks, tmp_ranks, vals, n, 32,  5,  8, ascending);
  dalib_build_new_ranks (ranks, tmp_ranks, vals, n, 32,  2,  4, ascending);
  dalib_build_new_ranks (ranks, tmp_ranks, vals, n, 32,  1,  1, -ascending);

} /* dalib_grade_real4 */

/**************************************************************************
*                                                                         *
*   void dalib_grade_real8 (ranks, vals, n)                               *
*                                                                         *
**************************************************************************/

static void dalib_grade_real8 (ranks, vals, n)

int    ranks[];
double vals[];
int n;

{ 
   dalib_internal_error ("sort of real*8 not available yet");
   dalib_stop ();

} /* dalib_grade_real8 */

/**************************************************************************
*                                                                         *
*   void dalib_grade_int4 (ranks, vals, n)                                *
*                                                                         *
**************************************************************************/

static void dalib_grade_int4 (ranks, vals, n)

int ranks[];
int vals[];
int n;

{ int *tmp_ranks;    /* size must be n */
  int i;

#ifdef DEBUG
  printf ("grade up of %d values\n", n);
#endif

  tmp_ranks = (int *) dalib_int_malloc (n, "grade_up_int4");

  /* make a default ranking */

  for (i=0; i<n; i++) ranks[i] = i+1;

  dalib_build_new_ranks (ranks, tmp_ranks, vals, n, 32, 29, 32, ascending);
  dalib_build_new_ranks (ranks, tmp_ranks, vals, n, 32, 25, 28, ascending);
  dalib_build_new_ranks (ranks, tmp_ranks, vals, n, 32, 21, 24, ascending);
  dalib_build_new_ranks (ranks, tmp_ranks, vals, n, 32, 17, 20, ascending);
  dalib_build_new_ranks (ranks, tmp_ranks, vals, n, 32, 13, 16, ascending);
  dalib_build_new_ranks (ranks, tmp_ranks, vals, n, 32,  9, 12, ascending);
  dalib_build_new_ranks (ranks, tmp_ranks, vals, n, 32,  5,  8, ascending);
  dalib_build_new_ranks (ranks, tmp_ranks, vals, n, 32,  2,  4, ascending);
  dalib_build_new_ranks (ranks, tmp_ranks, vals, n, 32,  1,  1, -ascending);

} /* dalib_grade_int4 */

/**************************************************************************
*                                                                         *
*   general routine with comparison function available                    *
*                                                                         *
*   - result and val array must be conform                                *
*   - dim must be available                                               *
*                                                                         *
**************************************************************************/

static void dalib_grade (result_array, val_array, dim, kind)

array_info result_array, val_array;
int        dim;
int        kind;

{ dd_type result_ddt, val_ddt;
  int size;
  int *ranks, *vals;
  int val_is, result_is;

  size = dalib_array_local_size (val_array);

  dalib_make_array_ddt (&result_ddt, result_array);
  dalib_make_array_ddt (&val_ddt, val_array);
 
  dalib_ddt_is_contiguous (result_ddt, &result_is, &ranks);

  if (!result_is)
     { dalib_internal_error ("grade, rank array not contiguous");
       dalib_stop ();
     }

  dalib_ddt_is_contiguous (val_ddt, &val_is, &vals);

  if (!val_is)
     { dalib_internal_error ("grade, array not contiguous");
       dalib_stop ();
     }

#ifdef DEBUG
  printf ("%d: grade, ascending=%d, size=%d, kind=%d\n", 
          pcb.i, ascending, size, kind);
#endif

  if (kind == 1) dalib_grade_int4  (ranks, vals, size);
  if (kind == 4) dalib_grade_real4 (ranks, (float *) vals, size);
  if (kind == 8) dalib_grade_real8 (ranks, (double *) vals, size);

} /* dalib_grade */

static void dalib_sort (result_array, val_array, dim, kind)

array_info result_array, val_array;
int        dim;
int        kind;

{ dalib_internal_error ("sort_up/down : not available yet");
  dalib_stop ();

} /* dalib_sort */

/**************************************************************************
*                                                                         *
*   general check of arrays for grade_up, grade_down                      *
*                                                                         *
**************************************************************************/

void dalib_grade_check (rank_array, val_array, dim)
array_info *rank_array;
array_info *val_array;
int        *dim;

{
  if (!dalib_is_array_info (*rank_array))
     { dalib_internal_error ("grade_up/down, result is not array info");
       dalib_stop ();
     }
  if (!dalib_is_array_info (*val_array))
     { dalib_internal_error ("grade_up/down, result is not array info");
       dalib_stop ();
     }

  if (!FUNCTION(dalib_present) (dim)) 
     { dalib_internal_error ("grade_up/down : dim must be available");
       dalib_stop ();
     }

} /* dalib_grade_check */

/**************************************************************************
*                                                                         *
*   general check of arrays for sort_up, sort_down                        *
*                                                                         *
**************************************************************************/

void dalib_sort_check (rank_array, val_array, dim)
array_info *rank_array;
array_info *val_array;
int        *dim;

{
  if (!dalib_is_array_info (*rank_array))
     { dalib_internal_error ("sort_up/down, result is not array info");
       dalib_stop ();
     }
  if (!dalib_is_array_info (*val_array))
     { dalib_internal_error ("sort_up/down, result is not array info");
       dalib_stop ();
     }

  if (!FUNCTION(dalib_present) (dim)) 
     { dalib_internal_error ("sort_up/down : dim must be available");
       dalib_stop ();
     }

} /* dalib_sort_check */

/**************************************************************************
*                                                                         *
*   dalib_grade_up                                                        *
*                                                                         *
**************************************************************************/

          /******************************************************
          *   dalib_grade_up_i                                  *
          ******************************************************/

void FUNCTION(dalib_grade_up_i) (result_data, array_data, dim_data,
                                 result_dsp,  array_dsp,  dim_dsp  ) 

void *result_data, *array_data;
int  *dim_data;

array_info *result_dsp, *array_dsp, *dim_dsp;

{ int size;

  ascending = 1;    /* set global value for grading up */

#ifdef CHECK
  dalib_grade_check (result_dsp, array_dsp, dim_data);
#endif

  size = (*array_dsp)->size;

  if (size == 4)

     dalib_grade (*result_dsp, *array_dsp, *dim_data, 1);

   else 

     { dalib_internal_error ("grade_up, only integer*4 allowed");
       dalib_stop ();
     }
 
} /* dalib_grade_up_i */

          /******************************************************
          *   dalib_grade_up_r                                  *
          ******************************************************/

void FUNCTION(dalib_grade_up_r) (result_data, array_data, dim_data,
                                 result_dsp,  array_dsp,  dim_dsp  ) 

void *result_data, *array_data;
int  *dim_data;

array_info *result_dsp, *array_dsp, *dim_dsp;

{ int size;

  ascending = 1;    /* set global value for grading up */

#ifdef CHECK
  dalib_grade_check (result_dsp, array_dsp, dim_data);
#endif

  size = (*array_dsp)->size;

  if (size == 4)

     dalib_grade (*result_dsp, *array_dsp, *dim_data, 4);

   else if (size == 8)

     dalib_grade (*result_dsp, *array_dsp, *dim_data, 8);

   else 

     { dalib_internal_error ("grade_up, only real*4 or real*8 allowed");
       dalib_stop ();
     }
 
} /* dalib_grade_up_r */

/**************************************************************************
*                                                                         *
*   dalib_grade_down                                                      *
*                                                                         *
**************************************************************************/

          /******************************************************
          *   dalib_grade_down_i                                *
          ******************************************************/

void FUNCTION(dalib_grade_down_i) (result_data, array_data, dim_data,
                                   result_dsp,  array_dsp,  dim_dsp  ) 

void *result_data, *array_data;
int  *dim_data;

array_info *result_dsp, *array_dsp, *dim_dsp;

{ int size;

  ascending = -1;    /* set global value for grading down */

#ifdef CHECK
  dalib_grade_check (result_dsp, array_dsp, dim_data);
#endif

  size = (*array_dsp)->size;

  if (size == 4)

     dalib_grade (*result_dsp, *array_dsp, *dim_data, 1);

   else 

     { dalib_internal_error ("grade_down, only integer*4 allowed");
       dalib_stop ();
     }
 
} /* dalib_grade_down_i */

          /******************************************************
          *   dalib_grade_down_r                                *
          ******************************************************/

void FUNCTION(dalib_grade_down_r) (result_data, array_data, dim_data,
                                   result_dsp,  array_dsp,  dim_dsp  ) 

void *result_data, *array_data;
int  *dim_data;

array_info *result_dsp, *array_dsp, *dim_dsp;

{ int size;

  ascending = -1;    /* set global value for grading down */

#ifdef CHECK
  dalib_grade_check (result_dsp, array_dsp, dim_data);
#endif

  size = (*array_dsp)->size;

  if (size == 4)
     dalib_grade (*result_dsp, *array_dsp, *dim_data, 4);
   else if (size == 8)
     dalib_grade (*result_dsp, *array_dsp, *dim_data, 8);
   else 
     { dalib_internal_error ("grade_down, only real*4 or real*8 allowed");
       dalib_stop ();
     }
 
} /* dalib_grade_down_r */

/**************************************************************************
*                                                                         *
*   dalib_sort_up                                                        *
*                                                                         *
**************************************************************************/

          /******************************************************
          *   dalib_sort_up_i                                  *
          ******************************************************/

void FUNCTION(dalib_sort_up_i) (result_data, array_data, dim_data,
                                 result_dsp,  array_dsp,  dim_dsp  ) 

void *result_data, *array_data;
int  *dim_data;

array_info *result_dsp, *array_dsp, *dim_dsp;

{ int size;

  ascending = 1;    /* set global value for grading up */

#ifdef CHECK
  dalib_sort_check (result_dsp, array_dsp, dim_data);
#endif

  size = (*array_dsp)->size;

  if (size == 4)

     dalib_sort (*result_dsp, *array_dsp, *dim_data, 1);

   else 

     { dalib_internal_error ("sort_up, only integer*4 allowed");
       dalib_stop ();
     }
 
} /* dalib_sort_up_i */

          /******************************************************
          *   dalib_sort_up_r                                  *
          ******************************************************/

void FUNCTION(dalib_sort_up_r) (result_data, array_data, dim_data,
                                 result_dsp,  array_dsp,  dim_dsp  ) 

void *result_data, *array_data;
int  *dim_data;

array_info *result_dsp, *array_dsp, *dim_dsp;

{ int size;

  ascending = 1;    /* set global value for grading up */

#ifdef CHECK
  dalib_sort_check (result_dsp, array_dsp, dim_data);
#endif

  size = (*array_dsp)->size;

  if (size == 4)

     dalib_sort (*result_dsp, *array_dsp, *dim_data, 4);

   else if (size == 8)

     dalib_sort (*result_dsp, *array_dsp, *dim_data, 8);

   else 

     { dalib_internal_error ("sort_up, only real*4 or real*8 allowed");
       dalib_stop ();
     }
 
} /* dalib_sort_up_r */

/**************************************************************************
*                                                                         *
*   dalib_sort_down                                                      *
*                                                                         *
**************************************************************************/

          /******************************************************
          *   dalib_sort_down_i                                *
          ******************************************************/

void FUNCTION(dalib_sort_down_i) (result_data, array_data, dim_data,
                                   result_dsp,  array_dsp,  dim_dsp  ) 

void *result_data, *array_data;
int  *dim_data;

array_info *result_dsp, *array_dsp, *dim_dsp;

{ int size;

  ascending = -1;    /* set global value for grading down */

#ifdef CHECK
  dalib_sort_check (result_dsp, array_dsp, dim_data);
#endif

  size = (*array_dsp)->size;

  if (size == 4)

     dalib_sort (*result_dsp, *array_dsp, *dim_data, 1);

   else 

     { dalib_internal_error ("sort_down, only integer*4 allowed");
       dalib_stop ();
     }
 
} /* dalib_sort_down_i */

          /******************************************************
          *   dalib_sort_down_r                                *
          ******************************************************/

void FUNCTION(dalib_sort_down_r) (result_data, array_data, dim_data,
                                   result_dsp,  array_dsp,  dim_dsp  ) 

void *result_data, *array_data;
int  *dim_data;

array_info *result_dsp, *array_dsp, *dim_dsp;

{ int size;

  ascending = -1;    /* set global value for grading down */

#ifdef CHECK
  dalib_sort_check (result_dsp, array_dsp, dim_data);
#endif

  size = (*array_dsp)->size;

  if (size == 4)
     dalib_sort (*result_dsp, *array_dsp, *dim_data, 4);
   else if (size == 8)
     dalib_sort (*result_dsp, *array_dsp, *dim_data, 8);
   else 
     { dalib_internal_error ("sort_down, only real*4 or real*8 allowed");
       dalib_stop ();
     }
 
} /* dalib_sort_down_r */

