/*******************************************************************
*                                                                  *
*  DEFINITIONS : dalib.h                                           *
*                                                                  *
*  global definitions for the runtime system DALIB                 *
*                                                                  *
*  contains:                                                       *
*                                                                  *
*   - global variables, like pcb, trace_flag                       *
*   - global constants like MAX_RANK, MAX_TOPS,                    *
*                                                                  *
*******************************************************************/

#define MAX_DIMENSIONS 7   /* maximal rank for one array               */
#define MAX_RANK       4   /* maximal rank for one topology            */
#define MAX_TOPS      30   /* maximal number of different topologies   */
#define MAX_GROUPS    65   /* should be > MAX_RANK * MAX_TOPS          */

#define NAME_LENGTH   10   /* relevant characters for variables        */
#define MAX_FNAME_LEN 256  /* maximum filename length                  */

#define MAX_TASKS     6    /* maximal nesting of tasks                 */

/*******************************************************************
*                                                                  *
*  array with task identifiers (used for some machines             *
*                                                                  *
*  for many architectures tids[i] == i holds                       *
*                                                                  *
*******************************************************************/

#ifndef SINGLE

#define MAXP  256             /* maximal number of node processes */
#define MAXP1 MAXP+1

extern int tids[MAXP1];

#endif

/*******************************************************************
*                                                                  *
*  Process Control Block, local copy for host and all node proc.   *
*                                                                  *
*******************************************************************/

typedef struct

 { int i, p;            /* 0 <= i <= p = number of processes */
   int p_rank;
   int p2_1, p2_2;
   int p3_1, p3_2, p3_3;

   int mpmd_flag;       /* multiple tasks if mpmd_flag > 0          */

   int trace_flag;      /* trace if trace_flag >      0             */
   int call_flag;       /* trace calls if call_flag > 0             */
   int redist_flag;     /* trace redistributions if redist_flag > 0 */
   int comm_flag;       /* generate communication statistics if = 1 */
   int time_flag;       /* generate time listings for subprograms   */

 } process_control_block;

extern process_control_block pcb;

/*******************************************************************
*                                                                  *
*  Timing Statistics Information                                   *
*                                                                  *
*******************************************************************/

extern int time_statistics_collect;  /* set to 1 for collection of info
                                        as not always collected          */


/*******************************************************************
*                                                                  *
*  Communication Statisitcs Information                            *
*                                                                  *
*******************************************************************/

extern int comm_statistics_collect;  /* set to 1 for collection of info
                                        as not always collected          */

/*******************************************************************
*                                                                  *
*  Internal Data Types                                             *
*                                                                  *
*******************************************************************/

typedef void dalib_routine();  /* pointer to a subroutine */

     /***********************************************
     *                                              *
     *    derived data types                        *
     *                                              *
     ***********************************************/

typedef union ddt_info *dd_type;
#define NO_DDT ((dd_type) 0)

     /***********************************************
     *                                              *
     *    schedule                                  *
     *                                              *
     ***********************************************/

typedef struct indirect_schedule *schedule_ptr;
#define NO_SCHEDULE ((schedule_ptr) 0)

typedef struct remote_schedule *remote_ptr;
#define NO_RSCHEDULE ((remote_ptr) 0)

     /***********************************************
     *                                              *
     *    arrays                                    *
     *                                              *
     ***********************************************/

typedef struct ArrayRecord *array_info;
#define NO_ARRAY ((array_info) 0)

     /***********************************************
     *                                              *
     *    sections                                  *
     *                                              *
     ***********************************************/

typedef struct SectionRecord *section_info;
#define NO_SECTION ((section_info) 0)

     /***********************************************
     *                                              *
     *    distributions                             *
     *                                              *
     ***********************************************/

typedef struct DistributionRecord *Distribution;
#define NO_DISTRIBUTION ((Distribution) 0)

typedef union DistDimRecord *DistDim;
#define NO_DIST_DIM  ((DistDim) 0)

     /***********************************************
     *                                              *
     *    alignments                                *
     *                                              *
     ***********************************************/

typedef struct AlignmentRecord *Alignment;
#define NO_ALIGNMENT ((Alignment) 0)

     /***********************************************
     *                                              *
     *    distributed shared data                   *
     *                                              *
     ***********************************************/

typedef struct SharedRecord *Shared;
#define NO_SHARED ((Shared) 0)

     /***********************************************
     *                                              *
     *    remote memory access                      *
     *                                              *
     ***********************************************/

typedef struct RemoteRecord *Remote;
#define NO_REMOTE ((Remote) 0)

     /***********************************************
     *                                              *
     *    unstructured schedules                    *
     *                                              *
     ***********************************************/

typedef struct indirect_schedule *SchedulePtr;

     /***********************************************
     *                                              *
     *  DimInfo  :   info about one dimension       *
     *                                              *
     *   global_size [0]  :  global lower bound     *
     *   global_size [1]  :  global upper bound     *
     *                                              *
     *  map_flag = 0 (block distribution)           *
     *                                              *
     *   local_size [0] : local_size [1] local part *
     *   is a subrange of the global range          *
     *                                              *
     *  map_flag = 1 (cyclic(n), indirect distrib)  *
     *                                              *
     *   there is a mapping for global indexes      *
     *   map : [global_lb:global_ub] -> 0..size-1   *
     *                                              *
     *   local_size [0] : local_size [1] local part *
     *   is a subrange of 0..size-1                 *
     *                                              *
     ***********************************************/

typedef struct

  { int global_size [2];
    int local_size  [3];
    int shadow      [2];
    int map_flag;         
  } DimInfo;

     /***********************************************
     *                                              *
     *   dsp_status_flag   (values)                 *
     *                                              *
     ***********************************************/

   /* DSP_UNDEFINED : descriptor created, but nothing defined 
      DSP_DEFINED   : array_define has been called
      DSP_OWN_DATA  : own data has been allocated
      DSP_PTR_DATA  : descriptor points to other data

      note : consistent among all processors

   */

#define DSP_UNDEFINED 0
#define DSP_DEFINED   1
#define DSP_OWN_DATA  2
#define DSP_PTR_DATA  3

     /***********************************************
     *                                              *
     *    shared data structure for arrays          *
     *                                              *
     ***********************************************/

struct ArrayRecord

 { int ident;                            /* identification of this record   */
   int size;                             /* size in bytes for one elem      */
   int rank;                             /* rank of the array               */
   int dsp_kind;                         /* 0: kIS_STATIC, 1: kIS_DYNAMIC   */
   int dynamic;                          /* set to 1 if HPF_DYNMAMIC        */
   int global_addressing;                /* global addresses also if mapped */
   int trace;                            /* array/template has trace attrib */
   int shadow_status;                    /* 1 if all shadow is valid        */
   char name[NAME_LENGTH];               /* name of the variable (error!)   */
   int dsp_status_flag;                  /* values see above                */
   int ref_counter;                      /* myself + aligned arrays         */
   int static_offset;                    /* offset for static pointer       */
   int reuse_bits;                       /* will be set to 0 for updates    */
                                         /* bit reuse_bits[MAX_SCHEDULES]   */
   unsigned char *c_data;                /* pointer to the allocated data   */
   unsigned char *f_data;                /* pointer aligned to FORTRAN data */
   int           n_data;                 /* number of bytes allocated       */
   Distribution DistributeInfo;          /* describing the distribution     */
   Alignment  AlignInfo;                  /* describing the alignment        */
   Shared     SharedInfo;                 /* describing how data is shared   */
   Remote     RemoteInfo;                 /* describing remote access        */
   array_info GlobalInfo;                 /* global array in LOCAL routines  */
   DimInfo    dimensions[MAX_DIMENSIONS]; /* info about the dims             */
 };

#define NO_DATA ((unsigned char *) 0)

     /***********************************************
     *                                              *
     *    shared data structure for sections        *
     *                                              *
     ***********************************************/

typedef struct

  { int is_range;           /* 0 stands for single element   */
    int global_range [3];
    int local_range  [3];
  } SecDimInfo;

struct SectionRecord

 { int ident;              /* identification of this record */
   array_info array_id;    /* pointer to the full array descriptor */
   SecDimInfo dimensions[MAX_DIMENSIONS]; /* info about the section dims   */
 };

     /***********************************************
     *                                              *
     *    kind of distributions for dimensions      *
     *                                              *
     *     (use same constants in ADAPTOR tool)     *
     *                                              *
     ***********************************************/

#define kANY_TOPOLOGY    -2

#define kSERIAL_DIM       0
#define kBLOCK_DIM        1
#define kCYCLIC_DIM       2
#define kGEN_BLOCK_DIM    3
#define kINDIRECT_DIM     4
#define kARBITRARY_DIM    5

#define kALIGNED_DIM      6
#define kMAPPED_DIM       7
#define kREPLICATED_DIM   8
#define kEMBEDDED_DIM     9
#define kSOURCE_DIM      10

#define kANY_BLOCK_DIM       11
#define kANY_CYCLIC_DIM      12
#define kANY_GEN_BLOCK_DIM   13
#define kANY_INDIRECT_DIM    14
#define kANY_ARBITRARY_DIM   15

#define kANY_DISTRIBUTED_DIM 16

     /***********************************************
     *                                              *
     *    STRUCTURE IDENTIFICATORS                  *
     *                                              *
     ***********************************************/

#define ARRAY_IDENTIFICATION    1919
#define SECTION_IDENTIFICATION  1999

     /***********************************************
     *                                              *
     *    pointer to the name of the program        *
     *                                              *
     ***********************************************/

extern char dalib_program_name[MAX_FNAME_LEN];

     /***********************************************
     *                                              *
     *    some prototypes (might be useful)         *
     *                                              *
     ***********************************************/

extern char *dalib_malloc ();
extern int  *dalib_int_malloc ();
extern array_info dalib_array_copy_dsp ();

     /***********************************************
     *                                              *
     *    some machine specific definitions         *
     *                                              *
     ***********************************************/

#if defined(CS) 
#define INDEX(a,b) index(a,b)
#define RINDEX(a,b) rindex(a,b)
#else
#define INDEX(a,b) strchr(a,b)
#define RINDEX(a,b) strrchr(a,b)
#endif

#define F_TRUE 1

     /****************************************************************
     *                                                               *
     *  definitions for TRACING (VT)                                 *
     *                                                               *
     *  - attention : do not use same names as FORTRAN routines      *
     *                (gives conflicts on CRAY systems)              *
     *                                                               *
     ****************************************************************/

#define TRACE_GROUP_BCAST    301
#define TRACE_GROUP_REDUCE   302

#define TRACE_OVERLAP_UPDATE 303

      /*****************************************************
      *                                                    *
      *  Constants used for different kinds of desrciptors *
      *                                                    *
      *    -> used for ArrayRecord.dsp_kind                *
      *                                                    *
      *****************************************************/

#define kIS_STATIC       0
#define kIS_DYNAMIC      1
