/**************************************************************************
*                                                                         *
*  Author      : Resi Hoever-Klier, GMD, SCAI.LAB                         *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Jul 97                                                   *
*  Last Update : May 98                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : commstat.m4                                              *
*                                                                         *
*  Function: Generating communication statistics info about the           *
*            communication between the processors.                        *
*                                                                         *
*            Communication statistics will be collected, if the executable*
*            is called with -comm.                                        *
*            Info on sent and received messages can be switched off and   *
*            on by calling dalib_commstat_off() and dalib_commstat_on()   *
*            respectively in the FORTRAN program.                         *
*            Suggestion at program start: dalib_commstat_on().            *
*                                                                         *
*                                                                         *
*  Export :  internal Interface                                           *
*  ============================                                           *
*                                                                         *
*   void dalib_init_commstat (void)                                       *
*                                                                         *
*      - called if executable was called with -comm or with -commstat     *
*      - enables collection of communication statistics info              *
*                                                                         *
*   void FUNCTION dalib_commstat_on (void)                                *
*                                                                         *
*      - called in FORTRAN program to restart collection of communication *
*        statistics                                                       *
*                                                                         *
*   void dalib_commstat_entry_send (sender, receiver, length)             *
*                                                                         *
*      - called at all dalib send operations unless them to the own proc  *
*        if dalib_commstat_on.                                            *
*                                                                         *
*   void dalib_commstat_entry_receive (receiver, sender, msg_length)      *
*                                                                         *
*      - called at all dalib receive operations unless them from the own  *
*        proc if dalib_commstat_on.                                       *
*                                                                         *
*   void FUNCTION dalib_commstat_off (void)                               *
*                                                                         *
*      - called in FORTRAN program to stop collection of communication    *
*        statistics                                                       *
*                                                                         *
*   void dalib_collect_commstat (void)                                    *
*                                                                         *
*      - called at end of program run if executable was called with -comm *
*        or with -commstat                                                *
*      - collects and prints out communication statistics info            *
*                                                                         *
*                                                                         *
*   Related dalib-modules:                                                *
*                                                                         *
*   dalib.m4 dalib.h: definition of pcb.comm_flag  and                    *
*                                   comm_statistics_collect               *
*                                                                         *
*   arguments.m4: sets pcb.comm_flag if executable called with -comm      *
*                                                                         *
*   initall.m4: dalib_init: call of dalib_init_commstat if pcb.comm_flag  *
*                           unset of comm_statistics_collect              *
*               dalib_exit: call of dalib_collect_commstat                *
*                                                                         *
*   mailbox.m4: if pcb.comm_flag and comm_statistics_collect both set:    *
*               dalib_send: call of dalib_commstat_entry_send             *
*               dalib_receive: dalib_commstat_entry_receive               *
*                                                                         *
**************************************************************************/
  
#include <stdio.h>
#include "dalib.h"

#undef  DEBUG


#define MAX(A,B)  ((A) > (B) ? (A) : (B))
#define MIN(A,B)  ((A) < (B) ? (A) : (B))

static char commstatfilename[100]; /* file for commumication statistics info */
static FILE *commstatfile;           


typedef struct

   { int num_sends;
     float suml_sends;
     float minl_sends;
     float maxl_sends;
     float averagel_sends;
     int num_recs;
     float suml_recs;
     float minl_recs;
     float maxl_recs;
     float averagel_recs;
   } commstat_info;

commstat_info *my_comm;        /* to hold info for each process */
int my_comm_size;              /* space allocated for my_comm */
commstat_info *commstat_all;   /* to collect info of all processes */
int commstat_all_size;         /* space allocated for commstat_all */
commstat_info **commstat;      /* to reach each processes' info */
int commstat_size;             /* space allocated for commstat */

int numinfos;                  /* num of commstat_info records to 
                                  keep by each proc (pcb.p+1) */


/*******************************************************************
*                                                                  *
*  dalib_print_commstat (dest,comm,header)                         *
*                                                                  *
*  print communication statistics information in array comm with   *
*  header header into file dest.                                   *
*                                                                  *
*******************************************************************/

static void dalib_print_commstat (dest,comm,header)
FILE *dest;
commstat_info *comm;
char *header;

{
 char *line;
 char cont[20];
 char nid[6];
 int divisor;                     /*for computation of fildwidth */
 int max;                         /* fieldwidth */
 int maxlinelen;
 int i,j;
 float fval;                      /* for use in macro PRTFLOATLINE */

#define PRTFLOATLINE(DEST,INFTXT,INFO)           \
        sprintf (line,"%s%s",INFTXT,nid);          \
        for (j = 1; j <= pcb.p; j++)             \
           { fval = comm[j].INFO / 1024.0;       \
             sprintf (cont," %*.3f",max,fval);   \
             sprintf (line,"%s%s",line,cont);    \
           }                                     \
        fval = comm[0].INFO / 1024.0;            \
        sprintf (cont," %*.3f",max,fval);        \
        fprintf (DEST,"%s%s\n",line,cont);

#define PRTINTLINE(DEST,INFTXT,INFO)                 \
        sprintf (line,"%s%s",INFTXT,nid);              \
        for (j = 1; j <= pcb.p; j++)                 \
           { sprintf (cont," %*d",max,comm[j].INFO); \
             sprintf (line,"%s%s",line,cont);        \
           }                                         \
        sprintf (cont," %*d",max,comm[0].INFO);      \
        fprintf (DEST,"%s%s\n",line,cont);


#ifdef DEBUG
printf("%d: dalib_print_commstat:\n",pcb.i);
#endif

  strcpy (nid,"");

#ifdef DEBUG
  sprintf (nid,"%d:",pcb.i);
#endif

  fprintf (dest,"%s%s\n",nid,header);   /* header */

/* compute fieldwidths */

  divisor = 10;            /* first suggestion: less than 10 Mbytes */
  max = 5;                 /* print-format n.nnn */
  while ((i = MAX(comm[0].suml_sends,comm[0].suml_recs) / 1024 / divisor) > 0)
       { divisor = divisor *10;
         max++;
       }

  maxlinelen = 14 + 5 + (pcb.p + 1 ) * max;   /* descript-string + nid (sugg.) + (pcb.p+1) * data */

  line = (char *) dalib_malloc(maxlinelen,"dalib_print_commstat");

  sprintf (line,"%scomm. with:   ",nid);      /* first line */
  for (j = 1; j <= pcb.p; j++)
     { sprintf (cont,"proc%d",j);
       sprintf (line,"%s %*s",line,max,cont);
     }
  sprintf (cont,"all");
  fprintf (dest,"%s %*s\n",line,max,cont);

/* print lines */

  PRTINTLINE(dest,"# sends to:   ",num_sends);
  PRTFLOATLINE(dest,"total Mbytes: ",suml_sends);
  PRTFLOATLINE(dest,"minl.  sent:  ",minl_sends);
  PRTFLOATLINE(dest,"maxl.  sent:  ",maxl_sends);
  PRTFLOATLINE(dest,"averl. sent:  ",averagel_sends);
  PRTINTLINE(dest,"# recs from:  ",num_recs);
  PRTFLOATLINE(dest,"total Mbytes: ",suml_recs);
  PRTFLOATLINE(dest,"minl.  rec'd: ",minl_recs);
  PRTFLOATLINE(dest,"maxl.  rec'd: ",maxl_recs);
  PRTFLOATLINE(dest,"averl. rec'd: ",averagel_recs);


  dalib_free (line,maxlinelen);
  
} /* dalib_print_commstat */


/*******************************************************************
*                                                                  *
*  dalib_print_proc_to_proc_communications (void)                  *
*                                                                  *
*******************************************************************/

static void dalib_print_proc_to_proc_communications (void)

{
 char *header;
 char *line;
 char cont[20];
 int divisor;          /* for comp. of fieldwidth */
 int max1,max2;        /* fieldwidths */
 int maxlinelen;
 int i,j,k;
 float fval;           /* for print macros */

#define PRTFLINE(DEST,INFO)                            \
        sprintf (cont,"all");                          \
        sprintf (line,"%-*s",max2,cont);               \
        for (j = 0; j <= pcb.p; j++)                   \
           {                                           \
            for (k = 1; k <= pcb.p; k++)               \
               {                                       \
                fval = commstat[j][k].INFO / 1024.0;   \
                sprintf (cont,"%*.3f",max1,fval);      \
                sprintf (line,"%s %s",line,cont);      \
               }                                       \
            fval = commstat[j][0].INFO / 1024.0;       \
            sprintf (cont,"%*.3f",max1,fval);          \
            fprintf (DEST,"%s %s\n",line,cont);        \
            sprintf (cont,"proc%d",j+1);               \
            sprintf (line,"%-*s",max2,cont);           \
           }


#define PRTILINE(DEST,INFO)                                    \
        sprintf (cont,"all");                                  \
        sprintf (line,"%-*s",max2,cont);                       \
        for (j = 0; j <= pcb.p; j++)                           \
           {                                                   \
            for (k = 1; k <= pcb.p; k++)                       \
               {                                               \
                sprintf (cont,"%*d",max1,commstat[j][k].INFO); \
                sprintf (line,"%s %s",line,cont);              \
               }                                               \
            sprintf (cont,"%*d",max1,commstat[j][0].INFO);     \
            fprintf (DEST,"%s %s\n",line,cont);                \
            sprintf (cont,"proc%d",j+1);                       \
            sprintf (line,"%-*s",max2,cont);                   \
           }


#ifdef DEBUG
printf("%d: dalib_print_proc_to_proc_communications",pcb.i);
#endif

/* compute fieldwidth for messagelength */

  divisor = 10;             /* first suggestion: less than 10 Mbytes */
  max1 = 5;                 /* data-format: n.nnn */
  while ((i = MAX(commstat[0][0].suml_sends,commstat[0][0].suml_recs) / 1024 / divisor) > 0)
       { divisor = divisor *10;
         max1++;
       }


/* compute fieldwidth for num of procs */

  divisor = 10;             /* first suggestion: less than 10 Mbytes */
  max2 = 5;                 /* strlen("proc") + 1  */
  while ((i = pcb.p / divisor) > 0)
       { divisor = divisor *10;
         max2++;
       }

  max1 = MAX(max1,max2);      /* for many proc but little data */

  maxlinelen = max2 + (pcb.p + 1 ) * (max1 + 1);   /* procxx + (pcb.p+1) * data */

  header = (char *) dalib_malloc(maxlinelen,"dalib_print_proc_to_proc_comm_h");
  line   = (char *) dalib_malloc(maxlinelen,"dalib_print_proc_to_proc_comm_l");

  sprintf (cont,"to:");
  sprintf (header,"%-*s",max2,cont);
  for (j = 1; j <= pcb.p; j++)
     { sprintf(cont,"proc%d",j);
       sprintf(header,"%s %*s",header,max1,cont);
     }
  sprintf(cont,"all");
  sprintf (header,"%s %*s",header,max1,cont);

 
   printf ("\n Communication Statistics Summary for program %s (%d procs):\n",
           dalib_program_name,pcb.p);  /* the only info going to stdout */
   printf (         " Number of Mega-Bytes sent by all procs:\n%s\n",header);
  PRTFLINE(stdout,suml_sends); 
   printf("... detailed communication statistics in file %s.\n",commstatfilename);


  fprintf(commstatfile," Number of Mega-Bytes sent by all procs:\n%s\n",header);
  PRTFLINE(commstatfile,suml_sends);
  
  fprintf (commstatfile," Number of sends of all procs:\n%s\n",header);
  PRTILINE(commstatfile,num_sends);


  sprintf (cont,"from:");
  sprintf (header,"%-*s",max2,cont);
  for (j = 1; j <= pcb.p; j++)
     { sprintf(cont,"proc%d",j);
       sprintf(header,"%s %*s",header,max1,cont);
     }
  sprintf(cont,"all");
  sprintf (header,"%s %*s",header,max1,cont);

  fprintf(commstatfile," Number of Mega-Bytes received by all procs:\n%s\n",header);
  PRTFLINE(commstatfile,suml_recs);

  fprintf (commstatfile," Number of receives of all procs:\n%s\n",header);
  PRTILINE(commstatfile,num_recs);


  dalib_free (header,maxlinelen);
  dalib_free (line,maxlinelen);

} /* dalib_print_proc_to_proc_communications */


/*******************************************************************
*                                                                  *
*  commstat_info dalib_clear_commstat (comm)                       *
*                                                                  *
*******************************************************************/

static commstat_info *dalib_clear_commstat (comm)
commstat_info *comm;

{ int j;

#ifdef DEBUG
  printf("%d: dalib_clear_commstat\n",pcb.i);
#endif

  for (j = 0; j <= pcb.p; j++)
     {
      comm[j].num_sends      = 0;
      comm[j].suml_sends     = 0;
      comm[j].minl_sends     = 0;
      comm[j].maxl_sends     = 0;
      comm[j].averagel_sends = 0;
      comm[j].num_recs       = 0;
      comm[j].suml_recs      = 0;
      comm[j].minl_recs      = 0;
      comm[j].maxl_recs      = 0;
      comm[j].averagel_recs  = 0;
     }

  return (comm);

 }/* dalib_clear_commstat */


/*******************************************************************
*                                                                  *
*  dalib_init_commstat ()                                          *
*                                                                  *
* Local communication statistics:                                  *
* my_comm[proc]: communication with proc proc                      *
* my_comm[0]:    summary of communication with all procs           *
*                                                                  *
* Global communication statistics:                                 *
* commstat[proc]: array of communication statistics (my_comm)      *
*                 of proc proc                                     *
* commstat[0]   : summary communication statistics over all procs  *
*                                                                  *
*******************************************************************/

void dalib_init_commstat ()

{ int j,k,m;
  char header[60];

#ifdef DEBUG
  printf("%d: dalib_init_commstat:\n",pcb.i);
#endif

  numinfos = pcb.p + 1;

  my_comm_size = 1 * numinfos * sizeof(commstat_info); /* for my proc only */
  my_comm = (commstat_info *) dalib_malloc (my_comm_size,"dalib_init_commstat");

  dalib_clear_commstat (my_comm);
     
#ifdef DEBUG
    printf("%d: dalib_init_commstat: cleared my_comm\n",pcb.i);
#endif

  if (pcb.i == 1)
    {
     strcpy (commstatfilename,dalib_program_name);
     strcat (commstatfilename,".commstat");

     if ((commstatfile = fopen(commstatfilename,"w")) == NULL)
        dalib_internal_error("Unable to open communications file for writing");

#ifdef DEBUG
     printf("opened communications file %s\n",commstatfilename);
#endif


     commstat_all_size = numinfos * numinfos  * sizeof(commstat_info); 
     commstat_all = (commstat_info *) dalib_malloc(commstat_all_size,"dalib_init_commstat");

     commstat_size = numinfos * sizeof(commstat_info*); 
     commstat = (commstat_info**) dalib_malloc (commstat_size,"dalib_init_commstat");

 
     for (j = 0; j <= pcb.p; j++)
           {
            m = j * numinfos;
            commstat[j] = commstat_all + m;
            dalib_clear_commstat(commstat[j]);

#ifdef DEBUG
    printf("%d: dalib_init_commstat: cleared commstat[%d]\n",pcb.i,j);
#endif

           }
    }


  comm_statistics_collect = 1; /* collect communication statistics info from now on */
 

} /*dalib_init_commstat */


/*************************************************************************
*   void FUNCTION dalib_commstat_on (void)                               *
*                                                                        *
*   called in FORTRAN program to restart collection of communication     *
*   statistics                                                           *
*                                                                        *
*************************************************************************/
void FUNCTION(dalib_commstat_on) ()

{

#ifdef DEBUG
  printf("%d: dalib_commstat_on:\n",pcb.i);
#endif

  if (pcb.comm_flag == 0)     return;   /* executable not called with -comm */

  comm_statistics_collect = 1;             /* set */

  return;

} /* dalib_commstat_on */


/*******************************************************************
*                                                                  *
*  void dalib_commstat_entry_send (sender, receiver, length)       *
*                                                                  *
*  - collect communication statistics of send                      *
*                                                                  *
*******************************************************************/

void dalib_commstat_entry_send (sender, receiver, length)
int sender;
int receiver;
int length;

{
 int i;
 char header[60];

#ifdef DEBUG
 printf("%d:dalib_commstat_entry_send: from: %d; to: %d; len: %d\n",pcb.i,sender,receiver,length);
#endif

/* count the sends to receiver receiver */

 if (my_comm[receiver].num_sends == 0)
   { my_comm[receiver].num_sends = 1;
     my_comm[receiver].suml_sends = length;
     my_comm[receiver].minl_sends = length;
     my_comm[receiver].maxl_sends = length;
     my_comm[receiver].averagel_sends = length;
   }

 else
   { my_comm[receiver].num_sends += 1;
     my_comm[receiver].suml_sends += length;
     my_comm[receiver].minl_sends = MIN(my_comm[receiver].minl_sends,length);
     my_comm[receiver].maxl_sends = MAX(my_comm[receiver].maxl_sends,length);
     my_comm[receiver].averagel_sends = my_comm[receiver].suml_sends / my_comm[receiver].num_sends;
   }


/* count the summary of all sends */

 if (my_comm[0].num_sends == 0)
   { my_comm[0].num_sends = 1;
     my_comm[0].suml_sends = length;
     my_comm[0].minl_sends = length;
     my_comm[0].maxl_sends = length;
     my_comm[0].averagel_sends = length;
   }

 else
   { my_comm[0].num_sends += 1;
     my_comm[0].suml_sends += length;
     my_comm[0].minl_sends = MIN(my_comm[0].minl_sends,length);
     my_comm[0].maxl_sends = MAX(my_comm[0].maxl_sends,length);
     my_comm[0].averagel_sends = my_comm[0].suml_sends / my_comm[0].num_sends;
   }

} /* dalib_commstat_entry_send */


/*******************************************************************
*                                                                  *
*  void dalib_commstat_entry_receive (receiver, sender, length)    *
*                                                                  *
*  - collect communication statistics of receive                   *
*                                                                  *
*******************************************************************/

void dalib_commstat_entry_receive (receiver, sender, length)
int receiver;
int sender;
int length;

{
int i;
int j;
char header[60];

#ifdef DEBUG
 printf("%d:dalib_commstat_entry_receive: %d; from: %d; len: %d\n",pcb.i,receiver,sender,length);
#endif

/* count the receives from sender sender */

 if (my_comm[sender].num_recs == 0)
   { my_comm[sender].num_recs  = 1;
     my_comm[sender].suml_recs = length;
     my_comm[sender].minl_recs = length;
     my_comm[sender].maxl_recs = length;
     my_comm[sender].averagel_recs = length;
   }

 else
   { my_comm[sender].num_recs += 1;
     my_comm[sender].suml_recs += length;
     my_comm[sender].minl_recs = MIN(my_comm[sender].minl_recs,length);
     my_comm[sender].maxl_recs = MAX(my_comm[sender].maxl_recs,length);
     my_comm[sender].averagel_recs = my_comm[sender].suml_recs / my_comm[sender].num_recs;
   }

/* count the summary of all receives */

 if (my_comm[0].num_recs == 0)
   { my_comm[0].num_recs  = 1;
     my_comm[0].suml_recs = length;
     my_comm[0].minl_recs = length;
     my_comm[0].maxl_recs = length;
     my_comm[0].averagel_recs = length;
   }

 else
   { my_comm[0].num_recs += 1;
     my_comm[0].suml_recs += length;
     my_comm[0].minl_recs = MIN(my_comm[0].minl_recs,length);
     my_comm[0].maxl_recs = MAX(my_comm[0].maxl_recs,length);
     my_comm[0].averagel_recs = my_comm[0].suml_recs / my_comm[0].num_recs;
   }

} /* dalib_commstat_entry_receive */


/***********************************************************************
*   void FUNCTION dalib_commstat_off (void)                            *
*                                                                      *
*   called in FORTRAN program to stop collection of communication      *
*   statistics.                                                        *
*                                                                      *
***********************************************************************/
void FUNCTION(dalib_commstat_off) ()

{
 int i;

#ifdef DEBUG
  printf("%d: dalib_commstat_off:\n",pcb.i);
#endif

  if (pcb.comm_flag == 0)   return;   /* executable called without -comm */

  comm_statistics_collect = 0;             /* unset */

  return;

} /* dalib_commstat_off */


/*******************************************************************
*                                                                  *
*  dalib_add_to_summary (commstat[j])                              *
*                                                                  *
*******************************************************************/

static void  dalib_add_to_summary(comm)
commstat_info *comm;

{
  int j,k,l;
  int length;
  char header[60];

#ifdef DEBUG
  printf("%d: dalib_add_to_summary:\n",pcb.i);
#endif

  for (j = 0; j <= pcb.p; j++)
     {   
       commstat[0][j].num_sends     += comm[j].num_sends;
       commstat[0][j].suml_sends    += comm[j].suml_sends;

       if (comm[j].minl_sends != 0)
          if (commstat[0][j].minl_sends == 0)
               commstat[0][j].minl_sends = comm[j].minl_sends;
            else
               commstat[0][j].minl_sends = MIN(commstat[0][j].minl_sends,comm[j].minl_sends);

          commstat[0][j].maxl_sends = MAX(commstat[0][j].maxl_sends,comm[j].maxl_sends);

       commstat[0][j].num_recs      += comm[j].num_recs;
       commstat[0][j].suml_recs     += comm[j].suml_recs;

       if (comm[j].minl_recs != 0)
          if (commstat[0][j].minl_recs == 0)
               commstat[0][j].minl_recs = comm[j].minl_recs;
            else
               commstat[0][j].minl_recs =  MIN(commstat[0][j].minl_recs,comm[j].minl_recs);

          commstat[0][j].maxl_recs =  MAX(commstat[0][j].maxl_recs,comm[j].maxl_recs);
     
	}


} /*dalib_add_to_summary */


/*******************************************************************
*                                                                  *
*  collection of communication statistics info                     *
*                                                                  *
* called by FUNCTION dalib_exit (initall.m4)                       *
*                                                                  *
*******************************************************************/

void dalib_collect_commstat()

{
  int j,k,l;
  int length;
  char header[60];

#ifdef DEBUG
  printf("%d: dalib_collect_commstat:\n",pcb.i);
  sprintf(header,"starting to send my_comm");
  dalib_print_commstat(stdout,my_comm,header);
#endif
  

  comm_statistics_collect = 0;  /*stop collection of communication statistics*/
                                /* already done in dalib_exit() */

  if (pcb.i > 1)
    {
     dalib_send (1, my_comm, numinfos * sizeof(commstat_info),1);

#ifdef DEBUG
     printf("%d: dalib_collect_commstat: SENT MY_STAT\n",pcb.i);
#endif
    }

  if (pcb.i == 1)    /* collect communication statistics */
    {

     for (j = 0; j <= pcb.p; j++)
        commstat[0][j] = commstat[1][j] = my_comm[j];


#ifdef DEBUG  
     printf("%d: collecting communication statistics of %d processes:\n",pcb.i,pcb.p-1);
#endif

     for (j = 2; j <= pcb.p; j++)
        {
#ifdef DEBUG
         printf("%d:WAITING FOR STATISTICS OF PROCESS no. %d.\n",pcb.i,j);
#endif

         dalib_receive (j, commstat[j], numinfos * sizeof(commstat_info));

#ifdef DEBUG
         sprintf(header,"GOT STATISTICS OF PROCESS no. %d:",j);
         dalib_print_commstat(stdout,commstat[j],header);
#endif
         
         dalib_add_to_summary(commstat[j]);

        }

  
     /* compute average lengths of sends and recs */

     for (j = 0; j <= pcb.p; j++)
        {     
         if (commstat[0][j].num_sends == 0)
            commstat[0][j].averagel_sends = 0;
         else
            commstat[0][j].averagel_sends = commstat[0][j].suml_sends / commstat[0][j].num_sends;
         if (commstat[0][j].num_recs == 0)
            commstat[0][j].averagel_recs = 0;
         else
            commstat[0][j].averagel_recs  = commstat[0][j].suml_recs  / commstat[0][j].num_recs;
        }
 

     /* print communication statistics */
   
     dalib_print_proc_to_proc_communications ();

     sprintf (header,"Communication statistics summary of all procs:");
     dalib_print_commstat(commstatfile,commstat[0],header);
 
     for (j = 1; j<= pcb.p; j++)
        {    
         sprintf (header,"Communication statistics of proc # %d:",j);
         dalib_print_commstat(commstatfile,commstat[j],header);
        }

     fclose (commstatfile);

     dalib_free (commstat,commstat_size);
     dalib_free (commstat_all,commstat_all_size);

     } /* if (pcb.i == 1) */

  dalib_free (my_comm,my_comm_size);

} /* dalib_collect_commstat */
