/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.WR                         *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Jan 92                                                   *
*  Last Update : Mar 98                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  MODULE : broadcast.m4                                                  *
*                                                                         *
*  Function: Realization of Broadcast Operations                          *
*                                                                         *
*  Export :  DALIB                                                        *
*  ===============                                                        *
*                                                                         *
*   void dalib_contex_broadcast (char *data, int size, int sendpid)       *
*                                                                         *
*   void dalib_top_brodcast (int topid, int dim, char *data, int size)    *
*                                                                         *
*  Export :  FORTRAN Interface                                            *
*  ===========================                                            *
*                                                                         *
*   void dalib_brodcast (char *data, int *size, int *sendpid)             *
*                                                                         *
* void FUNCTION(dalib_array_broadcast) (array_id, int *dim, int *ind_id, * 
*                                char *data, int *size           )        *
*                                                                         *
*  IMPORT :                                                               *
*                                                                         *
*    - dalib_top_subgroup   form topology                                 *
*    - dalib_group_bcast from groups                                      *
*                                                                         *
*  CHANGES:                                                               *
*                                                                         *
*    - broadcast among all processors only within context group           *
*                                                                         *
**************************************************************************/

#include "dalib.h"

# undef DEBUG

/*************************************************************************
*                                                                        *
*  void dalib_context_broadcast (char *data, int size, int pid)          *
*                                                                        *
*    - broadcast in the current subgroup                                 *
*    - pid is absolute processor id (not context position)               *
*                                                                        *
*  void dalib_all_broadcast (char *data, int size, int j)                *
*                                                                        *
*    - broadcast for all processes/processors (needed for init)          *
*                                                                        *
*************************************************************************/

void dalib_context_broadcast (data, size, pid)

char *data;     /* pointer to the data element          */
int pid, size;  /* process pid sends data of size bytes */

{ int group_id;
  int group_pos;

  group_id = dalib_context_group ();

  /* get the position in the group that is contiguous, step == 1 */

  group_pos = dalib_group_position (group_id, pid);

  dalib_group_bcast (data, size, group_pos, group_id);

} /* dalib_context_broadcast */

/*******************************************************************
*                                                                  *
*  void dalib_top_broadcast (int topid, int dim,                   *
*                            unsigned char *data, int size)        *
*                                                                  *
*******************************************************************/

void dalib_all_broadcast (data, size, pid)

char *data;   /* pointer to the data element */
int pid, size;  /* process pid sends data of size bytes */

{ int group_id;

  group_id = dalib_group_all ();
  dalib_group_bcast (data, size, pid, group_id);

} /* dalib_all_broadcast */

/*******************************************************************
*                                                                  *
*  void dalib_top_broadcast (int topid, int dim,                   *
*                            unsigned char *data, int size)        *
*                                                                  *
*******************************************************************/

void dalib_top_broadcast (topid, dim, data, size)

unsigned char *data; 
int      topid, dim, size;

{ int group_id, source;

  if (topid == 0)
     group_id = dalib_context_group ();
   else
     group_id = dalib_top_subgroup (topid, dim);

  source = 1;

  dalib_group_bcast (data, size, source, group_id);

} /* dalib_top_broadcast */

void dalib_top_broadcast_ddt (topid, dim, ddt)
dd_type  ddt;
int      topid, dim;

{ unsigned char *mem;
  int  NId, NP;
  int  size, is_contiguous;

  dalib_ddt_get_size (ddt, &size);
  if (size == 0) return;
  dalib_ddt_is_contiguous (ddt, &is_contiguous, &mem);

  if (is_contiguous)

     dalib_top_broadcast (topid, dim, mem, size);

   else

     { mem = (unsigned char *) dalib_malloc (size, "dalib_top_broadcast_ddt");
       dalib_top_info (topid, dim, &NP, &NId);
       if (NId == 1)
         dalib_ddt_pack (mem, ddt);
       dalib_top_broadcast (topid, dim, mem, size);
       if (NId != 1)
         dalib_ddt_unpack (ddt, mem, 0);  /* 0 for copy operation */
       dalib_free (mem, size);
     }

} /* dalib_top_broadcast_ddt */

/*******************************************************************
*                                                                  *
*  FORTRAN - Interface                                             *
*                                                                  *
*    dalib_broadcast (data, size, sendpid)                         *
*      - broadcast to host and all node from process sendpid       *
*                                                                  *
*    dalib_node_broadcast (data, size, sendpid)                    *
*      - broadcast to host and all node from process sendpid       *
*                                                                  *
*******************************************************************/

     /*********************************************************
     *                                                        *
     * FUNCTION(dalib_broadcast)                              * 
     *                                                        *
     *    (char *data, int *size, int *pid)                   *
     *                                                        *
     *********************************************************/

void FUNCTION(dalib_broadcast) (data, size, sendpid) 
int *data, *size, *sendpid;

{ int gid;

  gid = dalib_context_group ();
  dalib_group_bcast (data, *size, *sendpid, gid);

} /* dalib_broadcast */

     /*********************************************************
     *                                                        *
     * FUNCTION(dalib_string_broadcast)                       * 
     *                                                        *
     *    (char *data, int *size, int *pid)                   *
     *                                                        *
     *********************************************************/

void FUNCTION(dalib_string_broadcast)
     ARGS(`STRING_ARG(string_data), size, sendpid')

STRING_ARG_DECL(string_data);
int *size, *sendpid;

{ int gid;
  char *data;

  data = STRING_PTR(string_data);

  gid = dalib_context_group ();
  dalib_group_bcast (data, *size, *sendpid, gid);

} /* dalib_string_broadcast */

     /*********************************************************
     *                                                        *
     *  FUNCTION(dalib_array_broadcast) (...)                 * 
     *                                                        *
     *  - broadcast along a distributed dimension of an array *
     *                                                        *
     *********************************************************/

void FUNCTION(dalib_array_broadcast) (array_id, dim, ind_id, data, size) 

unsigned char *data; 
int **array_id;
int *dim, *ind_id, *size;

{  int topid, topdim, toppos;
   int group_id;

   dalib_array_topmap (*array_id, *dim, *ind_id, &topid, &topdim, &toppos);

   if (! dalib_in_topology (topid))
       return;

#ifdef DEBUG
   printf ("%d: dalib_array_broadcast (topid=%d, topdim=%d, toppos = %d) size = %d\n",
           pcb.i, topid, topdim, toppos, *size);
#endif

  group_id = dalib_top_subgroup (topid, topdim);
  dalib_group_bcast (data, *size, toppos, group_id);

} /* FUNCTION(dalib_array_broadcast) */ 
