/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.WR                         *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Aug 98                                                   *
*  Last Update : Nov 98                                                   *
*                                                                         *
*  This Module is part of the DALIB / UNILIB                              *
*                                                                         *
*  Module      : attach                                                   *
*                                                                         *
*  Function    : attaching common array descriptors                       *
*                (similiar to in_out for dummy arrays)                    *
*                                                                         *
**************************************************************************/

#undef DEBUG

#include "dalib.h"

static char *common_str_ptr;
static int  common_str_len;

/************************************************************************
*                                                                       *
*  ERROR message for COMMON conflicts                                   *
*                                                                       *
************************************************************************/

void dalib_common_mismatch (local_dsp, common_dsp, msg)

array_info local_dsp, common_dsp;
char msg[];

{ char msg1[256];
  int i;
  char cid [256];

  for (i=0; i<common_str_len; i++) cid[i] = common_str_ptr[i];
  cid[common_str_len] = '\0';

  sprintf (msg1, "COMMON /%s/ mismatch for array %s: %s\n", 
           cid, local_dsp->name, msg);

  dalib_internal_error (msg1);

  printf ("%d: descriptor found in COMMON = \n", pcb.i);
  dalib_print_array_info (common_dsp);

  printf ("%d: descriptor expected = \n", pcb.i);
  dalib_print_array_info (local_dsp);

  dalib_stop ();

} /* common_mismatch_error */

/*******************************************************************
*                                                                  *
*  FORTRAN Interface                                               *
*                                                                  *
*  FUNCTION(dalib_attach_in) (local_dsp, global_dsp, common_id)    *
*                                                                  *
*   - attach to a global array (COMMON) given by global_dsp        *
*   - assumed specification is given by local_dsp                  *
*   - verification of same size/type/dsp_kind/distribution/shadow  *
*                                                                  *
*   common /DATA_DSP/  A_DSP, B_DSP                                *
*   integer A_DSP, A_CDSP, B_DSP, B_CDSP                           *
*   A_CDSP = A_DSP; B_CDSP = B_DSP                                 *
*   ...  ! define expected A_DSP, B_DSP                            *
*   call dalib_attach_in (A_DSP, A_CDSP, 2, 'DATA')                *
*   call dalib_attach_in (B_DSP, B_CDSP, 2, 'DATA')                *
*                                                                  *
*   Attention: variable A_CDSP is no more used afterwards          *
*                                                                  *
*******************************************************************/

void FUNCTION(dalib_attach_in) 

     ARGS(`local_dsp, global_dsp, safety, STRING_ARG(common_id)')

array_info *local_dsp;
array_info *global_dsp;
int        *safety;           /* 2 : accept all, 1 : stop for changes */
STRING_ARG_DECL(common_id);

{ int info;
  int dsp_kind1, dsp_kind2;

  /* set name of common globally for all error messages */

  common_str_ptr = STRING_PTR(common_id);
  common_str_len = STRING_LEN(common_id);

  /* check correct descriptors */

  if (!dalib_is_array_info (*global_dsp))

     { char msg[128];

       /* we can assume that the array is used the first time */

       sprintf (msg, "%d: first array descriptor for common array %s",
                pcb.i, (*local_dsp)->name);

       if (pcb.redist_flag) printf (msg);
          
       /* dalib_internal_error (msg);  */

       *global_dsp = *local_dsp;

       return;
     }

  if (!dalib_is_array_info (*local_dsp))

     { /* that must be indeed a serious error */

       dalib_internal_error ("attach in: illegal local descriptor");
       dalib_stop ();

     }

  /* make some verifications with the defined local descriptor */

  info = dalib_match_global_shape (*local_dsp, *global_dsp);

  if (info == 0) 

      dalib_common_mismatch (*local_dsp, *global_dsp, 
                             "not same shape/rank/dims");

  dsp_kind1 = (*local_dsp)->dynamic;
  dsp_kind2 = (*global_dsp)->dynamic;

  if (dsp_kind1 != dsp_kind2)

      dalib_common_mismatch (*local_dsp, *global_dsp, 
                             "kind mismtach");

  /* now we are sure that existing COMMON descriptor has expected dsp_kind */

  if (!dalib_sufficient_shadow (*local_dsp, *global_dsp))

    { unsigned char *f_ptr;

      if (dsp_kind1 == kIS_STATIC)

        dalib_common_mismatch (*local_dsp, *global_dsp, 
          "insufficient shadow\n        (use !HPF$ SHADOW in program)");

      /* redistribute (new_dsp = local_dsp, old_dsp = global_dsp */

      dalib_do_redistribution (local_dsp, global_dsp,
                               "common array needs more shadow");

      *global_dsp = *local_dsp;   /* make consistent */

      /* old global descriptor does no more exist, is now equal to local */

      return;
   }

   /* verifications okay, so free the local descriptor and set global one */

#ifdef DEBUG
   printf ("%d attach in : local dsp = %p, global dsp = %p\n",
            pcb.i, *local_dsp, *global_dsp);
#endif

   dalib_free_descriptor (*local_dsp);

   *local_dsp = *global_dsp; 

} /* dalib_attach_in */
