/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Feb 95                                                   *
*  Last Update : Mar 96                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : assign1                                                  *
*                                                                         *
*  Function    : structured assignment with section descriptors           *
*                                                                         *
*  Export : FORTRAN Interface                                             *
*                                                                         *
*   void FUNCTION(dalib_assign) (target_section, source_section)          *
*                                                                         *
*   void FUNCTION(dalib_assign_permute) (target_section, source_section,  * 
*                                         p1, p2, p3, p4, p5, p6, p7)     *
*                                                                         *
*  UPDATES:                                                               *
*                                                                         *
*   10/95  :  assignment works now for aligned arrays                     *
*                                                                         *
**************************************************************************/

#undef DEBUG

#include "dalib.h"
#include <stdio.h>

/*******************************************************************
*                                                                  *
*   GLOBAL DATA FOR MOVING of SECTIONS                             *
*                                                                  *
*******************************************************************/

   /* two constants used for indexing the global data */

#define SOURCE 0
#define TARGET 1 

       /************************************************
       *                                               *
       *   Information about the both sections         *
       *                                               *
       ************************************************/

static section_info section_id [2];
static array_info   array_id   [2];
static int          array_dims [2][MAX_DIMENSIONS];

/* Note : section can have fixed elements in some dimensions, so
          rank of section might be smaller than rank of array           

          array_dims[...][sec_dim]  is array dim of sec_dim (0..rank-1)  */

       /************************************************
       *                                               *
       *   Permutation data                            *
       *                                               *
       ************************************************/

static int is_permute;
static int permute_vals [MAX_DIMENSIONS];

/*******************************************************************
*                                                                  *
*   SETTING GLOBAL DATA OF ONE SECTION                             *
*                                                                  *
*      set_section (A (N,3:N, 1, 2:N-1), kind)                     *
*                                                                  *
*         array_dims[kind] : (1,3)                                 *
*                                                                  *
*******************************************************************/

static int dalib_set_section (section, kind)
section_info section;
int kind;

{ int array_rank, sec_rank;
  int i;

  SecDimInfo *sec_dim;

  section_id [kind] = section;
  array_id   [kind] = section->array_id;
 
  array_rank     = array_id[kind]->rank;
  sec_dim        = section_id[kind]->dimensions;

#ifdef DEBUG
  printf ("%d: set section %d, kind = %d, array_rank = %d\n",
          pcb.i, section, kind, array_rank);
  dalib_print_array_info (array_id[kind]);
#endif 

       /*****************************************
       *    TRAVERSE SECTION by INDEX           *
       *****************************************/

  sec_rank = 0;

  for (i=0; i < array_rank; i++)

     { if (sec_dim->is_range == 1)

          { /* is a range */

            array_dims [kind][sec_rank] = i;
            sec_rank++;
          }

       sec_dim++;
     }

#ifdef DEBUG
  printf ("%d: array_dims [%d] = ( ", pcb.i, kind);
  for (i=0; i<sec_rank; i++) printf (" %d", array_dims[kind][i]);
  printf (" )\n");
#endif 

  return (sec_rank);

} /* dalib_set_section */

/*******************************************************************
*                                                                  *
*  dalib_check_sections                                            *
*                                                                  *
*******************************************************************/

void dalib_check_sections (rank)
int rank;

{ SecDimInfo *source_dim, *target_dim;
  int i, sn, tn;
  int error;

  error = 0;

  for (i=0; (i<rank) && (!error); i++)

   { source_dim = section_id[SOURCE]->dimensions + array_dims[SOURCE][i];
     target_dim = section_id[TARGET]->dimensions + array_dims[TARGET][i];

     sn = dalib_range_size (source_dim->global_range[0],
                            source_dim->global_range[1],
                            source_dim->global_range[2]);
    
     tn = dalib_range_size (target_dim->global_range[0],
                            target_dim->global_range[1],
                            target_dim->global_range[2]);

     if (sn != tn) error = 1;
   }

   if (error)

     { dalib_internal_error ("assign of section mismatch");
       printf ("source section : \n");
       dalib_print_section_info (section_id[SOURCE]);
       printf ("target section : \n");
       dalib_print_section_info (section_id[TARGET]);
       printf ("source section has size %d at dim %d\n", 
                sn, array_dims[SOURCE][i-1] + 1);
       printf ("target section has size %d at dim %d\n", 
                tn, array_dims[TARGET][i-1] + 1);
       if (is_permute)
          { printf ("permutation :  ");
            for (i=0; i < rank; i++) printf ("%d ", permute_vals[i] + 1);
            printf ("\n");
          }
       dalib_stop ();
     }

} /* dalib_check_sections */

/*******************************************************************
*                                                                  *
*  FUNCTION(dalib_assign) (target_section, source_section, size)   *
*                                                                  *
*  ASSIGNMENT FOR ARRAY SECTIONS: target_section = source-section  *
*                                                                  *
*******************************************************************/

static void dalib_do_assign (target_section, source_section, rank)

section_info target_section;
section_info source_section;
int rank;

{ dd_type ddt_source, ddt_target;
  char *vals;

  int size1, size2;

       /*****************************************
       *    check the two sections for compatib *
       *****************************************/

   dalib_check_sections (rank);

   if (is_permute)
      dalib_make_perm_section_ddt (&ddt_source, source_section, permute_vals);
     else
      dalib_make_section_ddt (&ddt_source, source_section);

   dalib_make_section_ddt (&ddt_target, target_section);

   dalib_ddt_get_size (ddt_source, &size1);
   dalib_ddt_get_size (ddt_target, &size2);

   if (size1 != size2)

      { dalib_internal_error ("size mismatch in do_assign");
        dalib_stop ();
      }

   vals = (char *) dalib_malloc (size1, "dalib_do_assign");

   dalib_ddt_pack   (vals, ddt_source);
   dalib_ddt_unpack (ddt_target, vals, 0);

   dalib_free (vals, size1);

   dalib_ddt_free (ddt_source);
   dalib_ddt_free (ddt_target);

} /* dalib_do_assign */

/*******************************************************************
*                                                                  *
*    FUNCTION(dalib_assign) (target_section, source_section)       * 
*                                                                  *
*******************************************************************/

void FUNCTION(dalib_assign) (target_section, source_section) 

section_info *target_section;
section_info *source_section;

{ int assign_rank; 
  section_info source_sec, target_sec;

#ifdef DEBUG
   printf ("%d: dalib_assign, section %d = section %d\n",
           pcb.i, *target_section, *source_section);
#endif

   dalib_section_full (&source_sec, *source_section);
   dalib_section_full (&target_sec, *target_section);

       /*****************************************
       *    set infos about section globally    *
       *****************************************/

  assign_rank = dalib_set_section (source_sec, SOURCE);
  if (dalib_set_section (target_sec, TARGET) != assign_rank)
     dalib_internal_error ("dalib_assign");

  is_permute = 0;

  dalib_do_assign (target_sec, source_sec, assign_rank);

  if (dalib_is_array_info (*target_section))
     FUNCTION(dalib_section_free) (&target_sec); 
 
  if (dalib_is_array_info (*source_section))
     FUNCTION(dalib_section_free) (&source_sec); 
 
} /* dalib_assign */

/*******************************************************************
*                                                                  *
* FUNCTION(dalib_assign_permute) (target_section, source_section, * 
*                         p1, p2, p3, p4, p5, p6, p7)              *
*                                                                  *
*******************************************************************/

void FUNCTION(dalib_assign_permute) (target_section, source_section, 
                            p1, p2, p3, p4, p5, p6, p7)

section_info *target_section;
section_info *source_section;

int *p1, *p2, *p3, *p4, *p5, *p6, *p7;

{ int assign_rank;
  section_info target_sec, source_sec;

#ifdef DEBUG
   printf ("%d: dalib_assign_permute, section %d = section %d\n",
           pcb.i, *target_section, *source_section);
#endif

   dalib_section_full (&source_sec, *source_section);
   dalib_section_full (&target_sec, *target_section);
 
       /*****************************************
       *    set infos about section globally    *
       *****************************************/

  assign_rank = dalib_set_section (source_sec, SOURCE);

  if (dalib_set_section (target_sec, TARGET) != assign_rank)
     dalib_internal_error ("dalib_assign_permute");

  is_permute = 1;

  /* note : FORTRAN permutation between 1 and n, but 0 .. n-1 in C */

  switch (assign_rank)  {

    case 7 : permute_vals[6] = *p7 - 1;
    case 6 : permute_vals[5] = *p6 - 1;
    case 5 : permute_vals[4] = *p5 - 1;
    case 4 : permute_vals[3] = *p4 - 1;
    case 3 : permute_vals[2] = *p3 - 1;
    case 2 : permute_vals[1] = *p2 - 1;
    case 1 : permute_vals[0] = *p1 - 1;
    case 0 : break;

  } /* switch */

  /* permute index information of source section */

  dalib_do_assign (target_sec, source_sec, assign_rank);

  if (dalib_is_array_info (*target_section))
     FUNCTION(dalib_section_free) (&target_sec); 
 
  if (dalib_is_array_info (*source_section))
     FUNCTION(dalib_section_free) (&source_sec); 
 
} /* FUNCTION(dalib_assign_permute) */ 

