/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*                                                                         *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Sep 94                                                   *
*  Last Update : Mar 96                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : array.m4                                                 *
*                                                                         *
*  Function: Creating descriptors for arrays                              *
*                                                                         *
*  Export :  FORTRAN Interface                                            *
*  ===========================                                            *
*                                                                         *
*  void FUNCTION(dalib_array_make_dsp) (array_id, rank, size)             *
*                                                                         *
*    - creates just a descriptor for an array or a template               *
*                                                                         *
*  void FUNCTION(dalib_array_define) (array_id,                           *
*                        lb1, ub1, lb2, ub2, lb3, ub3,                    *
*                        lb4, ub4, lb5, ub5, lb6, ub6, lb7, ub7)          *
*                                                                         *
*    - sets the shape and computes the local sizes                        *
*                                                                         *
*  void FUNCTION(dalib_array_create_copy) (new_array_id, old_array_id)    *
*                                                                         *
*    - creates an array descriptor with the same information as the old   *
*                                                                         *
*  void FUNCTION(dalib_darray_get) (array_id, rank, size,                 *
*                        lb1, ub1, lb2, ub2, lb3, ub3,                    *
*                        lb4, ub4, lb5, ub5, lb6, ub6, lb7, ub7)          *
*                                                                         *
*    - get the size of an array by the descriptor                         *
*                                                                         *
*  void FUNCTION(dalib_array_deallocate) (array_id)                       *
*                                                                         *
*    - frees allocated data for a descriptor                              *
*    - but will not destroy the corresponding descriptor                  *
*                                                                         *
*  void FUNCTION(dalib_array_free) (array_id)                             *
*                                                                         *
*    - free a descriptor (needs also some amount of memory)               *
*    - dalib_array_free frees also data / distribution / alignment        *
*                                                                         *
*  void FUNCTION(dalib_array_shadow) (array_id, rank,                     *
*                        lb1, ub1, lb2, ub2, lb3, ub3,                    *
*                        lb4, ub4, lb5, ub5, lb6, ub6, lb7, ub7)          *
*                                                                         *
*  void FUNCTION(dalib_array_access) (array_id, a_ptr,                    *
*                                a_zero, a_dim1, a_dim2, a_dim3,          *
*                                a_dim4, a_dim5, a_dim6, a_dim7)          *
*                                                                         *
*  void FUNCTION(dalib_array_allocate) (array_id, a_ptr,                  *
*                                a_zero, a_dim1, a_dim2, a_dim3,          *
*                                a_dim4, a_dim5, a_dim6, a_dim7)          *
*                                                                         *
*   INTERNAL ROUTINES                                                     *
*                                                                         *
*   int dalib_array_local_size (array_info array_id)                      *
*                                                                         *
**************************************************************************/

#include <stdio.h>
#include "dalib.h"

#undef DEBUG
#define CHECK

static int error;

/**************************************************************************
*                                                                         *
*  PREDICATE for testing array info                                       *
*                                                                         *
**************************************************************************/

int dalib_is_array_info (array_id)
array_info array_id;

{  if (array_id == (array_info) 0) return (0);
   return (array_id->ident == ARRAY_IDENTIFICATION);
} /* dalib_is_array_info */

/**************************************************************************
*                                                                         *
*  void dalib_print_array_info (array_info array_id)                      *
*                                                                         *
*   - print all information of an array descriptor                        *
*                                                                         *
**************************************************************************/

void dalib_print_array_info (array_id)

array_info array_id;

{ int i, rank;
  DimInfo     *dims;

#ifdef CHECK
  if (!dalib_is_array_info (array_id))
     { dalib_internal_error ("print_array_info, not array info");
       dalib_stop ();
     }
#endif 

  rank = array_id->rank;
 
  printf ("array_decriptor (%p) for %s : rank = %d ", array_id, 
           array_id->name, rank);

  if (array_id->dynamic == kIS_STATIC)
     printf (" static");
   else if (array_id->dynamic == kIS_DYNAMIC)
     printf (" dynamic");

  if (array_id->DistributeInfo != NO_DISTRIBUTION)
    printf (" distributed\n");
  if (array_id->AlignInfo != NO_ALIGNMENT)
    { array_info template;
      int top_id;
      dalib_alignment_info (array_id, &template, &top_id);
      printf (" aligned to %s\n", template->name);
    }
   else printf ("\n");

  dims = array_id->dimensions;
  printf ("  global size : (");
  for (i=0; i<rank; i++)
    { printf ("%d:%d", dims->global_size[0], dims->global_size[1]);
      if (i<rank-1) printf (",");
      dims++;
    }
  printf (")\n");

  dims = array_id->dimensions;
  printf ("  local size  : (");
  for (i=0; i<rank; i++)
    { printf ("%d:%d:%d", dims->local_size[0], 
                       dims->local_size[1], dims->local_size[2]);
      if (i<rank-1) printf (",");
      dims++;
    }
  printf (")\n");

  dims = array_id->dimensions;
  printf ("  shadow      : (");
  for (i=0; i<rank; i++)
    { printf ("%d:%d", dims->shadow[0], dims->shadow[1]);
      if (i<rank-1) printf (",");
      dims++;
    }
  printf (")\n");

  if (array_id->n_data > 0)
    printf ("  allocated data (addr=%p,size=%d), ",
            array_id->c_data, array_id->n_data);
   else
    printf ("  no allocated data, ");
  if (array_id->f_data != NO_DATA)
    printf ("  attached data (addr=%p)\n", array_id->f_data);
   else
    printf ("  no attached data\n");

} /* dalib_print_array_info */

/**************************************************************************
*                                                                         *
*  HELP FUNCTIONS                                                         *
*                                                                         *
**************************************************************************/

static void dalib_set_dim_info (dim, lb, ub)

DimInfo *dim;
int lb, ub;

{ dim->global_size[0] = lb;
  dim->global_size[1] = ub;

  /* default : local size is equal global size */

  dim->local_size[0] = lb;
  dim->local_size[1] = ub;
  dim->local_size[2] = 1;

  dim->map_flag = 0;

  /* attention : do not touch shadow */

} /* dalib_setdim_info */

   /********************************************************************
   *                                                                   *
   *   int dalib_dim_equal (DimInfo *dim1, DimInfo *dim2)              *
   *                                                                   *
   *   - -1 if dimensions have different shape                         *
   *   - 0  if same shape, but different lower/upper bounds            *
   *   - 1  if lb:ub is same for both dimensions                       *
   *                                                                   *
   ********************************************************************/

static int dalib_dim_equal (dim1, dim2)

DimInfo *dim1, *dim2;

{ int extent;

  extent = dim1->global_size[1] - dim1->global_size[0];
  if (dim2->global_size[1] - dim2->global_size[0] != extent) return (-1);
  if (dim1->global_size[0] != dim2->global_size[0]) return (0);
  return (1);

} /* dalib_dim_equal */

static void dalib_get_dim_info (dim, lb, ub)

DimInfo *dim;
int *lb, *ub;

{ int size;

  /* map the global range of the array to *lb : *ub, where *lb is fixed */

  size = dim->global_size[1] - dim->global_size[0] + 1;
  *ub  = *lb + size - 1;

} /* dalib_get_dim_info */

static void dalib_set_dim_shadow (dim, lv, rv)

DimInfo *dim;
int lv, rv;

{ dim->shadow [0] = lv;
  dim->shadow [1] = rv;
} /* dalib_set_dim_shadow */

/**************************************************************************
*                                                                         *
*  dalib_full_local_sizes (array_id)                                      *
*                                                                         *
*  - initializes the local size to be the global size                     *
*                                                                         *
**************************************************************************/

void dalib_full_local_sizes (array_id)

array_info array_id;

{ DimInfo *dimensions;

  int i, rank;
  int *global, *local;
 
  dimensions = array_id->dimensions;
  rank       = array_id->rank;

  for (i=0; i<rank; i++)
 
    { global = dimensions->global_size;
      local  = dimensions->local_size;
 
      local[0] = global[0];
      local[1] = global[1];
      local[2] = 1;
 
      dimensions ++;
 
    } /* for i */
 
} /* dalib_full_local_sizes */

void dalib_empty_local_sizes (array_id)

array_info array_id;

{ DimInfo *dimensions;

  int i, rank;
  int *global, *local;
 
  dimensions = array_id->dimensions;
  rank       = array_id->rank;

  for (i=0; i<rank; i++)
 
    { local  = dimensions->local_size;
 
      local[0] = 1;
      local[1] = 0;
      local[2] = 1;
 
      dimensions ++;
 
    } /* for i */
 
} /* dalib_empty_local_sizes */

/**************************************************************************
*                                                                         *
*   dalib_array_make_dsp (int rank, int size)                             *
*                                                                         *
**************************************************************************/

int dalib_array_dsp_size (rank)

{ int save;

  save = (MAX_DIMENSIONS - rank) * sizeof (DimInfo);

  return sizeof (struct ArrayRecord) - save;

} /* dalib_array_dsp_size */

/**************************************************************************
*                                                                         *
*   array_info dalib_array_new_dsp (int rank, int size)                   *
*                                                                         *
*   - creates a new array descriptor for given rank and size              *
*   - used internally in the UNILIB/DALIB                                 *
*                                                                         *
**************************************************************************/

array_info dalib_array_new_dsp (rank, size, kind)

int rank, size, kind;

{ array_info   descriptor;
 
  int i;
 
#ifdef DEBUG
  printf ("%d: dalib_array_new_dsp , rank %d, size %d \n", pcb.i, rank, size);
#endif
 
  descriptor = (array_info) dalib_malloc (dalib_array_dsp_size (rank),
                                          "array_new_dsp");
 
  descriptor->rank              = rank;
  descriptor->size              = size;
  descriptor->dsp_kind          = kind;
  descriptor->dynamic           = 0;
  if (kind == kIS_STATIC)
      descriptor->global_addressing = 1;
    else
      descriptor->global_addressing = 0;
  descriptor->ident             = ARRAY_IDENTIFICATION;
  descriptor->trace             = 0;
  descriptor->shadow_status     = 0;
  descriptor->ref_counter       = 1;          /* my own reference     */
  descriptor->dsp_status_flag   = DSP_UNDEFINED;
  descriptor->reuse_bits        = 0;

  descriptor->n_data          = 0;          /* no own/attached data */
  descriptor->c_data          = NO_DATA;
  descriptor->f_data          = NO_DATA;

  descriptor->AlignInfo       = NO_ALIGNMENT;
  descriptor->DistributeInfo  = NO_DISTRIBUTION;
  descriptor->SharedInfo      = NO_SHARED;
  descriptor->RemoteInfo      = NO_REMOTE;
  descriptor->GlobalInfo      = NO_ARRAY;
 
  /* default : no name for this descriptor (internal descriptors) */

  descriptor->name[0] = '\0';

  /* default : no overlap is given */

  for (i=0; i<rank; i++)

    {  descriptor->dimensions[i].shadow[0] = 0;
       descriptor->dimensions[i].shadow[1] = 0;
    }

  return (descriptor);

} /* dalib_array_new_dsp */

/**************************************************************************
*                                                                         *
*   dalib_array_compare_dsp (dp1, dsp2)                                   *
*                                                                         *
**************************************************************************/

static int dalib_array_compare_dsp (dsp1, dsp2)

array_info dsp1, dsp2;

{ DimInfo *dims1, *dims2;
  int     size1, size2;
  int     rank1, rank2;
  int     i, match, diff;

          /********************************************
          *   match rank against each other           *
          ********************************************/

  rank1 = dsp1->rank;
  rank2 = dsp2->rank;

  if (rank1 != rank2)
     { dalib_internal_error ("actual/dummy : rank mismatch");
       goto error;
     }

          /********************************************
          *   match size against each other           *
          ********************************************/

  size1 = dsp1->size;
  size2 = dsp2->size;

  if (size1 != size2)
     { dalib_internal_error ("actual/dummy : size mismatch");
       goto error;
     }

          /********************************************
          *   match shape against each other          *
          ********************************************/

  dims1 = dsp1->dimensions;
  dims2 = dsp2->dimensions;

  match = 1;

  for (i=0; i<rank1; i++)
 
     { diff = dalib_dim_equal (dims1+i,dims2+i);
       if (diff == 0)  match = 0;
       if (diff == -1) 
          { dalib_internal_error ("actual/dummy : shape mismatch");
            goto error;
          }
     }

  return (match);

          /********************************************
          *   ERROR: can never match against each     *
          ********************************************/

  error :

   printf ("ERROR : dummy/common array has not the expected size/shape\n\n");
   printf ("expected array size : \n");
   dalib_print_array_info (dsp1);
   printf ("actual array size : \n");
   dalib_print_array_info (dsp2);
   dalib_stop ();

   return (0);

} /* dalib_array_compare_dsp */

/**************************************************************************
*                                                                         *
*   array_info dalib_array_copy_dsp (array_id)                            *
*                                                                         *
*   - copies an array descriptor with all relevant information            *
*                                                                         *
**************************************************************************/
 
array_info dalib_array_copy_dsp (array_id)
 
array_info array_id;
 
{ array_info  new_descriptor;
 
  int rank, size;
 
#ifdef CHECK
  if (!dalib_is_array_info (array_id))
     { dalib_internal_error ("array_copy_dsp, not array info");
       dalib_stop ();
     }
#endif
 
  rank = array_id->rank;
 
  size = (MAX_DIMENSIONS - rank) * sizeof (DimInfo);
  size = sizeof (struct ArrayRecord) - size;
 
  new_descriptor = (array_info) dalib_malloc (size, "array_copy_dsp");
 
  dalib_memcopy (new_descriptor, array_id, size);

  /* note : the following attributes are inherited :
 
     new_descriptor->size
     new_descriptor->rank
     new_descriptor->dynamic 
     new_descriptor->global_addressing
 
     all dimensions (global shape, local shape)

  */

  new_descriptor->trace             = 0;
  new_descriptor->shadow_status     = 0;
  new_descriptor->reuse_bits        = 0;

  new_descriptor->ref_counter     = 1;      /* my own reference is given */

  new_descriptor->n_data          = 0;
  new_descriptor->c_data          = NO_DATA;
  new_descriptor->f_data          = NO_DATA;
  new_descriptor->AlignInfo       = NO_ALIGNMENT;
  new_descriptor->DistributeInfo  = NO_DISTRIBUTION;
  new_descriptor->SharedInfo      = NO_SHARED;
  new_descriptor->RemoteInfo      = NO_REMOTE;
  new_descriptor->GlobalInfo      = NO_ARRAY;
 
  /* status of new descriptor will be DEFINED unless old one was UNDEFINED */

  if (array_id->dsp_status_flag == DSP_UNDEFINED)

     new_descriptor->dsp_status_flag = DSP_UNDEFINED;
   else
     new_descriptor->dsp_status_flag = DSP_DEFINED;

  return (new_descriptor);
 
} /* dalib_array_copy_dsp */

/**************************************************************************
*                                                                         *
*  void dalib_array_global_shape (array_id, rank, dimsize)                *
*                                                                         *
**************************************************************************/
 
void dalib_array_global_shape (array_id, rank, lb, ub)
 
array_info array_id;
int *rank;
int lb[], ub[];
 
{ int i, array_rank;
  DimInfo *dims;

#ifdef CHECK
  if (!dalib_is_array_info (array_id))
    { dalib_internal_error ("array_global_shape : not an array");
      dalib_stop ();
    }
#endif

  array_rank = array_id->rank;
  dims       = array_id->dimensions;

  for (i=0; i<array_rank; i++, dims++)
 
    { lb[i] = dims->global_size[0];
      ub[i] = dims->global_size[1];
    }

  *rank = array_rank;
 
} /* dalib_array_global_shape */

/**************************************************************************
*                                                                         *
*                                                                         *
**************************************************************************/
 
int dalib_array_same_global_bounds (array_id1, array_id2)
 
array_info array_id1, array_id2;
 
{ int i, rank;
  DimInfo *dims1, *dims2;

  rank  = array_id1->rank;

  if (rank != array_id2->rank) return (0);

  dims1 = array_id1->dimensions;
  dims2 = array_id2->dimensions;

  for (i=0; i<rank; i++, dims1++, dims2++)
 
    { if (dims1->global_size[0] != dims2->global_size[0]) return (0);
      if (dims1->global_size[1] != dims2->global_size[1]) return (0);
    }

  return (1);

} /* dalib_array_same_global_bounds */

/**************************************************************************
*                                                                         *
*  void dalib_array_local_shape (array_info array_id, int *rank,          *
*                                int lb[], int ub[], int str[])           *
*                                                                         *
**************************************************************************/
 
void dalib_array_local_shape (array_id, rank, lb, ub, str)
 
array_info array_id;

int *rank;

int lb [];
int ub [];
int str[];
 
{ int i, array_rank;
  DimInfo *dims;

#ifdef CHECK
  if (!dalib_is_array_info (array_id))
    { dalib_internal_error ("array_local_shape : not an array");
      dalib_stop ();
    }
#endif

  array_rank = array_id->rank;
  dims       = array_id->dimensions;

  for (i=0; i<array_rank; i++, dims++)
 
    { lb [i] = dims->local_size[0];
      ub [i] = dims->local_size[1];
      str[i] = dims->local_size[2];
    }

  *rank = array_rank;
 
} /* dalib_array_local_shape */

/**************************************************************************
*                                                                         *
*  void dalib_array_local_embedding (array_info array_id, int *ierr,      *
*                                    int *rank, int lb[], int ub[])       *
*                                                                         *
*  - values in lb and ub are the lower and upper bounds in global         *
*    coordinates of the elements allocated on this processor              *
*                                                                         *
**************************************************************************/
 
void dalib_array_local_embedding (array_id, ierr, rank, lb, ub)
 
array_info array_id;

int *ierr, *rank;

int lb [];
int ub [];
 
{ int i, array_rank;
  DimInfo *dims;

#ifdef CHECK
  if (!dalib_is_array_info (array_id))
    { dalib_internal_error ("array_local_shape : not an array");
      dalib_stop ();
    }
#endif

  array_rank = array_id->rank;
  dims       = array_id->dimensions;

  for (i=0; i<array_rank; i++, dims++)
 
    { lb [i] = dims->local_size[0] - dims->shadow[0];
      ub [i] = dims->local_size[1] + dims->shadow[1];
    }

  *ierr = 0;
  *rank = array_rank;
 
} /* dalib_array_local_embedding */

/**************************************************************************
*                                                                         *
*  Functions for defining array/template descriptors                      *
*                                                                         *
*  void FUNCTION(dalib_make_dsp) (array_id, rank, size, dsp_kind,         *
*                                 hpf_dynamic, array_name)                *
*                                                                         *
*    - returns an internal descriptor for an array/template               *
*      of the rank 'rank' and 'size' bytes for one element                *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_array_make_dsp) 

  ARGS(`array_id, rank, size, dsp_kind, hpf_dynamic, STRING_ARG(array_name)')

array_info *array_id;
int *rank, *size, *dsp_kind, *hpf_dynamic;
STRING_ARG_DECL(array_name);

{ array_info   descriptor;
  DimInfo      *dims;

  char *name_ptr;
  int  i, name_len;

#ifdef DEBUG
  printf ("%d: dalib_array_make_dsp, rank %d \n", pcb.i, *rank);
#endif

  name_ptr = STRING_PTR(array_name);
  name_len = STRING_LEN(array_name);

  descriptor = dalib_array_new_dsp (*rank, *size, *dsp_kind);

  descriptor->dynamic = *hpf_dynamic;

  if (name_len >= NAME_LENGTH) name_len = NAME_LENGTH - 1;
  for (i=0; i<name_len; i++) descriptor->name[i] = name_ptr[i];
  descriptor->name[name_len] = '\0';

  *array_id = descriptor;

#ifdef DEBUG
  printf ("%d: created array descriptor %p for %s (rank=%d,size=%d)\n", 
           pcb.i, descriptor, descriptor->name, *rank, *size);
#endif

} /* dalib_array_make_dsp */

/**************************************************************************
*                                                                         *
*      of the rank 'rank' and 'size' bytes for one element                *
*                                                                         *
**************************************************************************/

int FUNCTION(dalib_size) (addr1, addr2, ass_size)

unsigned char *addr1, *addr2;
int           *ass_size;

{ int real_size;

  real_size = addr2 - addr1;

  if (real_size != *ass_size)

   { char msg [80];

     printf ("ATTENTION: fadapt assumed size = %d, compiler made size = %d\n",
              *ass_size, real_size);
   }

  return (real_size);

} /* dalib_size */

/**************************************************************************
*                                                                         *
*  void FUNCTION(dalib_array_define) (array_id,                           *
*                        lb1, ub1, lb2, ub2, lb3, ub3,                    *
*                        lb4, ub4, lb5, ub5, lb6, ub6, lb7, ub7)          *
*                                                                         *
*    - sets the shape in an existing descriptor, computes local sizes     *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_array_define) (array_id,
                          lb1, ub1, lb2, ub2, lb3, ub3,
                          lb4, ub4, lb5, ub5, lb6, ub6, lb7, ub7)

array_info *array_id;
int *lb1, *ub1, *lb2, *ub2, *lb3, *ub3, *lb4, *ub4,
    *lb5, *ub5, *lb6, *ub6, *lb7, *ub7;

{ array_info   descriptor;
  DimInfo      *dims;
  int          i, rank;

  descriptor = *array_id;

  dims = descriptor->dimensions;
  rank = descriptor->rank;

  switch (rank) {
     case 7 : dalib_set_dim_info (dims + 6, *lb7, *ub7);
     case 6 : dalib_set_dim_info (dims + 5, *lb6, *ub6);
     case 5 : dalib_set_dim_info (dims + 4, *lb5, *ub5);
     case 4 : dalib_set_dim_info (dims + 3, *lb4, *ub4);
     case 3 : dalib_set_dim_info (dims + 2, *lb3, *ub3);
     case 2 : dalib_set_dim_info (dims + 1, *lb2, *ub2);
     case 1 : dalib_set_dim_info (dims + 0, *lb1, *ub1);
  } /* switch */

  /* for underspecified mappings we cannot compute local sizes */

#ifdef DEBUG
  printf ("%d: array defined (rank=%d) : ", pcb.i, rank);
  for (i=0;i<rank;i++)
     printf ("%d = %d", dims[i].global_size[0], dims[i].global_size[1]);
  printf ("\n");
#endif

  if (dalib_array_map_underspecified (descriptor)) return;

  if (descriptor->DistributeInfo != NO_DISTRIBUTION)
     dalib_dist_local_sizes (descriptor);
   else if (descriptor->AlignInfo != NO_ALIGNMENT)
     dalib_align_local_sizes (descriptor);
   else
     dalib_full_local_sizes (descriptor);

#ifdef DEBUG
     printf ("%d: array define, local sizes computed\n", pcb.i);
     dalib_print_array_info (descriptor);
#endif
} /* dalib_array_define */

/**************************************************************************
*                                                                         *
*  void FUNCTION(dalib_array_create_copy) (new_array_id, old_array_id)    *
*                                                                         *
*    - creates an array descriptor with same information as the old one   *
*    - new descriptor has no distribution at all                          *
*                                                                         *
*    call DALIB_array_create_copy (B_NDSP,B_DSP)                          *
*    ...   ! define new distribution for B_DSP                            *
*    call DALIB_redistribute (B_DSP,B_NDSP)                               *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_array_create_copy) (new_array_id, old_array_id)

array_info *new_array_id, *old_array_id;

{ array_info old, new;

  old = *old_array_id;
  new = dalib_array_copy_dsp (old);

  /* inherit some attributes */

  new->dynamic = old->dynamic;
  new->trace   = old->trace;

  /* ATTENTION: do not change any ref_counter here */

  *new_array_id = old;
  *old_array_id = new;

  /* new_array_id : is now the old descriptor */
  /* old_array_id : is DEFINED with old sizes */

} /* dalib_array_create_copy */

/**************************************************************************
*                                                                         *
*  void FUNCTION(dalib_array_trace) (array_info *array_id)                *
*                                                                         *
*    - set the trace attribute for an array descriptor                    *
*                                                                         *
*  void FUNCTION(dalib_array_set_dirty) (array_info *array_id)            *
*                                                                         *
*    - array is updated, invalidate internal information                  *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_array_trace) (array_id)

array_info *array_id;

{ (*array_id)->trace = 1;

} /* dalib_array_trace */

/**************************************************************************
*                                                                         *
*  void FUNCTION(dalib_array_set_dirty) (array_info *array_id)            *
*                                                                         *
*    - array is updated, invalidate internal information                  *
*                                                                         *
*  Note: this routine will also be called if a descriptor becomes         *
*        invalid (to free schedules)                                      *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_array_set_dirty) (array_id)

array_info *array_id;

{ int *reuse_bits; 

#ifdef DEBUG
  printf ("made array with dsp = %d dirty, bits = %d\n", 
          *array_id, (*array_id)->reuse_bits);
#endif

  if (!(*array_id)->trace) return; /* in fact: it is an error */

  reuse_bits = & ( (*array_id)->reuse_bits );

  dalib_inspector_db_set_invalid (reuse_bits);

  (*array_id)->shadow_status = 0;

} /* dalib_array_set_dirty */

/**************************************************************************
*                                                                         *
*  Information about the global size of an array/template                 *
*                                                                         *
*  void FUNCTION(dalib_array_get_define) (local_dsp, argument_dsp,        * 
*                         is1, lb1, is2, lb2, is3, lb3,                   *
*                         is4, ub4, is5, lb5, is6, lb6, is7, lb7)         *
*                                                                         *
*   input values  : is1, is2, ..., isn                                    *
*                   lb1, lb2, ..., lbn                                    *
*                                                                         *
*        subroutine SUB (..., A, ...)                                     *
*        real A(lb1:, lb2:,....,lbn:)                                     *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_array_get_define) (local_dsp, argument_dsp, 
                        is1, lb1, is2, lb2, is3, lb3,
                        is4, lb4, is5, lb5, is6, lb6, is7, lb7)

array_info *local_dsp, *argument_dsp;
int *is1, *lb1, *is2, *lb2, *is3, *lb3, *is4, *lb4,
    *is5, *lb5, *is6, *lb6, *is7, *lb7;

{ array_info  descriptor;
  DimInfo     *dims;

  int  i, dummy_rank;
  int  lb [MAX_DIMENSIONS];
  int  ub [MAX_DIMENSIONS];
  int  str[MAX_DIMENSIONS];

  if (!FUNCTION(dalib_present) (argument_dsp))

     { /* dummy argument can be optional (so argument_dsp is not present), 
          so we cannot define the new local descriptor                     */

       return;

     }

  descriptor = *argument_dsp;

  if (dalib_is_array_info (descriptor))

     { /* get the values of ub1, ..., ub7 */

       dalib_array_global_shape (descriptor, &dummy_rank, lb, ub);

     }  /* descriptor was for a full array */

    else if (dalib_is_section_info (descriptor))

     { /* get the values of ub1, ..., ub7 */

       dalib_section_global_shape (descriptor, &dummy_rank, lb, ub, str);

       /* make the norm for the stride */

       for (i=0; i<dummy_rank; i++)
          if (str[i] != 1)
            { ub[i] = dalib_range_size (lb[i], ub[i], str[i]);
              lb[i] = 1;
            }
     }

    else

     { dalib_internal_error ("illegal dummy array (assumed-shape)");
       dalib_stop ();
     }

  /* move the shape if lower bounds are specified */

  switch (dummy_rank) {

      case 7 : if (*is7) { ub[6] = *lb7 + ub[6] - lb[6]; lb[6] = *lb7; }
      case 6 : if (*is6) { ub[5] = *lb6 + ub[5] - lb[5]; lb[5] = *lb6; }
      case 5 : if (*is5) { ub[4] = *lb5 + ub[4] - lb[4]; lb[4] = *lb5; }
      case 4 : if (*is4) { ub[3] = *lb4 + ub[3] - lb[3]; lb[3] = *lb4; }
      case 3 : if (*is3) { ub[2] = *lb3 + ub[2] - lb[2]; lb[2] = *lb3; }
      case 2 : if (*is2) { ub[1] = *lb2 + ub[1] - lb[1]; lb[1] = *lb2; }
      case 1 : if (*is1) { ub[0] = *lb1 + ub[0] - lb[0]; lb[0] = *lb1; }

   } /* switch */

   FUNCTION(dalib_array_define) (local_dsp, 
            lb, ub, lb+1, ub+1, lb+2, ub+2, lb+3, ub+3, 
                    lb+4, ub+4, lb+5, ub+5, lb+6, ub+6);

} /* dalib_array_get_define */

/**************************************************************************
*                                                                         *
*  Define a local descriptor from a global descriptor                     *
*                                                                         *
*  void FUNCTION(dalib_array_local_define) (array_id, dummy_array_id,     * 
*                         is1, lb1, is2, lb2, is3, lb3,                   *
*                         is4, ub4, is5, lb5, is6, lb6, is7, lb7)         *
*                                                                         *
*   input values  : is1, is2, ..., isn                                    *
*                   lb1, lb2, ..., lbn                                    *
*                                                                         *
*        HPF_LOCAL subroutine SUB (..., A, ...)                           *
*        real A(lb1:, lb2:,....,lbn:)                                     *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_array_local_define) (array_id, dummy_array_id, 
                        is1, lb1, is2, lb2, is3, lb3,
                        is4, lb4, is5, lb5, is6, lb6, is7, lb7)

array_info *array_id, *dummy_array_id;
int *is1, *lb1, *is2, *lb2, *is3, *lb3, *is4, *lb4,
    *is5, *lb5, *is6, *lb6, *is7, *lb7;

{ array_info  descriptor;
  DimInfo     *dims;

  int  i, dummy_rank;
  int  lb [MAX_DIMENSIONS];
  int  ub [MAX_DIMENSIONS];
  int  str[MAX_DIMENSIONS];

  descriptor = *dummy_array_id;

  if (dalib_is_section_info (descriptor))

     descriptor = ((section_info) descriptor)->array_id;

  if (dalib_is_array_info (descriptor))

     { if (descriptor->GlobalInfo != NO_ARRAY)
  
           descriptor = descriptor->GlobalInfo;

     }  /* descriptor was for a full array */

    else 

     { dalib_internal_error ("illegal dummy array (assumed-shape)");
       dalib_stop ();
     }

  /* set the global array descriptor for local array descriptor */

  (*array_id)->GlobalInfo = descriptor;

} /* dalib_array_local_define */

/**************************************************************************
*                                                                         *
*   void FUNCTION(dalib_array_shadow) (array_id, is_full_shadow,          *
*                         lov1, rov1, lov2, rov2, lov3, rov3,             *
*                         lov4, rov4, lov5, rov5, lov6, rov6, lov7, rov7) *
*                                                                         *
*   - define the size of the shadow area (will be added to local size     *
*     of every non-empty part)                                            *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_array_shadow) (array_id, is_full_shadow,
                          lov1, rov1, lov2, rov2, lov3, rov3,
                          lov4, rov4, lov5, rov5, lov6, rov6, lov7, rov7)

array_info *array_id; 

int *is_full_shadow; 

int *lov1, *rov1, *lov2, *rov2, *lov3, *rov3, *lov4, *rov4,
    *lov5, *rov5, *lov6, *rov6, *lov7, *rov7;

{ DimInfo  *Dims;
  int      rank;

#ifdef CHECK
  if (!dalib_is_array_info (*array_id))
     { dalib_internal_error ("array_shadow, not array info");
       dalib_stop ();
     }
#endif 

  /* Attention: overlap cannot be defined after reserving memory */

  Dims = (*array_id)->dimensions;
  rank = (*array_id)->rank;

#ifdef DEBUG
  printf ("%d: array_overlap , rank %d\n", pcb.i, rank);
#endif

  switch (rank) {
     case 7 : dalib_set_dim_shadow (Dims + 6, *lov7, *rov7);
     case 6 : dalib_set_dim_shadow (Dims + 5, *lov6, *rov6);
     case 5 : dalib_set_dim_shadow (Dims + 4, *lov5, *rov5);
     case 4 : dalib_set_dim_shadow (Dims + 3, *lov4, *rov4);
     case 3 : dalib_set_dim_shadow (Dims + 2, *lov3, *rov3);
     case 2 : dalib_set_dim_shadow (Dims + 1, *lov2, *rov2);
     case 1 : dalib_set_dim_shadow (Dims + 0, *lov1, *rov1);
  } /* switch */

} /* dalib_array_shadow */

/**************************************************************************
*                                                                         *
*  void dalib_array_free_own_data (array_info descriptor)                 *
*                                                                         *
*     - free own data if it has been allocated                            *
*                                                                         *
**************************************************************************/

void dalib_array_free_own_data (descriptor)

array_info descriptor;

{ if (descriptor->n_data == 0) return;

  /* descriptor has own data allocated, make some consisteny checks */

#ifdef CHECK
  if (descriptor->c_data == NO_DATA)
 
     { dalib_internal_error ("free array descriptor, no data");
       dalib_stop ();
     }

  if (descriptor->dsp_status_flag != DSP_OWN_DATA)

     { char msg [128];

       sprintf (msg, 
                "free array descriptor (%s), has own data, invalid status",
                descriptor->name);

       dalib_internal_error (msg);
       dalib_stop ();
     }
#endif

  if (descriptor->SharedInfo != NO_SHARED)

      dalib_array_shared_free (descriptor);

    else if (descriptor->RemoteInfo != NO_REMOTE)

      dalib_array_remote_free (descriptor);

    else

      dalib_free (descriptor->c_data, descriptor->n_data);

  descriptor->n_data = 0;
  descriptor->c_data = NO_DATA;

} /* dalib_array_free_own_data */

/**************************************************************************
*                                                                         *
*  void dalib_destroy_descriptor (array_id)                               *
*                                                                         *
*     - destroy the descriptor, no more reference to it                   *
*                                                                         *
**************************************************************************/

void dalib_destroy_descriptor (descriptor)

array_info descriptor;

{ /* free data structures for distribution      */

  if (descriptor->DistributeInfo != NO_DISTRIBUTION)
    dalib_dist_free (descriptor->DistributeInfo, descriptor->rank);

  if (descriptor->AlignInfo != NO_ALIGNMENT)
    dalib_align_free (descriptor->AlignInfo, descriptor->rank);

  if (descriptor->SharedInfo != NO_SHARED)
    dalib_shared_free (descriptor->SharedInfo);

  if (descriptor->RemoteInfo != NO_REMOTE)
    dalib_remote_free (descriptor->RemoteInfo);

  dalib_free (descriptor, dalib_array_dsp_size (descriptor->rank));

} /* dalib_destroy_descriptor */

/**************************************************************************
*                                                                         *
*  void dalib_free_descriptor (array_id)                                  *
*                                                                         *
*     - free own data if it has been allocated                            *
*     - decrease references and destroy if it is necessary                *
*                                                                         *
**************************************************************************/

void dalib_free_descriptor (descriptor)

array_info descriptor;

{ /* Step 1 : free own allocated data for the descriptor */

  /* if (descriptor->n_data > 0)  */

  dalib_array_free_own_data (descriptor);
        
  /* Step 2 :  descrease references     */

 if (descriptor->ref_counter <= 0)

    { dalib_internal_error ("free array descriptor, not valid reference");
      dalib_stop ();
    }

  descriptor->ref_counter --;

  if (descriptor->ref_counter == 0)

     dalib_destroy_descriptor (descriptor);

} /* dalib_free_descriptor */

/**************************************************************************
*                                                                         *
*   void FUNCTION(dalib_array_deallocate) (array_info *array_id)          *
*                                                                         *
*     - free descriptor data (called with DEALLOCATE statement)           *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_array_deallocate) (array_id) 

array_info *array_id;

{ array_info descriptor;

#ifdef CHECK
  if (!dalib_is_array_info (*array_id))
     { dalib_internal_error ("array_deallocate, not array info");
       dalib_stop ();
     }
#endif 

  descriptor = *array_id;

  dalib_array_free_own_data (descriptor);

  /* array sizes are no longer defined, so reset distribution       */

  if (descriptor->DistributeInfo != NO_DISTRIBUTION)
    
     dalib_distribution_reset (descriptor);

  /* now the descriptor is only defined, but no DATA */

  descriptor->dsp_status_flag = DSP_DEFINED;

} /* dalib_array_deallocate */

/**************************************************************************
*                                                                         *
* void FUNCTION(dalib_array_free) (array_id)                              *
*                                                                         *
*     - free descriptor, data and distribution/alingment info             *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_array_free) (array_id) 

array_info *array_id;

{ 

#ifdef CHECK
  if (!dalib_is_array_info (*array_id))
     { dalib_internal_error ("array_free, not array info");
       dalib_stop ();
     }
#endif 

  dalib_free_descriptor (*array_id);  /* delete also data */

} /* dalib_array_free */

/**************************************************************************
*                                                                         *
*  Functions for creating memory                                          *
*                                                                         *
*  real A(N1,N2,N3)    is the global array                                *
*                                                                         *
*  A (G_I1, G_I2, G_I3, G_I4)                                             *
*                                                                         *
*  A (L_I1, L_I2, L_I3, L_I4)                                             *
*                                                                         *
*   L_Ik = G_Ik  if it is a serial dimension                              *
*   L_Ik = G_Ik  if it is a block distributed dimension (but a_zero!)     *
*   L_Ik = G_Ik/Np  if it is a cyclic distributed dimension               *
*                   (but a_zero has also be reshifted)                    *
*                   Np is number of processors for this dimension         *
*                                                                         *
*  then it becomes real a(1)                                              *
*                                                                         *
*  dalib_array_allocate (a_dsp, a, 4, a_zero, a_dim1, a_dim2, a_dim3)     *
*                                                                         *
*  a (i,j,k) -> a (a_zero + L_I1 + a_dim1 * L_I2 + ... + a_dim3 * L_I4)   *
*                                                                         *
**************************************************************************/

void dalib_array_malloc (array_id)
 
array_info array_id;

{ int rank, size;

  int first;
  int total [MAX_DIMENSIONS+1];
  unsigned char *a_ptr, *dummy;

  rank = array_id->rank;
  size = array_id->size;

  dalib_array_addressing (array_id, pcb.i, &dummy, &first, total);
 
  /* total[rank] is the total size of the array, but add 1 element
     to allow abitrary alignment with FORTRAN structures            */

  size *= (total[rank] + 1);

  array_id->dsp_status_flag = DSP_OWN_DATA;  /* array has its own data */

  if (size <= 0)

     { array_id->n_data            = 0;
       array_id->c_data            = NO_DATA;

#ifdef DEBUG
       printf ("%d: 0 bytes allocated\n", pcb.i);
#endif
       return;
     }

  a_ptr = (unsigned char *) dalib_malloc (size, "dalib_array_malloc");

#ifdef DEBUG
  printf ("%d: %d bytes allocated, address = %p\n", pcb.i, size, a_ptr);
#endif

  array_id->n_data   = size;          /* number of bytes allocated   */
  array_id->c_data   = a_ptr;         /* data points to dynamic data */
  array_id->f_data   = NO_DATA;       /* not aligned to FORTRAN data */

} /* dalib_array_malloc */

/**************************************************************************
*                                                                         *
* void FUNCTION(dalib_array_access) (array_id, ....)                      * 
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_array_access) (array_id, a_ptr, a_zero, a_dim1, a_dim2, 
                           a_dim3, a_dim4, a_dim5, a_dim6, a_dim7)

array_info *array_id;
unsigned char *a_ptr;

void **a_zero;              /* fiktive zero element            */
int  *a_dim1, *a_dim2, *a_dim3, *a_dim4, *a_dim5, *a_dim6, *a_dim7; 

{ int rank, size;
  int first;
  long offset;
  int total[MAX_DIMENSIONS+1];
  unsigned char *data;

  array_info descriptor;

  descriptor = *array_id;

  if (dalib_is_section_info (descriptor))

     { /* check whether the same addressing scheme can be used */

       if (!dalib_is_full_array_section (descriptor))

          { dalib_internal_error ("no addressing of section possible");
            dalib_stop ();
          }

       descriptor = ((section_info) descriptor)->array_id; 

     }

#ifdef CHECK
  if (!dalib_is_array_info (descriptor))
     { dalib_internal_error ("array_access, not array info");
       dalib_stop ();
     }
#endif 

  /* if no data has been allocated, the access will not do anyting
     ATTENTION: can happen after redistribution of non allocated arrays */

  if (descriptor->f_data == NO_DATA) return;

  if (descriptor->RemoteInfo != NO_REMOTE)

     dalib_array_remote_init (descriptor);

  rank = descriptor->rank;
  size = descriptor->size;

#ifdef DEBUG
  printf ("%d: access array for descriptor %p, rank = %d\n",
          pcb.i, descriptor, rank);
#endif 

  dalib_array_addressing (descriptor, pcb.i, &data, &first, total);
 
  switch (rank) {
 
    case 7 : *a_dim7 = total[7];
    case 6 : *a_dim6 = total[6];
    case 5 : *a_dim5 = total[5];
    case 4 : *a_dim4 = total[4];
    case 3 : *a_dim3 = total[3];
    case 2 : *a_dim2 = total[2];
    case 1 : *a_dim1 = total[1];
    case 0 : break;
 
  } /* switch */

#ifdef DEBUG
  printf ("%d: will access %d * %d bytes\n", pcb.i, total[rank], size);
  printf ("%d: offset for first element = %d\n", pcb.i, first);
  printf ("%d: dyn addr = %p, static addr = %p\n", pcb.i, data, a_ptr);
#endif 

  /* that is pointer to the allocated memory */

  offset = data - a_ptr;

  if (offset % size)

    { printf ("array_access: static addr = %p, dyn addr = %p\n", a_ptr, data);
      printf ("              diff = %d is not multiple of size = %d\n",
                             offset, size);
      dalib_internal_error ("alignment problem, use flag -f for compilation");
      dalib_stop ();
    }

  offset = offset / size;

  *a_zero = (void *) (offset + 1 - first);

} /* dalib_array_access */

/**************************************************************************
*                                                                         *
*  void dalib_array_full_allocate (array_info dsp, char *f_ptr)           *
*                                                                         *
*   - allocates the memory for an array descriptor                        *
*   - set the f_data pointer of dsp aligned to f_ptr                      *
*                                                                         *
**************************************************************************/

void dalib_array_full_allocate (dsp, f_ptr)

array_info dsp;
unsigned char *f_ptr;

{ long offset;

#ifdef DEBUG
  printf ("%d: allocate/attach memory for descriptor %p\n", pcb.i, dsp);
#endif

  if (dsp->SharedInfo != NO_SHARED)
 
     dalib_array_shared_malloc (dsp);

   else if (dsp->RemoteInfo != NO_REMOTE)
 
     dalib_array_remote_malloc (dsp);

   else

     dalib_array_malloc (dsp);
 
  /* now calcuate dsp->f_data = dsp->c_data + offset in such a way
     that f_ptr = dsp_f_data + X * size                             */

  offset = (f_ptr - dsp->c_data);

#ifdef DEBUG
  printf ("%d: offset = %d\n", pcb.i, offset);
#endif

  offset = offset % dsp->size;
  if (offset < 0) offset += dsp->size;
  dsp->f_data = dsp->c_data + offset;

#ifdef DEBUG
  if (offset)
  printf ("%d: n_data = %d, c_data = %p, ptr = %p, offset = %d, f_data = %p\n",
          pcb.i, dsp->n_data, dsp->c_data, f_ptr, 
          offset, dsp->f_data);
#endif

} /* dalib_array_full_allocate */

/**************************************************************************
*                                                                         *
* void FUNCTION (dalib_array_allocate) (array_id, a_ptr)                  * 
*                                                                         *
*  - allocate memory for a given array_descriptor                         *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_array_allocate) (array_id, a_ptr)
 
array_info *array_id;
unsigned char *a_ptr;

{ array_info descriptor;

#ifdef CHECK
  if (!dalib_is_array_info (*array_id))
     { dalib_internal_error ("array_allocate, not array info");
       dalib_stop ();
     }
#endif 

#ifdef DEBUG
  printf ("%d: dalib_allocate for descriptor %p\n", pcb.i, *array_id);
#endif

  descriptor = *array_id;

  dalib_array_full_allocate (*array_id, a_ptr);

} /* dalib_array_allocate */

void FUNCTION(dalib_carray_allocate) 

 ARGS(`array_id, STRING_ARG(a_ptr)')

array_info *array_id;
STRING_ARG_DECL(a_ptr);

{ FUNCTION(dalib_array_allocate) (array_id, STRING_PTR(a_ptr));

} /* dalib_carray_allocate */

void FUNCTION(dalib_carray_access) 

 ARGS(`array_id, STRING_ARG(a_ptr), a_zero, a_dim1, a_dim2, 
       a_dim3, a_dim4, a_dim5, a_dim6, a_dim7')

array_info *array_id;
STRING_ARG_DECL(a_ptr);

void **a_zero;              /* fiktive zero element            */
int  *a_dim1, *a_dim2, *a_dim3, *a_dim4, *a_dim5, *a_dim6, *a_dim7; 

{ FUNCTION(dalib_array_access)
  (array_id, STRING_PTR(a_ptr), a_zero, a_dim1, a_dim2,
       a_dim3, a_dim4, a_dim5, a_dim6, a_dim7);

} /* dalib_carray_access */

/**************************************************************************
*                                                                         *
*  int dalib_array_has_data (array_info array_id)                         *
*                                                                         *
*  - global consistent flag if data has been set or allocated             *
*                                                                         *
*  Note :  (array_id->data == NO_DATA) is not globally consistent         *
*                                                                         *
**************************************************************************/

int dalib_array_has_data (array_id)

array_info array_id;

{ if (array_id->dsp_status_flag == DSP_UNDEFINED) return (0);
  if (array_id->dsp_status_flag == DSP_DEFINED) return (0);

  return (1);  /* so data has been allocated or set */

} /* dalib_array_has_data */

/**************************************************************************
*                                                                         *
*  int FUNCTION(dalib_allocated) (array_id)                               *
*                                                                         *
*  - Problem: must be consistent among all processors                     *
*                                                                         *
**************************************************************************/

int FUNCTION(dalib_allocated) (array_id)
array_info *array_id;

{ if ((*array_id)->dsp_status_flag == DSP_OWN_DATA) return (F_TRUE);
  return (0);

} /* dalib_allocated */

/**************************************************************************
*                                                                         *
* void FUNCTION(dalib_array_setdata) (array_id, data) * 
*                                                                         *
*   - data points to array (lb1-l_ov1, ...., lbk-l_ovk)                   *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_array_setdata) (array_id, a_ptr) 

array_info *array_id;
unsigned char *a_ptr;

{ /* dalib_array_setdata */

#ifdef CHECK
  if (!dalib_is_array_info (*array_id))
     { dalib_internal_error ("array_setdata, not array info");
       dalib_stop ();
     }
#endif 

  (*array_id)->f_data          = a_ptr;
  (*array_id)->dsp_status_flag = DSP_PTR_DATA;

} /* dalib_array_setdata */

void FUNCTION(dalib_carray_setdata) ARGS(`array_id, STRING_ARG(a_ptr)') 

array_info *array_id;
STRING_ARG_DECL(a_ptr);

{ FUNCTION(dalib_array_setdata) (array_id, STRING_PTR(a_ptr)); }

/**************************************************************************
*                                                                         *
*  int dalib_array_local_size (array_id)                                  *
*                                                                         *
*   - returns number of elements for the local part of the array          *
*                                                                         *
**************************************************************************/

int dalib_array_local_size (array_id)

array_info array_id;
 
{ int i, rank;
  DimInfo *dims;
 
  int low, high, stride, size;
 
#ifdef CHECK
  if (!dalib_is_array_info (array_id))
     { dalib_internal_error ("array_local_size, not array info");
       dalib_stop ();
     }
#endif 

  rank = array_id->rank;
  dims = array_id->dimensions;
 
  size = 1;
 
  for (i=0; i<rank; i++)
 
   { low    = dims->local_size[0];
     high   = dims->local_size[1];
     stride = dims->local_size[2];
     if (low <= high)
        size *= (high - low + stride) / stride;
      else
        size = 0;
     dims++;
   }
 
  return (size);
 
}  /* dalib_array_local_size */

/**************************************************************************
*                                                                         *
*  void dalib_ptr_set_array (array_info ptr_dsp, array_info arr_dsp)      *
*                                                                         *
**************************************************************************/

void dalib_ptr_set_array (pointer, target)

array_info pointer, target;

{ int i, rank;

  rank = target->rank;

  if (rank != pointer->rank)

     { dalib_internal_error ("ptr_define (array), rank mismatch");
       dalib_stop ();
     }

  /* now we copy the full size information of target */

  for (i=0; i<rank; i++)

     pointer->dimensions[i] = target->dimensions[i];

  if (dalib_array_map_underspecified (pointer)) 

     { dalib_internal_error ("ptr_define (array), must inherit mapping");
       dalib_stop ();
     }

  if (pointer->DistributeInfo != NO_DISTRIBUTION)
     dalib_dist_local_sizes (pointer);
   else if (pointer->AlignInfo != NO_ALIGNMENT)
     dalib_align_local_sizes (pointer);
   else
     dalib_full_local_sizes (pointer);

  /* pointer gets data pointer of the target */

  pointer->n_data = 0;
  pointer->c_data = NO_DATA;
  pointer->f_data = target->f_data;

  pointer->dsp_status_flag = DSP_PTR_DATA;

} /* dalib_ptr_set_array */

/**************************************************************************
*                                                                         *
*  void FUNCTION(dalib_ptr_define)(array_info *ptr_dsp, *arr_dsp)         *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_ptr_define) (ptr_dsp, arr_dsp)

array_info *ptr_dsp, *arr_dsp;

{ array_info pointer, target;

  int i, rank;

  pointer = *ptr_dsp;
  target  = *arr_dsp;

#ifdef CHECK
  if (!dalib_is_array_info (pointer))
     { dalib_internal_error ("ptr_define, pointer not array info");
       dalib_stop ();
     }
  if (!dalib_is_array_info (target))
     { dalib_internal_error ("ptr_define, target not array info");
       dalib_stop ();
     }
#endif
 
  rank = target->rank;

  if (rank != pointer->rank)

     { dalib_internal_error ("ptr_define, rank mismatch");
       dalib_stop ();
     }

  /* now we copy the full size information of target */

  for (i=0; i<rank; i++)

     pointer->dimensions[i] = target->dimensions[i];

  /* at this time we do not support pointer assignments for distr. stuff
     because freeing the descriptor of pointer might cause serious problems */

  if (target->DistributeInfo != NO_DISTRIBUTION)

     { dalib_internal_error ("ptr_define, distributed target");
       dalib_stop ();
     }

  if (target->AlignInfo != NO_ALIGNMENT)

     { dalib_internal_error ("ptr_define, aligned target");
       dalib_stop ();
     }

  if (target->dsp_status_flag == DSP_UNDEFINED)

     { dalib_internal_error ("ptr_define, target undefined");
       dalib_stop ();
     }

  /* pointer gets data pointer of the target */

  pointer->n_data = 0;
  pointer->c_data = NO_DATA;
  pointer->f_data = target->f_data;

  pointer->dsp_status_flag = DSP_PTR_DATA;

} /* dalib_ptr_define */

/**************************************************************************
*                                                                         *
*  void dalib_inc_array_ref (array_info array_dsp)                        *
*  void dalib_dec_array_ref (array_info array_dsp)                        *
*                                                                         *
*    - routines are called for align and dealign                          *
*                                                                         *
**************************************************************************/

void dalib_inc_array_ref (array_dsp)

array_info array_dsp;

{
#ifdef CHECK
  if (!dalib_is_array_info (array_dsp))
     { dalib_internal_error ("inc array ref, template not array descriptor");
       dalib_stop ();
     }
#endif 

  array_dsp->ref_counter ++;

} /* dalib_inc_array_ref */

void dalib_dec_array_ref (array_dsp)

array_info array_dsp;

{
#ifdef CHECK
  if (!dalib_is_array_info (array_dsp))
     { dalib_internal_error ("dec array ref, template not array descriptor");
       dalib_stop ();
     }
#endif 

  if (array_dsp->ref_counter <= 0)

    { dalib_internal_error ("dec array ref, not valid reference");
      dalib_stop ();
    }

  array_dsp->ref_counter --;

  if (array_dsp->ref_counter == 0)

    { /* the template is no longer used, so we can now free it */

      dalib_destroy_descriptor (array_dsp);

    }

} /* dalib_dec_array_ref */

