/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*                                                                         *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Jun 95                                                   *
*  Last Update : Jun 95                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : addressing                                               *
*                                                                         *
*  Function: addressing of distributed/shared/global arrays               *
*                                                                         *
*  dalib_array_local_addressing (array_info *array_id,                    *
*                                int *first, int total[])                 *
*                                                                         *
*  dalib_array_global_addressing (array_info *array_id,                   *
*                                 int *first, int total[])                *
*                                                                         *
*  void dalib_make_array_ddt (dd_type *new_ddt, array_info array_id)      *
*                                                                         *
*  void dalib_make_perm_section_ddt (dd_type *new_ddt,                    *
*                                 section_info section_id, int perm[])    *
*                                                                         *
**************************************************************************/

#include "dalib.h"

#undef DEBUG
#define CHECK

/**************************************************************************
*                                                                         *
*  int dalib_dim_local_extent (DimInfo)                                   *
*                                                                         *
*  - only this routine decides about packing or not                       *
*                                                                         *
**************************************************************************/
 
int dalib_dim_local_extent (dim)
 
DimInfo *dim;
 
{ int size;
  int lb, ub;
 
  lb = dim->local_size[0];
  ub = dim->local_size[1];
 
  if (lb > ub)
     size = 0;
   else
     size = (ub - lb + 1) + dim->shadow[0] + dim->shadow[1];
 
  return (size);
 
} /* dalib_dim_local_extent */
 
/**************************************************************************
*                                                                         *
*  int dalib_dim_global_extent (DimInfo)                                  *
*                                                                         *
**************************************************************************/
 
int dalib_dim_global_extent (dim)
 
DimInfo *dim;
 
{ int size;
  int lb, ub;
 
  lb = dim->global_size[0];
  ub = dim->global_size[1];
 
#ifdef DEBUG
  printf ("dalib_dim_global_extent : lb = %d, ub = %d\n", lb, ub);
#endif
 
  if (lb > ub)
     size = 0;
   else
     size = (ub - lb + 1) + dim->shadow[0] + dim->shadow[1];
 
  return (size);
 
} /* dalib_dim_global_extent */

/**************************************************************************
*                                                                         *
*  dalib_array_local_addressing (array_info *array_id,                    *
*                                int *first, int total[])                 *
*                                                                         *
*   first is difference between A->data and A(0,....,0)                   *
*    and A(lb1-l_ov1, ..., lbk-l_ovk)                                     *
*                                                                         *
*   total[i] is difference between A(...,xi,...) and A(...,xi+1,...)      *
*            at dim i (first dimension is 0)                              *
*                                                                         *
*   total[rank] is total size of the array (0 for no elements here)       *
*                                                                         *
**************************************************************************/
 
void dalib_array_local_addressing (array_id, first, total)
 
array_info array_id;
int        *first;
int        total[];
 
{ int i, rank;
  DimInfo *array_dim;
 
  array_dim  = array_id->dimensions;
 
  rank = array_id->rank;
 
  total[0] = 1;
  *first   = 0;
 
  for (i=0; i < rank; i++)
 
    { *first     +=  (array_dim->local_size[0] - array_dim->shadow[0])
                     * total[i];
 
      total[i+1]  = total[i] * dalib_dim_local_extent (array_dim);
 
      array_dim++;
 
#ifdef DEBUG
      printf ("%d: local addressing, i = %d, first = %d, total = %d\n",
               pcb.i, i, *first, total[i+1]);
#endif
 
    }  /* for all dimensions */
 
} /* dalib_array_local_addressing */

int dalib_array_local_data_size (array_id)
 
array_info array_id;
 
{ int i, rank, size;
  DimInfo *array_dim;
 
  array_dim  = array_id->dimensions;
  rank       = array_id->rank;
 
  size = 1;

  for (i=0; i < rank; i++)
 
    size *= dalib_dim_local_extent (array_dim++);
 
  if (size > 0) return size; 
     else       return 0;

} /* dalib_array_local_data_size */

/**************************************************************************
*                                                                         *
*  dalib_array_global_addressing (array_info *array_id,                   *
*                                int *first, int total[])                 *
*                                                                         *
**************************************************************************/

void dalib_array_global_addressing (array_id, first, total)
 
array_info array_id;
int        *first;
int        total[];
 
{ int i, rank;
  DimInfo *array_dim;
 
#ifdef DEBUG
  dalib_print_array_info (array_id);
#endif
 
  array_dim  = array_id->dimensions;
 
  rank = array_id->rank;
 
  total[0] = 1;
  *first   = 0;
 
  for (i=0; i < rank; i++)
 
    { *first     +=  (array_dim->global_size[0] - array_dim->shadow[0])
                     * total[i];
 
      total[i+1]  = total[i] * dalib_dim_global_extent (array_dim);
 
      array_dim++;
 
#ifdef DEBUG
      printf ("%d: global addressing, i = %d, first = %d, total = %d\n",
               pcb.i, i, *first, total[i+1]);
#endif
 
    }  /* for all dimensions */
 
} /* dalib_array_global_addressing */

int dalib_array_global_data_size (array_id)
 
array_info array_id;
 
{ int i, rank, size;
  DimInfo *array_dim;
 
  array_dim  = array_id->dimensions;
  rank       = array_id->rank;
 
  size = 1;

  for (i=0; i < rank; i++)
 
    size *= dalib_dim_global_extent (array_dim++);
 
  if (size > 0) return size; 
     else       return 0;

} /* dalib_array_global_data_size */

/**************************************************************************
*                                                                         *
*  DALIB interface : IMPORTANT                                            *
*                                                                         *
*  dalib_array_addressing (array_info array_id, int pid,                  *
*                          char **data, int *first, int total[])          *
*                                                                         *
*   data is pointer to local part of array on processor pid               *
*                                                                         *
*   first is difference between A->data and A(0,....,0)                   *
*    and A(lb1-l_ov1, ..., lbk-l_ovk)                                     *
*                                                                         *
*   total[i] is difference between A(...,xi,...) and A(...,xi+1,...)      *
*            at dim i (first dimension is 0)                              *
*                                                                         *
*   total[rank] is total size of the array (0 for no elements here)       *
*                                                                         *
**************************************************************************/

void dalib_array_addressing (array_id, pid, data, first, total)
 
array_info array_id;     /*  input        */
int        pid;          /*  parameters   */

unsigned char **data;
int        *first;
int        total[];
 
{  *data = array_id->f_data;

   if (array_id->global_addressing)
 
       dalib_array_global_addressing (array_id, first, total);

     else

       dalib_array_local_addressing (array_id, first, total);

} /* dalib_array_addressing */

int dalib_array_data_size (array_id)

array_info array_id;

{ if (array_id->SharedInfo != NO_SHARED)

       return dalib_array_global_data_size (array_id);

     else

       return dalib_array_local_data_size (array_id);

} /* dalib_array_data_size */

/**************************************************************************
*                                                                         *
*  void dalib_make_array_ddt (dd_type *new_ddt, array_info array_id)      *
*                                                                         *
*    - creates ddt for my local array part, one elem has size bytes       *
*                                                                         *
*   Note : usually new_ddt is contiguous, but most not be the             *
*          case (e.g. with overlapping, not packed cyclic dims)           *
*                                                                         *
**************************************************************************/
 
void dalib_make_array_ddt (new_ddt, array_id)
 
dd_type     *new_ddt;
array_info  array_id;
 
{ unsigned char *array_data;   /* points to first element of section */
  DimInfo  *array_dim;

  int      i, array_rank, array_size;
 
  int n    [MAX_DIMENSIONS];
  int inc  [MAX_DIMENSIONS];
 
  int first;      /* first will be the offset to A(l1-ov1,...,ln-ovn2)  */
  int total [MAX_DIMENSIONS + 1];

  int low;
  int lb, ub, str;
 
#ifdef DEBUG
  printf ("%d: dalib_make_array_ddt of %d\n", 
           pcb.i, array_id);
#endif 

#ifdef CHECK
  if (!dalib_is_array_info (array_id))
     { dalib_internal_error ("make_array_ddt, not array info");
       dalib_stop ();
     }
#endif

  array_dim  = array_id->dimensions;
  array_rank = array_id->rank;
  array_size = array_id->size;
 
#ifdef DEBUG
  printf ("%d: dalib_make_array_ddt of %d, rank = %d\n", 
           pcb.i, array_id, array_rank);
#endif 

  dalib_array_addressing (array_id, pcb.i, &array_data, &first, total);

  /* for section compute A(lb1,...,lbk) and ni, inci */

  low = 0;

       /************************************************
       *                                               *
       *    A(lb,....,lbk) = a_data + lb1 * 1          *
       *                       + lb2 * total[1]        *
       *                       + lb3 * total[2]        *
       *                 + ... + lbk * totalk[k]       *
       *                                               *
       ************************************************/
 
  for (i=0; i<array_rank; i++)

     { lb   = array_dim->local_size[0];
       ub   = array_dim->local_size[1];
       str  = array_dim->local_size[2];
       low    += lb * total[i];
       inc[i]  = total[i] * str;   /* is the stride */
       n[i]    = dalib_range_size (lb, ub, str);
       array_dim ++;
     }
 
#ifdef DEBUG
  printf ("%d: dalib_make_array_ddt of %d, low = %d, first =%d\n",
           pcb.i, array_id, low, first);
#endif 

  array_data += (low - first ) * array_size;

  if (total[array_rank] == 0)
 
      dalib_ddt_def_section (new_ddt, array_data, 0, 0);
 
    else
 
      dalib_ddt_def_section (new_ddt, array_data, array_size,
                             array_rank, inc, n);

} /* dalib_make_array_ddt */

/**************************************************************************
*                                                                         *
*  static void get_dim (dim, sdim, srank, inc, n, sdim, first, total)     *
*                                                                         *
*    dim : describes dimension of the full array                          *
*   sdim : describes dimension of the section of the array                *
*                                                                         *
*   first : will describe address offset for the first element            *
*   total : will describe dimension extionsion                            *
*                                                                         *
**************************************************************************/

static void get_dim (gdim, sdim, size, srank, inc, n, first, zero_section)

DimInfo *gdim;
SecDimInfo *sdim;
int size;             /* diff for two succesive elements in this dim */
int *srank;           /* increments for rank of section */
int inc[], n[];
int *first, *zero_section;

{  int lb, ub, str, sn;

   /* first  will be address of A (lb1,lb2,...,lbn) */

   lb   = sdim->local_range[0];
   ub   = sdim->local_range[1];
   str  = sdim->local_range[2];

   sn   = dalib_range_size (lb, ub, str);

   *first += lb * size;

   if (sdim->is_range == 0)

      {  /* A (....,val,....), only first has to be updated */

         if (sn == 0) *zero_section = 0;

      }

   else if (sn == 0)

      /* attention can be (gdim->local_size[0] > gdim->local_size[1]) */

      { *zero_section = 0;
        n [*srank] = 0;
        (*srank)++;
      }

    else
      { inc[*srank] = size * str;
        n  [*srank] = sn;              /* local section size */
#ifdef DEBUG
        printf ("inc[%d] = %d, n[%d] = %d\n", *srank, inc[*srank],
                                              *srank, n[*srank]);
#endif
        (*srank)++;
      }

#ifdef DEBUG

   printf ("%d: get_dim, lb = %d, ub = %d, size = %d, srank = %d, 1st = %d\n",
           pcb.i, lb, ub, size, *srank, *first);

#endif

}  /* get_dim */

/**************************************************************************
*                                                                         *
*  in  : section (dsp) A (lb1:ub1:str1, lb2:ub2:str2, ..., lbn:ubn:strn)  *
*                                                                         *
*  out : ->  first is A (lb1,lb2,...,lbn)                                 *
*  out : ->  n1 = size of lb1:ub1:str1, ...., nk = size of lbk:ubk:strk   *
*                                                                         *
**************************************************************************/

void dalib_section_offset_dsp (section_id, rank, first, n, inc)

section_info section_id;
int          *first;
int          *rank;
int          n[];
int          inc[];

{ array_info array_id;
  int        array_rank;
  char       *array_data;   /* not needed, dummy */
  DimInfo    *array_dims;

  int        array_first;
  int        array_total[MAX_DIMENSIONS];

  int        i;
  int        zero_section;

  SecDimInfo *section_dims;

  array_id   = section_id->array_id;

  /* start with descriptor of full array */

  dalib_array_addressing (array_id, pcb.i, 
                          &array_data, &array_first, array_total);

  /*

  printf ("array addressing, first = %d, total = %d\n", 
           array_first, array_total[0]);
  */

  array_dims = array_id->dimensions;
  array_rank = array_id->rank;

  section_dims = section_id->dimensions;

  zero_section = 1;   /* default is not a zero section           */

  *first  = -array_first;  /* start with no offset within array  */
  *rank   = 0;             /* increase only for sdim[j].is_range */

  for (i=0; i<array_rank; i++)

    { get_dim (array_dims++, section_dims++, array_total[i],
               rank, inc, n, first, &zero_section);
    }

  /*
  printf ("section addressing, first = %d\n", *first);
  */

  if (zero_section == 0) *rank = -1;   /* -1 for empty section */

} /* dalib_section_offset_dsp */

/**************************************************************************
*                                                                         *
*  void dalib_make_perm_section_ddt (dd_type *new_ddt,                    *
*                                 section_info section_id, int perm[])    *
*                                                                         *
*    - creates ddt for my local section, one elem has size bytes          *
*                                                                         *
*      A (lb1:ub1:str1, lb2:ub2:str2, ..., lbn:ubn:strn)                  *
*                                                                         *
*   ->  first is A (lb1,lb2,...,lbn)                                      *
*                                                                         *
*   ->  n1 = size of lb1:ub1:str1, ...., nk = size of lbk:ubk:strk        *
*   ->  inc1 = str1, inc2 = m1 * str1, ..., incn = m1 * .. * mn-1 * strn  *
*                                                                         *
**************************************************************************/

void dalib_make_perm_section_ddt (new_ddt, section_id, perm)

dd_type      *new_ddt;
section_info section_id;
int          perm[];

{ unsigned char *array_data;   /* points to first element of section */
  DimInfo  *array_dims;
  int      i, array_rank, section_rank, array_size;
  SecDimInfo *section_dims;

  array_info array_id;

  int n    [MAX_DIMENSIONS];
  int inc  [MAX_DIMENSIONS];
  int help [MAX_DIMENSIONS];

  int first;      /* first will be the offset to A(gl1-ov1,...,gln-ovn2)
                     but then it will be the offset to A(l1,l2,...,ln)    */

  int total [MAX_DIMENSIONS + 1];

  int zero_section;

#ifdef CHECK
  if (!dalib_is_section_info (section_id))
    { dalib_internal_error ("make_section_ddt : not a section");
      dalib_stop ();
    }
#endif

#ifdef DEBUG
  printf ("%d: dalib_make_perm_section_ddt\n", pcb.i);
  dalib_print_section_info (section_id);
#endif

  array_id = section_id->array_id;

  array_dims = array_id->dimensions;
  array_rank = array_id->rank;
  array_size = array_id->size;

  section_dims = section_id->dimensions;

  /* start with descriptor of full array */

  dalib_array_addressing (array_id, pcb.i, &array_data, &first, total);

  zero_section = 1;       /* default is not a zero section           */
  first        = -first;  /* subtract it from start point of section */

  section_rank = 0;  /* increase only for sdim[j].is_range */

  for (i=0; i<array_rank; i++)
    { get_dim (array_dims++, section_dims++, total[i],
               &section_rank, inc, n, &first, &zero_section);
    }

#ifdef DEBUG
  printf ("%d: get section, rank = %d of %d, first=%d, size=%d, zero=%d\n",
           pcb.i, section_rank, array_rank, first, array_size, zero_section);
#endif

  array_data += first * array_size;

  if (perm != (int *) 0)

     { /* apply permutation to the arrays inc and n */

#ifdef DEBUG
       printf ("%d: permute section, srank = %d, p = %d %d %d %d %d %d %d\n",
               pcb.i, section_rank, perm[0], perm[1], perm[2], perm[3],
                                    perm[4], perm[5], perm[6]);
#endif

       for (i=0; i<section_rank; i++)
           help[i] = inc[i];
       for (i=0; i<section_rank; i++)
           inc[i] = help[perm[i]];
       for (i=0; i<section_rank; i++)
           help[i] = n[i];
       for (i=0; i<section_rank; i++)
           n[i] = help[perm[i]];

     } /* permutations have been applied */

  if (zero_section == 0)

     dalib_ddt_def_section (new_ddt, array_data, 0, 0);

   else

     dalib_ddt_def_section (new_ddt, array_data, array_size, 
                            section_rank, inc, n);

} /* dalib_make_perm_section_ddt */

#undef DEBUG

void dalib_make_section_ddt (new_ddt, section_id)

dd_type      *new_ddt;
section_info section_id;

{ /* make section ddt without any permutation */

   dalib_make_perm_section_ddt (new_ddt, section_id, (int *) 0);

} /* dalib_make_section_ddt */

/*******************************************************************
*                                                                  *
*   dalib_make_global_offsets (global_offsets, nr, base_dsp,       *
*                              mask, indexes)                      *
*                                                                  *
*    input  :  nr       : number of values                         *
*              base_dsp : array that is indirectly accessed        *
*              mask     : index values are not used                *
*                                                                  *
*    output :  int global_offsets[]                                *
*                                                                  *
*******************************************************************/
 
void dalib_make_global_offsets (global_offsets, nr,
                                base_dsp, mask, indexes)
 
int global_offsets[];
int nr;
int *indexes[];
int mask[];
array_info base_dsp;
 
{ int i, dim, rank;
  int zero;
  int extensions[MAX_DIMENSIONS+1];
 
  rank = base_dsp->rank;
  dalib_array_global_addressing (base_dsp, &zero, extensions);
 
#pragma vdir nooverlap(global_offsets,indexes,extensions)

  if (mask == (int *) 0)
 
     { /* there is no mask */
 
#ifdef VECTOR

       for (i=0; i<nr; i++) global_offsets[i] = -zero;
       for (dim=0; dim < rank; dim++)
          { int ext, *ind;
            ext = extensions[dim];
            ind = indexes[dim];
#pragma vdir nooverlap (global_offsets,ind)
            for (i=0; i<nr; i++) global_offsets[i] += ind[i] * ext;
           }

#else
       for (i=0; i<nr; i++)
 
          { global_offsets[i] = -zero;
            for (dim=0; dim < rank; dim++)
                global_offsets[i] += *(indexes[dim]+i) * extensions[dim];
#ifdef DEBUG
            printf ("%d: global %d -> local %d\n",
                    pcb.i, *(indexes[0]+i), global_offsets[i]);
#endif
          } /* end for */
#endif

     }
 
    else
 
     { /* there is a mask avaialable */
 
#ifdef DEBUG
       printf ("%d: make_global_offsets with mask (nr = %d)\n", pcb.i, nr);
#endif
 
       for (i=0; i<nr; i++)
         if (mask[i])
            { global_offsets[i] = -zero;
              for (dim=0; dim < rank; dim++)
                  global_offsets[i] += *(indexes[dim]+i) * extensions[dim];
            } /* end for */
         else
              global_offsets[i] = -1;
     }
 
} /* dalib_make_global_offsets */

/*******************************************************************
*                                                                  *
*   dalib_addr_global_to_local (base_dsp, globals, locals)         *
*                                                                  *
*   - translate globals[i] to locals[i] for 0 <= i < nr)           *
*                                                                  *
*   global_address ->  -g_zero + i1 * g_dim1 + ...+ i<n> g_dim<n>  *
*   local_address  ->  -l_zero + i1 * l_dim1 + ...+ i<n> l_dim<n>  *
*                                                                  *
*******************************************************************/

void dalib_addr_global_to_local (base_dsp, nr, globals, locals)

array_info base_dsp;
int globals[];
int locals[];
int nr;

{ int g_zero, g_dims[MAX_DIMENSIONS+1];
  int g_lb[MAX_DIMENSIONS];
  int l_zero, l_dims[MAX_DIMENSIONS+1];
  int i, j, rank;
  int gaddr, laddr, index, haddr;

  char *dummy;

  rank = base_dsp->rank;

  /* beside the extensions and fictive zero we need the lower bounds
     (attention for shadow) in every dimension                       */

  for (i=0;i<rank;i++)
      g_lb[i] =    base_dsp->dimensions[i].global_size[0]
                 - base_dsp->dimensions[i].shadow[0];

  dalib_array_global_addressing (base_dsp, &g_zero, g_dims);

  /* look how I do my addressing here */

  dalib_array_addressing  (base_dsp, pcb.i, &dummy, &l_zero, l_dims);

  /* g_dims[2] = N1*N2, g_dims[1] = N1, g_dims[0] = 1 */

  if (rank == 1)

    {
#pragma vdir nooverlap(globals,locals)
  for (i=0; i<nr; i++)
       locals[i] = globals[i] + g_lb[0] - l_zero;
     }

   else

  {
    for (i=0; i<nr; i++)

     { gaddr = globals[i];
       haddr = gaddr;
       laddr = 0;

       for (j=rank-1;j>=0;j--)

          { /* determine i<j> */
            index = gaddr / g_dims[j];
            gaddr -= index * g_dims[j];
            /* index is now in 0..N-1, move to lb..ub */
            index += g_lb[j];
            laddr += index * l_dims[j];
          }
       locals[i] = laddr - l_zero;
#ifdef DEBUG
       printf ("%d: global addr %d -> local addr %d\n", 
                pcb.i, haddr, locals[i]);
#endif
     }

   }

} /* dalib_addr_global_to_local */

/**************************************************************************
*                                                                         *
*  int dalib_local_offset ()                                              *
*                                                                         *
**************************************************************************/

int dalib_local_offset (array_id, global_indices)
 
array_info array_id;
int global_indices[];
 
{ int lb, ub;     /* lb:ub is global size of one dimension   */
  int offset;
  int i, rank;
  int zero;  
  int total[MAX_DIMENSIONS + 1];
 
  rank = array_id->rank;

  dalib_array_local_addressing (array_id, &zero, total);
 
  offset = -zero;

  for (i=0; i<rank; i++)
    offset += global_indices[i] * total[i];
 
  return (offset);
 
} /* dalib_local_offset */

/*******************************************************************
*                                                                  *
*  dalib_make_remote_offsets (local_offsets, nr, base_dsp,         *
*                             owner, mask, indexes)                *
*                                                                  *
*   in   :  nr (number of indexes)                                 *
*           base_dsp (descriptor of accessed array)                *
*           owner[nr] are the owner of the indexes                 *
*           mask[nr] if access is masked                           *
*           indexes*[rank] pointers to the index arrays            *
*                                                                  *
*   out  :  local_offsets is array of local offsets                *
*                                                                  *
*******************************************************************/

void dalib_make_remote_offsets (local_offsets, nr, base_dsp,
                                owner, mask, indexes)

int local_offsets[];
int nr;
int *indexes[];
int owner[];
int mask[];
array_info base_dsp;

{ int i, dim, rank;
  int zero;
  int extensions[MAX_DIMENSIONS+1];

  /* can be that this is called for computing schedule */

#ifdef DEBUG
  printf ("%d: make_remote_offsets for %d pointers\n",
           pcb.i, nr;
#endif

  dalib_array_remote_init (base_dsp); 

  rank = base_dsp->rank;

  if (mask == (int *) 0)

     { /* there is no mask */

       for (i=0; i<nr; i++)

          { dalib_array_remote_addressing (base_dsp, owner[i],
                                           &zero, extensions);
            local_offsets[i] = -zero;
            for (dim=0; dim < rank; dim++) 
                local_offsets[i] += *(indexes[dim]+i) * extensions[dim];

#ifdef DEBUG
  printf ("%d: global %d -> local %d\n",
           pcb.i, *(indexes[0]+i), local_offsets[i]);
#endif
          } /* end for */
     }

    else 

     { /* there is a mask avaialable */

       for (i=0; i<nr; i++)

         if (mask[i])

            { dalib_array_remote_addressing (base_dsp, owner[i],
                                             &zero, extensions);
              local_offsets[i] = -zero;
              for (dim=0; dim < rank; dim++) 
                  local_offsets[i] += *(indexes[dim]+i) * extensions[dim];
            } /* end for */
         else
              local_offsets[i] = -1;
     }

} /* dalib_make_remote_offsets */

