/*
 *   pt2pt.m4  --  part of DALIB
 *
 *   DALIB is run-time library of Adaptor.
 *   Adaptor is implementation of High Performance Fortran
 *   Copyright(c) GMD St. Augustin, Germany
 *
 *   System-dependent point-to-point communications over Embedded Parix system.
 *   Written by Ilya Evseev, CSA, Russia~. Date: Nov 1998
 *   Contact e-mail: evseev@csa.ru
 */

#include <stdio.h>
#include <string.h>

#include "dalib_epx.h"


/*------------------------------------------------------*
 *  Data and routines common for all DALIB_EPX modules  *
 *------------------------------------------------------*/

mailbox_t mailbox[MAXP];  /* mailboxes for connectivity with each node */

int topoId = -1;          /* virtual topology ident */


/*
 *  Print formatted message and abort execution.
 */
void dalib_epx_stop( char *format, ... )
{
    char buf[256];
    va_list ap;
    va_start( ap, format );
    vsprintf( buf, format, ap );
    va_end( ap );
    dalib_internal_error( buf );
    dalib_stop();
}

/*-------------------------------*
 *  end of common routines/data  *
 *-------------------------------*/


/*
 *  Note: don't use memset() !
 *  It destroys contents of LinkCB field !
 */
void machine_mpi_init( void )
{
    int i;
    mailbox_t *p;
    for( i=0, p=mailbox; i < ELEMS(mailbox); i++, p++ )
        p->msgbuf=NULL;
}


/*-------------------------------------------------*
 *  waiting for completion of last sending to pid  *
 *-------------------------------------------------*/

void machine_wait_send( int pid )
{
    ASync( topoId, pid );
    if( mailbox[pid].msgbuf )
        dalib_free( mailbox[pid].msgbuf, mailbox[pid].msglen );
    mailbox[pid].msgbuf = NULL;
}


void machine_mpi_exit( void )
{
    int pid;
    for( pid=0; pid < ELEMS(mailbox); pid++ )
        if( mailbox[pid].msgbuf )
            machine_wait_send( pid );
}


/*-----------------------------------------------*
 *  blocking send                                *
 *                                               *
 *    kind = 0  really blocking until receive    *
 *    kind = 1  blocking until msg copied        *
 *    kind = 2  msg buffer can directly be used  *
 *-----------------------------------------------*/

void machine_mpi_send( int to, char *message, int length, int kind)
{
    int target = to-1;    /*  to = 1..MAXP1  */
    int error;
    char *buffer = message;

    switch( kind )
    {
    case 0:
     /*     call a blocking send, buffering not necessary
      */
        if(( error = SendLink( mailbox[target].LinkCB, message, length )) < 0 )
        {
#ifdef DEBUG
            int i;
            printf( "%d. Used links:", pcb.i );
            for( i=0; i<pcb.p; i++ )
                printf( "  %p", mailbox[target].LinkCB );
            printf("\n");
#endif
            dalib_epx_stop(
                "SendLink() failed: "
                "pcb.i=%d, link=%p, kind=%d, to=%d, len=%d, errno=%d",
                 pcb.i, mailbox[target].LinkCB, kind, to, length, errno );
        }
        break;

    case 1:
     /*     copy the message for non-blocking send, then as for kind==2
      */
        buffer = (char* )dalib_malloc( length, "machine_mpi_send" );
        dalib_memcopy( buffer, message, length );

    case 2:

     /* wait for the last non-blocking send */

        if( mailbox[target].msgbuf )
            machine_wait_send( target );

     /* start non-blocking send */

        ASend( topoId, target, buffer, length, &error );
        if( error < 0 )
            dalib_epx_stop("ASend() failed: "
                "pcb.i=%d, kind=%d, to=%d, len=%d, error=%d",
                 pcb.i, kind, to, length, error );

     /* save message params for mpi_wait_send() */

        mailbox[target].msgbuf = buffer;
        mailbox[target].msglen = length;

        break;

    default:
        dalib_epx_stop("Invalid `kind' value: "
            "pcb.i=%d, kind=%d, to=%d, len=%d",
             pcb.i, kind, to, length );
    } /* switch */

} /* machine_mpi_send */


/*--------------------------------------------------------------------*
 *  BLOCKING receive - blocks really until message has been received  *
 *--------------------------------------------------------------------*/

void machine_mpi_receive( int from, char *message, int length )
{
    int source = from-1;
    int error;

    if( from == -1 )
    {     /* receive any message */
        dalib_internal_error("EPX don't support jokers");
    }
    if(( error = RecvLink( mailbox[source].LinkCB, message, length )) < 0 )
        dalib_epx_stop("RecvLink() failed: "
            "pcb.i=%d, link=%p, from=%d, len=%d, errno=%d",
             pcb.i, mailbox[source].LinkCB, from, length, errno );
}


/*
 *  blocking send/receive in one step
 */
void machine_mpi_sendreceive( int to,   char *out_msg, int out_len,
                              int from, char *in_msg,  int in_len )
{
    dalib_internal_error ("machine_mpi_sendreceive not supported");
}


/*
 *  non blocking send
 */
int machine_mpi_isend( int to, char *message, int length )
{
    dalib_internal_error ("NON BLOCKING mode not available for EPX");
    return 0;
}


/*
 *  non blocking receive
 */
int machine_mpi_ireceive( int from, char *message, int length)
{
    dalib_internal_error ("NON BLOCKING mode not available for EPX");
    return 0;
}


/*
 *  Ask status of an immediate send/receive
 *   - msg_id is free if machine_mpi_done == 1
 *   - msg_id has been given by immediate send or receive
 */
int machine_mpi_done( int msg_id )
{
    dalib_internal_error ("NON BLOCKING mode not available for EPX");
    return 0;
}


/*
 *  wait for completion of an immediate send/receive
 */
void machine_mpi_wait( int msg_id )
{
    /* msgwait (msg_id); */
    dalib_internal_error ("NON BLOCKING mode not available for EPX");
}

/* EOF */
