/*************************************************************************
*                                                                        *
*  Name : gmdhpfarguments.c                                              *
*                                                                        *
*  Purpose : Evaluation of command line arguments for gmdhpf             *
*                                                                        *
*  Author : Resi Hoever-Klier, GMD, SCAI.LAB                             *
*                                                                        *
*  created     : Sep. 1996                                               *
*                                                                        *
*  Last Update : Nov. 1996                                               *
*                                                                        *
*************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "ratc.h"             /* bool, true, false */
#include "gmdhpfglobal.h"

# ifdef alliant
# include <fcntl.h>             /* for testing fileaccess */
# else
# include <unistd.h>            /* for testing fileaccess */
# endif

#define LINELENGTH 1024
 
void print_gmdhpf_help (void);
void print_gmdhpf_settings (void);
 
/****************************************************************
*                                                               *
*  print_gmdhpf_help ()                                         *
*                                                               *
*   - prints list of available options (very generic)           *
*                                                               *
****************************************************************/
 
void print_gmdhpf_help ()
{ int i;
 
#ifdef DEBUG
  printf ("print_gmdhpf_help:\n");
#endif
 
 printf ("\n");
 
 printf ("gmdhpf [gmdhpf_option]... [filename]... \n");
 printf ("\n");
 printf ("gmdhpf_option may be: \n");
 printf ("\n");
 
 printf ("  -verbose or -v   verbose\n");
 printf ("  -help            printout usage information\n");
 printf ("  -settings        printout actual settings for fadapt-call\n");
 printf ("  -keep            do not erase intermediate files\n");
 printf ("\n");
 
 printf ("  -list            generate listfile of fadapt messages\n");
 printf ("\n");
 
 printf ("  -dryrun          show but not exec internal commands; \n");
 printf ("                   check existence of essential executable and library files.\n");
 printf ("\n");
 
 printf ("  -Wcpp\"[cpp_opt]...\"  additional options for cpp-call\n");
 printf ("  -Wa\"[fadapt_opt]...\" additional options for fadapt-call\n");
 printf ("  -Wf\"[fc_opt]...\"     additional options for fc-call\n");
 printf ("  -Wl\"[link_opt]...\"   additional options for call of linker\n");
 printf ("                       (generation of SPMD program or single node program)\n");
 printf ("\n");
 
 printf ("  -c               do not link\n");
 printf ("          default: translate and link if no error occured\n");
 printf ("\n");
 
 printf ("  -uniproc         generate single node program\n");
 printf ("          default: generate SPMD program\n");
 printf ("\n");
 
 printf ("  -o <outfile>     name of generated executable\n");
 printf ("          default: a.out\n");
 printf ("\n");

 printf ("Initial configuration settings are read from: $PHOME/.adprc  and\n");
 printf ("                                               $HOME/.adprc  and\n");
 printf ("                                                   ./.adprc ,\n");
 printf ("later settings overriding earlier ones.\n");
 printf ("\n");

 printf ("fadapt -help       will list all possible options for fadapt-call\n");
 printf ("For possible options for cpp- fc- and ld-call please refer to related man-pages.\n");
 printf ("\n");

} /* print_gmdhpf_help */
 

/*************************************************************************
*                                                                        *
*  void set_gmdhpf_compilers ()                                          *
*                                                                        *
*    - set the machine specific predefined options                       *
*                                                                        *
*************************************************************************/
 
static void set_gmdhpf_compilers ()

{
  strcpy (CPP,        environment [target_machine][0]);
  strcpy (CPP_OPT,    environment [target_machine][1]);
  strcpy (FADAPT_OPT, environment [target_machine][2]);
  strcpy (FADAPT_MP,  environment [target_machine][3]);
  strcpy (FC,         environment [target_machine][4]);
  strcpy (FC_OPT,     environment [target_machine][5]);
  strcpy (FC_MP,      environment [target_machine][6]);
  strcpy (LD,         environment [target_machine][7]);
  strcpy (LD_OPT,     environment [target_machine][8]);
  strcpy (LD_MP,      environment [target_machine][9]);
  strcpy (LD1,        environment [target_machine][10]);
  strcpy (LD1_OPT,    environment [target_machine][11]);
  strcpy (LD1_MP,     environment [target_machine][12]);
 
} /* set_gmdhpf_compilers */
 

/*************************************************************************
*                                                                        *
*  set_gmdhpf_commmunication()                                           *
*                                                                        *
*  - compute name of DALIB if not set by user in .adprc-file             *
*    depending upon target_communication, target_machine and whether     *
*    single node program or SPMD program is to be generated.             *
*                                                                        *
*************************************************************************/
 
static void set_gmdhpf_communication()

{ char dalibname[MAXPATHLEN];
 
#ifdef DEBUG
  printf ("set_gmdhpf_commmunication:\n target_comm: %s; target_mach: %s\n",communication_items[target_communication],machine_items[target_machine]);
  printf ("set_gmdhpf_commmunication: PHOME:<%s>\n",PHOME);
#endif


  if (!(USER_DEFINED_DALIB_FLAG))   /* compute default name of DALIB */
    {
     switch (target_communication)  
       {
        case NATIVE: switch (target_machine)
                       {
                        case CM5:  strcat (DALIB,"CM5");
                        break;
                        case PGON: strcat (DALIB,"PGON"); 
                        break;
                        case CS:   strcat (DALIB,"CS");
                        break;
                        case RS6K: strcat (DALIB,"SP"); 
                        break;
                        default:   printf ("ERROR: no support of target_communication NATIVE for target_machine %s .\n",machine_items[target_machine]);
                                   printf (" target_communication NATIVE only supported for following target_machines:\n");
                                   printf (" cm5, pgon, cs, and rs6k.\n");
                                   error_exit(-1);
                       } /* switch (target_machine) */
                    break;

        case SHM:   strcat (DALIB,"SHM");
                    break;

        case PVM3:  strcat (DALIB,"PVM3");
                    break;

        case MPI:   strcat (DALIB,"MPI");
                    break;

        default:    printf("INTERNAL ERROR: not recognized target_communication\n");
                    error_exit(-1);

       } /* switch (target_communication) */

#ifdef DEBUG
printf("after switch (target_communication):\n"); 
#endif

     if (uniproc_flag) 
       {
        strcat (DALIB,"/unilib.a");
       }
     else 
       {
        strcat (DALIB,"/dalib.a");
       }

#ifdef DEBUG
printf("!(USER_DEFINED_DALIB_FLAG); computed: %s.\n",DALIB); 
#endif

    }  /* if (!(USER_DEFINED_DALIB_FLAG)) */

#ifdef DEBUG
printf("DALIB: %s\n",DALIB);
#endif

/* in any case: determine MESS_PASS_LIB: */

   if (uniproc_flag) strcpy (MESS_PASS_LIB,"");
     else
      { switch (target_communication)
          {
           case NATIVE:  strcpy (MESS_PASS_LIB,"");
                         break;
           case SHM:     strcpy (MESS_PASS_LIB,SHM_LIB);
                         break;
           case PVM3:    strcpy (MESS_PASS_LIB,PVM_LIB);
                         break;
           case MPI:     if (target_machine == RS6K) strcpy (MPI_LIB,"");
                         strcpy (MESS_PASS_LIB,MPI_LIB);
                         break;
           default:      fprintf(stderr,"INTERNAL ERROR: not recognized target_communication\n");
                         error_exit(-1);
          } /*switch (target_communication) */
      } /* if (uniproc_flag) */


#ifdef DEBUG
  printf ("end of set_gmdhpf_commmunication:\n target_comm: %s; target_mach: %s\n DALIB: %s; MESS_PASS_LIB: %s\n",communication_items[target_communication],machine_items[target_machine],DALIB,MESS_PASS_LIB);
#endif


} /* set_gmdhpf_communication */
 
 
 
/*************************************************************************
*                                                                        *
*  set_gmdhpf_defaults ()                                                *
*                                                                        *
*************************************************************************/
 
static void set_gmdhpf_defaults ()
 
{
  
#ifdef DEBUG
  printf ("set_gmdhpf_defaults:\n");
#endif
 
  USER_DEFINED_DALIB_FLAG = 0; /* default: compute name of dalib */
  verbose_flag    = 0;
  dryrun_flag = 0;
 
  ld_flag = 1;
  uniproc_flag = 0;     /* default: generate SPMD-program */

  target_machine        = predefined_target_machine;
  target_communication  = predefined_target_communication;
 
     if (getenv(PHOME_envvar))
         strcpy(PHOME,getenv(PHOME_envvar));
      else
       { fprintf(stderr,"environment variable PHOME is not set; gmdhpf aborting.\n");
         error_exit (-1);
       }

#ifdef DEBUG
  printf("PHOME (getenv PHOME): %s\n",PHOME);
#endif
 
 
  strcpy (FADAPT, PHOME);
  strcat (FADAPT, "/bin/fadapt");
  strcpy (GENLIST,  PHOME);
  strcat (GENLIST, "/bin/genlist");
  strcpy (DALIB, PHOME);
  strcat (DALIB, "/dalib/");
  strcpy (MPI_LIB, predefined_MPI_LIB);         
  strcpy (PVM_LIB, predefined_PVM_LIB);         
  strcpy (SHM_LIB, predefined_SHM_LIB);         

  set_gmdhpf_compilers();

#ifdef DEBUG
  printf("after set_gmdhpf_defaults:\n");
  print_gmdhpf_settings();
#endif

/*set_gmdhpf_communication (); we will do it after read_gmdhpf_adprc_values()*/

 
} /* set_gmdhpf_defaults */
 
 
/****************************************************************
*                                                               *
*  print_gmdhpf_settings ()                                     *
*                                                               *
*   - prints list of settings for call of gmdhpf                *
*                                                               *
****************************************************************/
 
void print_gmdhpf_settings (void)
{
  
#ifdef DEBUG
  printf ("print_gmdhpf_settings:\n");
#endif

  printf ("\n");
  if (strlen(ADPRC_FILES) == 0)
    printf ("No configuration file found: taking predefined defaults.\n");
  else
    printf ("Default settings read in from following configuration files:\n%s\n",ADPRC_FILES);

  printf ("\n");
  printf ("ADAPTOR Home Directory : %s\n", PHOME);
  printf ("target_machine:        : %s\n", machine_items[target_machine]);
  printf ("target_communication   : %s\n", communication_items[target_communication]);
  printf ("DALIB:                 : %s\n", DALIB);
  printf ("Mess.-Pass.-Lib.       : %s\n", MESS_PASS_LIB);
 
  printf ("\n");
  printf ("Verbose = %d,    Dryrun = %d,   Keep = %d\n", verbose_flag, dryrun_flag, keep_flag);
 
  printf ("\n");
  printf ("invoke cpp for cpp input-files\n");
  printf ("             CPP       : %s\n",CPP);
  printf ("             CPP_OPT   : %s\n",CPP_OPT);

  printf ("\n");
  printf ("invoke fadapt for fadapt input-files\n");
  printf ("             FADAPT    : %s\n",FADAPT);
  printf ("             FADAPT_OPT: %s\n",FADAPT_OPT);
/*  printf ("             FADAPT_MP : %s\n",FADAPT_MP);*/

  printf ("\n");
  printf ("invoke fc for fc input-files\n");
  printf ("             FC        : %s\n",FC);
  printf ("             FC_OPT    : %s\n",FC_OPT);
/*  printf ("             FC_MP     : %s\n",FC_MP);*/
 
  printf ("\n");

  if (ld_flag) 
    {
     printf ("LINK STEP WILL BE INVOKED if no error occurres at compilation step.\n");
     if (uniproc_flag)
        printf ("LD1 will be taken for generation of SINGLE NODE program %s\n",output_file_name);
     else
        printf ("LD will be taken for generation of SPMD program %s\n",output_file_name);
    }
  else
    printf ("LINK STEP will NOT be envoked\n");
 
  printf ("\n");

  printf ("Generation of SPMD program with:\n");
  printf ("             LD        : %s\n",LD);
  printf ("             LD_OPT    : %s\n",LD_OPT);
/* printf ("             LD_MP     : %s\n",LD_MP);*/
  printf ("\n");
  printf ("Generation of SINGLE NODE program with:\n");
  printf ("             LD1       : %s\n",LD1);
  printf ("             LD1_OPT   : %s\n",LD1_OPT);
/*  printf ("             LD1_MP    : %s\n",LD1_MP);*/
 
  printf ("\n");
    
} /* print_gmdhpf_settings */

 
/*************************************************************************
*                                                                        *
* int find_item (item_list, item_no, item_string)                        *
*                                                                        *
*    looks for string <item_string> in <item_list> of <item_no> elements.*
*          returns position of <item_string> in <item_list>   rsp.       *
*                  -1  if <item_string> not found in <item_list>.        *
*                                                                        *
*************************************************************************/
 
int find_item (item_list, item_no, item_string)
char *item_list[], *item_string;
int  item_no;
 
{ int i, pos;
 
#ifdef DEBUG
  printf ("find_item: item_no(%d)\n",item_no);
#endif
 
  pos = -1;
 
  for (i=0; i<item_no; i++)
    {
#ifdef DEBUG
printf("comparing %s with item_list[%d]: %s\n",item_string,i,item_list[i]);
#endif
     if (strcmp(item_string,item_list[i]) == 0)
        pos = i;
    }
 
  return (pos);
 
}  /* find_item;
 

/*************************************************************************
*                                                                        *
*  bool get_next_input(char *line, char* ident, char *value);            *
*                                                                        *
*    - handles a line (of .adprc-file):                                  *
*                                                                        *
*    If <line> is a definition line, get_next_input returns:             *
*    - the first string lefthand of "=" as <ident>    and                *
*    - all righthand of "=" whithout leading or trailing blank or tabs   *
*      and without trailing "\n" as <value>.                             *
*                                                                        *
*    - returns true  if <line> is found to be a definition line          *
*              false  else.                                              *
*                                                                        *
*************************************************************************/
 
# define COMMENT '#'
 
bool get_next_input (char *line, char* ident, char *value)
 
{ char   *valptr;
  char   *pointer;
  char   help_ident[LINELENGTH];  /* needed for sscan */
  int    idlen,vallen;
  int    i;
 
#ifdef DEBUG
  printf ("get_next_input:\n");
#endif
 
  if (line[0] == COMMENT) return(false);      /*  <comment>  */
 
  if ((pointer = strchr(line,'=')) == 0)
 
      return(false);     /* no assignment*/

  /***********************
  * handle identifier    *
  ***********************/
 
  idlen = pointer - line;


  strncpy(help_ident," ",LINELENGTH);  /* eleminate garbage */
  strncpy(help_ident, line, idlen);  /* eleminate leading and trailing blanks */
 
  sscanf (help_ident,"%s",ident);
 
  /***********************
  * handle value         *
  ***********************/
 
  strcpy(value,pointer+1);   /* clear leading blanks and tabs ... */
 
  for (vallen = 0; vallen <= strlen(value); vallen++)
    if (value[vallen] != ' ' && value[vallen] != '\t') break;
 
  valptr = &value[vallen];
 
  if (vallen > 0)                 /* first valid element of value */
       strcpy (value,valptr);
 
  /*eleminate trailing \n and blanks and tabs ... */
 
  for (vallen = strlen(value)-1; vallen >= 0; vallen--)
    if (   value[vallen] != ' ' && value[vallen] != '\t'
        && value[vallen] != '\n')
          break;
 
  value[vallen+1] = '\0';
  return (true);
 
} /* get_next_input */
 

/*************************************************************************
*                                                                        *
*  bool my_str_eq (str1, str2)                                           *
*                                                                        *
*    - compares lower case str1 against str2                             *
*                                                                        *
*************************************************************************/
 
#define LINELENGTH  1024
 
bool my_str_eq (str1, str2)
 
char *str1, *str2;
 
{ char hstr1 [LINELENGTH];
  int i;
 
#ifdef DEBUG
  printf ("my_str_eq of :%s: with :%s:\n",str1, str2);
#endif

  strcpy (hstr1, str1);
 
  for (i=0; i<strlen (hstr1); i++)
      hstr1[i] = tolower (hstr1[i]);
 
  if (strcmp (hstr1, str2) == 0) return (true);
 
  return (false);
 
} /* my_str_eq */
 
 
/*************************************************************************
*                                                                        *
*  read_gmdhpf_adprc_values (adp_fp)                                     *
*                                                                        *
*    - read .adprc file to overwrite options                             *
*                                                                        *
*************************************************************************/
 
static void read_gmdhpf_adprc_values (adp_fp)
 
FILE *adp_fp;
 
{
 
    char   line[LINELENGTH];
    char   ident[LINELENGTH];
    char   value[LINELENGTH];
    char   cmpvalue[LINELENGTH];   /*for comparision with strings in global.h*/
    char   *filler = "               "; /*15 blanks */
    int    pos;
    int    linecntr;
 
/* local copies to avoid overwrite of values read in from .adprc 
   by set_gmdhpf_compilers and set_gmdhpf_communication 
   after reading target_machine resp. target_communictions from .adprc
   rsp. to avoid empty name for fadapt
*/

    char   cpp[MAXPATHLEN];
    char   cpp_opt[160+MAXPATHLEN];
    char   fc[MAXPATHLEN];
    char   fc_opt[160+MAXPATHLEN];
    char   ld[MAXPATHLEN];
    char   ld_opt[160+MAXPATHLEN];
    char   ld1[MAXPATHLEN];
    char   ld1_opt[160+MAXPATHLEN];
	
    char   dalib[MAXPATHLEN];

    bool   def_cpp, def_cpp_opt, def_fc, def_fc_opt, def_ld, def_ld_opt;
    bool   def_ld1, def_ld1_opt;
 
/*  char   fadapt[MAXPATHLEN];    */
 
#ifdef DEBUG
  printf ("read_gmdhpf_adprc_values:\n");
#endif
 
linecntr = 0;
 
    def_cpp = def_cpp_opt = false;
    def_fc  = def_fc_opt  = false;
    def_ld  = def_ld_opt  = false;
    def_ld1 = def_ld1_opt = false;


    while ((fgets(line,LINELENGTH,adp_fp)) != NULL)
 
     { /* next line till EOF */
 
       linecntr+=1;
 
       if (!get_next_input(line, ident, value)) continue;
 
#ifdef DEBUG
 printf("line %d: handling >%s< = >%s<\n",linecntr,ident,value);
#endif
                                                   /*put it into global vars*/
 
                               /* values influenced by set_gmdhpf_compilers */
       if (my_str_eq (ident,"cpp")) 
          { strcpy (cpp,value);
            def_cpp = true;
          }
       else if (my_str_eq(ident,"cpp_opt")) 
              { strcpy (cpp_opt,value);
                def_cpp_opt = true;
              }
       else if (my_str_eq (ident,"fc")) 
              { strcpy (fc,value);
                def_fc = true;
              }
       else if (my_str_eq(ident,"fc_opt")) 
              { strcpy (fc_opt,value);
                def_fc_opt = true;
              }
       else if (my_str_eq (ident,"ld")) 
              { strcpy (ld,value);
                def_ld = true;
              }
       else if (my_str_eq(ident,"ld_opt")) 
              { strcpy (ld_opt,value);
                def_ld_opt = true;
              }
       else if (my_str_eq(ident,"ld1")) 
              { strcpy (ld1,value);
                def_ld1 = true;
              }
       else if (my_str_eq(ident,"ld1_opt")) 
              { strcpy (ld1_opt,value);
                def_ld1_opt = true;
              }

                               /* values influenced by set_gmdhpf_defaults  */
       else if (my_str_eq(ident,"fadapt")) strcpy (FADAPT,value);
       else if (my_str_eq(ident,"fadapt_opt")) strcpy (FADAPT_OPT,value);
       else if (my_str_eq(ident,"mpi_lib")) strcpy (MPI_LIB,value);
       else if (my_str_eq(ident,"pvm_lib")) strcpy (PVM_LIB,value);
       else if (my_str_eq(ident,"shm_lib")) strcpy (SHM_LIB,value);

                          /* values influenced by set_gmdhpf_communication  */
       else if (my_str_eq(ident,"dalib")) 
            { strcpy (dalib,value);
              USER_DEFINED_DALIB_FLAG = 1;
            }

 
       else if (my_str_eq(ident,"target_machine"))
 
        { strcpy (cmpvalue,"-");
          strcat (cmpvalue,value);
 
          if( (pos = find_item(machine_options,no_machine,cmpvalue)) >= 0)
 
             target_machine = pos;
 
          else
 
            { strcpy(cmpvalue,value);
              strncat(cmpvalue,filler,15-strlen(value)); /*trail. blanks*/
              if( (pos=find_item(machine_items,no_machine,cmpvalue)) >= 0)
 
                 target_machine = pos;
 
                else               /*no valid target-machine*/
 
                 fprintf (stderr,"%s is no valid target machine (ignored)\n",
                          value);
            }
 
          set_gmdhpf_compilers ();
 
         }
 

       else if (my_str_eq(ident,"target_communication"))

         { strcpy (cmpvalue,"-");
           strcat (cmpvalue,value);

#ifdef DEBUG
printf("looking for target_communication %s in communication_options\n",cmpvalue);
#endif

           if ((pos = find_item(communication_options,no_communication,cmpvalue)
) >= 0)
              { 
               target_communication = pos;

#ifdef DEBUG
printf("found target_communication: %d\n",pos);
#endif
              } 
           else
 
             { strcpy(cmpvalue,value);
               strncat(cmpvalue,filler,11-strlen(value)); /*trail. blanks*/

#ifdef DEBUG
printf("looking for target_communication %s in communication_items\n",cmpvalue);
#endif

               if( (pos=find_item(communication_items,no_communication,cmpvalue)
) >= 0)
                { 
                 target_communication = pos;

#ifdef DEBUG
printf("found target_communication: %d\n",pos);
#endif
                }
                else               /*no valid target-communication*/
 
                 fprintf (stderr,"%s is no valid target communication (ignored)\n",value);
             }

/*        set_gmdhpf_communication ();  We will do it at at the end of 
                                        eval_gmdhpf_arg to be sure to take all 
                                        desired libraries (if target_comm. is
                                        NATIVE the names of the libs depend
                                        upon target_machine) and to know the
                                        correct uniproc option.      */
 
         } /* ident == "target_communication */
 
 
       else    /* not recognized identifier */
 
         { fprintf (stderr,"line %d: \"%s = %s\" not recognized .adprc-option\n", linecntr, ident, value);
         }
 
      /* clear variables */
 
      strncpy(ident,"",strlen(ident));
      strncpy(value,"",strlen(value));
      strncpy(line ,"",LINELENGTH);
 
     } /* while */

/* copy processor options read in from .adprc into global vars */
 
    if (def_cpp == true )        strcpy (CPP,cpp);
    if (def_cpp_opt == true )    strcpy (CPP_OPT,cpp_opt);
    if (def_fc == true )         strcpy (FC,fc);  
    if (def_fc_opt == true )     strcpy (FC_OPT,fc_opt);
    if (def_ld == true )         strcpy (LD,ld);  
    if (def_ld_opt == true )     strcpy (LD_OPT,ld_opt);
    if (def_ld1 == true )        strcpy (LD1,ld1);  
    if (def_ld1_opt == true )    strcpy (LD1_OPT,ld1_opt);

/* copy dalib read in from .adprc into global vars */
 
    if (USER_DEFINED_DALIB_FLAG == 1)  strcpy (DALIB,dalib);


/*  set_gmdhpf_communication(); */

#ifdef DEBUG
  printf("returning from read_gmdhpf_adprc_values!\n");
#endif

} /* read_gmdhpf_adprc_values */
 
 
/*************************************************************************
*                                                                        *
*  void reset_gmdhpf_defaults ()                                         *
*                                                                        *
*    - read existant .adprc files to overwrite default options:          *
*                                     - first trying in directory $PHOME *
*                                     - then in                   $HOME  *
*                                     - at last in                .      *
*                                                                        *
*  ADPRC_FILES is filled with the names of the configuration files read. *
*                                                                        *
*************************************************************************/
 
static void reset_gmdhpf_defaults ()
 
{ FILE *adp_fp;
  int i;
  char *adprc_file;

#define RC_SUFFIX "/.adprc"

#ifdef DEBUG
  printf ("reset_gmdhpf_defaults:\n");
#endif

  strcpy(ADPRC_FILES,"");

/* test auf $PHOME/.adprc */

try_phome:

  if ((adprc_file = getenv("PHOME")) == NULL)
    {
#ifdef DEBUG
printf("$PHOME (home directory of ADAPTOR) not set; no configuration file found
.\n");
#endif
     goto try_home;
    }
 
  strcat (adprc_file,RC_SUFFIX);

  if ((adp_fp = fopen(adprc_file,"r")) != NULL)
     {
#ifdef DEBUG
printf("reading configuration file %s\n",adprc_file);
#endif
       read_gmdhpf_adprc_values (adp_fp);
       fclose (adp_fp);

       strcat (ADPRC_FILES,adprc_file);
       strcat (ADPRC_FILES," ");
     }


/* test auf $HOME/.adprc */

try_home: 

  if ((adprc_file = getenv("HOME")) == NULL)
    {
#ifdef DEBUG
printf ("$HOME (installation directory) not set\n");
#endif
     goto try_work;
    }

  strcat (adprc_file, RC_SUFFIX);

  if ((adp_fp = fopen(adprc_file,"r")) != NULL)
     {
#ifdef DEBUG
printf("reading configuration file %s\n",adprc_file);
#endif
       read_gmdhpf_adprc_values (adp_fp);
       fclose (adp_fp);

       strcat (ADPRC_FILES,adprc_file);
       strcat (ADPRC_FILES," ");
      }


try_work:

  strcpy (adprc_file,".");
  strcat (adprc_file, RC_SUFFIX);

  if ((adp_fp = fopen(adprc_file,"r")) != NULL)
     {
#ifdef DEBUG
printf("reading configuration file %s\n",adprc_file);
#endif
 
       read_gmdhpf_adprc_values (adp_fp);
       fclose (adp_fp);

       strcat (ADPRC_FILES,adprc_file);
     }


#ifdef DEBUG
  printf("after reset_gmdhpf_defaults:\n");
  print_gmdhpf_settings();
#endif
 
} /* reset_gmdhpf_defaults */


/*************************************************************************
*                                                                        *
*  int eval_gmdhpf_arg (argc, argv)                                      *
*                                                                        *
*   - evaluates command line arguments                                   *
*   - returns number of not evaluated arguments                          *
*                                                                        *
*************************************************************************/

int eval_gmdhpf_arg (argc, argv)
int argc;
char **argv;

{ int i, j, no_arg;
  int print_settings_flag;
  char *comm;
  void print_gmdhpf_help ();
 
#ifdef DEBUG
  printf ("eval_gmdhpf_arg:\n");
#endif
 
  set_gmdhpf_defaults ();
  reset_gmdhpf_defaults ();    /* use .adprc file */

  print_settings_flag = 0;     /*don't just print actual settings*/
  no_arg = 0;           /* number of not evaluated arguments */

  i=1;
  j = 0;                /* index for input_file_names */

  while (i<argc)
    {
      comm = argv[i];

      if ('-' != comm[0])
         {                           /*seems to be a filename*/
          input_file_name[j] = comm; 

#ifdef DEBUG
  printf("got input_file_name[%d]: %s\n",j,input_file_name[j]);
#endif
          j += 1;
         }  

      else if (strcmp(comm,"-help") == 0)
           { print_gmdhpf_help ();
             error_exit (0);
           }

      else if (strcmp(comm,"-settings") == 0)
            print_settings_flag = 1;

      else if (strcmp(comm,"-dryrun") == 0)
           dryrun_flag = 1;


      else if ((strcmp(comm,"-verbose") == 0) || (strcmp(comm,"-v") == 0))
           {  verbose_flag = 1;
           }

      else if (strcmp(comm,"-list") == 0)
           {  list_flag = 1;
              strcat(FADAPT_OPT," -list");
           }
 
      else if (strcmp(comm,"-mp") == 0)

           {  strcat(FADAPT_OPT, " ");
              strcat(FADAPT_OPT, FADAPT_MP);
              strcat(FC_OPT, " ");
              strcat(FC_OPT, FC_MP);
              strcat(LD_OPT, " ");
              strcat(LD_OPT, LD_MP);
              strcat(LD1_OPT, " ");
              strcat(LD1_OPT, LD1_MP);
           }
 
      else if (strcmp(comm,"-keep") == 0)
           keep_flag = 1;

      else if (strcmp(comm,"-c") == 0)
           ld_flag = 0;

      else if ((strcmp(comm,"-uniproc") == 0) || (strcmp(comm,"-1") == 0))
           {  uniproc_flag = 1;
/*            strcat(FADAPT_OPT," -uniproc"); doesnt work actually*/
              strcat(FADAPT_OPT," -1");
           }

      else if (strcmp(comm,"-o") == 0)
         {
             if ((argc > (i+1)) && (sscanf(argv[i+1],"%s",output_file_name)))
                i++;
	     else
	     {
		fprintf(stderr,"argument missing or argument type wrong\n");
		fprintf(stderr,"needed for argument -o\n");
		if (dryrun_flag == 0) error_exit(-1);
	     }
         }


      else if (strncmp(comm,"-Wcpp",5) == 0)
             { strcat (CPP_OPT," ");
               strcat (CPP_OPT,comm+5); 
             }

      else if (strncmp(comm,"-Wa",3) == 0)
             {  strcat (FADAPT_OPT," "); 
                strcat (FADAPT_OPT,comm+3); 
             }

      else if (strncmp(comm,"-Wf",3) == 0)
             { strcat (FC_OPT," ");
               strcat (FC_OPT,comm+3); 
             }

      else if (strncmp(comm,"-Wl",3) == 0)
             { strcat (LD_OPT," ");
               strcat (LD_OPT,comm+3); 
               strcat (LD1_OPT," ");
               strcat (LD1_OPT,comm+3); 
             }

      else if (strncmp(comm,"-I",2) == 0)
             { strcat (CPP_OPT," ");
               strcat (CPP_OPT,comm);
               strcat (FADAPT_OPT," ");
               strcat (FADAPT_OPT,comm);
               strcat (FC_OPT," ");
               strcat (FC_OPT,comm);
             }


      else { /* not recognized option */
             no_arg += 1;
             argv[no_arg] = argv[i];
           }
      i++;

    } /*while */

   set_gmdhpf_communication();

   if (print_settings_flag)
      { print_gmdhpf_settings ();
        error_exit (0);
      }

#ifdef DEBUG
  printf("after eval_gmdhpf_arg:\n");
  print_gmdhpf_settings();
#endif
 

    return (no_arg);   /* not evaluated arguments */

} /* eval_gmdhpf_arg */
