/*************************************************************************
*                                                                        *
*  Name : gmdhpf.c                                                       *
*                                                                        *
*  Purpose : Driver program for the ADAPTOR translation tool             *
*                                                                        *
*  Author : Resi Hoever-KLier, GMD, I1.HR                                *
*                                                                        *
*  created :     Aug. 1996                                               *
*                                                                        *
*  Last Update : Nov. 1996                                               *
*                                                                        *
*************************************************************************/

# include <stdio.h>   /* stderr */
#include <stdlib.h>
#include <string.h>
# include "gmdhpfglobal.h"
# include "ratc.h"
 
# ifdef alliant
# include <fcntl.h>             /* for testing fileaccess */
# else
# include <unistd.h>            /* for testing fileaccess */
# endif
 
extern int errno;             /*should be defined in unistd.h*/

/* external variables defined in this module */
 
int interactive = 0;
 
#define TMP_FILE_LENGTH 30
#define TMP "/tmp/gmdhpf"
 
static char tempfile [TMP_FILE_LENGTH];
 
#define CMD_LENGTH 4096

#define LINELENGTH 1024
 
int infile_typ;            /* for typ of input file (f.e. cpp or fc etc.) */
#define CPP_FILE    1
#define FADAPT_FILE 2
#define FC_FILE     3
#define LD_FILE     4
#define UNDEF_FILE  0

static char command[CMD_LENGTH];
static int  command_len;

static void release_workspace (void);
void error_exit (int);
 
 
/*************************************************************************
*                                                                        *
*  bool  split_filename(filename,main_name,suffix)                       *
*                                                                        *
*   - splits filename into suffix (chars behind last point) and          *
*                          main_name (chars before last point)           *
*   - returns false if no point found in filename                        *
*             true  else                                                 *
*                                                                        *
*************************************************************************/
 
bool  split_filename(filename,main_name,suffix)
  char *main_name, *suffix;
  char *filename;
 
{
  char *last_point;
  int   length,main_len;
 
#ifdef DEBUG
  printf ("split_filename:\n");
#endif
 
  last_point = strrchr(filename,'.');
 
  if (last_point == 0)  return (false);  /* no point in filename */
 
  length = strlen(filename);
  main_len = last_point - filename;
 
  strcpy(suffix,last_point+1);
 
  strcpy (main_name, filename);
  main_name[main_len] = '\0';
 
  return(true);
 
} /*split_filename*/
 
 
/************************************************************************
*                                                                       *
*  bool is_cpp_source_file (string)                                     *
*                                                                       *
*  - checks whether string is an allowed cpp input filename             *
*  - returns true if is and   else   else                               *
*                                                                       *
************************************************************************/
 
bool is_cpp_source_file (string)
char *string;
{  int length;
 
#ifdef DEBUG
  printf ("is_cpp_source_file:\n");
#endif
 
   length = strlen (string);
 
   if (strcmp(string+length-2,".F9") == 0) return (true);
   if (strcmp(string+length-4,".F90") == 0) return (true);
   if (strcmp(string+length-4,".FCM") == 0) return (true);
   if (strcmp(string+length-4,".HPF") == 0) return (true);
   if (strcmp(string+length-4,".H90") == 0) return (true);
 
   return(false);

} /* is_cpp_source_file */
 
 
/************************************************************************
*                                                                       *
*  bool is_fadapt_source_file (string)                                  *
*                                                                       *
*  - checks whether string is an allowed fadapt input filename          *
*  - returns true if is and   else   else                               *
*                                                                       *
************************************************************************/
 
bool is_fadapt_source_file (string)
char *string;
{  int length;
 
#ifdef DEBUG
  printf ("is_fadapt_source_file:\n");
#endif
 
   length = strlen (string);
 
   if (strcmp(string+length-2,".f9") == 0) return (true);
   if (strcmp(string+length-4,".f90") == 0) return (true);
   if (strcmp(string+length-4,".fcm") == 0) return (true);
   if (strcmp(string+length-4,".hpf") == 0) return (true);
   if (strcmp(string+length-4,".h90") == 0) return (true);
 
   return(false);

} /* is_fadapt_source_file */
  
 
/************************************************************************
*                                                                       *
*  bool is_fc_source_file (string)                                      *
*                                                                       *
*  - checks whether string is an allowed fc input filename              *
*  - returns true if is and   else   else                               *
*                                                                       *
************************************************************************/
 
bool is_fc_source_file (string)
char *string;
{  int length;
 
#ifdef DEBUG
  printf ("is_fc_source_file:\n");
#endif
 
   length = strlen (string);
 
   if (strcmp(string+length-2,".f") == 0) return (true);
   if (strcmp(string+length-4,".for") == 0) return (true);
   if (strcmp(string+length-2,".F") == 0) return (true);
   if (strcmp(string+length-4,".FOR") == 0) return (true);
   if (strcmp(string+length-2,".e") == 0) return (true);  /* ??? */
   if (strcmp(string+length-2,".r") == 0) return (true);
   if (strcmp(string+length-2,".s") == 0) return (true);
   if (strcmp(string+length-2,".S") == 0) return (true);
   if (strcmp(string+length-3,".il") == 0) return (true);
   if (strcmp(string+length-5,".tcov") == 0) return (true);
 
   return(false);
 
} /* is_fc_source_file */

 
/************************************************************************
*                                                                       *
*  bool is_ld_source_file(string)                                       *
*                                                                       *
*  - checks whether string is  an allowed filename for fc-link-step     *
*  - returns true if is and   else   else                               *
*                                                                       *
************************************************************************/
 
bool is_ld_source_file (string)
char *string;
{  int length;
 
#ifdef DEBUG
  printf ("is_ld_source_file:\n");
#endif
 
   length = strlen (string);
 
   if (strcmp(string+length-2,".o") == 0) return (true);
   if (strcmp(string+length-2,".a") == 0) return (true);
 
   return(false);
 
} /*is_ld_source_file */

 
/************************************************************************
*                                                                       *
*  concat_cmd ()                                                        *
*                                                                       *
*  - concatenates parts of a command together                           *
*  - exits -1 if any error occured                                      *
*                                                                       *
************************************************************************/
 
void concat_cmd(what)
char *what;
 
{
 
#ifdef DEBUG
  printf ("concat_cmd:\n");
#endif
 
   if (( command_len += strlen(what) + 1) > CMD_LENGTH)
      { fprintf(stderr,"ERROR: resulting command too large; max len: %d\n",CMD_LENGTH);
        fprintf(stderr,"contents of command:\n%s\n",command);
        fprintf(stderr,"to be added:\n%s\n",what);
        error_exit (-1);
      }
 
  strcat (command," ");
  strcat (command,what);

  return;
 
} /* concat_cmd */
 

/************************************************************************
*                                                                       *
*  bool remove_intermediate_file(intermediate_file)                     *
*                                                                       *
*  - composes rm command to remove intermediate_file.                   *
*  - returns false if any error occured                                 *
*                                                                       *
************************************************************************/
 
bool  remove_intermediate_file(intermediate_file)
char *intermediate_file;
 
{
  int i;
  bool returnvalue;
  char rmcmd[MAXPATHLEN];

 
#ifdef DEBUG
  printf ("remove_intermediate_file: intermediate_file:%s\n",intermediate_file);
#endif

  returnvalue = true;
 
  strcpy (rmcmd,"rm -f ");
  strcat (rmcmd,intermediate_file);

  if (dryrun_flag)
     printf("submitting: %s\n",rmcmd);
   else
    {
     if (verbose_flag)  printf ("%s\n",rmcmd);
     i = system (rmcmd);

     if (i != 0)
       { printf ("ERROR,unable to remove %s!\n",intermediate_file);
         returnvalue = false;
       }
     }
 
  return (returnvalue);

} /* remove_intermediate_file */
 
 
/*************************************************************************
*                                                                        *
*  char *generate_new_name(gen_name)                                     *
*                                                                        *
*  places gen_name  in array generated_file_names for later use          *
*  - returns pointer to filename in array generated_file_names           *
*  - exits -1 if no more space for new filename                          *
*                                                                        *
*************************************************************************/
 
char *generate_new_name (gen_name)
char * gen_name;
 
{ char *ptr;
 
#ifdef DEBUG
  printf ("generate_new_name:\n");
#endif
 
  if (no_generated_files >= no_input_files * file_factor)
     { fprintf(stderr,"INTERNAL ERROR: no more space for new file_name!\n");
       error_exit (-1);
     }
 
  ptr = generated_file_names + no_generated_files * MAXPATHLEN;
  strcpy(ptr , gen_name);
  no_generated_files += 1;
 
  return (ptr);
 
} /* generate_new_name */
 
 
/*************************************************************************
*                                                                        *
*  void  determine_cpp_outfile_name(gen_name,main_name,suffix)           *
*                                                                        *
*   - returns   name of cpp output-file from name of cpp input-file      *
*     in variable gen_name                                               *
*                                                                        *
*************************************************************************/
 
void  determine_cpp_outfile_name(gen_name,main_name,suffix)
char *gen_name,*main_name, *suffix;
 
{ int i;
 
#ifdef DEBUG
  printf ("determine_cpp_outfile_name; main_name:<%s>, suffix:<%s>:\n",main_name,suffix);
#endif
 
  strcpy (gen_name,main_name);
  strcat (gen_name,".");
 
  for (i = 0; i < strlen(suffix); i++)
       suffix[i] = tolower(suffix[i]);
  strcat (gen_name,suffix);
 
} /* determine_cpp_outfile_name */


/************************************************************************
*                                                                       *
*  bool call_cpp (cpp_infile,cpp_outfile)                               *
*                                                                       *
*  - composes flags and files for cpp-call; then calls cpp.             *
*  - returns false if any error occured                                 *
*                                                                       *
************************************************************************/
 
bool call_cpp (cpp_infile,cpp_outfile)
char *cpp_infile, *cpp_outfile;

{
  int i;
  char main_name[MAXPATHLEN];
  char suffix[MAXPATHLEN];
  bool returnvalue;


#ifdef DEBUG
  printf ("call_cpp: cpp_infile:%s:\n",cpp_infile);
#endif

  returnvalue = true;
 
  command_len = strlen(CPP);
  strcpy (command,CPP);

  concat_cmd (CPP_OPT);
  concat_cmd (cpp_infile);
  concat_cmd (cpp_outfile);
 
  if (dryrun_flag)
     {  printf("submitting: %s\n",command);
        return (true);
     }

  if (verbose_flag)  printf("%s\n",command);

  i = system(command);
#ifdef DEBUG
  printf("back from system <%s>. returned:%d\n",command,i);
#endif
  if (i != 0)
     { fprintf(stderr, "cpp command returned error %d\n",i);
       returnvalue = false;
      }
#ifdef DEBUG
  printf("returning from call_cpp\n");
#endif

  return (returnvalue);

} /* call_cpp */
 
 
/*************************************************************************
*                                                                        *
*  void  determine_adp_outfile_name(gen_name,main_name,suffix)           *
*                                                                        *
*   - returns    name of fadapt output-file:                             *
*     <FADAPT_PREFIX><main_name>.f in gen_name                           *
*                                                                        *
*************************************************************************/
 
void  determine_adp_outfile_name(gen_name,main_name,suffix)
char *gen_name,*main_name, *suffix;
 
{  int i;
 
#ifdef DEBUG
  printf ("determine_adp_outfile_name; main_name:<%s>, suffix:<%s>:\n",main_name,suffix);
#endif
 
  strcpy (gen_name,FADAPT_PREFIX);
  strcat (gen_name,main_name);
  strcpy (main_name,gen_name);    /* remember new main_name for followers */
  strcat (gen_name,".f");
 
#ifdef DEBUG
  printf ("gen_name: <%s>\n",gen_name);
#endif
 
} /* determine_adp_outfile_name */
 
 
/************************************************************************
*                                                                       *
*  bool call_fadapt (adp_infile,adp_outfile)                            *
*                                                                       *
*  - composes flags and files for fadapt-call; then calls fadapt.       *
*    generates list of fadapt messages, if gmdhpf was called with       *
*    option -list                                                       *
*  - returns false if any error occured                                 *
*                                                                       *
************************************************************************/
 
bool call_fadapt (adp_infile,adp_outfile)
char *adp_infile, *adp_outfile;
 
{
  int i;
  bool returnvalue;
  char rmcmd[MAXPATHLEN];

 
#ifdef DEBUG
  printf ("call_fadapt: adp_infile:%s;adp_outfile:%s:\n",adp_infile,adp_outfile);
#endif

  returnvalue = true;

  if (access(adp_outfile, F_OK) == 0) /*does it exist?*/
    {
      strcpy( rmcmd,"rm -f ");
      strcat (rmcmd,adp_outfile);
      if (dryrun_flag)  printf("submitting: %s\n",rmcmd);
        else
        {
         if (verbose_flag)  printf("%s\n",rmcmd);
         i = system(rmcmd);
#ifdef DEBUG
  printf("back from system <%s>. returned:%d\n",rmcmd,i);
#endif
         if (i != 0)
           {
            printf("ERROR: unable to remove %s!\n",adp_outfile);
            return (false);
           }
        }
     }


  command_len = strlen(FADAPT);
  strcpy (command,FADAPT);

  concat_cmd ("-f");             /* indicate seperate adaption...   */
  concat_cmd (FADAPT_OPT);
  concat_cmd (adp_infile);
 
  if (dryrun_flag)  printf("submitting: %s\n",command);
   else
    {
     if (verbose_flag)  printf("%s\n",command);

     i = system(command);
#ifdef DEBUG
  printf("back from system <%s>. returned:%d\n",command,i);
#endif
     if (i != 0)
       { fprintf(stderr, "fadapt command returned error %d\n",i);
         returnvalue = false;
       }
     }

#ifdef DEBUG
  printf("returning from call_fadapt\n");
#endif

  if (list_flag)
    {
      command_len = strlen(GENLIST);
      strcpy (command,GENLIST);

      if (dryrun_flag) printf ("submitting: %s\n",command);
        else
         {
          if (verbose_flag) printf("%s\n",command);

          i = system (command);
#ifdef DEBUG
  printf("back from system <%s>. returned:%d\n",command,i);
#endif
          if (i != 0)
            { fprintf(stderr,"genlist command returned error %d\n",i);
              returnvalue = false;
            }
         }
     } /*if (list_flag)*/

  return (returnvalue);
 
 
} /* call_fadapt */
 
 
/*************************************************************************
*                                                                        *
*  void determine_fc_outfile_name(gen_name,main_name)                    *
*                                                                        *
*   - returns    name of fc output-file:                                 *
*     <main_name>.o                                                      *
*                                                                        *
*************************************************************************/
 
void  determine_fc_outfile_name(gen_name,main_name)
char *gen_name, *main_name;
 
{
  int i;
 
#ifdef DEBUG
  printf ("determine_fc_outfile_name; main_name:<%s>\n",main_name);
#endif
 
  strcpy (gen_name,main_name);
  strcat (gen_name,".o");
 
  return;
 
}/* gen_fc_outfile_name */

 
/************************************************************************
*                                                                       *
*  bool call_fc (fc_infile,fc_outfile)                                  *
*                                                                       *
*  - composes flags and files for fc-call; then calls fc.               *
*  - returns false if any error occured                                 *
*                                                                       *
************************************************************************/
 
bool call_fc (fc_infile,fc_outfile)
char *fc_infile;
 
{ bool returnvalue;
  int i;
 
#ifdef DEBUG
  printf ("call_fc: fc_infile:%s.\n",fc_infile);
#endif

  returnvalue = true;
 
  command_len = strlen(FC);
  strcpy (command,FC);
 
  concat_cmd ("-c");                     /* don't link */
  concat_cmd (FC_OPT);
  concat_cmd (fc_infile);

  if (!(target_machine == CRAY_T3E))
    {
     concat_cmd ("-o");                 /* CRAY T3E would link despite of -c */
     concat_cmd (fc_outfile);
    }

  if (dryrun_flag)
     {  printf("submitting: %s\n",command);
        return (true);
     }
 
  if (verbose_flag)  printf("%s\n",command);
 
  i = system(command);
#ifdef DEBUG
  printf("back from system <%s>. returned:%d\n",command,i);
#endif
  if (i != 0)
     { fprintf(stderr, "fc command returned error %d\n",i);
       returnvalue = false;
     }

#ifdef DEBUG
  printf("returning from call_fc\n");
#endif
  return (returnvalue);
 
} /* call_fc */
 
 
/************************************************************************
*                                                                       *
*  bool is_readable(file_name)                                          *
*                                                                       *
*  - tests whether file file_name is readable by trying to cat the file *
*    to   /dev/null                                                     *
*  - returns true   if no problem at cat-command                        *
*            false  else                                                *
*                                                                       *
************************************************************************/
 
bool is_readable(file_name)
char *file_name;
 
{
  int i;
  char testcmd[160+MAXPATHLEN];
  bool returnvalue;

 
#ifdef DEBUG
  printf ("is_readable (%s):\n",file_name);
#endif

  returnvalue = true;

  strcpy (testcmd,"cat ");
  strcat (testcmd,file_name);
  strcat (testcmd," > /dev/null");

  i = system(testcmd);
#ifdef DEBUG
  printf("back from system <%s>. returned:%d\n",testcmd,i);
#endif
  if (i != 0)
    {
     printf("NO READ ACCESS ON %s!\n",file_name);
     returnvalue = false;
    }

  return (returnvalue);

} /* is_readable(file_name) */
     
/************************************************************************
*                                                                       *
*  bool call_ld ()                                                      *
*                                                                       *
*  - composes flags and files for ld-call; then calls ld.               *
*  - returns false if any error occured                                 *
*                                                                       *
************************************************************************/
 
bool call_ld ()
 
{
  int i;
  bool r, returnvalue;
  char testcmd[MAXPATHLEN];

 
#ifdef DEBUG
  printf ("call_ld:\n");
#endif

  returnvalue = true;
 
  command_len = strlen(LD);
  strcpy (command,LD);

  concat_cmd ("-o");
  concat_cmd (output_file_name);
 
  for (i = 0; i < no_input_files; i++)
       concat_cmd (ld_input_file[i]);

   if (strlen(DALIB) == 0)
     {
      fprintf(stderr,"INTERNAL ERROR: empty filename for DALIB!\n");
      returnvalue = false;
     }
   else
      returnvalue = is_readable(DALIB);


  concat_cmd (DALIB); 
  concat_cmd (MESS_PASS_LIB);
  concat_cmd (LD_OPT);
 
  if (dryrun_flag)
    {
      printf("submitting: %s\n",command);
      return (returnvalue);
    }

  if (returnvalue == false) return (false);

 
  if (verbose_flag)  printf("%s\n",command);
 
  i = system(command);
#ifdef DEBUG
  printf("back from system <%s>. returned:%d\n",command,i);
#endif
  if (i != 0)
     { fprintf(stderr, "ld command returned error %d\n",i);
       returnvalue = false;	
     }

#ifdef DEBUG
  printf("returning from call_ld\n");
#endif
  return (returnvalue);
 
} /* call_ld */

 
/************************************************************************
*                                                                       *
*  bool call_ld1()                                                      *
*                                                                       *
*  - composes flags and files for ld1-call; then calls ld1.             *
*  - returns false if any error occured                                 *
*                                                                       *
************************************************************************/
 
bool call_ld1()
 
{
  int i;
  bool returnvalue;


#ifdef DEBUG
  printf ("call_ld1:\n");
#endif

  returnvalue = true;
 
  command_len = strlen(LD1);
  strcpy (command,LD1);

  concat_cmd ("-o");
  concat_cmd (output_file_name);


  for (i = 0; i < no_input_files; i++)
       concat_cmd (ld_input_file[i]);

  if (strlen(DALIB) == 0)
    {
     fprintf(stderr,"INTERNAL ERROR: empty filename for DALIB!\n");
     returnvalue = false;
    }
  else
     returnvalue = is_readable(DALIB);
  
 
  concat_cmd (DALIB);
  concat_cmd (LD1_OPT);
 
  if (dryrun_flag)
    {
      printf("submitting: %s\n",command);
      return (returnvalue);
    }
 
  if (returnvalue == false)    return (false);
 
 
  if (verbose_flag)  printf("%s\n",command);

  i = system(command);
#ifdef DEBUG
  printf("back from system <%s>. returned:%d\n",command,i);
#endif
  if (i != 0)
     { fprintf(stderr, "ld1 command returned error %d\n",i);
       returnvalue = false;
     }

#ifdef DEBUG
  printf("returning from call_ld1\n");
#endif
  return (returnvalue);
 
} /* call_ld1 */


/*************************************************************************
*                                                                        *
*  bool translate_inputfiles(void)                                       *
*                                                                        *
*   - generates input-(and output-)filenames for requested processors    *
*   - returns true  if all processors return 0                           *
*             false else                                                 *
*   - exits -1 if any severe error occurs                                *
*                                                                        *
*************************************************************************/
 
bool translate_inputfiles(void)
 
{ int i,j;
  char main_name[MAXPATHLEN];
  char suffix[MAXPATHLEN];
  char cpp_outfile[MAXPATHLEN];
  char adp_outfile[MAXPATHLEN];
  char fc_outfile[MAXPATHLEN];
  bool returnvalue;
  char *ptr;
 
#ifdef DEBUG
  printf ("translate_inputfiles:\n");
#endif
 
  returnvalue = true;
 
  for ( i = 0; i < no_input_files; i++)
      {
       if (split_filename(input_file_name[i],main_name,suffix) == false )
          {
           printf ("ERROR: unable to split filename <%s>\n",input_file_name[i]);
           if (!(dryrun_flag) ) error_exit(-1);
          }
       infile_typ = UNDEF_FILE;
 
       if (is_cpp_source_file(input_file_name[i]))
          {
           infile_typ = CPP_FILE;
 
           determine_cpp_outfile_name(cpp_outfile,main_name,suffix);
           returnvalue = call_cpp(input_file_name[i],cpp_outfile);
#ifdef DEBUG
  printf("back from call_cpp\n");
#endif
           if (returnvalue == false)   continue;         /* next input file */
 
           determine_adp_outfile_name(adp_outfile,main_name,suffix);
           returnvalue = call_fadapt(cpp_outfile,adp_outfile);
#ifdef DEBUG
  printf("back from call_fadapt\n");
#endif
           if (returnvalue == false)    continue;        /* next input file */
 
           if (keep_flag == 0)
             remove_intermediate_file (cpp_outfile);
 
           determine_fc_outfile_name(fc_outfile,main_name);
           returnvalue = call_fc(adp_outfile,fc_outfile);
#ifdef DEBUG
  printf("back from call_fc\n");
#endif
           if (returnvalue == false)    continue;         /* next input file */
 
           if (keep_flag == 0)
             remove_intermediate_file (adp_outfile);
 
           ld_input_file[i] = generate_new_name(fc_outfile);
 
#ifdef DEBUG
  printf("is_cpp_file: ld_input_file[%d]:%s\n",i,ld_input_file[i]);
#endif
          }
 
 
       else if (is_fadapt_source_file(input_file_name[i]))
          {
           infile_typ = FADAPT_FILE;
 
           determine_adp_outfile_name(adp_outfile,main_name,suffix);
           returnvalue = call_fadapt(input_file_name[i],adp_outfile);
#ifdef DEBUG
  printf("back from call_fadapt\n");
#endif
           if (returnvalue == false)     continue;        /* next input file */
 
 
           determine_fc_outfile_name(fc_outfile,main_name);
           returnvalue = call_fc(adp_outfile,fc_outfile);
#ifdef DEBUG
  printf("back from call_fc\n");
#endif
           if (returnvalue == false)     continue;        /* next input file */
 
           if (keep_flag == 0)
             remove_intermediate_file (adp_outfile);
 
           ld_input_file[i]  = generate_new_name(fc_outfile);
 
#ifdef DEBUG
  printf("is_adp_file: ld_input_file[%d]:%s\n",i,ld_input_file[i]);
#endif
          }
 
 
       else if (is_fc_source_file(input_file_name[i]))
          {
           infile_typ = FC_FILE;
 
           determine_fc_outfile_name(fc_outfile,main_name);
           returnvalue = call_fc(input_file_name[i],fc_outfile);
#ifdef DEBUG
  printf("back from call_fc\n");
#endif
           if (returnvalue == false)      continue;       /* next input file */
 
#ifdef DEBUG
  printf("retrunvalue = true; fc_outfile: %s\n",fc_outfile);
#endif
 
           ld_input_file[i]  = generate_new_name(fc_outfile);
 
#ifdef DEBUG
  printf("is_fc_file: ld_input_file[%d]:%s\n",i,ld_input_file[i]);
#endif
          }
 
 
       else if (is_ld_source_file(input_file_name[i]))
          {
           infile_typ = LD_FILE;
           ld_input_file[i]  = input_file_name[i];
 
#ifdef DEBUG
  printf("is_ld_file: ld_input_file[%d]:%s\n",i,ld_input_file[i]);
#endif
          }
 
      else
             { fprintf(stderr,"%s : no valid inputfilename for any processor!\n",input_file_name[i]);
               returnvalue = false;
             }
 
       }  /* for */
 
#ifdef DEBUG
  printf("returning from translate_input_files\n");
#endif
 
  return (returnvalue);
 
} /*  translate_inputfiles()  */
 
 
/************************************************************************
*                                                                       *
*  void create_workspace (argc,argv)                                    *
*                                                                       *
*  - creates workspace for arrays of input- and output-filenames for    *
*    each processor                                                     *
*                                                                       *
************************************************************************/
 
void create_workspace (argc,argv)
int argc;
char ** argv;

{ int i;
  char *comm;
 
#ifdef DEBUG
  printf ("create_workspace:\n");
#endif
 
  for (i = 1; i < argc; i++)
     {  comm = argv[i];
        if ('-' != comm[0])
          {                                /* seems to be a filename */
#ifdef DEBUG
  printf ("<%s> seems to be a filename.\n",comm);
#endif
           if ((strcmp(argv[i-1],"-o") != 0) && (strcmp(argv[i-1],"-home") != 0))
             {                 /*input filename*/
#ifdef DEBUG
  printf ("<%s> seems to be an input file name!\n",comm);
#endif
              if (access (comm, R_OK) != 0)
                { fprintf (stderr,"No access on %s!\n", comm);
                  exit (-1);
                }
              no_input_files += 1;
             }  /* if name of output file or phome */
          } /* if opt without - */
     }  /* for */


/* series of pointers to filenames placed in argv or in the arrays below */
 
  input_file_name  = (char **) malloc (sizeof(char*) * no_input_files);
  ld_input_file    = (char **) malloc (sizeof(char*) * no_input_files);
 
/* space for generated filenames */
 
  file_factor = no_input_files * 1;  /* for ld_input_files only */
  generated_file_names  = (char *) malloc (sizeof(char) * file_factor * MAXPATHLEN);

  no_generated_files = 0;
 
} /* create_workspace */


/************************************************************************
*                                                                       *
*  void release_workspace ()                                            *
*                                                                       *
*  - releases workspace for arrays of input- and                        *
*    outputfilenames for each processor                                 *
*                                                                       *
************************************************************************/
 
static void release_workspace ()
{
 
#ifdef DEBUG
  printf ("release_workspace:\n");
#endif
 
/* series of pointers to filenames placed in the arrays below */
 
  free (input_file_name);
  free (ld_input_file);
 
/* space for generated filenames */
 
  free (generated_file_names);
 

} /* release_workspace */


/************************************************************************
*                                                                       *
*  void error_exit (int:number)                                         *
*                                                                       *
*  - releases workspace  and  exits with number number                  *
*                                                                       *
************************************************************************/
 
void error_exit (number)
int number;
{
 
#ifdef DEBUG
  printf ("error_exit:\n");
#endif
  
   release_workspace();
   exit (number);
 
} /* error_exit */
 

/************************************************************************
*                                                                       *
*           main                                                        *
*                                                                       *
************************************************************************/
 
main(argc, argv)
int argc;
char **argv;
 
{  int ErrorCount;
   int i, no_arg;
   bool returnvalue;

  INITIALIZED_FLAG = 1;    /* indicate fadapt is called by gmdhpf */
 
  strcpy(output_file_name,"a.out");

  no_input_files = 0;

  create_workspace (argc,argv);
 
   no_arg = eval_gmdhpf_arg (argc, argv);
 
   if (no_arg > 0)
 
      { fprintf (stderr, "Unrecognized options\n");
        for (i=1; i<=no_arg; i++)
           fprintf (stderr, "invalid option = %s\n", argv[i]);
        printf ("  gmdhpf -help\n");
        error_exit (-1);
      }
 
  if (no_input_files == 0)
    {
     printf("gmdhpf: no input files\n");
     error_exit (-1);
    }
 
#ifdef INTERACTIVE
/*       { interactive = 1;    */
/*         xmenu (argc, argv); */  /* widget management */
/*         exit (0);           */
/*       }                     */
#endif

#ifdef DEBUG
  printf("no_input_files: %d\n",no_input_files);
  for (i = 0; i < no_input_files; i++)
   printf("input_file_name[%d]: %s\n",i,input_file_name[i]);
#endif

   returnvalue = translate_inputfiles();

   if (returnvalue == false)
     {
      printf ("gmdhpf: no link because of error at translation step!\n");
      error_exit (-1);
     }

#ifdef DEBUG
  printf("no_input_files: %d\n",no_input_files);
  for (i = 0; i < no_input_files; i++)
   printf("ld_input_file[%d]: %s\n",i,ld_input_file[i]);
#endif

   if (ld_flag)
      if (uniproc_flag)
         returnvalue = call_ld1 ();
        else
         returnvalue = call_ld (); 

   if (returnvalue == false)
     {
      printf ("gmdhpf: error at link step!\n");
      error_exit (-1);
     }


   release_workspace (); 

   exit (0);
}
 

