/*************************************************************************
*                                                                        *
*  Name : translate.c                                                    *
*                                                                        *
*  Purpose : Translation of one file                                     *
*                                                                        *
*  Author : Dr. Thomas Brandes, GMD, SCAI.LAB                            *
*                                                                        *
*  Last Update : Aug  1993                                               *
*                                                                        *
*************************************************************************/

#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>

#include "global.h"
#include "protocol.h"      /* protocol_errors  */

#include "MySource.h"
#include "Parser.h"
#include "Tree.h"
#include "MakeDefs.h"      /* SEMANTIC Phase 1 */
#include "Semantic.h"      /* SEMANTIC Phase 2 */
#include "ControlFlow.h"   /* SEMANTIC Phase 3 */
#include "Modules.h"       /* WriteModules     */
#include "Unparse.h"

#include "CallGraphFns.h"
#include "Calling.h"

#include "ImplMapping.h"     /* PARALLEL Phase 1 */
#include "Normalization.h"
#include "AnaOverlap.h"
#include "InterProc.h"
#include "AdaptAnalysis.h"

#include "MakeArguments.h"
#include "MakeParLoops.h"
#include "MakeHome.h"
#include "MakeLocal.h"
#include "Classification.h"
#include "Optimization.h"
#include "CodeGeneration.h"
#include "FinalCode.h"

/*************************************************************************
*                                                                        *
*  strip_file (infile_name, outfile_name, strip_length)                  *
*                                                                        *
*  Purpose : Splitting Fortran Source programs with longer ines          *
*                                                                        *
*  Author : James Cownie, Meiko, Bristol
*                                                                        *
*  Last Update : June 1993                                               *
*                                                                        *
*                                                                        *
*   The behaviour of this code was not actually safe. It will break      *
*   if it has to split lines with long character string definitions,     *
*   as it is not aware that that is what it is doing.                    *
*                                                                        *
*   It would actually be safer not to try to choose a good place to      *
*   split, but always split at the correct column, and ASSUME that the   *
*   Fortran compiler will re-assemble the lines at the same width.       *
*                                                                        *
*************************************************************************/

#define MAXLEN 200

void strip_file (infile_name, strip_length)

char * infile_name;
int strip_length;

{ char buffer [MAXLEN+100];
  char outfile_name [50];
  FILE *infile, *outfile;
  int c;
  int buflen;
  int error;
   
  /* return; */

  /* get the file name for the outfile */

  sprintf (outfile_name, "xwyvz_%d.tmp", getpid());

  if ((strip_length < 50) || (strip_length > 150))
    { 
      fprintf (stderr, "strip_length must be in range [50 - 150]\n");
      exit (-1);
    }
    
  infile  = fopen (infile_name, "r");
    
  if (infile == NULL)
    { fprintf (stderr, "Input File %s could not be opened\n", infile_name);
      exit (-1);
    } 
    
  /* outfile will be a temporary file */

  outfile = fopen (outfile_name, "w");
    
  if (outfile == NULL)
    { fprintf (stderr, "Output File %s could not be opened\n", outfile_name);
      exit (-1);
    } 
    
  buflen = 0;  /* actual buffer length */
    
  while ((c = fgetc(infile)) != EOF)

  { int i;

    if (c == '\n')

      { /* new line, print current line */

        for (i=0; i<buflen; i++) putc (buffer[i], outfile);
        putc ('\n', outfile);
	buflen = 0;
      }

     else 

      { /* set new character */

        buffer[buflen++] = c;

        if (buflen == strip_length)

           { /* print out buffer */

             for (i=0; i<buflen; i++) putc (buffer[i], outfile);
	     putc ('\n', outfile);

             /* make continuation line, but attention to compiler 
                directives that require same first characters      */

             if (   (buffer[0] == ' ')
                 || ((buffer[0] >= '0') && (buffer[0] <= '9')) )

                { for (i=0;i<5;i++) buffer[i] = ' '; }

             buffer[5] = '&';

             buflen = 6;

           }
     }

  }  /* while not EOF */
    
  /* print rest of the buffer (last line if no \n in input) */

  if (buflen > 0)

    { int i;

      for (i=0; i<buflen; i++) putc (buffer[i], outfile);
      putc ('\n', outfile);

    }
    
  fclose (infile);
  fclose (outfile);

#ifdef WIN32
  sprintf (buffer, "copy %s %s", outfile_name, infile_name);
  sprintf (buffer, "del %s", outfile_name);
#else
  sprintf (buffer, "mv -f %s %s", outfile_name, infile_name);
#endif

  error = system (buffer);

  if (error != 0)

    { fprintf (stderr, "ATTENTION: first attempt to replace %s failed\n",
                        infile_name);
      error = system ("sleep 10");
      error = system (buffer);
      if (error != 0)
       { fprintf (stderr, "ERROR: could not replace old file %s after strip\n",
                           infile_name);
          exit (-1);
       }
    }

  if (verbose_flag)

     printf ("Fortran file %s has been generated (strip = %d)\n", 
              infile_name, strip_length);
     
} /* strip_file */

/*************************************************************************
*                                                                        *
*  UnparseTree (filename, root)                                          *
*                                                                        *
*   - unparse the tree 'root' to file with name 'filename'               *
*   - make a strip on the file                                           *
*                                                                        *
*************************************************************************/

void UnparseTree (filename, root)
char  * filename;
tTree root;

{ FILE *f;

  f = fopen (filename, "w");

  if (f == (FILE *) NULL)
   { printf ("Error: could not open file %s for unparsing\n", filename);
     printf ("Please check permissions\n");
     exit (-1);
   }

  FileUnparse (f, root);

  fclose (f);

  strip_file (filename, strip_length);

}  /* UnparseTree */

/*************************************************************************
*                                                                        *
*  int SourceParser (char *file_name)                                    *
*                                                                        *
*    - parses fortran file file_name, returns number of syntax errors    *
*    - -1 if file could not be opened                                    *
*    - syntax tree is globally available in TreeRoot                     *
*                                                                        *
*************************************************************************/
 
int SourceParser (fname)
char *fname;

{  int Errors;

   filename = fname;    /* for makefile write, will know the file name */
   BeginFile (fname);
   Errors = Parser () + SourceFileErrors;
   if (Errors == 0) 
       sprintf (last_message, "File %s succesfully parsed", fname);
     else
       sprintf (last_message, "File %s parsed with %d Errors", fname, Errors);
   return (Errors);
} /* SourceParser */

int SourceCheck ()
{  if (!CheckTree (TreeRoot))
     { sprintf (last_message, "Tree is not okay");
       return 1;
     }
    else
     { sprintf (last_message, "Tree is good");
       return 0;
     }
}

int SourceSemantic ()

{ int phase;
  int errors;

  if (verbose_flag) printf ("SEMANTIC PHASE 1 : Make Definitions\n");
  phase = 1;
  MakeDefs (TreeRoot); 
  errors = protocol_errors ();
  if (errors > 0)
    { sprintf (last_message, 
        "%d Errors in SEMANTIC 1 : MakeDefs (see adaptor.def)", errors);
      goto Ende;
    }

  if (!CheckTree (TreeRoot))
    { sprintf (last_message, "Illegal Tree after PHASE 1 : MakeDefs");
      goto Ende;
    }

  if (verbose_flag) printf ("SEMANTIC PHASE 2 : Checking\n");
  phase = 2;
  BeginSemantic (); 
  Semantic (TreeRoot); 
  CloseSemantic (); 
  errors = protocol_errors ();
  if (errors > 0)
    { sprintf (last_message, 
         "%d Errors in SEMANTIC 2 : Checks (see adaptor.sem)", errors);
      goto Ende;
    }
  if (!CheckTree (TreeRoot))
    { sprintf (last_message, "Illegal Tree after PHASE 2 : Semantic");
      goto Ende;
    }

  if (verbose_flag) printf ("SEMANTIC PHASE 3 : Control Flow\n");
  phase = 3;
  ControlFlow (TreeRoot); 
  errors = protocol_errors ();
  if (errors > 0)
    { sprintf (last_message, 
         "%d Errors in SEMANTIC 3 : Control Flow (see adaptor.cf)", errors);
      goto Ende;
    }

  sprintf (last_message, "Semantic Analysis was successful");

  WriteModules (TreeRoot);

  phase = 0;

Ende: return (phase);
}

int SourceWrite ()

{  FILE *myFile;
   printf ("Write Tree on File test.out \n");
   myFile = fopen ("test.out","w");
   if (myFile == (FILE *) NULL)
    { printf ("Adaptor failed to open file test.out\n");
      printf ("Please check permissions\n");
      exit (-1);
    }
   WriteTree (myFile, TreeRoot);
   fclose (myFile);
   printf ("Writing Tree is ready\n");
   sprintf (last_message, "Abtract Tree written to file test.out");
}

int SourceCalling (filename)
char *filename;

{  FILE *myFile;

   BeginCalling ();
   printf ("Starting Calling Analysis \n");
   Calling (TreeRoot); 
   printf ("Ending Calling Analysis\n");

   /* import CGFile from Calling.h */

   printf ("Start Writing CallGraph to file %s\n", filename);

   CGFile = fopen (filename,"w");

   if (CGFile == (FILE *) NULL)
    { printf ("Adaptor failed to open file %s\n", filename);
      printf ("Please check permissions\n");
      exit (-1);
    }
   OutCallGraph (TheCallGraph);
   fclose (CGFile);
   printf ("End Writing CallGraph to File %s\n", filename);
   sprintf (last_message, "Call Graph has been generated");
}

void SourceUnparse ()

{ BeginUnparse ();
  Unparse (TreeRoot);
  CloseUnparse ();
}

int SourceInterface (interface_file)
char * interface_file;

{  int errors;

   if (verbose_flag) printf ("MAKE INTERFACE\n");
   MakeInterface (TreeRoot);

   errors = protocol_errors ();

   if (errors > 0)
     { sprintf (last_message,
          "%d Errors for Interface (see adaptor.intf)", errors);
       goto Ende;
     }
   if (!CheckTree (TreeRoot))
     { sprintf (last_message, "Illegal Tree after Making Interface");
       goto Ende;
     }

   UnparseTree (interface_file, TreeRoot);

   sprintf (last_message, "Interface File %s created\n", interface_file);

   Ende: return (errors);
}

int SourceParallelization ()

{ int phase;
  int errors;

  /* PHASE 1 : DEFAULT DISTRIBUTION */

     if (verbose_flag) printf ("PARALLELIZATION 1 : Distribution\n");
     phase = 1;
     BeginImplMapping ();
     ImplMapping (TreeRoot); 
     CloseImplMapping ();
     errors = protocol_errors ();
     if (errors > 0)
       { sprintf (last_message, 
            "%d Errors in PHASE 1 : Distribution (see adaptor.dist)", errors);
         goto Ende;
       }
     if (!CheckTree (TreeRoot))
       { sprintf (last_message, "Illegal Tree after PHASE 1 : Distribution");
         goto Ende;
       }

  /* PHASE 2 : Normalization  */

     if (verbose_flag) printf ("PARALLELIZATION 2 : Normalization\n");
     phase = 2;
     BeginNormalization ();
     Normalization (TreeRoot); 
     CloseNormalization ();
     errors = protocol_errors ();
     if (errors > 0)
       { sprintf (last_message, 
            "%d Errors in PHASE 2 : Normalizations (see adaptor.normal)", 
            errors);
         goto Ende;
       }
     if (!CheckTree (TreeRoot))
       { sprintf (last_message, "Illegal Tree after PHASE 2 : Normalization ");
         goto Ende;
       }

  /* PHASE 3 : Overlapping */

     if (verbose_flag) printf ("PARALLELIZATION 3 : Overlapping\n");
     phase = 3;
     BeginAnaOverlap ();
     OverlapAnalysis (TreeRoot); 
     CloseAnaOverlap ();
     errors = protocol_errors ();
     if (errors > 0)
       { sprintf (last_message, 
            "%d Errors in PHASE 3 : Overlapping (see adaptor.ovl)", 
            errors);
         goto Ende;
       }
     if (!CheckTree (TreeRoot))
       { sprintf (last_message, "Illegal Tree after PHASE 3 : Overlapping ");
         goto Ende;
       }

  /* PHASE 4 : Interprocedural Analysis */

     if (verbose_flag) 
        printf ("PARALLELIZATION 4 : Interprocedural Analysis\n");
     phase = 4;
     BeginInterProc ();
     Interprocedural (TreeRoot); 
     CloseInterProc ();
     errors = protocol_errors ();
     if (errors > 0)
       { sprintf (last_message, 
            "%d Errors in PHASE 4 : Overlapping (see adaptor.ipa)", 
            errors);
         goto Ende;
       }
     if (!CheckTree (TreeRoot))
       { sprintf (last_message, "Illegal Tree after PHASE 4 : Interprocedural");
         goto Ende;
       }

     /* make an intermediate output */

     if (debug_flag)
        UnparseTree ("normal_unparse.f", TreeRoot);

  /* PHASE 5 : ADAPTOR Analysis */

     if (verbose_flag) 
        printf ("PARALLELIZATION 5 : ADAPTOR Analysis\n");
     phase = 5;
     BeginAdaptAnalysis ();
     AdaptAnalysis (TreeRoot); 
     CloseAdaptAnalysis ();
     errors = protocol_errors ();
     if (errors > 0)
       { sprintf (last_message, 
            "%d Errors in PHASE 5 : Analysis (see adaptor.ana)", 
            errors);
         goto Ende;
       }
     if (!CheckTree (TreeRoot))
       { sprintf (last_message, "Illegal Tree after PHASE 5 : Analysis");
         goto Ende;
       }

     sprintf (last_message, "Parallelization without errors");
     phase = 0;   /* no errors happened */

Ende: return(phase);

} /* SourceParallelization */

int SourceAdapt (node_file)
char * node_file;

{  FILE *f;
   tTree HostRoot;
   int errors;
   int phase;

   /* PHASE 1 : ADAPTOR Arguments */

     if (verbose_flag) printf ("ADAPTOR PHASE 1 : Arguments\n");
     phase = 1;
     MakeArguments (TreeRoot); 
     errors = protocol_errors ();
     if (errors > 0)
       { sprintf (last_message, 
            "%d Errors in PHASE 1 : Arguments (see adaptor.args)", errors);
         goto Ende;
       }
     if (!CheckTree (TreeRoot))
       { sprintf (last_message, "Illegal Tree after PHASE 1 : Arguments ");
         goto Ende;
       }

     /* make an intermediate output */

     if (debug_flag)
        UnparseTree ("arg_unparse.f", TreeRoot);

   /* PHASE 2 : ADAPTOR Make Loops */

     if (verbose_flag) printf ("ADAPTOR PHASE 2 : Make Loops \n");
     phase = 2;
     MakeParLoops (TreeRoot); 
     errors = protocol_errors ();
     if (errors > 0)
       { sprintf (last_message,
            "%d Errors in PHASE 2 : MakeLoops (see adaptor.loops)", errors);
         goto Ende;
       }
     if (!CheckTree (TreeRoot))
       { sprintf (last_message, "Illegal Tree after PHASE 2 : Make Loops");
         goto Ende;
       }

     /* make an intermediate output */

     if (debug_flag)
        UnparseTree ("loop_unparse.f", TreeRoot);

   /* PHASE 3 : ADAPTOR Find Homes */

     if (verbose_flag) printf ("ADAPTOR PHASE 3 : Find Homes \n");
     phase = 3;
     MakeHome (TreeRoot);  
     errors = protocol_errors ();
     if (errors > 0)
       { sprintf (last_message, 
            "%d Errors in PHASE 3 : FindHomes (see adaptor.home)", errors);
         goto Ende;
       }
     if (!CheckTree (TreeRoot))
       { sprintf (last_message, "Illegal Tree after PHASE 3 : Find Homes ");
         goto Ende;
       }

     /* make an intermediate output */

     if (debug_flag)
        UnparseTree ("home_unparse.f", TreeRoot);

   /* PHASE 4 : ADAPTOR Extract Communication */

     if (verbose_flag) printf ("ADAPTOR PHASE 4 : Extract Communication \n");
     phase = 4;
     MakeLocal (TreeRoot);  
     errors = protocol_errors ();
     if (errors > 0)
       { sprintf (last_message, 
            "%d Errors in PHASE 4 : ExtractComm (see adaptor.local)", errors);
         goto Ende;
       }
     if (!CheckTree (TreeRoot))
       { sprintf (last_message, "Illegal Tree after PHASE 4 : ExtractComm ");
         goto Ende;
       }

     /* make an intermediate output */

     if (debug_flag)
        UnparseTree ("loc_unparse.f", TreeRoot);

   /* PHASE 5 : ADAPTOR Classification */

     if (verbose_flag) printf ("ADAPTOR PHASE 5 : Classification\n");
     phase = 5;
     Classification (TreeRoot); 
     errors = protocol_errors ();
     if (errors > 0)
       { sprintf (last_message, 
            "%d Errors in PHASE 5 : Serial (see adaptor.class)", errors);
         goto Ende;
       }
     if (!CheckTree (TreeRoot))
       { sprintf (last_message, "Illegal Tree after PHASE 5 : Classification");
         goto Ende;
       }

   /* PHASE 6 : ADAPTOR Optimization */

     if (verbose_flag) printf ("ADAPTOR PHASE 6 : Optimization\n");
     phase = 6;
     Optimization (TreeRoot); 
     errors = protocol_errors ();
     if (errors > 0)
       { sprintf (last_message, 
            "%d Errors in PHASE 6 : Optimization (see adaptor.opt)", errors);
         goto Ende;
       }
     if (!CheckTree (TreeRoot))
       { sprintf (last_message, "Illegal Tree after PHASE 6 : Optimization");
         goto Ende;
       }

     /* make an intermediate output */

     if (debug_flag)
        UnparseTree ("opt_unparse.f", TreeRoot);

   /* PHASE 7 : ADAPTOR Code Generation */

     if (verbose_flag) printf ("ADAPTOR PHASE 7 : Code Generation\n");
     phase = 7;

     NodeAdapt (TreeRoot); 

     errors = protocol_errors ();
     if (errors > 0)
       { sprintf (last_message, 
          "%d Errors in PHASE 7 : Code Generation (see adaptor.code)", errors);
         goto Ende;
       }
     if (!CheckTree (TreeRoot))
       { sprintf (last_message,
                  "Illegal Node Tree after PHASE 7 : Code Generation");
         goto Ende;
       }

     FinalCode (TreeRoot); 

     UnparseTree (node_file, TreeRoot);

     sprintf (last_message, "Translation successful, %s generated", node_file);
     phase = 0;

Ende: return(phase);
}

/*************************************************************************
*                                                                        *
*  translate_file (source_file, target_file)                             *
*                                                                        *
*************************************************************************/

void translate_file (source_filename, target_filename)

char *source_filename;
char *target_filename;

{ int ErrorCount;

  char node_file [10];  /* name of file for node/cube/node1 program */

  ErrorCount = SourceParser (source_filename);
 
  if (ErrorCount > 0)
    { printf ("%s\n", last_message);
      exit (-1);
    }
  else if (verbose_flag)
      printf ("%s\n", last_message);

  if (!CheckTree (TreeRoot))
    { sprintf (last_message, "Illegal Tree after Parsing");
      exit(-1);
    }

  ErrorCount = SourceSemantic ();
 
  if (ErrorCount > 0)
    { printf ("%s\n", last_message);
      exit (-1);
    }
  else if (verbose_flag)
      printf ("%s\n", last_message);

  ErrorCount = SourceParallelization ();
 
  if (ErrorCount > 0)
    { printf ("%s\n", last_message);
      exit (-1);
    }
  else if (verbose_flag)
      printf ("%s\n", last_message);

  if (strcmp (target_filename, "") == 0)

   {  strcpy (node_file, model_srcfile [target_model]);
      ErrorCount = SourceAdapt (node_file);
   }

   else
 
      ErrorCount = SourceAdapt (target_filename);

  if (ErrorCount > 0)
    { printf ("%s\n", last_message);
      exit (-1);
    }
  else if (verbose_flag)
      printf ("%s\n", last_message);

}

void interface_file (source_filename, target_filename)

char *source_filename;
char *target_filename;

{ int ErrorCount;

  char node_file [10];  /* name of file for node/cube/node1 program */

  ErrorCount = SourceParser (source_filename);
 
  if (ErrorCount > 0)
    { printf ("%s\n", last_message);
      exit (-1);
    }
  else if (verbose_flag)
      printf ("%s\n", last_message);

  ErrorCount = SourceSemantic ();
 
  if (ErrorCount > 0)
    { printf ("%s\n", last_message);
      exit (-1);
    }
  else if (verbose_flag)
      printf ("%s\n", last_message);

  if (strcmp (target_filename, "") == 0)
      ErrorCount = SourceInterface ("interface.h");
   else
      ErrorCount = SourceInterface (target_filename);

  if (ErrorCount > 0)
    { printf ("%s\n", last_message);
      exit (-1);
    }
  else if (verbose_flag)
      printf ("%s\n", last_message);

}

void callgraph_file (source_filename, target_filename)

char *source_filename;
char *target_filename;

{ int ErrorCount;

  char node_file [10];  /* name of file for node/cube/node1 program */

  ErrorCount = SourceParser (source_filename);
 
  if (ErrorCount > 0)
    { printf ("%s\n", last_message);
      exit (-1);
    }
  else if (verbose_flag)
      printf ("%s\n", last_message);

  ErrorCount = SourceSemantic ();
 
  if (ErrorCount > 0)
    { printf ("%s\n", last_message);
      exit (-1);
    }
  else if (verbose_flag)
      printf ("%s\n", last_message);

  if (strcmp (target_filename, "") == 0)
      ErrorCount = SourceInterface ("interface.h");
   else
      ErrorCount = SourceCalling ("test.call");

  if (ErrorCount > 0)
    { printf ("%s\n", last_message);
      exit (-1);
    }
  else if (verbose_flag)
      printf ("%s\n", last_message);

}
