/*************************************************************************
*                                                                        *
*  Name : arguments.c                                                    *
*                                                                        *
*  Purpose : Evaluation of command line arguments for hpf / fadapt       *
*                                                                        *
*  Author : Dr. Thomas Brandes, GMD, SCAI.LAB                            *
*           Resi Hoever-Klier, GMD, SCAI.LAB                             *
*                                                                        *
*  created     : Aug  1993                                               *
*                                                                        *
*  Last Update : May  1997                                               *
*                                                                        *
*   Mar 96  : reading .adprc files inserted (by Resi)                    *
*   May 97  : option -I<include_dir>                                     *
*                                                                        *
*************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include "ratc.h"             /* bool, true, false */
#include "global.h"
#include <string.h>

#undef DEBUG

# ifdef WIN32
# include <io.h>             /* for testing fileaccess */
# else
# include <unistd.h>            /* for testing fileaccess */
# endif

/****************************************************************
*                                                               *
*  is_source_file()                                             *
*                                                               *
*  INPUT:       string for filename                             *
*                                                               *
*  OUTPUT:      true, if string is an allowed fortran filename  *
*               false, if string isn`t a fortran filename       *
*                                                               *
****************************************************************/

bool is_source_file (string)
char *string;
{  int length;
   
   length = strlen (string);

   if (strcmp(string+length-2,".f") == 0) return (true);
   if (strcmp(string+length-3,".f9") == 0) return (true);
   if (strcmp(string+length-4,".f90") == 0) return (true);
   if (strcmp(string+length-4,".fcm") == 0) return (true);
   if (strcmp(string+length-4,".hpf") == 0) return (true);
   if (strcmp(string+length-4,".h90") == 0) return (true);

   return(false);
}

/****************************************************************
*                                                               *
*  print_version ()                                             *
*                                                               *
****************************************************************/

void print_version ()

{  printf (VERSION_STRING);
   printf ("\n");

} /* print_version */

/****************************************************************
*                                                               *
*  print_help ()                                                *
*                                                               *
*   - prints list of available options (very generic)           *
*                                                               *
****************************************************************/

void print_help ()
{ int i;
 printf ("\n");

 printf ("fadapt [options] \n");
 printf ("fadapt [options]    filename ... filename \n");
 printf ("fadapt [options] -f filename ... filename \n");

 for (i=0; i<no_slanguage; i++)
   printf (" %15s     source language = %s\n", slanguage_options[i], 
                                             slanguage_items[i]);
 printf ("  \n");

 for (i=0; i<no_tlanguage; i++)
   printf (" %15s     target language = %s\n", tlanguage_options[i], 
                                             tlanguage_items[i]);
 printf ("  \n");

 for (i=0; i<no_model; i++)
   printf (" %15s     model = %s\n", model_options[i], model_items[i]);
 printf ("  \n");

 for (i=0; i<no_smp; i++)
   printf (" %15s     sm parallel = %s\n", smp_options[i], smp_items[i]);
 printf ("  \n");

 for (i=0; i<no_vec; i++)
   printf (" %15s     vectorization = %s\n", vec_options[i], vec_items[i]);
 printf ("  \n");

 for (i=0; i<no_ddefault; i++)
   printf (" %15s     default distribution = %s\n", ddefault_options[i], 
                                                    ddefault_items[i]);
 printf ("  \n");

 printf ("  [-mp]              shared memory parallelization of output\n");
 printf ("  [-d <n>]           maximal number of distributed dimensions\n");
 printf ("  [-strip <n>]       number of columns in generated Fortran sources\n");
 printf ("  [-fix <n>]         number of relevant columns in input file\n");
 printf ("  [-free]            free source format\n");
 printf ("  [-v]               verbose\n");
 printf ("  [-debug]           generates intermediate files\n");
 printf ("  [-list]            do not delete protocol files\n");
 printf ("  [-G]               abbreviation for -v -debug -list\n");
 printf ("\n");

 printf ("  [-interface]       Generate Interface Specification\n");
 printf ("  [-call]            Generate Call Graph\n");

 printf ("  [-i4|-i8]          default size of integer is 4 or 8 bytes\n");
 printf ("  [-r4|-r8]          default size of real    is 4 or 8 bytes\n");
 printf ("  [-a4|-a8]          default size of address is 4 or 8 bytes\n");
 printf ("  [-sp|-dp]          REAL is single / double precision \n");

 printf ("  [-O]               optimization\n");
 printf ("  [-noopt]           no optimizations\n");

 printf ("  [-make]            generate Makefile\n");

 printf ("  [-safety n]        run time checking (n=0, 1 or 2)\n");

 printf ("  [-nostrict]        no check of arguments\n");

 printf ("  [-I<include_dir>]  include directories\n");

 printf ("\n");
} /* print_help */

void print_settings ()

{ int i;

  printf ("\n");

  printf ("\n");
  printf ("Default Distribution : %s [%s]\n", ddefault_items[ddefault_kind],
                                            ddefault_options[ddefault_kind]);

  printf ("Source Language      : %s [%s]\n", slanguage_items[source_language],
                                           slanguage_options[source_language]);

  printf ("Target Language      : %s [%s]\n", tlanguage_items[target_language],
                                           tlanguage_options[target_language]);

  printf ("Target Model         : %s [%s]\n", model_items[target_model],
                                            model_options[target_model]);

  printf ("SM parallelization   : %s [%s]\n", smp_items[sm_parallelization],
                                              smp_options[sm_parallelization]);

  printf ("Vectorization        : %s [%s]\n", vec_items[vectorization],
                                              vec_options[vectorization]);

  printf ("\n");

  printf ("default integer size : %d\n", default_int_size);
  printf ("default real    size : %d\n", default_real_size);
  printf ("default address size : %d\n", default_addr_size);
  if (is_double_precision)
     printf ("REAL is double precision \n");
   else
     printf ("REAL is single precision \n");
  printf ("\n");

  printf ("Maximal number of distributed dims : %d\n", MaxDistributedDims);
  printf ("\n");
  if (optimization > 0)
     printf ("Optimization is switched on\n");
   else
     printf ("Optimization is switched off\n");
  printf ("\n");
  printf ("Safety level is                    : %d\n", safety);
  printf ("\n");
  printf ("Relevant columns for input sources : %d\n", fix_length);
  printf ("Strip length for generated sources : %d\n", strip_length);
  printf ("\n");
  printf ("Verbose = %d,  Debug = %d List = %d\n",
           verbose_flag, debug_flag, list_flag);

  for (i=0; i<no_include_dirs; i++)

    printf ("Include Directory : %s\n", include_dirs[i]);

  printf ("\n");

} /* print_settings */

/*************************************************************************
*                                                                        *
* int find_item (item_list, item_no, item_string)                        *
*                                                                        *
*************************************************************************/

int find_item (item_list, item_no, item_string)
char *item_list[], *item_string;
int  item_no;

{ int i, pos;
 
  pos = -1;

  for (i=0; i<item_no; i++)
     if (strcmp(item_string,item_list[i]) == 0)
        pos = i;

  return (pos);

}  /* find_item;

/*************************************************************************
*                                                                        *
*  set_defaults ()                                                       *
*                                                                        *
*************************************************************************/

static void set_defaults ()

{
  source_language       = predefined_source_language;
  target_language       = predefined_target_language;
  target_model          = predefined_target_model;
  sm_parallelization    = predefined_sm_parallelization;
  vectorization         = predefined_vectorization;
  ddefault_kind         = predefined_ddefault_kind;
  MaxDistributedDims    = predefined_MaxDistributedDims;
  optimization          = predefined_optimization;
  test_flag             = 0;
  safety                = predefined_safety;
 
  default_int_size  = predefined_default_int_size;
  default_real_size = predefined_default_real_size;
  default_addr_size = predefined_default_addr_size;
  is_double_precision = 0;
 
  strip_length    = predefined_strip_length;
  fix_length      = predefined_fix_length;
 
  verbose_flag    = 0;
  debug_flag      = 0;
  list_flag       = 0;
 
  adapt_command   = FULL_ADAPT;
 
} /* set_defaults */

/*************************************************************************
*                                                                        *
*  eval_arg (argc, argv)                                                 *
*                                                                        *
*   - evaluates command line arguments                                   *
*   - returns number of not evaluated arguments                          *
*                                                                        *
*************************************************************************/

int eval_arg (argc, argv)
int argc;
char **argv;

{ int i, no_arg, pos;
  char *comm;
  void print_help ();
  void print_settings ();
  void print_version ();

  set_defaults ();

  no_arg = 0;           /* number of not evaluated arguments */

  /* count number of source files in variable no_input_files */

  no_input_files = 0;

  for (i=1; i<argc; i++)
    {  comm = argv[i];
       if ('-' != comm[0])
           no_input_files += 1;
    }

  input_file_name = (char **) malloc (sizeof(char*) * no_input_files);

  no_input_files = 0;
  i=1;
  while (i<argc)
    { comm = argv[i];
      if ('-' != comm[0])
         { /* seems to be a file name */
#ifdef WIN32
           if (_access (comm, 0) != 0)
#else
           if (access (comm, R_OK) != 0)
#endif
                { fprintf(stderr,"No access on %s!\n", comm);
                  exit(-1);
                }
           else if (!is_source_file(comm))
                { fprintf(stderr,"%s : no Fortran filename\n", comm);
                  exit(-1);
                }
           else 
                { input_file_name [no_input_files] = comm;
                  no_input_files += 1;
                }
         }

      else if (strcmp(comm,"-d") == 0)
         { if ((argc >= (i+1)) && (sscanf (argv[i+1], "%d", &MaxDistributedDims)))
             	i++;
	     else
	     { fprintf(stderr,"argument missing or argument type wrong\n");
	       fprintf(stderr,"needed for argument -d\n");
	       exit(-1);
	     }
         }

      else if (strcmp(comm,"-strip") == 0)
         { if ((argc >= (i+1)) && (sscanf (argv[i+1], "%d", &strip_length)))
             	i++;
	     else
	     { fprintf(stderr,"argument missing or argument type wrong\n");
	       fprintf(stderr,"needed for argument -strip\n");
	       exit(-1);
	     }
         }
      else if (strcmp(comm,"-fix") == 0)
         { if ((argc >= (i+1)) && (sscanf (argv[i+1], "%d", &fix_length)))
             	i++;
	     else
	     { fprintf(stderr,"argument missing or argument type wrong\n");
	       fprintf(stderr,"needed for argument -fix\n");
	       exit(-1);
	     }
         }

      else if (strcmp(comm,"-free") == 0)
           fix_length = 0;

      else if (strcmp(comm,"-test") == 0)
           test_flag = 1;

      else if (strcmp(comm,"-O") == 0)
           optimization = 4;
      else if (strcmp(comm,"-O1") == 0)
           optimization = 1;
      else if (strcmp(comm,"-O2") == 0)
           optimization = 2;
      else if (strcmp(comm,"-O3") == 0)
           optimization = 3;
      else if (strcmp(comm,"-O4") == 0)
           optimization = 4;
      else if (strcmp(comm,"-noopt") == 0)
           optimization = 0;

      else if (strcmp(comm,"-safety") == 0)
         { if ((argc >= (i+1)) && (sscanf (argv[i+1], "%d", &safety)))
             	i++;
	     else
	     { fprintf(stderr,"argument missing or argument type wrong\n");
	       fprintf(stderr,"needed for argument -safety\n");
	       exit(-1);
	     }
         }

      else if (strcmp(comm,"-nostrict") == 0)
           semantic_check = 0;

      else if ( (pos=find_item (slanguage_options, no_slanguage, comm)) >= 0)
           source_language = pos;

      else if ( (pos=find_item (tlanguage_options, no_tlanguage, comm)) >= 0)
           target_language = pos;

      else if ( (pos=find_item (model_options, no_model, comm)) >= 0)
           target_model         = pos;

      else if ( (pos=find_item (smp_options, no_smp, comm)) >= 0)
           sm_parallelization   = pos;

      else if ( (pos=find_item (vec_options, no_vec, comm)) >= 0)
           vectorization        = pos;

      else if (strcmp(comm,"-i4") == 0)
           default_int_size = 4;
      else if (strcmp(comm,"-i8") == 0)
           default_int_size = 8;
      else if (strcmp(comm,"-r4") == 0)
           default_real_size = 4;
      else if (strcmp(comm,"-r8") == 0)
           default_real_size = 8;
      else if (strcmp(comm,"-a4") == 0)
           default_addr_size = 4;
      else if (strcmp(comm,"-a8") == 0)
           default_addr_size = 8;
      else if (strcmp(comm,"-dp") == 0)
           is_double_precision = 1;
      else if (strcmp(comm,"-sp") == 0)
           is_double_precision = 0;
      else if (strcmp(comm,"-v") == 0)
           verbose_flag = 1;
      else if (strcmp(comm,"-list") == 0)
           list_flag = 1;
      else if (strcmp(comm,"-debug") == 0)
           debug_flag = 1;
      else if (strcmp(comm,"-G") == 0)
           { verbose_flag = 1; list_flag = 1; debug_flag = 1; }

      else if ( (pos=find_item (ddefault_options, no_ddefault, comm)) >= 0)
           ddefault_kind = pos;

      else if (strcmp (comm, "-interface") == 0)
           adapt_command = MAKE_INTERFACE;

      else if (strcmp (comm, "-call") == 0)
           adapt_command = MAKE_CALLGRAPH;

      else if (strcmp(comm,"-f") == 0)
           adapt_command = SINGLE_ADAPT;  /* no linking */

      else if (strcmp(comm,"-o") == 0)
         {
             if (argc > i+1)
             { i++;
               output_file_name = argv[i];
             }
	     else
	     {
		fprintf(stderr,"argument missing or argument type wrong\n");
		fprintf(stderr,"needed for argument -o\n");
		exit(-1);
	     }
         }

      else if (strcmp(comm,"-help") == 0)
           { print_help ();
             exit (0);
           }
      else if (strcmp(comm,"-version") == 0)
           { print_version ();
             exit (0);
           }
      else if (strcmp(comm,"-settings") == 0)
           { print_settings ();
             exit (0);
           }
      else if (comm[1] == 'I')   /* is -I */
           SetIncludeDir (comm+2);
      else { /* not recognized option */
             no_arg += 1;
             argv[no_arg] = argv[i];
           }
      i++;
    }

    return (no_arg);   /* not evaluated arguments */
}
