/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Feb 96                                                   *
*  Last Update : Feb 96                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : shared.m4                                                *
*                                                                         *
*  Function: realization of distributed shared array by rpc's             *
*                                                                         *
*  Export :  internal Interface                                           *
*  ============================                                           *
*                                                                         *
*  Export :  FORTRAN  Interface                                           *
*  ============================                                           *
*                                                                         *
**************************************************************************/

#undef DEBUG

#include "dalib.h"

/**************************************************************************
*                                                                         *
* for remote procedure calls the shared information is zero               *
*                                                                         *
**************************************************************************/

struct SharedRecord
 
{ int  shared_kind;

};

/**************************************************************************
*                                                                         *
*   int dalib_is_globally_shared (Shared info)                            *
*                                                                         *
*     - returns 0 if local memory on each processor (local addresses)     *
*     - returns 1 if global memory for all processors (global addresses)  *
*                                                                         *
**************************************************************************/

int dalib_is_globally_shared (info)

Shared info;

{ 

  return (0);

} /* dalib_is_globally_shared */

/**************************************************************************
*                                                                         *
*   void dalib_array_shared (array_id)                                    *
*                                                                         *
*    - set descriptor in such a way that array can be accessed            *
*                                                                         *
**************************************************************************/
 
void FUNCTION(dalib_array_shared) (array_id)

array_info *array_id;

{ Shared shared_ptr;

  /* allocate the pointer of Shared Record structure */

  shared_ptr = (Shared) dalib_malloc (sizeof (struct SharedRecord),
                                      "dalib_array_shared");

  /* assign the array_id->SharedInfo */

  (*array_id)->SharedInfo = shared_ptr;
 
} /* dalib_array_shared */

/**************************************************************************
*                                                                         *
*  void dalib_array_vshared (array_id, dummy_id)                          *
*                                                                         *
*    - verify that dummy is a shared array (otherwise make                *
*      a new descriptor with shared information)                          *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_array_vshared) (array_id, dummy_id)

array_info *array_id;
 
{
}

/**************************************************************************
*                                                                         *
*                                                                         *
**************************************************************************/

void dalib_array_shared_alloc (array_id)
 
array_info array_id;

{ /* memory is created as for a usual array */

  dalib_array_malloc (array_id);

  /* nothing else is to do */

}  /* dalib_array_shared_alloc */
 
/**************************************************************************
*                                                                         *
*  dalib_shared_free (descriptor, data_flag)                              *
*                                                                         *
*   - delete data only if data_flag is set                                *
*                                                                         *
**************************************************************************/

void dalib_shared_free (descriptor, data_flag)

array_info descriptor;
int        data_flag;

{ Shared shared_ptr;

  shared_ptr  = descriptor->SharedInfo;
 
  /* detach/deallocate distributed memory region */

  free (descriptor->data);
  free (shared_ptr);

  if (descriptor->data != NO_DATA)
 
     { /* deallocate local data of the array if data_flag */
 
       if (data_flag) free (descriptor->data);
     }

} /* dalib_shared_free */


typedef struct {

  int kind;             /* Attention: must not be 0 (internally used) */
  array_info *array_id;
  int offset;

  } question_type;

/**************************************************************************
*                                                                         *
*   void dalib_rpc_get (data, remote_pid, array_id, offset)               *
*                                                                         *
*     - read remote data of array_id (offset) on remote_pid processor     *
*                                                                         *
**************************************************************************/

void dalib_rpc_get (data, remote_pid, array_id, offset)

int *data;
int remote_pid;
array_info *array_id;
int offset;

{  question_type send_data;

   int rc;
   int tag;
   int source;
   int mstat;

          /********************************************
          *                                           *
          *   prepare the request                     *
          *                                           *
          ********************************************/

   send_data.kind     = 1;           /* remote read */
   send_data.array_id = array_id;
   send_data.offset   = offset;

#ifdef DEBUG
   printf ("%d: call of rpc_get on %d with offset %d\n",
            pcb.i, remote_pid, offset);
#endif

   rpc_call (remote_pid, &send_data, sizeof (send_data),
                         data, sizeof (int));

} /* dalib_rpc_get */

/**************************************************************************
*                                                                         *
*   void dalib_rpc_put (op, data, remote_pid, array_id, offset)           *
*                                                                         *
*     - write remote data to array_id (offset) on remote_pid processor    *
*                                                                         *
**************************************************************************/
 
void dalib_rpc_put (op, data, remote_pid, array_id, offset)
 
int *data;
int remote_pid;
int op;
array_info *array_id;
int offset;
 
{  struct {

     question_type query;
     char          data[128];  /* data will be sent with the query */

     } send_data;
 
   int rc;
   int tag;
   int source;
   int mstat;
   int size;
   int dummy;
 
          /********************************************
          *                                           *
          *   prepare the request                     *
          *                                           *
          ********************************************/
 
   send_data.query.kind     = 2+op;        /* remote put */
   send_data.query.array_id = array_id;
   send_data.query.offset   = offset;

   size = (*array_id)->size;
   dalib_memcopy (&(send_data.data), data, size);
 
#ifdef DEBUG
   printf ("%d: call of rpc_op on %d with offset %d\n",
            pcb.i, remote_pid, offset);
#endif
 
   rpc_call (remote_pid, &send_data, sizeof (question_type) + size,
                         &dummy, 4);
 
} /* dalib_rpc_put */

/**************************************************************************
*                                                                         *
*  dalib_array_local_read (char *data,                                    *
*                          array_info array_id, int local_offset)         *
*                                                                         *
*     - read in local data (at the given offset)                          *
*                                                                         *
**************************************************************************/

void dalib_array_local_read (data, array_id, local_offset)

unsigned char *data;
array_info *array_id;
int local_offset;

{ unsigned char *pdata;
  int size;

  pdata = (*array_id)->data;
  size  = (*array_id)->size;

  pdata += local_offset * size;
 
#ifdef DEBUG
  printf ("%d: read in my array at local_offset %d\n", pcb.i, local_offset);
#endif

  dalib_memcopy (data, pdata, size);
 
} /* dalib_array_local_read */

/**************************************************************************
*                                                                         *
*  dalib_array_local_put (int op, char *data,                             *
*                         array_info array_id, int local_offset)          *
*                                                                         *
**************************************************************************/

void dalib_array_local_put (op, data, array_id, local_offset)

unsigned char *data;
int op;
array_info *array_id;
int local_offset;

{ unsigned char *pdata;
  int size;

  pdata = (*array_id)->data;
  size  = (*array_id)->size;

  pdata += local_offset * size;
 
#ifdef DEBUG
  printf ("%d: put into my array at offset %d, op = %d\n", 
           pcb.i, local_offset, op);
#endif

  switch (op) {

    case  0 : dalib_memcopy (pdata, data, size); break;
    case  1 : min_ints (pdata, data); break;
    case  2 : min_reals (pdata, data); break;
    case  3 : min_doubles (pdata, data); break;
    case  4 : max_ints (pdata, data); break;
    case  5 : max_reals (pdata, data); break;
    case  6 : max_doubles (pdata, data); break;
    case  7 : add_ints (pdata, data); break;
    case  8 : add_reals (pdata, data); break;
    case  9 : add_doubles (pdata, data); break;
    case 10 : mult_ints (pdata, data); break;
    case 11 : mult_reals (pdata, data); break;
    case 12 : mult_doubles (pdata, data); break;
    case 13 : and_ints (pdata, data); break;
    case 14 : or_ints (pdata, data); break;
    case 15 : eor_ints (pdata, data); break;
    case 16 : and_bools (pdata, data); break;
    case 17 : or_bools (pdata, data); break;
    case 18 : neq_bools (pdata, data); break;
    default : dalib_internal_error ("unknown op for dalib_array_local_put");
              dalib_stop ();

  } /* end switch */
 
} /* dalib_array_local_put */

/**************************************************************************
*                                                                         *
*  dalib_answer_question (question_type *question,                        *
*                         char **answer, int *size)                       *
*                                                                         *
*    - server function to answer requests from other processors           *
*                                                                         *
**************************************************************************/

void dalib_answer_question (question, answer, size)

question_type *question;
unsigned char **answer;
int *size;

{ 

#ifdef DEBUG
  printf ("%d: answer question, kind = %d\n", pcb.i, question->kind);
#endif

  if (question->kind == 1)

    { array_info array_id;

      array_id = *(question->array_id);
      *size    = array_id->size;
      *answer  = (unsigned char *) 
                 dalib_malloc (*size, "dalib_answer_question");

      dalib_array_local_read (*answer, question->array_id,
                              question->offset);

    }

   else if (question->kind > 1)

    { array_info array_id;
      int op;
      char *data;

      op       = question->kind - 2;
 
      array_id = *(question->array_id);
      *size    = array_id->size;
 
      data     = (char *) question;
      data    += sizeof (question_type);   /* data from other proc */

      dalib_array_local_put (op, data, question->array_id,
                                       question->offset);

      /* make a fictive acknowledge */

      *size    = 4;
      *answer  = (unsigned char *) 
                 dalib_malloc (*size, "dalib_answer_question");
 

    }

   else

    { dalib_internal_error ("answer_question: illegal kind");
      dalib_stop ();
    }

} /* dalib_answer question */

/**************************************************************************
*                                                                         *
*  void dalib_remote_access (nr, a1, ..., a<nr>)                          *
*                                                                         *
*    - allow remote access to the arrays a1, ..., a<nr>                   *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_remote_access) (nr, a1, a2, a3, a4, a5, a6, a7)

int *nr;
array_info *a1, *a2, *a3, *a4, *a5, *a6, *a7;

{ if (*nr > 1)

    { dalib_internal_error ("remote_access: only 1 array allowed");
      dalib_stop ();
    }

  switch (*nr) {
    case 7: dalib_array_remote_init (*a7);
    case 6: dalib_array_remote_init (*a6);
    case 5: dalib_array_remote_init (*a5);
    case 4: dalib_array_remote_init (*a4);
    case 3: dalib_array_remote_init (*a3);
    case 2: dalib_array_remote_init (*a2);
    case 1: dalib_array_remote_init (*a1);
  } /* end switch */

  rpc_init ();

  FUNCTION(dalib_barrier) ();

} /* dalib_remote_access */

/**************************************************************************
*                                                                         *
*  void dalib_remote_stop ()                                              *
*                                                                         *
*    - stop remote access to all the arrays                               *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_remote_stop) ()

{ 

  rpc_exit ();

} /* dalib_remote_stop */
 
/**************************************************************************
*                                                                         *
*   void dalib_array_read (char *data, array_info *array_id,              *
*                          int *ind1, ..., int *ind7)                     *
*                                                                         *
*     - read array[ind1,...,indk] (maybe on remote processor)             *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_array_read) (data, array_id, ind1, ind2, ind3, ind4,
                                                  ind5 ,ind6, ind7)
unsigned char *data;
array_info *array_id;
int *ind1, *ind2, *ind3, *ind4, *ind5, *ind6, *ind7;
 

{ int owner;
  int offset;

  int global_indices[MAX_DIMENSIONS];
  int i, rank;

  int top_id;
  array_info template_dsp;

  unsigned char *pdata;

  /* pack indices for local and remote address computation */

  rank = (*array_id)->rank;

  dalib_array_info (*array_id, &template_dsp, &top_id);

  /* initialization */

  for (i=0; i<MAX_DIMENSIONS; i++) global_indices[i] = 0;
 
  switch (rank) {

    case 7: global_indices[6] = *ind7;
    case 6: global_indices[5] = *ind6;
    case 5: global_indices[4] = *ind5;
    case 4: global_indices[3] = *ind4;
    case 3: global_indices[2] = *ind3;
    case 2: global_indices[1] = *ind2;
    case 1: global_indices[0] = *ind1;

  } /* end switch */
 
  /* compute ownership */

  if (pcb.p == 1)

      { owner = 1;
        offset = dalib_local_offset (*array_id, global_indices);
      }
       
    else

      { owner  = dalib_multidim_owner (*array_id, global_indices);
        offset = dalib_remote_offset (*array_id, owner, global_indices);
        owner  = dalib_top_elem (top_id, owner);
      }

  if (owner == pcb.i)          /* then it is a local data access */

    { 

#ifdef DEBUG
      printf ("%d: dalib_array_read, index = %d, local, offset = %d\n",
               pcb.i, *ind1, offset);
#endif
  
      dalib_array_local_read (data, array_id, offset);

    }

   else              /* remote data access */

    { 

#ifdef DEBUG
      printf ("%d: dalib_array_read, index = %d, remote on = %d, offset = %d\n",
               pcb.i, *ind1, owner, offset);
#endif

      dalib_rpc_get (data, owner, array_id, offset);
 
    } /* end rpc */

}  /* dalib_array_read */

/**************************************************************************
*                                                                         *
*   void dalib_array_put (char *data, array_info *array_id, int *op,      *
*                          int *ind1, ..., int *ind7)                     *
*                                                                         *
*     - read array[ind1,...,indk] (maybe on remote processor)             *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_array_put) (data, array_id, op,
                                ind1, ind2, ind3, ind4, ind5 ,ind6, ind7)

unsigned char *data;
array_info *array_id;
int        *op;
int *ind1, *ind2, *ind3, *ind4, *ind5, *ind6, *ind7;
 

{ int owner;
  int offset;

  int global_indices[MAX_DIMENSIONS];
  int i, rank;

  int top_id;
  array_info template_dsp;

  unsigned char *pdata;

  /* pack indices for local and remote address computation */

  rank = (*array_id)->rank;

  dalib_array_info (*array_id, &template_dsp, &top_id);

  /* initialization */

  for (i=0; i<MAX_DIMENSIONS; i++) global_indices[i] = 0;
 
  switch (rank) {

    case 7: global_indices[6] = *ind7;
    case 6: global_indices[5] = *ind6;
    case 5: global_indices[4] = *ind5;
    case 4: global_indices[3] = *ind4;
    case 3: global_indices[2] = *ind3;
    case 2: global_indices[1] = *ind2;
    case 1: global_indices[0] = *ind1;

  } /* end switch */
 
  /* compute ownership */

  if (pcb.p == 1)

      { owner = 1;
        offset = dalib_local_offset (*array_id, global_indices);
      }
       
    else

      { owner  = dalib_multidim_owner (*array_id, global_indices);
        offset = dalib_remote_offset (*array_id, owner, global_indices);
        owner  = dalib_top_elem (top_id, owner);
      }

  if (owner == pcb.i)          /* then it is a local data access */

    { 

#ifdef DEBUG
      printf ("%d: dalib_array_put, index = %d, local, offset = %d\n",
               pcb.i, *ind1, offset);
#endif
  
      dalib_array_local_put (*op, data, array_id, offset);

    }

   else              /* remote data access */

    { 

#ifdef DEBUG
      printf ("%d: dalib_array_put, index = %d, remote on = %d, offset = %d\n",
               pcb.i, *ind1, owner, offset);
#endif

      dalib_rpc_put (*op, data, owner, array_id, offset);
 
    } /* end rpc */

}  /* dalib_array_put */
