/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*                Stephan Springstubbe, GMD, SCAI.LAB                      *
*                                                                         *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : May 95                                                   *
*  Last Update : May 95                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : shared.c                                                 *
*                                                                         *
*  Function: Creating descriptors for (distributed) shared data           *
*                                                                         *
*  EXPORT:                                                                *
*                                                                         *
*   a) FORTRAN interface                                                  *
*                                                                         *
* FUNCTION(dalib_array_shared) (array_id) * 
* FUNCTION(dalib_array_distributed_shared) (array_id) * 
*                                                                         *
* FUNCTION(dalib_array_read) (data, array_id, ind1, ind2, ind3, ind4, * 
*                                            ind5 ,ind6, ind7)            *
*                                                                         *
* FUNCTION(dalib_array_write) (data, array_id, ind1, ind2, ind3, ind4, * 
*                                             ind5 ,ind6, ind7)           *
*                                                                         *
*   b) DALIB interface                                                    *
*                                                                         *
*   dalib_array_shared_alloc (array_id, a_ptr, a_zero, a_dim1, a_dim2,    *
*                              a_dim3, a_dim4, a_dim5, a_dim6, a_dim7)    *
*                                                                         *
*   dalib_shared_free (array_id)                                          *
*                                                                         *
**************************************************************************/
 
#undef DEBUG

#include "dalib.h"

     /***********************************************
     *                                              *
     *   record describing how data is shared       *
     *                                              *
     ***********************************************/

#define MEM_GLOBAL_SHARED        2
#define MEM_DISTRIBUTED_SHARED   1

struct SharedRecord
 
 { int   shared_kind;
   int   shm_id;        
   char  *local_remote_addresses[MAXP];
 };
 
/**************************************************************************
*                                                                         *
* FUNCTION(dalib_array_shared) (array_id) * 
* FUNCTION(dalib_array_vshared) (array_id, dummy_array_id) * 
* FUNCTION(dalib_array_distributed_shared) (array_id) * 
*                                                                         *
**************************************************************************/

     /***********************************************
     *                                              *
* FUNCTION(dalib_array_shared) (array_id) * 
     *                                              *
     ***********************************************/

void FUNCTION(dalib_array_shared) (array_id, kind) 
 
array_info *array_id;
int        *kind;
 
{ Shared shared_ptr;

  shared_ptr       = (Shared) dalib_malloc (sizeof (int), "dalib_array_shared");

  shared_ptr->shared_kind = MEM_GLOBAL_SHARED;

  (*array_id)->SharedInfo = shared_ptr;

} /* dalib_array_shared */

     /***********************************************
     *                                              *
* FUNCTION(dalib_array_vshared) (array_id) * 
     *                                              *
     ***********************************************/

void FUNCTION(dalib_array_vshared) (array_id, dummy_array_id, kind) 
 
array_info *array_id, *dummy_array_id;
int *kind;
 
{ Shared shared_ptr;

  if (*array_id != *dummy_array_id)

    { /* there is already a new descriptor */

FUNCTION(dalib_array_shared) (array_id, kind); 
      return;
    }

  shared_ptr = (*array_id)->SharedInfo;

  if (shared_ptr != NO_SHARED)  return;

  /* make new descriptor */

  *array_id = dalib_array_copy_dsp (*array_id);  
       
  /* take the same distribution */

  (*array_id)->DistributeInfo = (*dummy_array_id)->DistributeInfo;
  (*array_id)->AlignInfo      = (*dummy_array_id)->AlignInfo;

FUNCTION(dalib_array_shared) (array_id, kind); 

} /* dalib_array_vshared */

     /***********************************************
     *                                              *
* FUNCTION(dalib_array_distributed_shared) (array_id) * 
     *                                              *
     ***********************************************/

void FUNCTION(dalib_array_distributed_shared) (array_id) 
 
array_info *array_id;
 
{ Shared shared_ptr;

  array_info template_id;
  int        top_id;

  int NP;

  dalib_array_info (*array_id, &template_id, &top_id);

  NP = dalib_top_size (top_id);

  shared_ptr       = (Shared) dalib_malloc (sizeof (int) + NP * sizeof (char *),
                                            "dalib_array_distributed_shared");

  shared_ptr->shared_kind = MEM_DISTRIBUTED_SHARED;

  (*array_id)->SharedInfo = shared_ptr;

} /* dalib_array_shared */
 
/**************************************************************************
*                                                                         *
*                                                                         *
**************************************************************************/

static int dalib_shared_allocate ();

/**************************************************************************
*                                                                         *
*                                                                         *
**************************************************************************/

void dalib_array_shared_alloc (array_id)
 
array_info array_id;
 
{ Shared shared_ptr;
  
  int rank;
  int size;             /* number of bytes for one element */

  int first;
  int total [MAX_DIMENSIONS+1];
  int elems;

  int shmid;
 
  unsigned char *dummy, *a_ptr; 

  shared_ptr = array_id->SharedInfo;
  
  size = array_id->size;
  rank = array_id->rank;

  switch (shared_ptr->shared_kind) {

   case MEM_GLOBAL_SHARED : 

     { /* creating global shared data */

       dalib_array_addressing (array_id, pcb.i, &dummy, &first, total);

       elems = total[rank] * size;
 
       dalib_shm_allocate (elems, &a_ptr, &shmid);

       shared_ptr->shm_id = shmid;

       /* that is pointer to the allocated memory */
 
       array_id->data = a_ptr;

       break;
     }

   case MEM_DISTRIBUTED_SHARED : 

    { /* creating distributed shared data */

      dalib_internal_error ("shared_alloc : distr. shared not yet");
      dalib_stop ();
      break;
    }

   default :

    { dalib_internal_error ("shared_alloc : unknown type");
      dalib_stop ();
    }

   } /* switch */

}  /* dalib_array_shared_alloc */

/*******************************************************************
*                                                                  *
*                                                                  *
*                                                                  *
*******************************************************************/

void dalib_shared_free (array_id)

array_info array_id;

{ Shared shared_ptr;
  unsigned char *shared_data;

  shared_ptr  = array_id->SharedInfo;
  shared_data = array_id->data;

   if (shared_ptr->shared_kind == MEM_GLOBAL_SHARED)

    { /* detach/deallocate shared memory region */

      dalib_shm_deallocate (shared_ptr->shm_id, shared_data);

    }

  free (shared_ptr);

} /* dalib_shared_free */

/*******************************************************************
*                                                                  *
*   local  access of array data                                    *
*                                                                  *
*******************************************************************/

void dalib_array_local_read (data, array_id, ind1, ind2, ind3, ind4, 
                                                   ind5 ,ind6, ind7)

char *data;
array_info array_id;
int *ind1, *ind2, *ind3, *ind4, *ind5, *ind6, *ind7;

{ int first;
  int total[MAX_DIMENSIONS+1];
  int size;
  int offset;

  char *ptr;

  size = array_id->size;

#ifdef DEBUG
  printf ("%d: local read , ind = %d\n", pcb.i, *ind1);
#endif

  /* addressing depends on how data is shared */

  dalib_array_addressing (array_id, pcb.i, &ptr, &first, total);
 
  offset = *ind1;

#ifdef DEBUG
  printf ("%d: local read, offset = %d, first = %d, size = %d\n",
           pcb.i, offset, first, size);
#endif

  ptr += (offset - first) * size;

  dalib_memcopy (data, ptr, size);

  /* a (ind1, ind2, ind3)
 
     calculate : a_dim1, a_dim2, a_dim3
                 with a_dimi is distance between to elements im dim i+1
 
     a (a_zero + L_I1 + a_dim1 * L_I2 + ... + a_dim3 * L_I4)
 
     dalib_array_xxx_addressing (array_id, &first, total);
 
     attention : total and first is dependent of processor id
 
     total[i] is exactly dim_i
 
     offset = ind1 + total[1] * ind2 + total[2] * ind3
 
     array->data += (offset - first) * size
 
  */

} /* dalib_array_local_read */

/*******************************************************************
*                                                                  *
*   remote access of array data                                    *
*                                                                  *
*******************************************************************/

void FUNCTION(dalib_array_read) (data, array_id, ind1, ind2, ind3, ind4, 
                                              ind5 ,ind6, ind7)

char *data;
array_info *array_id;
int *ind1, *ind2, *ind3, *ind4, *ind5, *ind6, *ind7;

{ int pid;
 
  int top_id;     /* topology to which array_id is mapped    */
  int top_dim;    /* dimension of topology / processor array */
  int lb, ub;     /* lb:ub is global size of one dimension   */
  int kind;       /* shows how dimension is distributed      */
  int base;       /* only used as dummy                      */
  int stride;     /* only used as dummy                      */
  int NP;         /* number of processors for the dimension  */
  int NId;        /* my position in dim, 1 <= NId <= NP      */
 
  /* following assumptions are made :

     a) rank of array = 1
     b) dim is block distributed
     c) number of elements is multiple of number of processors

     so offset calculation is very easy 

  */

  /* step 1 : compute owner in pid */

   dalib_array_dim_info (*array_id, 1, &top_id, &top_dim,
                         &base, &stride, &lb, &ub, &kind);
 
   dalib_top_info (top_id, top_dim, &NP, &NId);
 
   if (kind == kBLOCK_DIM)
      pid = dalib_block_owner (NP, lb, ub, *ind1);
    else if (kind == kCYCLIC_DIM)
      pid = dalib_cyclic_owner (NP, lb, ub, *ind1);
    else
      pid = 1;

#ifdef DEBUG
  printf ("%d: array get of index %d, val is on pid = %d\n",
           pcb.i, *ind1, pid);
#endif

  /* step 2 : offset calculation */

  /* a (ind1, ind2, ind3)

     calculate : a_dim1, a_dim2, a_dim3 
                 with a_dimi is distance between to elements im dim i+1 

     a (a_zero + L_I1 + a_dim1 * L_I2 + ... + a_dim3 * L_I4)   

     dalib_array_xxx_addressing (array_id, &first, total);

     attention : total and first is dependent of processor id 

     total[i] is exactly dim_i

     offset = ind1 + total[1] * ind2 + total[2] * ind3

     array->data += (offset - first) * size

  */

  /* possible solution :

     rpc (pid, "READ", *array_id, *ind1, *ind2, *ind3, *ind4, 
                                         *ind5, *ind6, *ind7)

  */

  /* this is solution for global shared memory segments */

  dalib_array_local_read (data, *array_id, ind1, ind2, ind3, ind4, 
                                                 ind5, ind6, ind7);

} /* dalib_array_read */

void FUNCTION(dalib_array_write) (data, array_id, ind1, ind2, ind3, ind4, 
                                               ind5 ,ind6, ind7)

char *data;
array_info *array_id;
int *ind1, *ind2, *ind3, *ind4, *ind5, *ind6, *ind7;

{
  dalib_internal_error ("array write not implemented yet");
  dalib_stop ();

} /* FUNCTION(dalib_array_write) */ 
