/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Feb 96                                                   *
*  Last Update : Feb 96                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : rpc_sp.m4                                                *
*                                                                         *
*  Function: machine dependent remote procedure calls                     *
*                                                                         *
*      CENJU  version       (realization)                                 *
*                                                                         *
*  Export :  internal Interface                                           *
*  ============================                                           *
*                                                                         *
*   void rpc_init ()                                                      *
*                                                                         *
*     - starts the server mode which allows recvncalls                    *
*                                                                         *
*   void rpc_exit ()                                                      *
*                                                                         *
*     - shutdown the server mode                                          *
*                                                                         *
*   void rpc_call (int pid, char *query, int query_size,                  *
*                           char *answer, int answer_size)                *
*                                                                         *
*    - remote call of dalib_answer_question                               *
*                                                                         *
*  Note: first 4 bytes of query must not be 0  (halt request)             *
*                                                                         *
**************************************************************************/

#undef DEBUG

#include "dalib.h"

/**************************************************************************
*                                                                         *
*                                                                         *
**************************************************************************/

void dalib_answer_question ();

int rpc_handler1 (arg1, arg2, arg3, arg4, arg5)

int arg1;
int arg2;
int arg3;
int arg4;
int arg5;

{ int question[5];
  int **answer;
  int answer_size;
  int result;

  question[0] = arg1;
  question[1] = arg2;
  question[2] = arg3;
  question[3] = arg4;
  question[4] = arg5;

  dalib_answer_question (question, &answer, &answer_size);

  /*
  rtn = CJrmwrite (pdata, from_pid, global_data, size);
  if (rtn != size) {
         dalib_internal_error ("CJrmwrite failed; %d\n", rtn);
       } 
  */

  result = **answer;

  free (*answer);

  return (result);

} /* rpc_handler1 */

/**************************************************************************
*                                                                         *
*  rpc_call (int pid, char *query, int query_size,                        *
*                     char *answer, int answer_size)                      *
*                                                                         *
*    - remote call of dalib_answer_question                               *
*                                                                         *
**************************************************************************/

void rpc_call (pid, query, query_size, answer, answer_size)

int pid;
unsigned char *query, *answer;
int query_size, answer_size;

{ /* note: CENJU numbers proc from 0 to n-1 instead of 1 to n */

  int no_params;
  int size;
  int *params;

  size = sizeof (int);

  params    = (int *) query;
  no_params = (query_size + size - 1) / size;

  rtn = CJrpc (pid-1,
               rpc_handler1,
               5,              /* number of parameter */
               query[0],
               query[1],
               query[2],
               query[3],
               query[4]);

  /* rtn has result only for integer sizes or smaller otherwise
     the data is available via global_data                      */

  if (answer_size > size)

     { dalib_internal_error ("rpc_call (CENJU): not handled yet");
       dalib_stop ();
     }

   else if (answer_size > 0)

     {  /* rtn is the result */

        dalib_memcopy (answer, &rtn, answer_size);

     }

   /* note that no answer is also possible */

} /* rpc_call */

/**************************************************************************
*                                                                         *
*                                                                         *
*                                                                         *
**************************************************************************/

void rpc_handler (int *inmsg)

{ /* answering request */

  int rc;
  int my_client;

  size_t nbytes;

  size_t length;
  int    answer_tag;

  int i;

#ifdef DEBUG
  printf ("%d: rpc_handler activated, msg_id = %d\n", pcb.i, *inmsg);
#endif

  my_client = -1;

  for (i=0; i<pcb.p; i++)

    if (i+1 != pcb.i)

       if (*inmsg == msgid[i]) my_client = i;

  rc = mpc_wait (inmsg, &nbytes);

  if (rc)
    { dalib_internal_error ("rpc_handler: message not available");
      dalib_stop ();
    }

#ifdef DEBUG
  printf ("%d: got a request of client %d, size = %d\n", 
           pcb.i, my_client+1, nbytes);
#endif

  if (question[my_client][0] == 0)

     {  /* I can stop the server for this processor */

        servers = servers - 1;

#ifdef DEBUG
  printf ("%d: stop service for processor %d, %d servers running\n",
           pcb.i, my_client+1, servers);
#endif

     }

   else

     { unsigned char *answer;
       int size;

       /* now we can answer the request */

       dalib_answer_question (question+my_client, &answer, &size);

       /* send answer back */

       rpc_send_answer (my_client+1, answer, size);

       free (answer);   /* give memory of answer free */

       /* restart server mode */

       rpc_start_server (my_client + 1);
 
     }

} /* rpc_handler */

/**************************************************************************
*                                                                         *
*   void rpc_init ()                                                      *
*                                                                         *
*     - starts the server mode which allows recvncalls                    *
*                                                                         *
**************************************************************************/

void rpc_init ()

{ /* nothing necessary on the CENJU-3 */

} /* rpc_init */

/**************************************************************************
*                                                                         *
*   void rpc_exit ()                                                      *
*                                                                         *
*     - shutdown the server mode                                          *
*                                                                         *
**************************************************************************/

void rpc_exit ()

{ /* trivial case on the CENJU */

} /* rpc_exit */

#undef DEBUG

#include <stdlib.h>   /* NULL */
#include "dalib.h"

struct SharedRecord
 
{ int  shared_kind;

};

/**************************************************************************
*                                                                         *
*   global_data is needed to exchange data via CJmread, CJmwrite          *
*                                                                         *
*   IMPORTANT:  global_data must have the same address on all processors  *
*                                                                         *
**************************************************************************/

#define MAX_SIZE 64 

unsigned char global_data [MAX_SIZE];

/**************************************************************************
*                                                                         *
*   int dalib_is_globally_shared (Shared info)                            *
*                                                                         *
*     - returns 0 if local memory on each processor (local addresses)     *
*     - returns 1 if global memory for all processors (global addresses)  *
*                                                                         *
**************************************************************************/

int dalib_is_globally_shared (info)

Shared info;

{ 

  return (0);

} /* dalib_is_globally_shared */

/**************************************************************************
*                                                                         *
*                                                                         *
**************************************************************************/
 
void FUNCTION(dalib_array_shared) (array_id, kind)
 
array_info *array_id;
int        *kind;
 
{ Shared shared_ptr;

  /* allocate the pointer of Shared Record structure */

  shared_ptr = (Shared) dalib_malloc (sizeof (struct SharedRecord),
                                      "dalib_array_shared");
  /* set the shared_kind */

  shared_ptr->shared_kind = *kind;

  /* assign the array_id->SharedInfo */

  (*array_id)->SharedInfo = shared_ptr;
 
  FUNCTION(dalib_barrier) ();

  /* make remote access possible */

  dalib_array_remote_init (*array_id);

} /* dalib_array_shared */

void FUNCTION(dalib_array_vshared) (array_id, dummy_id, kind)
 
array_info *array_id;
 
{
}

/**************************************************************************
*                                                                         *
*                                                                         *
**************************************************************************/

void dalib_array_shared_alloc (array_id)
 
array_info array_id;

{ /* memory is created as for a usual array */

  dalib_array_malloc (array_id);

  /* nothing else is to do */

}  /* dalib_array_shared_alloc */
 
/**************************************************************************
*                                                                         *
*                                                                         *
**************************************************************************/

int dalib_rpc_read (array_id, from_pid, offset)
 
array_info *array_id;
int from_pid;
int offset;
 
{ int global_indices[MAX_DIMENSIONS];

  int rank, size;   /* info from array_id */
  int rtn;

  unsigned char *pdata;    /* calculation of array data address */
  int  *idata;             /* help pointer for type conversion  */

  rank         = (*array_id)->rank;
  size         = (*array_id)->size;
 
#ifdef DEBUG
  printf ("%d: dalib_rpc_read from %d, offset = %d\n", 
            pcb.i, from_pid+1, offset);
#endif
 
  pdata = (*array_id)->data;

  pdata += offset * (*array_id)->size;
 
  /* give pdata to the calling processor */

  if (size <= sizeof(int)) 

     { idata = (int *) pdata;
       return (*idata);
     }

   else

     { rtn = CJrmwrite (pdata, from_pid, global_data, size);
      
       if (rtn != size) {
         dalib_internal_error ("CJrmwrite failed; %d\n", rtn);
       } 

       return (0);
     }
  
} /* dalib_rpc_read */

/**************************************************************************
*                                                                         *
*                                                                         *
**************************************************************************/

void dalib_array_rpc_read (data, array_id, offset, to_pid)

unsigned char *data;
array_info *array_id;
int offset;
int to_pid;     /* owner */

{ int size, rank;
  int rtn;

  size = (*array_id)->size;
  rank = (*array_id)->rank;

  { /* note: CENJU numbers proc from 0 to n-1 instead of 1 to n */

    rtn = CJrpc (to_pid-1,
                 dalib_rpc_read,
                 3,                    /* number of parameter */
                 array_id,  /* evtl. &array_id; doppelt indirekt */
                 pcb.i-1,
                 offset);

     /* rtn has result only for integer sizes or smaller otherwise
        the data is available via global_data                      */

     if (size <= sizeof (int))

       { dalib_memcopy (data, &rtn, size);
         
       }

       else

       { /* copy the global data written by remote processor to_pid */

         dalib_memcopy (data, global_data, size);

       }
 
     }

} /* dalib_array_rpc_read */

/**************************************************************************
*                                                                         *
*                                                                         *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_array_read) (data, array_id, ind1, ind2, ind3, ind4,
                                        ind5 ,ind6, ind7)
unsigned char *data;
array_info *array_id;
int *ind1, *ind2, *ind3, *ind4, *ind5, *ind6, *ind7;
 

{ int owner;
  int offset;

  int global_indices[MAX_DIMENSIONS];
  int i, rank;

  int top_id;
  array_info template_dsp;

  unsigned char *pdata;

  /* pack indices for local and remote address computation */

  rank = (*array_id)->rank;

  dalib_array_info (*array_id, &template_dsp, &top_id);

  /* initialization */

  for (i=0; i<MAX_DIMENSIONS; i++) global_indices[i] = 0;
 
  switch (rank) {
    case 7: global_indices[6] = *ind7;
    case 6: global_indices[5] = *ind6;
    case 5: global_indices[4] = *ind5;
    case 4: global_indices[3] = *ind4;
    case 3: global_indices[2] = *ind3;
    case 2: global_indices[1] = *ind2;
    case 1: global_indices[0] = *ind1;
  } /* end switch */
 
  /* compute ownership */

  if (pcb.p == 1)

      { owner = 1;
        offset = dalib_local_offset (*array_id, global_indices);
      }
       
    else

      { owner  = dalib_multidim_owner (*array_id, global_indices);
        offset = dalib_remote_offset (*array_id, owner, global_indices);
        owner  = dalib_top_elem (top_id, owner);
      }

  if (owner == pcb.i)          /* then it is a local data access */

    { 

#ifdef DEBUG
      printf ("%d: dalib_array_read, index = %d, local, offset = %d\n",
               pcb.i, *ind1, offset);
#endif
  
      pdata = (*array_id)->data;

      pdata += offset * (*array_id)->size;

      dalib_memcopy (data, pdata, (*array_id)->size);

    }

   else              /* remote data access */

    { 

#ifdef DEBUG
      printf ("%d: dalib_array_read, index = %d, remote on = %d, offset = %d\n",
               pcb.i, *ind1, owner, offset);
#endif

      dalib_array_rpc_read (data, array_id, offset, owner);
 
    } /* end rpc */

}  /* dalib_array_read */

/**************************************************************************
*                                                                         *
*  dalib_shared_free (descriptor, data_flag)                              *
*                                                                         *
*   - delete data only if data_flag is set                                *
*                                                                         *
**************************************************************************/

void dalib_shared_free (descriptor, data_flag)

array_info descriptor;
int        data_flag;

{ Shared shared_ptr;

  FUNCTION(dalib_barrier) ();

  shared_ptr  = descriptor->SharedInfo;
 
  /* detach/deallocate distributed memory region */

  free (descriptor->data);
  free (shared_ptr);

  if (descriptor->data != NO_DATA)
 
     { /* deallocate local data of the array if data_flag */
 
       if (data_flag) free (descriptor->data);
     }

} /* dalib_shared_free */
 
