/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Nov 93                                                   *
*  Last Update : Nov 93                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : unstructured.c                                           *
*                                                                         *
*  Function    : unstructured schedules                                   *
*                                                                         *
*  Export : Internal Interface                                            *
*                                                                         *
*                                                                         *
*  void dalib_new_indirect (schedule_ptr *new_schedule,                   *
*                           int source_topid, int target_topid,           *
*                           int N, owner_pid[], target_indexes[])         *
*                                                                         *
*    source_topid  : identifier for topology of source                    *
*    target_topid  : identifier for topology of target                    *
*                                                                         *
*    N           : number of accesses from source to target               *
*                                                                         *
*    owner_pid [i]      : target processor for i-th element               *
*                         (<= 0 implies no access)                        *
*                                                                         *
*    target_indexes [i] : target address for i-th element                 *
*                                                                         *
*   output:                                                               *
*                                                                         *
*    - new_schedule is pointer to the unstructured comm. schedule         *
*                                                                         *
*  void dalib_free_indirect (schedule_ptr sptr)                           *
*                                                                         *
*  void dalib_indirect_source_info (schedule_ptr sptr, int *no_values,    *
*                                   int **indexes)                        *
*                                                                         *
*  void dalib_indirect_target_info (schedule_ptr sptr, int *no_values,    *
*                                   int **indexes)                        *
*                                                                         *
*  void dalib_indirect_send_target (sptr, char *addr, int size)           *
*  void dalib_indirect_recv_target (sptr, int op, char *addr, int size)   *
*  void dalib_indirect_send_source (sptr, char *addr, int size)           *
*  void dalib_indirect_recv_source (sptr, char *addr, int size)           *
*                                                                         *
*  schedule_ptr sptr;   // pointer to the schedule                        *
*                                                                         *
*  char *data; -- data area for sending/receiving the values              *
*  int size;   -- specifies the size in bytes of one element              *
*                                                                         *
*  Definition of a new schedule for remote accesses                       *
*  ================================================                       *
*                                                                         *
*  void dalib_new_remote (remote_ptr *new_remote,                         *
*                         int topid, int N,                               *
*                         int owners[], int offsets[])                    *
*                                                                         *
*  dalib_remote_info (remote_ptr rptr, int *topid,                        *
*                     int *no_indexes, int *no_remote[]                   *
*                     int *rem_offsets[], int *inverse_map[])             *
*                                                                         *
*  void dalib_free_remote (remote_ptr rptr)                               *
*                                                                         *
**************************************************************************/

#undef DEBUG

#include "dalib.h"
#include <stdio.h>

/*********************************************************
*                                                        *
*  global data that is used to set up a schedule         *
*                                                        *
*********************************************************/

  /* source_no[pid] is number of indexes into target topology */

static int *source_no;          /* size is top_size (target_topid)      */
static int *source_offset;      /* size is top_size (target_topid)      */

  /* target_no[pid] is number of pointers from source topology */

static int *target_no;          /* size is top_size (source_topid)      */
static int *target_offset;      /* size is top_size (source_topid)      */

/*********************************************************
*                                                        *
*  Data structure for a schedule used for indirect addr  *
*                                                        *
*********************************************************/

struct indirect_schedule {

  int source_topid;
  int target_topid;

  int no_source_indexes;   /* number of elements of source indexes */

  int *source_indexes;     /* pointers from the source array     
                              size of array is no_source_indexes   */

  int no_target_indexes;   /* indexes accessed by other processors */

  int *target_indexes;     /* pointers into the target topology    */

  dd_type *source_ddt;     /* descriptor for sending from source   */
  dd_type *target_ddt;     /* descriptor for receiving in target   */

};

/*******************************************************************
*                                                                  *
*  Data structure for a schedule used for remote access            *
*                                                                  *
*               0   1   2   3   4   5   6   7   8   9  10          *
*                                                                  *
*  owners  :    0   1   1   0   2   3   0   1   2   1   3          *
*  offsets :   09  13  15  08  21  35  07  11  24  17  33          *
*                                                                  *
*                                                                  *
*  no_indexes  : 11                                                *
*  no_remote   :  3 (0)   4 (1)  2 (2)  2(3)                       *
*                                                                  *
*                   0   0   0   1   1   1   1   2   2   3   3      *
*                                                                  *
*  rem_offsets :   09  08  07  13  15  11  17  21  24  35  33      *
*  inverse_map :    0   3   6   1   2   7   9   4   8   5  10      *
*                                                                  *
*******************************************************************/

struct remote_schedule {

  int topid;               /* topology that will be remotely accessed */

  int no_indexes;          /* total number of remote accesses */

  int *no_remote;          /* no_remote[i] is number of access to pid i */

  int *rem_offsets;        /* remote offsets [i], 0 <= i < no_indexes   */

  int *inverse_map;        /* inverse pointers                          */

};

/*******************************************************************
*                                                                  *
*  Printing Information of a indirect schedule                     *
*                                                                  *
*******************************************************************/

#ifdef DEBUG

void print_indirect_schedule (sptr)
schedule_ptr sptr;

{ 
  printf ("%d: output of indirect schedule\n", pcb.i);
  printf ("%d: source_topid = %d, target_topid = %d\n",
           pcb.i, sptr->source_topid, sptr->target_topid);
  printf ("%d: no of source indexes = %d, target indexes = %d\n",
           pcb.i, sptr->no_source_indexes, sptr->no_target_indexes);

} /* print_indirect_schedule */

void print_remote_schedule (rptr)
remote_ptr rptr;

{ int i, NP;

  printf ("%d: output of remote schedule\n", pcb.i);
  printf ("%d: topid = %d, indexes = %d\n",
           pcb.i, rptr->topid, rptr->no_indexes);
  printf ("%d: remote ", pcb.i);
  NP = dalib_top_size (rptr->topid);
  for (i=0; i<NP; i++) printf (" %d (on %d)", rptr->no_remote[i], i);
  printf ("\n");
  NP = rptr->no_indexes;
  printf ("%d: offsets ", pcb.i);
  for (i=0; i<NP; i++) printf (" %d", rptr->rem_offsets[i]);
  printf ("\n");
  printf ("%d: inverse ", pcb.i);
  for (i=0; i<NP; i++) printf (" %d", rptr->inverse_map[i]);
  printf ("\n");
  
} /* print_remote_schedule */

#endif

/*******************************************************************
*                                                                  *
*  Help FUNCTION used for the definition of a schedule             *
*                                                                  *
*******************************************************************/

        /*************************************************
        *                                                *
        *  set_local_offset (offset, number, N)          *
        *                                                *
        *  int offset[N], number[N], N;                  *
        *                                                *
        *************************************************/

static void set_local_offset (offset, number, N)
int number[], offset[], N;

{ int k;
  offset[0] = 0;
  for (k=1; k<N; k++)
    { offset [k] = offset[k-1] + number [k-1];
    }
}

        /*************************************************
        *                                                *
        *  dalib_count_procids (procids, N, counter, NP) *
        *                                                *
        *     note: procids[k] == -1, not used           *
        *                                                *
        *************************************************/

static void dalib_count_procids (procids, N, counter, NP, Nreal)

int procids[], N, counter[], NP, *Nreal;

{ int i, pid;

  *Nreal = 0;

  for (pid = 0; pid < NP; pid++)
    counter[pid] = 0;

#ifdef DEBUG
   printf ("%d: dalib_count_procids procids[%d] -> counter[%d]\n",
           pcb.i, N, NP);
   /*
   for (i=0; i<N; i++)
      printf ("procid [%d] = %d\n", i, procids[i]); 
   */
#endif

  for (i=0; i<N; i++)
    { pid = procids[i];
      if (pid >= 0)
        { (counter[pid])++;
          (*Nreal)++;
        }
    }
#ifdef DEBUG
   printf ("%d: dalib_count_procids procids[%d] -> counter[%d], real = %d\n", 
           pcb.i, N, NP, *Nreal);
#endif
} /* dalib_count_procids */

/*******************************************************************
*                                                                  *
*               0   1   2   3   4   5   6   7   8   9  10          *
*                                                                  *
*  owners  :    0   1   1   0   2   3   0   1   2   1   3          *
*  offsets :   09  13  15  08  21  35  07  11  24  17  33          *
*                                                                  *
*               0   0   0   1   1   1   1   2   2   3   3          *
*                                                                  *
*  sorted  :   09  08  07  13  15  11  17  21  24  35  33          *
*  inverse :    0   3   6   1   2   7   9   4   8   5  10          *
*                                                                  *
*******************************************************************/

static void dalib_offset_sorting (N, owners, offsets,
                                  NP, counter, pointer,
                                  sorted_offsets, inverse_map)

int N, NP;
int owners[], offsets[];
int counter[], pointer[];
int sorted_offsets[], inverse_map[];

{ int j, pid;

  set_local_offset (pointer, counter, NP);
 
  for (j=0; j<N; j++)

    { /* collect offsets in sorted offsets and make inverse */

      pid = owners [j];

      if (pid >= 0)                   /* mask is hidden in the target_proc */

        { sorted_offsets [pointer[pid]] = offsets[j];
          inverse_map    [pointer[pid]] = j;

          pointer[pid] += 1;

        }
    }

} /* offset_sorting */
 
        /*************************************************
        *                                                *
        *  dalib_send_needed_ptrs (scounter, target_top) *
        *                                                *
        *************************************************/

static void dalib_send_needed_ptrs (scounter, target_topid)
int scounter[], target_topid;


{ int j, pid, tNP;

  tNP = dalib_top_size (target_topid);

  /* send every target processor the number of pointers to it */

  for (j=0; j < tNP; j++)

    { /* for all processors in target topology */

      pid = dalib_top_elem (target_topid, j);
      dalib_send (pid, scounter + j, sizeof(int), 1);

#ifdef DEBUG
      printf ("%d: has %d pointers to target processor %d\n",
              pcb.i, scounter[j], pid);
#endif

    } /* for all processors in target topology */

} /* dalib_send_needed_ptrs */

        /*************************************************
        *                                                *
        *  dalib_recv_needed_ptrs (tcounter, source_top) *
        *                                                *
        *************************************************/

static int dalib_recv_needed_ptrs (tcounter, source_topid)
int tcounter[], source_topid;


{ int j, pid;
  int sum;
  int sNP;

  sNP = dalib_top_size (source_topid);

  sum = 0;

  /* recv from every source processor the number of pointers to me */

  for (j=0; j < sNP; j++)
    { pid = dalib_top_elem (source_topid, j);
      dalib_receive (pid, tcounter + j, sizeof(int));
      sum += tcounter[j];
#ifdef DEBUG
      printf ("%d: has %d pointers from source processor %d\n",
              pcb.i, tcounter[j], pid);
#endif
    }

  return (sum);

} /* dalib_recv_needed_ptrs */

        /*************************************************
        *                                                *
        *  exchange_size (scounter, tNP, tcounter, tNP)  *
        *                                                *
        *************************************************/

static void send_remote_offsets (s_number, s_offset, target_topid,
                                 source_ddt,
                                 source_indexes, Nreal,
                                 remote_offsets, owner_pid, N)

int s_number[], s_offset[], target_topid;
int source_indexes[], Nreal;
int remote_offsets[], owner_pid[], N;
dd_type source_ddt[];

{ int *target_pointers;
  int j, pid;
  int target_pid, tNP;

  tNP = dalib_top_size (target_topid);

  target_pointers = (int *) dalib_int_malloc (Nreal, "send_remote_offsets");

  dalib_offset_sorting (N, owner_pid, remote_offsets,
                        tNP, s_number, s_offset,
                        target_pointers, source_indexes);

  /* reset the s_offset array */

  set_local_offset (s_offset, s_number, tNP);

  /* send the index values that point to other processors  */

  for (pid=0; pid<tNP; pid++)
    if (s_number[pid] > 0)
      {  
#ifdef DEBUG
         printf ("%d: sends %d about %d target pointers\n", pcb.i, pid+1,
                 s_number[pid]);
#endif
         target_pid = dalib_top_elem (target_topid, pid);
         dalib_send (target_pid, target_pointers + s_offset[pid], 
                     s_number[pid] * sizeof(int), 1);
      }

  for (pid = 0; pid < tNP; pid++)

     dalib_ddt_def_indexed (source_ddt+pid, s_number[pid], 
                            source_indexes + s_offset[pid]);

  dalib_free (target_pointers, Nreal * sizeof(int));

} /* send_remote_offsets */

static void recv_target_pointers (target_pointers, source_topid, 
                                  target_ddt, number, offset)

int target_pointers[], source_topid, number[], offset[];
dd_type target_ddt[];

{ int pid;
  int sNP, source_pid;

  sNP = dalib_top_size (source_topid);

  /* receive the target pointers of all source processors */

  set_local_offset (offset, number, sNP);
 
  for (pid=0; pid<sNP; pid++)
    if (number[pid] > 0)
       { source_pid = dalib_top_elem (source_topid, pid);
#ifdef DEBUG
         printf ("%d: recv %d global addrs from %d (rel = %d)\n", 
                  pcb.i, number[pid], source_pid, pid);
#endif
         dalib_receive (source_pid, target_pointers + offset[pid], 
                        number[pid] * sizeof(int));
       }

  for (pid = 0; pid < sNP; pid++)
    { dalib_ddt_def_indexed (target_ddt+pid, number[pid], 
                             target_pointers + offset[pid]);
    }
}

/*******************************************************************
*                                                                  *
*  Definition of a new schedule                                    *
*                                                                  *
*  void dalib_new_indirect (schedule_ptr *new_schedule,            *
*                           int source_topid, int target_topid,    *
*                           int N, owner_pid[], target_indexes[])  *
*                                                                  *
*    source_topid  : identifier for topology of source             *
*    target_topid  : identifier for topology of target             *
*                                                                  *
*    N           : number of accesses from source to target        *
*                                                                  *
*    owner_pid [i]    : owner processor for i-th element           *
*                         (<= 0 implies no access)                 *
*                                                                  *
*    target_indexes [i] : target address for i-th element          *
*                                                                  *
*  output:                                                         *
*                                                                  *
*    - new_schedule is pointer to the unstructured comm. schedule  *
*                                                                  *
*******************************************************************/

void dalib_new_indirect (new_schedule,
                         source_topid, target_topid,
                         N, owner_pid, target_indexes)

schedule_ptr *new_schedule;
int source_topid, target_topid, N;
int owner_pid[], target_indexes[];

{ schedule_ptr s;
  int Nreal, sNP, tNP;

#ifdef DEBUG
  printf ("%d: dalib_new_indirect, top %d (N=%d) -> top %d\n", 
           pcb.i, source_topid, N, target_topid);
#endif

     /******************************************************
     *                                                     *
     *  create a new record for the schedule               *
     *                                                     *
     ******************************************************/

  s = (schedule_ptr) dalib_malloc (sizeof (struct indirect_schedule),
                                          "dalib_new_indirect");

  s->source_topid = source_topid;
  s->target_topid = target_topid;

  sNP = dalib_top_size (source_topid);

#ifdef DEBUG
  printf ("%d: dalib_new_indirect, source_topid %d has %d processors\n",
           pcb.i, source_topid, sNP);
#endif

  if (dalib_in_topology (target_topid))
    { target_no  = (int *) dalib_int_malloc (sNP,"dalib_new_indirect");
      target_offset = (int *) dalib_int_malloc (sNP,"dalib_new_indirect");
    }

  tNP = dalib_top_size (target_topid);

#ifdef DEBUG
  printf ("%d: dalib_new_indirect, target_topid %d has %d processors\n",
           pcb.i, target_topid, tNP);
#endif

     /******************************************************
     *                                                     *
     *  get number of values needed from any other proc    *
     *                                                     *
     ******************************************************/

  if (dalib_in_topology (source_topid))

   { source_no  = (int *) dalib_int_malloc (tNP,"dalib_new_indirect");
     source_offset = (int *) dalib_int_malloc (tNP,"dalib_new_indirect");

     /* step 1 : count for every target processor 
                 how many indexes point to it    */

     dalib_count_procids (owner_pid, N, source_no, tNP, &Nreal);
     s->no_source_indexes = Nreal;

     dalib_send_needed_ptrs (source_no, target_topid);
   }

  else

   s->no_source_indexes = 0;

     /******************************************************
     *                                                     *
     *  recv number of values needed by any other proc     *
     *                                                     *
     ******************************************************/

  if (dalib_in_topology (target_topid))
     
     s->no_target_indexes = dalib_recv_needed_ptrs (target_no, source_topid);

   else
 
     s->no_target_indexes = 0;

     /******************************************************
     *                                                     *
     *  send pointers into the target topology             *
     *                                                     *
     ******************************************************/

  if (dalib_in_topology (source_topid))

    { s->source_indexes = (int *) dalib_int_malloc (s->no_source_indexes, 
                                    "dalib_new_indirect (source_indexes)");

      s->source_ddt = (dd_type *) dalib_malloc (sizeof (dd_type) * tNP,
                                    "dalib_new_indirect (source_ddt)");

      send_remote_offsets (source_no, source_offset, target_topid,
                           s->source_ddt,
                           s->source_indexes, Nreal,
                           target_indexes, owner_pid, N);
    }

     /******************************************************
     *                                                     *
     *  recv pointers from the source topology             *
     *                                                     *
     ******************************************************/

  if (dalib_in_topology (target_topid))

   { s->target_indexes = (int *) dalib_int_malloc (s->no_target_indexes, 
                                  "dalib_new_indirect (target_indexes)");

     s->target_ddt = (dd_type *) dalib_malloc (sizeof (dd_type) * sNP,
                                  "dalib_new_indirect (target_ddt)");

     /* step 5 : receive index values accessed by other processors */

     recv_target_pointers (s->target_indexes, source_topid,
                           s->target_ddt,
                           target_no, target_offset);
   }

#ifdef DEBUG
  print_indirect_schedule (s);
#endif

  if (dalib_in_topology (source_topid))

   { dalib_free (source_no, sizeof(int) * tNP);
     dalib_free (source_offset, sizeof(int) * tNP);
   }

  if (dalib_in_topology (target_topid))
   { dalib_free (target_no, sizeof(int) * sNP);
     dalib_free (target_offset, sizeof(int) * sNP);
   }
 
  *new_schedule = s;

} /* dalib_new_indirect */

/*******************************************************************
*                                                                  *
*  RELEASING of a schedule                                         *
*                                                                  *
*******************************************************************/

void dalib_free_indirect (sptr)
schedule_ptr sptr;

{ int p, NP;

  if (sptr == NO_SCHEDULE)
     dalib_internal_error ("free_indirect: no schedule");

  if (dalib_in_topology (sptr->source_topid))
    { dalib_free (sptr->source_indexes, sizeof(int) * sptr->no_source_indexes);
      NP = dalib_top_size (sptr->target_topid);
      for (p=0; p<NP; p++) dalib_ddt_free (sptr->source_ddt[p]);
      dalib_free (sptr->source_ddt, sizeof(dd_type) * NP);
    }

  if (dalib_in_topology (sptr->target_topid))
    { dalib_free (sptr->target_indexes, sizeof(int) * sptr->no_target_indexes);
      NP = dalib_top_size (sptr->source_topid);
      for (p=0; p<NP; p++) dalib_ddt_free (sptr->target_ddt[p]);
      dalib_free (sptr->target_ddt, sizeof(dd_type) * NP);
    }

  dalib_free (sptr, sizeof(struct indirect_schedule));

} /* dalib_free_indirect */

/*******************************************************************
*                                                                  *
*  Asking for Informations from the INDIRECT SCHEDULE              *
*                                                                  *
*******************************************************************/

void dalib_indirect_source_info (sptr, no_values, indexes)
schedule_ptr sptr;
int *no_values, **indexes;

{ 
  *no_values = sptr->no_source_indexes;
  *indexes   = sptr->source_indexes;
}

void dalib_indirect_target_info (sptr, no_values, indexes)
schedule_ptr sptr;
int *no_values, **indexes;

{ 
  *no_values = sptr->no_target_indexes;
  *indexes   = sptr->target_indexes;
}

/*******************************************************************
*                                                                  *
*  SENDING/RECEIVING VALUES using the schedule                     *
*                                                                  *
*******************************************************************/

        /*************************************************
        *                                                *
        *  dalib_indirect_send_target                    *
        *                                                *
        *  values[i] = TARGET [target_indexes[i]]        *
        *                                                *
        *************************************************/

void dalib_indirect_send_target (sptr, data, size)

schedule_ptr sptr;
int size;
char *data;

{ int k, NP, pid, topid;
  dd_type ddt;

#ifdef DEBUG
  printf ("%d: dalib_indirect_send_target %d -> source_top = %d, size = %d\n", 
          pcb.i, sptr->target_topid, sptr->source_topid, size);
#endif 

  if (!dalib_in_topology (sptr->target_topid))
     return;

  topid = sptr->source_topid;

  NP = dalib_top_size (topid);

  for (k=0; k<NP; k++)

    { ddt = sptr->target_ddt[k];
      dalib_ddt_set_data (ddt, data, size);
      pid = dalib_top_elem (topid, k);
#ifdef DEBUG
      printf ("%d: send, pid = %d, data = %d, size = %d\n",
               pcb.i, pid, data, size);
      dalib_ddt_print (ddt);
#endif
      dalib_send_ddt (pid, ddt);
    }

} /* dalib_indirect_send_target */

        /*************************************************
        *                                                *
        *  dalib_indirect_recv_target                    *
        *                                                *
        *  TARGET [target_indexes[i]] = values[i]        *
        *                                                *
        *************************************************/

void dalib_indirect_recv_target (sptr, op, data, size)

schedule_ptr sptr;
int size, op;
unsigned char *data;

{ int k, NP, pid, topid;
  dd_type ddt;
  
#ifdef DEBUG
  printf ("%d: dalib_indirect_recv_target %d <- source_top = %d, size = %d\n", 
          pcb.i, sptr->target_topid, sptr->source_topid, size);
#endif 

  if (!dalib_in_topology (sptr->target_topid)) return;

  topid = sptr->source_topid;

  NP = dalib_top_size (topid);

  for (k=0; k<NP; k++)  /* receive from every proc in source topology */

    { ddt = sptr->target_ddt[k];
      dalib_ddt_set_data (ddt, data, size);
      pid = dalib_top_elem (topid, k);
      dalib_recv_ddt_op (pid, ddt, op); 
    }

} /* dalib_indirect_recv_target */

        /*************************************************
        *                                                *
        *  dalib_indirect_send_source                    *
        *                                                *
        *  values[i] = SOURCE [source_indexes[i]]        *
        *                                                *
        *************************************************/

void dalib_indirect_send_source (sptr, data, size)

schedule_ptr sptr;
int size;
unsigned char *data;

{ int k, NP, pid, topid;
  dd_type ddt;

#ifdef DEBUG
  printf ("%d: dalib_indirect_send_source %d -> target_top = %d, size = %d\n", 
          pcb.i, sptr->source_topid, sptr->target_topid, size);
#endif 

  if (!dalib_in_topology (sptr->source_topid)) return;

  topid = sptr->target_topid;
  NP = dalib_top_size (topid);

  for (k=0; k<NP; k++)  /* send to every proc in target topolgy */

    { ddt = sptr->source_ddt[k];
      dalib_ddt_set_data (ddt, data, size);
      pid = dalib_top_elem (topid, k);
#ifdef DEBUG
      printf ("%d: send source to %d in top %d, pid = %d\n",
               pcb.i, k, topid, pid);
#endif
      dalib_send_ddt (pid, ddt);

    } /* for */

} /* dalib_indirect_send_source */

        /*************************************************
        *                                                *
        *  dalib_indirect_recv_source                    *
        *                                                *
        *  data [source_indexes[i]] = values[i]          *
        *                                                *
        *************************************************/

void dalib_indirect_recv_source (sptr, data, size)

schedule_ptr sptr;
int size;
char *data;

{ int k, NP, pid, topid;
  dd_type ddt;

#ifdef DEBUG
  printf ("%d: dalib_indirect_recv_source %d <- target_top = %d, size = %d\n", 
          pcb.i, sptr->source_topid, sptr->target_topid, size);
#endif 

  if (!dalib_in_topology (sptr->source_topid)) return;

  topid = sptr->target_topid;
  NP = dalib_top_size (topid);

  for (k=0; k<NP; k++)  /* recv from every proc in target topology */

    { ddt = sptr->source_ddt[k];
      dalib_ddt_set_data (ddt, data, size);
      pid = dalib_top_elem (topid, k);
#ifdef DEBUG
      printf ("%d: recv, pid = %d, data = %d, size = %d\n",
               pcb.i, pid, data, size);
      dalib_ddt_print (ddt);
#endif
      dalib_recv_ddt_op (pid, ddt, 0);
    } 

} /* dalib_indirect_recv_source */

/*******************************************************************
*                                                                  *
*  Definition of a new schedule for remote accesses                *
*                                                                  *
*  void dalib_new_remote (remote_ptr *new_remote,                  *
*                         int topid, int N,                        *
*                         int owners[], int offsets[])             *
*                                                                  *
*    N           : number of accesses from source to target        *
*                                                                  *
*    owners [i]  : owner processor for i-th element                *
*                         (<= 0 implies no access)                 *
*                                                                  *
*    offsets [i] : target address for i-th element                 *
*                                                                  *
*  output:                                                         *
*                                                                  *
*    - new schedule for unstructured remote read/write             *
*                                                                  *
*******************************************************************/
 
void dalib_new_remote (new_remote, topid,
                       N, owners, offsets)
 
remote_ptr *new_remote;
int topid, N;
int owners[], offsets[];   /* owners and offsets have N entries */

{ remote_ptr r;          /* help pointer for final result         */
  int NP;                /* number of processor in topology topid */
  int *sorted_offsets;   /* to filled in for remote schedule      */
  int *inverse_map;      /* to filled in for remote schedule      */
  int pid;

  int *counter;
  int *pointer;

  int Nreal;             /* really used values of original indexes */

     /******************************************************
     *                                                     *
     *  create a new record for the schedule               *
     *                                                     *
     ******************************************************/
 
  r = (remote_ptr) dalib_malloc (sizeof (struct remote_schedule),
                                 "dalib_new_remote");

  r->topid = topid;

  NP = dalib_top_size (topid);

#ifdef DEBUG
  printf ("%d: dalib_new_remote, topid %d (%d procs), N=%d)\n",
           pcb.i, topid, NP, N);
#endif

     /******************************************************
     *                                                     *
     *  get number of values needed from any other proc    *
     *                                                     *
     ******************************************************/
 
  counter  = (int *) dalib_int_malloc (NP, "dalib_new_remote");
  pointer  = (int *) dalib_int_malloc (NP, "dalib_new_remote");

  /* step 1 : count for every target processor
              how many indexes I have to it    */

  dalib_count_procids (owners, N, counter, NP, &Nreal);

  r->no_indexes = Nreal;

  /* step 2 : sort owners and offsets */

  sorted_offsets = (int *) dalib_int_malloc (Nreal, "dalib_new_remote");
  inverse_map    = (int *) dalib_int_malloc (Nreal, "dalib_new_remote");
  
  dalib_offset_sorting (N, owners, offsets, 
                        NP, counter, pointer,
                        sorted_offsets, inverse_map);

  r->no_remote = counter;
  r->rem_offsets = sorted_offsets;
  r->inverse_map = inverse_map;

#ifdef DEBUG
  print_remote_schedule (r);
#endif
 
  free (pointer);

  *new_remote = r;

} /* dalib_new_remote */

/*******************************************************************
*                                                                  *
*  dalib_remote_info (remote_ptr rptr, int *topid,                 *
*                     int *no_indexes, int *no_remote[]            *
*                     int *rem_offsets[], int *inverse_map[])      *
*                                                                  *
*******************************************************************/

void dalib_remote_info (rptr, topid, no_indexes, no_remote,
                        rem_offsets, inverse_map)

remote_ptr rptr;

int *topid;
int *no_indexes;
int **no_remote;
int **rem_offsets;
int **inverse_map;

{ *topid       = rptr->topid;
  *no_indexes  = rptr->no_indexes;
  *no_remote   = rptr->no_remote;
  *rem_offsets = rptr->rem_offsets;
  *inverse_map = rptr->inverse_map;

} /* dalib_remote_info */

/*******************************************************************
*                                                                  *
*  void dalib_free_remote (remote_ptr rptr)                        *
*                                                                  *
*******************************************************************/

void dalib_free_remote (rptr)

remote_ptr rptr;

{ /* free all allocated data */

  free (rptr->no_remote);
  free (rptr->rem_offsets);
  free (rptr->inverse_map);
  free (rptr);

} /* dalib_free_remote */
