/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Sep 93                                                   *
*  Last Update : Mar 96                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : topology                                                 *
*                                                                         *
*  Function: Definition of Processor Topologies                           *
*                                                                         *
*    - handles many different torus specifications                        *
*    - supports one-, two- or three-dimensional tori                      *
*                                                                         *
*  Export :  FORTRAN Interface                                            *
*  ===========================                                            *
*                                                                         *
*  void FUNCTION(dalib_top_create) (int *topid, int *first, int *rank,    *
*                                   int *n1, int *n2, int *n3)            *
*                                                                         *
*   - all the routines define new topologies that will be identified      *
*     with the integer vaule of 'topid'                                   *
*                                                                         *
*   - note: some values can be non-positive for free choices              *
*                                                                         *
*  Export :  local to DALIB                                               *
*  ========================                                               *
*                                                                         *
*  void dalib_top_info (int topid, int dim, int *N, int *I)               *
*                                                                         *
*    - number of processors and relative position for dimension 'dim'     *
*      in processors topology specified by 'topid'                        *
*                                                                         *
*  int dalib_top_size (int topid)                                         *
*                                                                         *
*    - returns number of processors in topology topid                     *
*                                                                         *
*  int dalib_top_elem (int topid, int k)                                  *
*                                                                         *
*    - processor id of kth processor in topology (0 <= k < top_size)      *
*                                                                         *
*  int dalib_in_topology (topid)                                          *
*                                                                         *
*   - returns true if processor is in topology topid                      *
*                                                                         *
* Changes:                                                                *
* ========                                                                *
*                                                                         *
*  03/96 : all routines are extended to deal with arbitrary dimensions    *
*                                                                         *
**************************************************************************/

#include <stdio.h> 
#include "dalib.h"

#undef DEBUG
#define CHECK

/**************************************************************************
*                                                                         *
* Definition of the Data Structures for Topologie/Processor (Sub-)Groups  *
*                                                                         *
**************************************************************************/

     /*********************************************************
     *                                                        *
     *  Definition of a processor group / topology            *
     *                                                        *
     *  RESTRICTION !!!   must always be contiguous           *
     *                                                        *
     *********************************************************/

typedef struct

  { int group;        /* subset of processors belonging to topology    */

    int rank;         /* rank of the topology                          */

    int extension [MAX_RANK];

    /* local information about position of me in this group            */

    int within;                    /* 0 : processor itself is not in 
                                          this topololgy               */

    int relpos    [MAX_RANK];      /* my own coordinates in this top   */

    int sub_group [MAX_RANK];      /* subgroup for every row, col, ...
                                      has extensions[i] processors     */

  } topology_entry;

     /*********************************************************
     *                                                        *
     *  All available topologies (ever defined)               *
     *                                                        *
     *********************************************************/

topology_entry *topologies [MAX_TOPS];
int            topology_top;

/**************************************************************************
*                                                                         *
*  ERROR Messages (if CHECK is defined)                                   *
*                                                                         *
*   void dalib_top_check_valid (int topid)                                *
*   void dalib_top_check_rank (int topid, int dim)                        *
*                                                                         *
**************************************************************************/

#ifdef CHECK

void dalib_top_check_valid (topid)

int topid;

{  char msg[80];
   if ((topid < 1) || (topid > topology_top))
   { sprintf (msg,"dalib_top_check_valid, %d is illegal (must be in 1:%d)\n",
       topid, topology_top);
     dalib_internal_error (msg);
     dalib_stop ();
   }

} /* dalib_top_check_valid */

void dalib_top_check_rank (topid, dim)

int topid, dim;

{  int trank;
   char msg[150];
   dalib_top_check_valid(topid);
   trank = topologies[topid-1]->rank;
   if ((dim <= 0) || (dim > trank))
     { sprintf (msg, "topology - rank : dim (%d) of top %d not in 1:%d\n",
       dim, topid, trank);
       dalib_internal_error (msg);
       dalib_stop ();
     }

} /* dalib_top_check_rank */

void dalib_top_rank_error (message, must_rank, is_rank)
char message[];
int must_rank, is_rank;
{ char msg[80];
  sprintf (msg, "(topology, at %s): rank is %d, but must be %d\n",
       message, is_rank, must_rank);
  dalib_internal_error (msg);
} /* dalib_top_rank_error */

#endif 

     /*********************************************************
     *                                                        *
     *  Printing out all topologies                           *
     *                                                        *
     *********************************************************/

#ifdef DEBUG

void dalib_top_print ()

{ topology_entry *entry;

  int k, i, rank;
  int first, step, size;

  for (k=0; k < topology_top; k++)

   { printf ("Topology (%d): ", k+1);
     entry = topologies[k];
     rank  = entry->rank;
     dalib_group_info (entry->group, &first, &size, &step);
     printf (" group = %d (%d:%d:%d), size = ", entry->group,
               first, first+(size-1)*step, step);
     for (i = 1; i<=rank; i++)
        printf ("%d ", entry->extension[i-1]);
     printf (" my pos = ");
     if (entry->within == 0)
        printf (" --  not within");
      else
        { for (i = 1; i<=rank; i++)
            printf ("%d ", entry->relpos[i-1]);
        }
     printf ("\n");
   }
}
#endif

     /*************************************************************
     *                                                            *
     *  Help functions for setting up a topolgy entry             *
     *                                                            *
     *  void dalib_top_get_relpos (p, first, rank, shape, relpos) *
     *                                                            *
     *  find the position of a processor in the given torus       *
     *                                                            *
     *  Example (n1=2, n2=3)                                      *
     *                                                            *
     *            1        2        3                             *
     *          |----------------------|                          *
     *       1  | 1        3        5  |                          *
     *          |                      |                          *
     *       2  | 2        4        6  |                          *
     *          ------------------------                          *
     *                                                            *
     *************************************************************/

static void dalib_top_get_relpos (pid1, rank, shape, relpos)
 
int pid1;
int rank;
int shape[];
int relpos[];

{ int pid;
  int dims[MAX_RANK];
  int i;
  int index;

  pid = pid1 - 1;

  /* now 0 <= pid < N = shape[0] * ... * shape[rank-1] */

  dims[0] = 1;

  for (i=1; i<rank; i++)
     dims [i] = dims[i-1] * shape[i-1];

  /* dims[0] = 1, dims[1] = n1, dims[2] = n1 * n2, ... */

  for (i = rank - 1; i >= 0; i--)

     { index     = pid / dims[i];
       relpos[i] = index + 1;
       pid      -= index * dims[i];
     }

#ifdef DEBUG
  printf ("dalib_top_get_relpos computed, %d:%d:%d of %d:%d%:%d (rank=%d)\n",
          relpos[0], relpos[1], relpos[2], shape[0], shape[1], shape[2], rank);
#endif

} /* dalib_top_get_relpos */

     /*************************************************************
     *                                                            *
     *  defining groups for every dimension of the topology       *
     *                                                            *
     *************************************************************/

static void dalib_define_groups (topid)

int topid;

{ int ntasks, first, step;
  int rank, dim;
  topology_entry *entry;
  int gid;

  entry = topologies[topid-1];

  rank = entry->rank;   /* rank of the topology */

  /* every processor will belong to a group for one dimension */

  for (dim=0; dim<rank; dim++)

   {  /* find number of tasks in the dimension */

      ntasks = entry->extension[dim];
      first  = dalib_top_dimpos (topid, dim+1, 1);
      step   = dalib_top_dimpos (topid, dim+1, 2) - first;

      gid    = dalib_group_create (ntasks, first, step);

      entry->sub_group [dim] = gid;

#ifdef DEBUG
      printf ("%d: top = %d, dim = %d, gid = %d, first = %d, step = %d\n", 
              pcb.i, topid, dim+1, gid, first, step);
#endif

   }

} /* dalib_define_groups */

         /****************************************************
         *                                                   *
         *  dalib_top_setentry (topology)                    *
         *                                                   *
         *   - put entry on the big stack                    *
         *                                                   *
         ****************************************************/

void dalib_top_setentry (entry)
topology_entry *entry;

{ /* set the new entry at the top of the table */

  if (topology_top >= MAX_TOPS)
    { fprintf (stderr, "too many topologies, limit is %d\n", MAX_TOPS);
      exit(-1);
    }

  topologies[topology_top] = entry;
  topology_top++;

  dalib_define_groups (topology_top);

}

         /****************************************************
         *                                                   *
         *  dalib_top_set (entry, group, rank, shape[])      *
         *                                                   *
         ****************************************************/
 
static void dalib_top_set (entry, group, rank, shape)

topology_entry *entry;
int group, shape[];
 
{ int i;       /* counter variable                          */
  int pid;     /* processor id in group, 1<=pid<=group_size */
 
  entry->rank = rank;

  for (i=0; i<rank; i++)
     entry->extension[i] = shape[i];

  entry->group = group;

  /* get my own position in this new topology */

  pid = dalib_group_relpos (group, pcb.i);   /* get my position in group */

  if (pid)

     { entry->within = 1;
       dalib_top_get_relpos (pid, rank, shape, entry->relpos);
     }

    else

     entry->within   = 0;
 
} /* dalib_top_set */

     /*********************************************************
     *                                                        *
     *  Make a new entry at the end of the topology table     *
     *                                                        *
     *  dalib_top_define (group, rank, shape)                 *
     *                                                        *
     *********************************************************/

/**************************************************************************
*                                                                         *
*  void dalib_top_define (int group, int rank, int shape[])               *
*                                                                         *
*    - define a new topology for processors in group                      *
*                                                                         *
**************************************************************************/

static void dalib_top_define (group, rank, shape)

int group, rank, shape[];

{ topology_entry *entry;
  int i;

  entry = (topology_entry *) dalib_malloc (sizeof (topology_entry),
                                           "dalib_top_define");

#ifdef DEBUG
  printf ("%d: dalib_top_define, group = %d, rank = %d, shape = ",
           pcb.i, group, rank);
  for (i=0; i<rank; i++) printf ("%d ", shape[i]);
  printf ("\n");
#endif

  dalib_top_set (entry, group, rank, shape);
  dalib_top_setentry (entry);

} /* dalib_top_define */

     /*********************************************************
     *                                                        *
     *  int dalib_find_topology (group, rank, int shape[])    *
     *                                                        *
     *  Searching a topology entry in the table               *
     *                                                        *
     *  (pos == topology_top) stands for not found            *
     *                                                        *
     *  - called by dalib_top_create                          *
     *                                                        *
     *********************************************************/

static int dalib_find_topology (group, rank, shape)

int rank, group;
int shape[];

{ int found, pos;
  topology_entry *entry;
  int i;

  pos = 0; found = 0;

  while ((pos < topology_top) && (!found))

   { entry = topologies[pos];

     found = (entry->rank == rank);

     if (found)
        found = (entry->group == group);

     for (i=0; i<rank; i++)
        if (found) 
           found = (entry->extension[i] == shape[i]);

     if (!found) pos++;
   }

   return (pos);

} /* dalib_find_topology */

/**************************************************************************
*                                                                         *
*                                                                         *
*                                                                         *
**************************************************************************/

int dalib_top_make (group, rank, shape)

int group, rank, shape[];

{ int id;

  id = dalib_find_topology (group, rank, shape);

  /* if not found, then define a new topology */

  if (id == topology_top)

     dalib_top_define (group, rank, shape);

  return (id+1);

} /* dalib_top_make */

     /*********************************************************
     *                                                        *
     *  Initialization of Topogies                            *
     *                                                        *
     *   Topology 1 :    NP                                   *
     *   Topology 2 :    N1 x N2                              *
     *   Topology 3 :    N1 x N2 x N3                         *
     *                                                        *
     *  Every processor is contained (within) the topology    *
     *                                                        *
     *********************************************************/

void dalib_top_exit ()

{ int i;

  for (i=0; i<topology_top; i++)
     dalib_free (topologies[i], sizeof(topology_entry));
     
  dalib_pop_context ();

} /* dalib_top_exit */

     /*********************************************************
     *                                                        *
     *  local export for DALIB                                *
     *                                                        *
     *********************************************************/

void dalib_top_info (topid, dim, N, I)

int topid, dim;
int *N, *I;

{ topology_entry *entry;

  /* topology 0 is topology 1 + host processor */

  if (topid == -1)
     { *N = 1;
       if (pcb.i == 0) 
          *I = 1;
         else
          *I = 0;
       return;
     }

  if (topid == 0)
     { dalib_top_info (1, dim, N, I);
       return;
     }

#ifdef CHECK
   dalib_top_check_rank (topid, dim);
#endif

  entry = topologies[topid-1];
  *N = entry->extension[dim-1];
  if (entry->within)
    *I = entry->relpos[dim-1];
   else
    *I = 0;
}

int dalib_top_rank (topid)
int topid;

{ 
  if (topid == 0) return (0);
  if (topid == -1) return (0);
  dalib_top_check_valid (topid);
  return (topologies[topid-1]->rank);
}

int dalib_top_group (topid)
int topid;

{ 
  if (topid == 0) return (0);
  if (topid == -1) return (0);
  dalib_top_check_valid (topid);
  return (topologies[topid-1]->group);
}

void dalib_top_shape (topid, shape)
int shape[];

{ int i;
  topology_entry *entry;

  if (topid == 0) return;
  if (topid == -1) return;
  dalib_top_check_valid (topid);
  if (topid == 0) return;
  if (topid == -1) return;
  entry = topologies[topid-1];
  for (i=0; i<entry->rank; i++) shape[i] = entry->extension[i];
}

int dalib_top_first (topid)

int topid;

{ 
  if (topid == 0) return (1);
  if (topid == -1) return (1);

#ifdef CHECK
  dalib_top_check_valid (topid);
#endif

  return dalib_group_first (topologies[topid-1]->group);

} /* dalib_top_first */

         /****************************************************
         *                                                   *
         *  int dalib_top_size (int topid)                   *
         *                                                   *
         *  - returns number of processors in topology       *
         *                                                   *
         ****************************************************/

int dalib_top_size (topid)
int topid;

{ topology_entry *entry;
  int i, size, rank;

  if (topid == 0) return (pcb.p);  /* ATTENTION: is important to say */
  if (topid == -1) return (1);

  dalib_top_check_valid (topid);
  entry = topologies[topid-1];
  size = 1;
  rank = entry->rank;
  for  (i=0; i<rank; i++)
     size *= entry->extension[i];
  return (size);
} /* dalib_top_size */

/**************************************************************************
*                                                                         *
*   int dalib_top_elem (int topid, int k)                                 *
*                                                                         *
*   - returns k-th element of a topology (0 <= k < size(topid))           *
*                                                                         *
**************************************************************************/

int dalib_top_elem (topid, k)
int topid, k;

{ /* no check for 0 <= k < top_size (topid) */

  if (topid == 0) return (1+k);   
  if (topid == -1) return (1);

  dalib_top_check_valid (topid);
  return dalib_group_abspos (topologies[topid-1]->group, k+1);

} /* dalib_top_elem */

/**************************************************************************
*                                                                         *
*   int dalib_top_subgroup (int topid, int topdim)                        *
*                                                                         *
*   - returns subgroup of topology topid for all processors with          *
*     same coordinates except in dimension topdim                         *
*                                                                         *
*   - calling processor must be in this subgroup                          *
*                                                                         *
**************************************************************************/

int dalib_top_subgroup (topid, topdim)

int topid, topdim;

{ topology_entry *entry;
  int gid;

#ifdef CHECK
  dalib_top_check_rank (topid, topdim);
#endif

  entry = topologies[topid-1];
  gid   = entry->sub_group[topdim-1];

  return (gid);

}  /* dalib_top_subgroup */

/**************************************************************************
*                                                                         *
*  FUNCTION(dalib_top_create) (int *topid, array_info *top_dsp,           *
*                              int *first)                                *
*                                                                         *
*   - returns in topid an internal id for the new defined topology        *
*   - shape is needed for building subsections                            *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_top_create) (top_id, top_dsp, first)

int        *top_id;     /* out: ADAPTORs internal topology identification   */
array_info *top_dsp;    /* in : array descriptor for processor array of HPF */
int *first;             /* in : first processor in current context          */

{ int rank;
  int current_group;
  int new_group;

  int N, P;

  int i;
  int shape[MAX_RANK];

  char msg[256];

  /* get shape / rank of the processor array */

  dalib_secarray_shape (top_dsp, &rank, shape);

  if (rank > MAX_RANK)

    { sprintf (msg, "top_create with rank = %d, maximal rank is %d",
                     rank, MAX_RANK);
      dalib_internal_error (msg);
      dalib_stop ();
    }

  /* with the array descriptor we can get the shape to define topology */

  /* step 1 : make useful values of topology */

  current_group = dalib_context_group ();

  N  = dalib_group_size (current_group);

  P  = N - *first + 1;   /* available processors */

  dalib_make_torus (P, rank, shape);

  /* now we have found the final value of shape */

  P = 1;
  for (i=0; i<rank; i++) P *= shape[i];

  /* step 2 : verify that there are sufficient processors */

  if ((P <= 0) || (P + *first - 1 > N))
    { fprintf (stderr, 
       "DALIB: top_create (P=%d, first = %d) fails (only %d processors)\n",
         P, *first, N);
      exit(-1);
    }

  /* define a new group of processors for the new topology */

  new_group = dalib_subgroup_create (current_group, P, *first, 1);

  /* make the new topology (defines it if not available) */

  *top_id = dalib_top_make (new_group, rank, shape);

  /* hide the topology id also in the array descriptor */

  (*top_dsp)->size = *top_id;

#ifdef DEBUG
  dalib_top_print ();
#endif

} /* dalib_top_create */ 

/**************************************************************************
*                                                                         *
*  Functions for defining sub topologies                                  *
*                                                                         *
*  void FUNCTION(dalib_subtop_create) (subtopid, topid,                   *
*                               kind1, lb1, ub1, inc1,                    *
*                               ...                                       *
*                               kind7, lb7, ub7, inc7)                    *
*                                                                         *
*    - returns an internal descriptor for a subtopology                   *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_subtop_create) (subtop_id, top_dsp,
                            kind1, lb1, ub1, str1, kind2, lb2, ub2, str2,
                            kind3, lb3, ub3, str3, kind4, lb4, ub4, str4,
                            kind5, lb5, ub5, str5, kind6, lb6, ub6, str6,
                            kind7, lb7, ub7, str7)

int        *subtop_id;
array_info *top_dsp;

int *kind1, *lb1, *ub1, *str1, *kind2, *lb2, *ub2, *str2,
    *kind3, *lb3, *ub3, *str3, *kind4, *lb4, *ub4, *str4,
    *kind5, *lb5, *ub5, *str5, *kind6, *lb6, *ub6, *str6,
    *kind7, *lb7, *ub7, *str7;

{ int top_id;
  int top_rank, subtop_rank;
  int top_group, subtop_group;
  int subtop_first;

  int i;

  int n   [MAX_DIMENSIONS];
  int inc [MAX_DIMENSIONS];

  section_info sub_array;

  FUNCTION(dalib_section_create) (&sub_array, top_dsp,
           kind1, lb1, ub1, str1, kind2, lb2, ub2, str2,
           kind3, lb3, ub3, str3, kind4, lb4, ub4, str4,
           kind5, lb5, ub5, str5, kind6, lb6, ub6, str6,
           kind7, lb7, ub7, str7);

  top_id    = (*top_dsp)->size;    /* topology id was hidden in descriptor */
  top_rank  = dalib_top_rank (top_id);
  top_group = dalib_top_group (top_id);

  dalib_section_offset_dsp (sub_array, &subtop_rank, &subtop_first, n, inc);

  /* verify that it is not an empty section */

  if (subtop_rank < 0)

     { dalib_internal_error ("processor subset is empty");
       dalib_print_section_global (sub_array);
       dalib_stop ();
     }

#ifdef DEBUG
  dalib_print_section_global (sub_array);
  printf ("%d: subtop offset, rank = %d, first = %d, (n,inc) = ",
           pcb.i, subtop_rank, subtop_first);
  for (i=0; i<subtop_rank; i++) printf ("%d %d , ",n[i],inc[i]);
  printf ("\n");
#endif

  subtop_group = dalib_subgroup_make (dalib_top_group (top_id), 
                                      subtop_first+1, subtop_rank, n, inc);

  /* special case : single processor -> make one-dimensional proc array */

  if (subtop_rank == 0)

     { subtop_rank = 1;
       n[0] = 1;
     }

  *subtop_id = dalib_top_make (subtop_group, subtop_rank, n);

#ifdef DEBUG
  printf ("%d: subtop (1=%d,group=%d) of top(id=%d,rank=%d,group=%d) -> %d\n",
           pcb.i, subtop_first, subtop_group, top_id, top_rank, top_group,
           *subtop_id);
#endif

} /* dalib_subtop_create */

     /*************************************************************
     *                                                            *
     *  Getting absolute processor ids in topology                *
     *                                                            *
     *  Example (n1=2, n2=3)                                      *
     *                                                            *
     *            1        2        3                             *
     *          |----------------------|                          *
     *       1  | first    3        5  |                          *
     *          |                      |                          *
     *       2  | 2        4        6  |                          *
     *          ------------------------                          *
     *                                                            *
     *   p  =  (i2 - 1) * n1 + i1                                 *
     *   p  =  (i3 - 1) * n1 * n2  + (i2 - 1) * n1 + i1           *
     *                                                            *
     *************************************************************/

int dalib_top_abspos (topid, pos)

int topid, pos[];
 
{ topology_entry *entry;
  int i, rank;

  int g_pos;
  int a_pos;

  if (topid <= 0)  return (1);   /* return processor 1 as representant */
 
  entry = topologies[topid-1];

  rank  = entry->rank;

  g_pos = dalib_linear1_abspos (rank, entry->extension, pos);
  a_pos = dalib_group_abspos (entry->group, g_pos);

#ifdef DEBUG
  printf ("%d: top_abspos (top=%d,group=%d), ", pcb.i, topid, entry->group);
  for (i=0; i<rank; i++) printf ("%d ", pos[i]);
  printf (" of ");
  for (i=0; i<rank; i++) printf ("%d ", entry->extension[i]);
  printf (" is g_pos=%d, a_pos=%d\n", g_pos, a_pos);
#endif

  return a_pos;

} /* dalib_top_abspos */
 
     /*************************************************************
     *                                                            *
     *  int dalib_in_topology (topid)                             *
     *                                                            *
     *   - returns true if processor is in topology topid         *
     *                                                            *
     *************************************************************/

int dalib_in_topology (topid)
int topid;

{  if (topid < 0)

      { /* host topology */

        if (pcb.i == 0)   
           return (1);
         else
           return (0);
      }

    else if (topid == 0)

      return (1);    /* replicated */

    else

      return (topologies[topid-1]->within);
}

int FUNCTION(dalib_is_in_procs) (topology)

int *topology;

{ return dalib_in_topology (*topology);

} /* dalib_is_in_procs */


     /*************************************************************
     *                                                            *
     *  int dalib_top_dimpos (topid, dim, pos)                    *
     *                                                            *
     *   - returns the process id of the processor in the         *
     *     same topology only in dimension dim at position pos    *
     *                                                            *
     *   This function is needed for global communication         *
     *   in processor topologies (reductions, broadcasts)         *
     *                                                            *
     *************************************************************/

int dalib_top_dimpos (topid, dim, pos) 
int topid, dim, pos;

{ topology_entry *entry;
  int my_pos[MAX_RANK];
  int i, rank;
  int pid;

  entry = topologies[topid-1];
  rank  = entry->rank;

  for (i=0; i<rank; i++)
      my_pos[i] = entry->relpos[i];

  my_pos[dim-1] = pos;

  pid = dalib_top_abspos (topid, my_pos);

#ifdef DEBUG
  printf ("dalib_top_dimpos, topid = %d, dim = %d, pos = %d -> pid = %d\n",
           topid, dim, pos, pid);
#endif

  return (pid);

} /* dalib_top_dimpos */

     /*************************************************************
     *                                                            *
     *  int dalib_top_relpos (topid, dim, pos)                    * 
     *                                                            *
     *   - returns the process id of the processor in the         *
     *     same topology but +pos relative positions              *
     *                                                            *
     *   This function is needed for local neighbored communic.   *
     *                                                            *
     *************************************************************/
 
int dalib_top_relpos (topid, dim, pos)
int topid, dim, pos;
 
{ topology_entry *entry;
  int my_pos[MAX_RANK];
  int i, rank;
  int *dimpos;
  int n;
 
  entry = topologies[topid-1];
  rank  = entry->rank;
 
  for (i=0; i<rank; i++)
      my_pos[i] = entry->relpos[i];
 
  dimpos = my_pos + dim - 1;
 
  n = entry->extension[dim-1];

  *dimpos += pos;
  if (*dimpos < 1) *dimpos += n;
  if (*dimpos > n) *dimpos -= n;

  return (dalib_top_abspos (topid, my_pos));
 
} /* dalib_top_relpos */
 
 
     /*************************************************************
     *                                                            *
     *  void dalib_top_neighbors (p_left, p_right, topid, dim)    *
     *                                                            *
     *************************************************************/
 
void dalib_top_neighbors (p_left, p_right, topid, dim) 
int *p_left, *p_right;
int topid, dim;

{ int pos;

  *p_left  = dalib_top_relpos (topid, dim, -1);
  *p_right = dalib_top_relpos (topid, dim, 1);

#ifdef DEBUG
  printf ("%d: top_neighbors (topid=%d,dim=%d) left = %d, right = %d\n", 
          pcb.i, topid, dim, *p_left, *p_right);
#endif

} /* dalib_top_neighbors */ 

/**************************************************************************
*                                                                         *
*  int dalib_top_distance (int topid, int dim)                            *
*                                                                         *
*   - topid must be valid topology identifier, 1 <= dim <= dalib_top_rank *
*   - returns difference between two neighbored processors                *
*                                                                         *
**************************************************************************/

int dalib_top_distance (topid, topdim)
int topid;
int topdim;

{ topology_entry *entry;
  int i, dist;

#ifdef CHECK
  dalib_top_check_rank (topid, topdim);
#endif

  entry = topologies[topid-1];

  dist = dalib_group_step (entry->group);

  /* topdim == 1 : dist = 1
     topdim == 2 : dist = entry->extension[0];
     topdim == 3 : dist = entry->extension[0] * entry->extension[1];
  */

  for (i=1; i<topdim; i++)
     dist = dist * entry->extension[i-1];

  return (dist);

} /* dalib_top_distance */

     /*************************************************************
     *                                                            *
     *  void dalib_top_position (int top_id, int pid,             *
     *                           bool *is_int_top, int relpos[])  *
     *                                                            *
     *  - find the relative position of pid in a topology         *
     *                                                            *
     *************************************************************/

void dalib_top_position (topid, pid, is_in_top, relpos)

int topid;
int pid;
int *is_in_top;
int relpos [];

{ topology_entry *entry;

  int group;
  int pos;

#ifdef CHECK
  dalib_top_check_valid (topid);
#endif 

  entry = topologies[topid-1];

  pos = dalib_group_relpos (entry->group, pid);   /* 1 <= pid <= size */

  if (!pos)   /* pid does not belong to the topology */

     { *is_in_top = 0;
       return;
     }

  *is_in_top = 1;

  dalib_top_get_relpos (pos, entry->rank, entry->extension, relpos);

} /* dalib_top_position */

     /*************************************************************
     *                                                            *
     *  int dalib_top_distinct (int topid1, int topid2)           *
     *                                                            *
     *   - returns 1 if topologies have no common processor       *
     *                                                            *
     *************************************************************/

int dalib_top_distinct (topid1, topid2)

{ int size1, size2;
  int first1, first2;

  size1 = dalib_top_size (topid1);
  size2 = dalib_top_size (topid2);

  first1 = dalib_top_first (topid1);
  first2 = dalib_top_first (topid2);

  if (first1 <= first2)

    { if (first1 + size1 -1 < first2) return 1;
      return 0;
    }

   else

    { if (first2 + size2 -1 < first1) return 1;
      return 0;
    }

} /* dalib_top_distinct */

/**************************************************************************
*                                                                         *
*  int dalib_top_restrict (int topology, int n, int dims[n])              *
*                                                                         *
*  - creates a new topology where dims are fixed to my dimensions         *
*                                                                         *
**************************************************************************/

int dalib_top_restrict (topology, n, dims)

int topology;
int n;
int dims[];

{ topology_entry *entry;

  int group;
  int rank;

  int full      [MAX_RANK];
  int new_shape [MAX_RANK];
  int new_inc   [MAX_RANK];

  int new_group;
  int new_rank;
  int first;
  int new_top;

  int i;

  if (topology == 0) return (topology);
  if (topology == -1) return (topology);

#ifdef CHECK
  dalib_top_check_valid (topology);
#endif

  entry = topologies[topology-1];

  rank  = entry->rank;
  group = entry->group;

  /* fix dimensions according dims       */

  for (i=0; i<rank; i++) full[i] = 1;
  for (i=0; i<n; i++)    full[dims[i]-1] = 0;
 
  dalib_linear1_restrict (rank, entry->extension, entry->relpos, full,
                          &new_rank, &first, new_shape, new_inc);

  new_group = dalib_subgroup_make (group, first, new_rank, new_shape, new_inc);

  if (new_rank == 0)

     { new_rank     = 1;
       new_shape[0] = 1;
     }

  new_top = dalib_top_make (new_group, new_rank, new_shape);

  return (new_top);

} /* dalib_top_restrict */

/**************************************************************************
*                                                                         *
*  void dalib_pack_topology (char buffer[], int top_id => int length)     *
*                                                                         *
*  void dalib_unpack_topology (char buffer[] => int top_id, int length)   *
*                                                                         *
*  - pack/unpack information of topology (first, rank, shape)             *
*  - returns number of bytes needed for the coding in buffer              *
*                                                                         *
*  Note: exchanging topology information might result in different        *
*        topology identifiers on the processors                           *
*                                                                         *
**************************************************************************/

     /*************************************************************
     *   dalib_pack_topology (buffer, top_id => length)           *
     *      [group, rank, extensions]                             *
     *************************************************************/

void dalib_pack_topology (buffer, top_id, length)

char buffer[];
int  top_id;
int  *length;

{ topology_entry *entry;

  int rank;
  int top_len;

  dalib_top_check_valid (top_id);

  entry = topologies[top_id-1];

  dalib_pack_group (buffer, entry->group, &top_len);

  rank = entry->rank;

  dalib_memcopy (buffer+top_len, &rank, sizeof(int));
  top_len += sizeof (int);

  dalib_memcopy (buffer+top_len, entry->extension, rank * sizeof(int));
  top_len += rank *sizeof (int);

  *length += top_len;

#ifdef DEBUG
  printf ("%d: packed top %d (group=%d, rank=%d) into buffer (%d bytes)\n",
          pcb.i, top_id, entry->group, entry->rank, *length);
#endif

} /* dalib_pack_topology */

     /*************************************************************
     *   dalib_unpack_topology (buffer => top_id, length)         *
     *************************************************************/

void dalib_unpack_topology (buffer, top_id, length)

char buffer[];
int  *length;
int  *top_id;

{ int group;
  int rank;
  int extension [MAX_RANK];
  int glen;

  dalib_unpack_group (buffer, &group, &glen);

  dalib_memcopy (&rank, buffer+glen, sizeof(int));
  glen += sizeof(int);

#ifdef DEBUG
  printf ("%d: unpack topology, group = %d, rank = %d\n",
           pcb.i, group, rank);
#endif

  dalib_memcopy (extension, buffer+glen, rank * sizeof(int));

  *length = glen + rank * sizeof(int);
  *top_id = dalib_top_make (group, rank, extension);

#ifdef DEBUG
  printf ("%d: unpacked top %d (group=%d, rank=%d) from buffer (%d bytes)\n",
          pcb.i, *top_id, group, rank, *length);
#endif

} /* dalib_unpack_topology */

