/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Mar 95                                                   *
*  Last Update : Mar 95                                                   *
*                                                                         *
*  Acknowledgement : Nicolas Bert for Solaris 2.4 port                    *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : timing.m4                                                *
*                                                                         *
*  Function    : some more FORTRAN 90 intrinsics for DALIB                *
*                                                                         *
*  Export :  FORTRAN Interface                                            *
*                                                                         *
*   void FUNCTION(dalib_date_and_time) (date, time, zone, values)         *
*                                                                         *
*   void FUNCTION(dalib_system_clock) (count, count_rate, count_max)      *
*                                                                         *
**************************************************************************/

#include <sys/types.h>
#include <sys/times.h>
#include <sys/time.h>

#if defined(RS6K) || defined(PowerPC) || defined(SUN4SOL2)
#include <time.h>
#endif

#include <sys/resource.h>

#if defined(CS)
#include <sys/rusage.h>
#endif
 
#if defined(SX4)
/* need files for variable HZ (clock ticks per second) */
#include <sys/param.h>
#endif
 
#if defined(SUN4SOL2)
/* need files for sysconf(_SC_CLK_TCK) */
#include <unistd.h> 
#include <limits.h>
#endif

/**************************************************************************
*                                                                         *
*  DATE_AND_TIME ([DATE] [,TIME] [,ZONE] [,VALUES])                       *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_date_and_time) 
     ARGS(`STRING_ARG(date), STRING_ARG(time), STRING_ARG(zone), values') 

STRING_ARG_DECL(date);
STRING_ARG_DECL(time);
STRING_ARG_DECL(zone);
int  *values;

{ struct tm *my_time;
  time_t clock;

  struct timeval tp;
#if defined(PowerPC)
  long int tzp;
#else
  struct timezone tzp;
#endif

  char date_string[8];
  char time_string[10];
  char zone_string[5];

  /* t = tp.tv_sec * 100000 + tp.tv_usec / 10; */

  int i, vals[8];
  int zone_hour, zone_min;
 
  (void) gettimeofday (&tp, &tzp);

  /* tp.tv_sec, tp.tv_usec is the current time expressed in elapsed  
     seconds and microseconds since 00:00 GMT, January 1, 1970 (zero hour) */

  my_time = localtime (&tp.tv_sec);

  /* localtime returns pointer to a structure containing the time, 
     broken down into variaous components of that time             */

  vals[0] = my_time->tm_year+1900;   /* int tm_year    : year - 1900          */
  vals[1] = my_time->tm_mon+1;       /* int tm_mon     : month of year (0-11) */
  vals[2] = my_time->tm_mday;        /* int tm_mday    : day of month (1-31)  */

  /* external variable timezone contains difference, in seconds, between
     UTC and local standard time                                              */

#if defined(SUN4SOL2) || defined(RS6K)
  vals[3] = timezone / 60;
#else
  vals[3] =  -32767;
#endif

  /* vals[3] = - 60;  can be used as defalut            */

  vals[4] = my_time->tm_hour;        /* int tm_hour    : hours (0-23)         */
  vals[5] = my_time->tm_min;         /* int_tm_min     : minutes (0-59)       */
  vals[6] = my_time->tm_sec;         /* int tm_sec     : seconds (0-59)       */
  vals[7] = tp.tv_usec / 1000;

  if (FUNCTION(dalib_present) (STRING_PTR(date)))

     { /* set CCYYMMDD to date */

       sprintf (date_string, "%4d%2d%2d", vals[0], vals[1], vals[2]);

       if (date_string[4]==' ') date_string[4]='0';
       if (date_string[6]==' ') date_string[6]='0';

       dalib_set_fstring (STRING_PTR(date), 8, date_string);
     }

  if (FUNCTION(dalib_present) (STRING_PTR(time)))

     { /* set hhmmss.sss to time */

       sprintf (time_string, "%2d%2d%2d.%3d", 
                vals[4], vals[5], vals[6], vals[7]);

       if (time_string[0]==' ') time_string[0]='0';
       if (time_string[2]==' ') time_string[2]='0';
       if (time_string[4]==' ') time_string[4]='0';
       if (time_string[7]==' ') time_string[7]='0';
       if (time_string[8]==' ') time_string[8]='0';

       dalib_set_fstring (STRING_PTR(time), 10, time_string);
     }

  if (FUNCTION(dalib_present) (STRING_PTR(zone)))

     { /* set +/- hhmm to zone */
 
       zone_min  = vals[3];

       if (zone_min >= 0) 
         { zone_hour = zone_min / 60;
           zone_min  = zone_min -60 * zone_hour;
           sprintf (zone_string,"+%2d%2d", zone_hour, zone_min);
         }
        else if (zone_min == -32767)
           sprintf (zone_string,"     ");
        else
         { zone_min = -zone_min;
           zone_hour = zone_min / 60;
           zone_min  = zone_min -60 * zone_hour;
           sprintf (zone_string,"-%2d%2d", zone_hour, zone_min);
         }

       for (i=1; i<5; i++)
         if (zone_string[i]==' ') zone_string[i]='0';

       dalib_set_fstring (STRING_PTR(zone), 5, zone_string);
     }

  if (FUNCTION(dalib_present) (values)) 

     {
       for (i=0; i<8; i++) values[i] = vals[i];
     }

} /* FUNCTION(dalib_date_and_time) */ 

/**************************************************************************
*                                                                         *
*  Walltime Timer                                                         *
*                                                                         *
*  SYSTEM_CLOCK ( [COUNT] [,COUNT_RATE] [,COUNT_MAX] )                    *
*                                                                         *
*  - we take a 24-hour clock with a resolution of 1000 ticks per second   *
*                                                                         *
**************************************************************************/

#define COUNT_RATE 1000
#define DAY_SECS   24*3600
#define COUNT_MAX  DAY_SECS*COUNT_RATE - 1

int dalib_timestamp ()

{ int count;

#if defined(CENJU3)
   int CJgettmr ();
#elif defined(PowerPC)
   struct timeval tp;
   long int tzp;
#else
   struct timeval tp;
   struct timezone tzp;
#endif

#if defined(CENJU3)

     count = CJgettmr ();

#else

    /* t = tp.tv_sec * 100000 + tp.tv_usec / 10; */

    { (void) gettimeofday (&tp, &tzp);

      count = (tp.tv_sec % (DAY_SECS)) * COUNT_RATE + 
              (tp.tv_usec / (1000000 / COUNT_RATE));

    }

#endif

  return (count);

} /* dalib_timestamp */

int dalib_timerate ()

{
   return (COUNT_RATE);

} /* dalib_timerate */

/**************************************************************************
*                                                                         *
*  SUBROUTINE dalib_system_clock ([int count], [int count_rate],          *
*                                 [int count_max])                        *
*                                                                         *
*  Note: no doubling of arguments                                         *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_system_clock) (count, count_rate, count_max) 

int *count, *count_rate, *count_max;

{ if (FUNCTION(dalib_present) (count_rate)) 

     *count_rate = dalib_timerate();

  if (FUNCTION(dalib_present) (count_max)) 

     *count_max  = COUNT_MAX;

  if (FUNCTION(dalib_present) (count)) 

     *count = dalib_timestamp ();

} /* FUNCTION(dalib_system_clock) */ 

/**************************************************************************
*                                                                         *
*  CPU timer                                                              *
*                                                                         *
**************************************************************************/

#if defined(CENJU3)
void FUNCTION(dalib_cpu_time) (time) 
float *time;
{ dalib_internal_error ("dalib_cpu_time not supported");
  dalib_stop ();
} /* FUNCTION(dalib_cpu_time) */ 
#else
void FUNCTION(dalib_cpu_time) (time) 

float *time;

{ 

#if defined (SUN4SOL2)
    struct tms r;					
    times(&r);					
    *time = (float)r.tms_utime / (float)sysconf(_SC_CLK_TCK);	
#elif defined (CRAY) 
    struct tms r;					
    times(&r);					
    *time = (float)r.tms_utime ;
#elif defined (SX4) 
    struct tms r;					
    times(&r);					
    *time = (float)r.tms_utime / HZ;
#else					

    struct rusage r;

    getrusage (0, &r);

#if defined(CS)
    *time =    (r.ru_utime.tv_sec) * 1.0 
             + (r.ru_utime.tv_nsec)/ 1000000.0;
#else

    *time =    (r.ru_utime.tv_sec) * 1.0 
             + (r.ru_utime.tv_usec)/ 1000000000.0;
#endif
  
#endif

} /* dalib_cpu_time */
#endif
