/**************************************************************************
*                                                                         *
*  Author      : Stephan Springstubbe, GMD, SCAI.LAB                      *
*                Thomas Brandes, GMD, SCAI.LAB                            *
*                                                                         *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : May 95                                                   *
*  Last Update : May 95                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : shm.c                                                    *
*                                                                         *
*  Function: SYSTEM V support of shared memory segments                   *
*                                                                         *
*   int shm_allocate (size)                                               *
*                                                                         *
*     - one processor defines a shared memory segment, returns its id     *
*                                                                         *
*   char *shm_attach (shmid)                                              *
*                                                                         *
*     - one processor defines a shared memory segment, returns its id     *
*                                                                         *
*   void shm_detach (char *shmaddr)                                       *
*                                                                         *
*    - process will no longer use the shared memory (starting at shmaddr) *
*                                                                         *
*   void shm_deallocate (int shmid)                                       *
*                                                                         *
*    - gives shared memory region free (only called by 1 processor)       *
*                                                                         *
*   EXPORTED for array.c                                                  *
*   ====================                                                  *
*                                                                         *
*   dalib_shm_allocate (shm_size => char *shm_ptr, int shm_id)            *
*                                                                         *
*   dalib_shm_deallocate (char *shm_ptr, int shm_id)                      *
*                                                                         *
**************************************************************************/

#undef  DEBUG

#include "dalib.h" 

#include <stdio.h>          /* printf */
 
#if defined(SHM)

#include <sys/types.h>      /* shmget */
#include <sys/ipc.h>        /* shmget */
#include <sys/shm.h>        /* shmget */
#include <sys/errno.h>      /* error numbers */
 
#define PAGE_SIZE           4096
 
/**************************************************************************
*                                                                         *
*   int shm_allocate (size)                                               *
*                                                                         *
*     - one processor defines a shared memory segment, returns its id     *
*                                                                         *
**************************************************************************/

int shm_allocate (int size)

{
  int shmid;
  extern int errno;
  int memsize;
  int shmflag;

  memsize = ( (size + PAGE_SIZE - 1) / PAGE_SIZE ) * PAGE_SIZE;

  shmflag = 255*256 + 255;  /* set last 9 lower bits all to true */

  shmid = shmget (IPC_PRIVATE, memsize, shmflag); 

  if (shmid < 0) {
    printf ("error in shm_allocate operation: <%d>!\n", errno);
    printf ("size was : %d, chosen memsize : %d\n", size, memsize);
    dalib_internal_error ("shm_allocate");
    if (errno == EINVAL)
       printf ("invalid size for shared memory segment\n");
    if (errno == ENOMEM)
       printf ("not enough physical memory\n");
    if (errno == ENOSPC)
       printf ("system imposed limit exceeded\n");
    dalib_stop ();
  }

#ifdef DEBUG
    printf ("shared_mem_allocate, shmid =%d\n", shmid);
#endif

  return (shmid);
 
} /* shm_allocate */
 
/**************************************************************************
*                                                                         *
*   char *shm_attach (shmid)                                              *
*                                                                         *
*     - one processor defines a shared memory segment, returns its id     *
*                                                                         *
**************************************************************************/

char *shm_attach (int shmid)

{ char *shmaddr;
  extern int errno;
 
#ifdef DEBUG
  printf ("%d: attach to shmid = %d\n", pcb.i, shmid);
#endif

  /* SHM_RND stands for random attach, included from shm.h */

  shmaddr = (char *) shmat (shmid, (char *) 0, SHM_RND);

  if ((long) shmaddr == -1) {
    printf ("error in shmat operation, error number = %d\n", errno);
    dalib_internal_error ("shm_attach failed");
    if (errno == EACCES)
       printf ("no permission to access shmid %d\n", shmid);
    if (errno == EINVAL)
       printf ("%d is not a valid shared memory id to attach\n", shmid);
    if (errno == EMFILE)
       printf ("exceed system limit\n");
    if (errno == ENOMEM)
       printf ("not enough memory to attach to shared memory %d\n", shmid);
    dalib_stop ();
  }

  return (shmaddr);
 
} /* shm_attach */
 
/**************************************************************************
*                                                                         *
*  void shm_detach (char *shmaddr)                                        *
*                                                                         *
*  - process will no longer use the shared memory (starting at shmaddr)   *
*                                                                         *
**************************************************************************/

void shm_detach (char *shmaddr)

{ extern int errno;
  char msg[128];
  int shmrtn;
 
#ifdef DEBUG
  printf ("%d: detach from shared memory, addr = %d\n", pcb.i, shmaddr);
#endif

  shmrtn = shmdt (shmaddr);

  if (shmrtn < 0) 

   { sprintf (msg, "shm_detach fails, errno = %d\n", errno);
     dalib_internal_error (msg);
     if (errno == EINVAL)
        printf ("not address of a shared memory segment\n");
     dalib_stop ();
   }
 
} /* end shm_detach */
 
/**************************************************************************
*                                                                         *
*  void shm_deallocate (int shmid)                                        *
*                                                                         *
*  - gives shared memory region free (only called by 1 processor)         *
*                                                                         *
**************************************************************************/

void shm_deallocate (int shmid)

{ extern int errno;
  char msg[128];
  int shmrtn;
 
  shmrtn = shmctl (shmid, IPC_RMID, NULL);

#ifdef DEBUG
    printf ("shared_mem_deallocate operation: <%d>!\n", shmrtn);
#endif 

  if (shmrtn < 0)
   { sprintf (msg, "shm_deallocate fails, errno = %d\n", errno);
     dalib_internal_error (msg);
     dalib_stop ();
   }
 
}  /* shm_deallocate */

/**************************************************************************
*                                                                         *
*   dalib_shm_allocate (shm_size => char *shm_ptr, int shm_id)            *
*                                                                         *
**************************************************************************/

void dalib_shm_allocate (shm_size, shm_ptr, shm_id)

int shm_size;
int *shm_id;
char **shm_ptr;
 
{ int size, sendpid;
  
#ifdef DEBUG
  printf ("%d: dalib_shm_allocate, size = %d\n", pcb.i, shm_size);
#endif

  /* allocation of the shared memory region */
 
  if (pcb.i == 1) 
     *shm_id = shm_allocate (shm_size);
 
  /* broadcast shm_id to all processors */

  size    = sizeof(shm_id);
  sendpid = 1;

  FUNCTION(dalib_broadcast) (shm_id, &size, &sendpid); 
 
#ifdef DEBUG
  printf ("%d: dalib_shm_allocate, attach to %d\n", pcb.i, *shm_id);
#endif

  /* attaching the shared memory region */

  *shm_ptr = shm_attach (*shm_id);
 
} /* dalib_shm_allocate */

/**************************************************************************
*                                                                         *
*   dalib_shm_deallocate (char *shm_ptr, int shm_id)                      *
*                                                                         *
**************************************************************************/

void dalib_shm_deallocate (shm_id, shm_ptr)

int shm_id;
char *shm_ptr;

{ 

#ifdef DEBUG
  printf ("%d: dalib_shm_deallocate from shm_id = %d, addr = %d\n", 
           pcb.i, shm_id, shm_ptr);
#endif

  /* detaching must be done by all processors */

  shm_detach (shm_ptr);
 
  /* barrier before detaching */
 
  FUNCTION(dalib_barrier) (); 
 
  /* deallocation of the shared memory region only by first processor */
 
  if (pcb.i == 1) shm_deallocate (shm_id);

} /* dalib_shm_deallocate */

#elif defined(GM)

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/mppg.h>
#include <sys/types.h>
#include <unistd.h>
#include <mpi.h>
 
#define PAGE_SIZE           4096
 
/**************************************************************************
*                                                                         *
*   void gm_allocate (int gm_size, int *p_id, int **gm_ptr)               *
*                                                                         *
*     - one processor defines a global memory region,                     *
*       returns the address and the process id (UNIX)                     *
*                                                                         *
**************************************************************************/

void gm_allocate (gm_size, p_id, gm_ptr)

int gm_size;
int *p_id;
int **gm_ptr;

{ int *adr;
  int *gm_address;

  char err_msg [32];

  adr = (int *) 0;

  gm_address = (int *) dp_xmalloc(adr, gm_size);

  if (gm_address == (void *) -1)

     { printf ("(%d) dp_xmalloc returned errno %d; meaning: ",pcb.i,errno);

       switch (errno) {
           case EINVAL:{ strcpy (err_msg,"EINVAL");break;}
           case ENOMEM:{ strcpy (err_msg,"ENOMEM");break;}
           case EAGAIN:{ strcpy (err_msg,"EAGAIN");break;}
           default: strcpy (err_msg,"strange");
               }
          printf (" %s\n",err_msg);
        dalib_internal_error ("GM ERROR");
        dalib_stop ();
     }

#ifdef DEBUG
     printf ("(%d) dp_xmalloc returned: gm_address = %p; gm_size %d\n",
             pcb.i, gm_address, gm_size);
#endif

  *gm_ptr = gm_address;
  *p_id   = getpid();

}  /* gm_allocate */

/**************************************************************************
*                                                                         *
*   char *shm_attach (shmid)                                              *
*                                                                         *
*     - one processor defines a shared memory segment, returns its id     *
*                                                                         *
**************************************************************************/

int *gm_attach (p_id, gm_ptr)

int p_id;
int *gm_ptr;

{ int *gm_address;
  int *adr;

  char err_msg [32];

  adr = (int *) 0;

  gm_address = dp_xmatt(p_id, gm_ptr, adr);

  if (gm_address ==  (void *) -1)

       { printf ("%d: error %d occurred at dp_xmatt. meaning: ",
                  pcb.i, errno);

         switch (errno)
               {
                case EINVAL:{ strcpy (err_msg,"EINVAL");break;}
                case ESRCH:{ strcpy (err_msg,"ESRCH");break;}
                default: strcpy (err_msg,"strange");
               }
          printf (" %s\n",err_msg);
          dalib_stop ();
        }

  return (gm_address);

} /* gm_attach */
 
/**************************************************************************
*                                                                         *
*  void gm_free (int *gm_ptr)                                             *
*                                                                         *
*  - gives global memory region free (must be called by all processors)   *
*                                                                         *
**************************************************************************/

void gm_free (gm_ptr)

int *gm_ptr;

{ int rc;

  char err_msg [32];

#ifdef DEBUG
  printf ("%d: gm_free of gm_ptr = %p\n", pcb.i, gm_ptr);
#endif

  rc = dp_xmfree (gm_ptr);

  if (rc == -1)

      { printf ("(%d) dp_xmfree returned errno: %d meaning: ", 
                 pcb.i, errno);
        switch (errno)
              {
               case EINVAL:{ strcpy (err_msg,"EINVAL");break;}
               default: strcpy (err_msg,"strange");
              }
        printf (" %s\n",err_msg);
        dalib_stop ();
      }

#ifdef DEBUG
   printf ("%d: dp_xmfree successful!\n", pcb.i);
#endif

}  /* gm_free */

/**************************************************************************
*                                                                         *
*   dalib_shm_allocate (shm_size => char *shm_ptr, int shm_id)            *
*                                                                         *
**************************************************************************/

void dalib_shm_allocate (shm_size, shm_ptr, shm_id)

int shm_size;
int *shm_id;
int **shm_ptr;
 
{ int gm_pid;
  int *gm_address;

  int size, sendpid;
  
#ifdef DEBUG
  printf ("%d: dalib_shm_allocate (GM), size = %d\n", pcb.i, shm_size);
#endif

  /* allocation of the global memory region */
 
  if (pcb.i == 1) 

     gm_allocate (shm_size, &gm_pid, &gm_address);
 
  /* broadcast shm_id/shm_ptr to all processors */

  size    = sizeof(gm_pid);
  sendpid = 1;

  FUNCTION(dalib_broadcast) (&gm_pid, &size, &sendpid); 
 
  size    = sizeof(gm_address);
  sendpid = 1;

  FUNCTION(dalib_broadcast) (&gm_address, &size, &sendpid); 
 
#ifdef DEBUG
  printf ("%d: dalib_shm_allocate (GM), attach to pid = %d, addr = %p\n", 
            pcb.i, gm_pid, gm_address);
#endif

  /* attaching the global memory region */

  if (pcb.i == 1)

     { *shm_ptr = gm_address;
       *shm_id  = gm_pid;
     }

   else

     { *shm_ptr = gm_attach (gm_pid, gm_address);
       *shm_id  = gm_pid;
     }
 
} /* dalib_shm_allocate */

/**************************************************************************
*                                                                         *
*   dalib_shm_deallocate (char *shm_ptr, int shm_id)                      *
*                                                                         *
**************************************************************************/

void dalib_shm_deallocate (shm_id, shm_ptr)

int shm_id;
int *shm_ptr;

{ 
#ifdef DEBUG
  printf ("%d: dalib_shm_deallocate (GM) from p_id = %d, addr = %p\n", 
           pcb.i, shm_id, shm_ptr);
#endif
 
  /* barrier before memory is really freed  */
 
 
  /* deallocation of the global memory region only by first
     or by all processor                                        */
 
  /* if (pcb.i == 1)  */

/*  FUNCTION(dalib_barrier) (); */
  MPI_Barrier(MPI_COMM_WORLD);

     gm_free (shm_ptr);

/*  FUNCTION(dalib_barrier) ();  */
  MPI_Barrier(MPI_COMM_WORLD);

} /* dalib_shm_deallocate */

#else

/**************************************************************************
*                                                                         *
*   NO SUPPORT OF SYSYEM_V FEATURES !!!!!!!!!!!!!!!!!!                    *
*                                                                         *
**************************************************************************/

 /* make never calls for all the routines */

 void shm_never (routine)
 char *routine;

 { dalib_internal_error
     ("shared memory segments / semaphores not supported on your system");
   printf ("do not use SHARED directive or -shared flag\n");
   dalib_stop ();
 }

 int shm_allocate ()
 { shm_never ("shm_allocate"); return(0); }

 char *shm_attach ()
 { shm_never ("shm_attach"); return ((char *) NULL); }

 void shm_detach ()
 { shm_never ("shm_detach"); }

 void shm_deallocate ()
 { shm_never ("shm_deallocate"); }

#endif
