/**************************************************************************
*                                                                         *
*  Author      : Thomas Brandes, GMD, SCAI.LAB                            *
*                                                                         *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Jun 95                                                   *
*  Last Update : Jun 95                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*        S E M A P H O R E S                                              *
*                                                                         *
*  Module      : sem.c                                                    *
*                                                                         *
*  Function: SYSTEM V support of semaphores                               *
*                                                                         *
*   int sem_allocate (number)                                             *
*                                                                         *
*    - allocation of number semaphores whose value is initialized         *
*      with init                                                          *
*                                                                         *
*   void sem_deallocate (int semid)                                       *
*                                                                         *
*    - gives semaphores free                                              *
*                                                                         *
*   void sem_lock (int semid, int num)                                    *
*                                                                         *
*    - set semaphore to 0 (but wait in any case until it is one           *
*                                                                         *
*   void sem_unlock (int semid, int num)                                  *
*                                                                         *
*    - set semaphore back to 1                                            *
*                                                                         *
*   void dalib_define_barrier ()                                          *
*   void dalib_free_barrier ()                                            *
*                                                                         *
*    - defining and deleting of barrier                                   *
*                                                                         *
*   EXPORTED for FORTRAN                                                  *
*   ====================                                                  *
*                                                                         *
* FUNCTION(dalib_sem_barrier) () * 
*                                                                         *
**************************************************************************/

#undef DEBUG

#if defined(SHM)

#include "dalib.h" 

#include <stdio.h>          /* printf */
 
#include <sys/types.h>      /* shmget */
#include <sys/ipc.h>        /* shmget */
#include <sys/sem.h>        /* semaphores    */
#include <sys/errno.h>      /* error numbers */
 
static int barrier_id = -1;   /* semaphore id for barrier */

/**************************************************************************
*                                                                         *
*   void sem_set_value (semid, number, val)                               *
*                                                                         *
**************************************************************************/

void sem_set_value (semid, number, val)

int semid, number, val;

{ extern int errno;
  int ret;

#if defined(SUN4) || defined(SGI)
  union semun arg;
#endif 

#if defined(SUN4) || defined(SGI)
  arg.val = val;
  ret = semctl (semid, number, SETVAL, arg); 
#else
  ret = semctl (semid, number, SETVAL, val); 
#endif

  if (ret < 0)

    { dalib_internal_error ("could not set value of semaphore");
      printf ("sem = %d, number = %d, val = %d, error = %d\n",
               semid, number, val, errno);
      dalib_stop ();
    }

} /* sem_set_value */

/**************************************************************************
*                                                                         *
*   int sem_allocate (number)                                             *
*                                                                         *
*   - allocation of number semaphores whose value is initialized          *
*     with init                                                           *
*                                                                         *
**************************************************************************/

int sem_allocate (number)

int number;

{ int sem_id;
  int sem_flag;

  extern int errno;

  sem_flag = 256*255 + 255;
  sem_id   = semget (IPC_PRIVATE, number, sem_flag);

  if (sem_id == -1)
   { dalib_internal_error ("creating semaphores, semget failed");
     if (errno == ENOSPC)
        printf ("system-imposed limit on the maximum number exceeded\n");
     if (errno == ENOENT) 
        printf ("key not existent\n");
     if (errno == EINVAL) 
        printf ("number %d out of range\n", number);
     if (errno == EACCES) 
        printf ("access problems\n");
     dalib_stop ();
   }

#ifdef DEBUG
  printf ("semaphores allocated\n");
#endif

  return (sem_id);

} /* sem_allocate */

/**************************************************************************
*                                                                         *
*  void sem_deallocate (int semid)                                        *
*                                                                         *
*  - gives semaphores free                                                *
*                                                                         *
**************************************************************************/

void sem_deallocate (int semid)

{ extern int errno;
  char msg[128];
  int semrtn;
 
#ifdef DEBUG
    printf ("deallocate semaphores, semid = %d\n", semid);
#endif 

  semrtn = semctl (semid, 0, IPC_RMID, NULL);

  if (semrtn < 0)
   { sprintf (msg, "sem_deallocate fails, errno = %d\n", errno);
     dalib_internal_error (msg);
     dalib_stop ();
   }
 
}  /* sem_deallocate */

/**************************************************************************
*                                                                         *
*  void sem_lock (int semid, int num)                                     *
*                                                                         *
*   - set semaphore to 0 (but wait in any case until it is one            *
*                                                                         *
**************************************************************************/

void sem_lock (semid, num)
int semid, num;

{ /* set semaphore to zero if one or wait for one */
 
   struct sembuf op;
   int status;
   extern int errno;
   int nsops;
 
   op.sem_num = num;
   op.sem_op = -1;
   op.sem_flg = SEM_UNDO;
 
   nsops = 1;   /* number of operations */

   status = semop (semid, &op, nsops);
 
   if (status == -1)

    { dalib_internal_error ("could not lock semaphore");
      printf ("semaphore = %d, num = %d, error status = %d\n", 
               semid, num, errno);
      dalib_stop ();
    }

} /* sem_lock */

/**************************************************************************
*                                                                         *
*  void sem_unlock (int semid, int num)                                   *
*                                                                         *
*   - set semaphore back to 1                                             *
*                                                                         *
**************************************************************************/

void sem_unlock (semid, num)
int semid, num;

{ /* set semaphore to zero if one or wait for one */
 
   struct sembuf op;
   int status;
   extern int errno;
 
   op.sem_num = num;
   op.sem_op  = 1;
   op.sem_flg = SEM_UNDO;
 
   status = semop (semid, &op, 1);
 
   if (status == -1)

    { dalib_internal_error ("could not unlock semaphore");
      printf ("semaphore = %d, num = %d, error status = %d\n", 
               semid, num, errno);
      dalib_stop ();
    }

} /* sem_unlock */

void sem_op (semid, num, val)
int semid, num, val;

{ /* general operation */
 
   struct sembuf op;
   int status;
   extern int errno;
 
   op.sem_num = num;
   op.sem_op  = val;
   op.sem_flg = SEM_UNDO;
 
   status = semop (semid, &op, 1);
 
   if (status == -1)

    { dalib_internal_error ("could not operate on semaphore");
      printf ("semaphore = %d, num = %d, val = %d, error status = %d\n", 
               semid, num, val, errno);
      dalib_stop ();
    }

} /* sem_unlock */

/**************************************************************************
*                                                                         *
*  void sem_barrier (int semid, int num)                                  *
*                                                                         *
*   - waits until all processes have entered the barrier                  *
*                                                                         *
**************************************************************************/

void sem_barrier (semid)
int semid;

{  extern int errno;
 
   int np;

#ifdef DEBUG
   printf ("%d: enters barrier 0\n", pcb.i);
#endif

   np = pcb.p;

   if (pcb.i == 1)
      sem_set_value (semid, 0, np);

   sem_op (semid, 0, -1);   /* pass barrier 0      */
   sem_op (semid, 0, 0);    /* wait for all passed */

#ifdef DEBUG
   printf ("%d: enters barrier 1\n", pcb.i);
#endif

   /* needs a second barrier to make sure that value of semaphore
      will not be set before all processes have passed the first barrier */

   if (pcb.i == 1)
      sem_set_value (semid, 1, np);

   sem_op (semid, 1, -1);   /* pass barrier 1      */
   sem_op (semid, 1, 0);    /* wait for all passed */

#ifdef DEBUG
   printf ("%d: leaves barrier\n", pcb.i);
#endif

} /* sem_barrier */
 
/**************************************************************************
*                                                                         *
*   dalib_define_barrier ()                                               *
*                                                                         *
**************************************************************************/

void dalib_define_barrier ()

{ /* creates a semaphore for barriers */

  int sendpid, size;

  sendpid = 1;

  if (pcb.i == sendpid)

    { barrier_id = sem_allocate (2);  /* needs 2 semaphores */
      sem_set_value (barrier_id, 0, 0);
      sem_set_value (barrier_id, 1, 0);

#ifdef DEBUG
  printf ("barrier defined, semid = %d, barval = %d\n", barrier_id, val);
#endif
    }

  /* broadcast barrier_id to all processors */

  size    = sizeof(barrier_id);

  FUNCTION(dalib_broadcast) (&barrier_id, &size, &sendpid); 
 
} /* dalib_define_barrier */

/**************************************************************************
*                                                                         *
*   dalib_define_barrier ()                                               *
*                                                                         *
**************************************************************************/

void dalib_free_barrier ()

{ /* synchronize before the semaphore is freed */
 
  FUNCTION(dalib_barrier) (); 

  if (pcb.i == 1)
     sem_deallocate (barrier_id);

} /* dalib_free_barrier */

/**************************************************************************
*                                                                         *
* FUNCTION(dalib_sem_barrier) () * 
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_sem_barrier) () 

{ 
  /* use number 0 of global semaphore barrier_id */

  sem_barrier (barrier_id);
}

#else

/**************************************************************************
*                                                                         *
*   NO SUPPORT OF SYSYEM_V FEATURES !!!!!!!!!!!!!!!!!!                    *
*                                                                         *
**************************************************************************/

 /* make never calls for all the routines */

 void sem_never (routine)
 char *routine;

 { dalib_internal_error
     ("semaphores not supported on your system");
   printf ("do not use SHARED directive or -shared flag\n");
   dalib_stop ();
 }

 int sem_allocate ()

 { sem_never ("sem_allocate");
   return(0);
 }

 void sem_detach ()
 { sem_never ("sem_detach"); }

 void sem_lock ()
 { sem_never ("sem_lock"); }

 void sem_unlock ()
 { sem_never ("sem_unlock"); }

 void dalib_define_barrier ()
 { sem_never ("dalib_define_barrier"); }

 void dalib_free_barrier ()
 { sem_never ("dalib_define_barrier"); }

void FUNCTION(dalib_sem_barrier) () 
 { sem_never ("dalib_sem_barrier"); }

#endif
