/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*                                                                         *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Jan 95                                                   *
*  Last Update : Feb 95                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : section.c                                                *
*                                                                         *
*  Function: Creating descriptors for sections of arrays                  *
*                                                                         *
*  Export :  FORTRAN Interface                                            *
*  ===========================                                            *
*                                                                         *
*   void FUNCTION(dalib_section_create) (section_id, array_id,            *
*                               kind1, lb1, ub1, inc1,                    *
*                               ...                                       *
*                               kind7, lb7, ub7, inc7)                    *
*                                                                         *
*   void FUNCTION(dalib_section_point) (section_id, array_id,             *
*                           ind1, ind2, ind3, ind4, ind5, ind6, ind7)     *
*                                                                         *
*   void FUNCTION(dalib_section_free) (section_info *section_id)          *
*                                                                         *
*   void dalib_section_full (section_info *section_id,                    *
*                            array_info   array_id)                       *
*                                                                         *
*     input  : array_id                                                   *
*     output : section_id, exactly describing the full array section      *
*                                                                         *
*   void dalib_section_reset (section_info section_id)                    *
*                                                                         *
*     - compute again the local size by global size                       *
*                                                                         *
*   int dalib_section_rank (section_info section_id)                      *
*                                                                         *
*   void dalib_section_global_shape (section_info section_id, int *rank,  *
*                                    int lb[], int ub[], int stride[] )   *
*                                                                         *
*    - computes global shape, e.g. A(3:5,2,6:1:-1,1:7:3) -> 3, [3,6,3]    *
*                                                                         *
*   void dalib_section_local_bounds (section_dsp, dim, *lb, *ub, *inc)    *
*                                                                         *
*   void dalib_section_local_shape (section_info section_id,              *
*                                   int *rank, int lb[], int ub[])        *
*                                                                         *
*    - computes rank as well as lower/upper bounds of local section part  *
*    - this routine fails if there is any stride                          *
*                                                                         *
*   int dalib_is_section_info (section_info section_id)                   *
*                                                                         *
*     -  PREDICATE for testing array info                                 *
*                                                                         *
*   void dalib_print_section_info (section_info section_id)               *
*                                                                         *
*     -  PRINTING section information record                              *
*                                                                         *
*   Updates                                                               *
*   =======                                                               *
*                                                                         *
*    - packing/unpacking of section descriptors                           *
*                                                                         *
***************************************************************************/

#include <stdio.h>
#include "dalib.h"

#undef DEBUG
#define CHECK

/**************************************************************************
*                                                                         *
*  int dalib_is_section_info (section_info section_id)                    *
*                                                                         *
*    -  PREDICATE for testing array info                                  *
*                                                                         *
**************************************************************************/
 
int dalib_is_section_info (section_id)
section_info section_id;
 
{  if (section_id == (section_info) 0) return (0);
   return (section_id->ident == SECTION_IDENTIFICATION);
} /* dalib_is_array_info */

/**************************************************************************
*                                                                         *
*  void dalib_print_section_info (section_info section_id)                *
*                                                                         *
*  PRINTING section information record                                    *
*                                                                         *
**************************************************************************/

void dalib_print_section_info (section_id)
section_info section_id;

{ int i, rank;
  array_info array_id;

  SecDimInfo  *sdims;

#ifdef CHECK
  if (!dalib_is_section_info (section_id))
    { dalib_internal_error ("print_section_info : not a section");
      dalib_stop ();
    }
#endif 

  array_id = section_id->array_id;
  rank = array_id->rank;
  fprintf (stdout, "%d: section (dsp=%d,rank=%d) is ", 
           pcb.i, section_id, rank);

  sdims = section_id->dimensions;

  fprintf (stdout, " global (");
  for (i=0; i<rank; i++)
    { if (sdims->is_range)
         fprintf (stdout,"%d:%d:%d", sdims->global_range[0], 
                  sdims->global_range[1], sdims->global_range[2]);
       else
         fprintf (stdout,"%d", sdims->global_range[0]);
      if (i<rank-1) fprintf (stdout,",");
      sdims++;
    }
  fprintf (stdout, ") ");

  sdims = section_id->dimensions;
  fprintf (stdout, " local (");
  for (i=0; i<rank; i++)
    { fprintf (stdout,"%d:%d:%d", sdims->local_range[0],
                       sdims->local_range[1], sdims->local_range[2]);
      if (i<rank-1) fprintf (stdout,",");
      sdims++;
    }
  fprintf (stdout, ")\n");

} /* dalib_print_section_info */

/**************************************************************************
*                                                                         *
*  void dalib_print_section_global (section_id)                           *
*                                                                         *
*    - prints global information of a section/array                       *
*                                                                         *
*      section [3:5,3] of P [2:4,3:5]                                     *
*                                                                         *
**************************************************************************/

void dalib_print_section_global (section_id)

section_info section_id;

{ array_info array_id;
  int        i, rank;

  DimInfo    *array_dims;
  SecDimInfo *section_dims;

  array_id = section_id->array_id;
  rank     = array_id->rank;

  section_dims = section_id->dimensions;
  array_dims   = array_id->dimensions;

  fprintf (stdout, "%d: section [", pcb.i);

  for (i=0; i<rank; i++, section_dims++)

    { if (!section_dims->is_range)

         fprintf (stdout,"%d", section_dims->global_range[0]);

       else if (section_dims->global_range[2] == 1)

         fprintf (stdout,"%d:%d", section_dims->global_range[0], 
                  section_dims->global_range[1]);
       else

         fprintf (stdout,"%d:%d:%d", 
                  section_dims->global_range[0], 
                  section_dims->global_range[1], 
                  section_dims->global_range[2]);

      if (i<rank-1) fprintf (stdout,",");
    }

  fprintf (stdout, "] of %s [", array_id->name);

  for (i=0; i<rank; i++, array_dims++)

    { fprintf (stdout,"%d:%d", array_dims->global_size[0],
                  array_dims->global_size[1]);
      if (i<rank-1) fprintf (stdout,",");
    }

  fprintf (stdout, "]\n");

} /* dalib_print_section_global */

/**************************************************************************
*                                                                         *
*  HELP FUNCTIONS                                                         *
*                                                                         *
**************************************************************************/

static void dalib_set_secdim_info (dim, kind, lb, ub, stride)

SecDimInfo *dim;
int kind, lb, ub, stride;

{ if (kind == 0)

    { dim->is_range = 0;
      dim->global_range[0] = lb;
      dim->global_range[1] = lb;
      dim->global_range[2] = 1;
    }

   else 

    { dim->is_range = 1;
      dim->global_range[0] = lb;
      dim->global_range[1] = ub;
      dim->global_range[2] = stride;
    }

} /* dalib_set_secdim_info */

static void dalib_set_secdim_point (dim, val)

SecDimInfo *dim;
int val;

{
  dim->is_range = 0;
  dim->global_range[0] = val;
  dim->global_range[1] = val;
  dim->global_range[2] = 1;

} /* dalib_set_secdim_point */

static void dalib_local_section_size (section, sec_dim, array_dim)

section_info  section;
SecDimInfo    *sec_dim;
DimInfo       *array_dim;

{ int *array_global, *sec_global, *overlap;
  int size;

  array_global = array_dim->global_size;
  overlap      = array_dim->overlap;
  sec_global   = sec_dim->global_range;

  size = dalib_range_size (sec_global[0], sec_global[1], 
                           sec_global[2]);

  /* set upper bound correctly, e.g. 3:N+1:6 of 1:N can be correct */

  if (size > 0)
     sec_global[1] = sec_global[0] + (size-1) * sec_global[2];

#ifdef CHECK

  /* attention:  right overlap on left side might be possible */

  if (   (sec_global[0] < array_global[0] - overlap[1] ) 
      || (sec_global[0] > array_global[1] + overlap[0])
      || (sec_global[1] < array_global[0] - overlap[1])
      || (sec_global[1] > array_global[1] + overlap[0]))

    { /* error only if section is not empty */

      if (size > 0)
        { dalib_internal_error ("section out of global array range");
          dalib_print_section_global (section);
          dalib_stop ();
       }
    }
#endif

  /* local range = global section * local size    */

  dalib_intersect_sections (sec_global, array_dim->local_size,
                            sec_dim->local_range);

} /* dalib_local_section_size */

/**************************************************************************
*                                                                         *
*  Functions for defining section descriptors                             *
*                                                                         *
*  void FUNCTION(dalib_section_create) (section_id, array_id,             *
*                               kind1, lb1, ub1, inc1,                    *
*                               ...                                       *
*                               kind7, lb7, ub7, inc7)                    *
*                                                                         *
*    - returns an internal descriptor for a section                       *
*                                                                         *
**************************************************************************/

static int dalib_section_dsp_size (rank)
int rank;

{ int save;

  save = (MAX_DIMENSIONS - rank) * sizeof (SecDimInfo);

  return sizeof (struct SectionRecord) - save;

} /* dalib_section_dsp_size */

void FUNCTION(dalib_section_create) (section_id, array_id, 
                            kind1, lb1, ub1, str1, kind2, lb2, ub2, str2, 
                            kind3, lb3, ub3, str3, kind4, lb4, ub4, str4, 
                            kind5, lb5, ub5, str5, kind6, lb6, ub6, str6, 
                            kind7, lb7, ub7, str7)

section_info *section_id;
array_info *array_id;

int *kind1, *lb1, *ub1, *str1, *kind2, *lb2, *ub2, *str2, 
    *kind3, *lb3, *ub3, *str3, *kind4, *lb4, *ub4, *str4, 
    *kind5, *lb5, *ub5, *str5, *kind6, *lb6, *ub6, *str6, 
    *kind7, *lb7, *ub7, *str7;

{ section_info   descriptor;
  SecDimInfo     *sdims;
  DimInfo        *adims;

  int i, rank;

#ifdef CHECK
  if (!dalib_is_array_info (*array_id))
     { dalib_internal_error ("section_create, not array info");
       dalib_stop ();
     }
#endif

  rank = (*array_id)->rank;

  descriptor = (section_info)
    dalib_malloc (dalib_section_dsp_size (rank), "section_create");

  descriptor->ident    = SECTION_IDENTIFICATION;
  descriptor->array_id = (*array_id);

  sdims = descriptor->dimensions;
  adims = (*array_id)->dimensions;

  switch (rank) {

     case 7 : dalib_set_secdim_info (sdims + 6, *kind7, *lb7, *ub7, *str7);
     case 6 : dalib_set_secdim_info (sdims + 5, *kind6, *lb6, *ub6, *str6);
     case 5 : dalib_set_secdim_info (sdims + 4, *kind5, *lb5, *ub5, *str5);
     case 4 : dalib_set_secdim_info (sdims + 3, *kind4, *lb4, *ub4, *str4);
     case 3 : dalib_set_secdim_info (sdims + 2, *kind3, *lb3, *ub3, *str3);
     case 2 : dalib_set_secdim_info (sdims + 1, *kind2, *lb2, *ub2, *str2);
     case 1 : dalib_set_secdim_info (sdims + 0, *kind1, *lb1, *ub1, *str1);

  } /* switch */

  /* now set the local size of the section */

  for (i=0; i<rank; i++)
      dalib_local_section_size (descriptor, sdims+i, adims+i);

  *section_id = descriptor;

#ifdef DEBUG
  printf ("%d: section descriptor %d of rank %d created, array_dsp = %d\n", 
           pcb.i, descriptor, rank, *array_id);
  dalib_print_section_info (*section_id);
#endif

} /* FUNCTION(dalib_section_create) */ 

/**************************************************************************
*                                                                         *
*  void FUNCTION(dalib_section_point) (section_id, array_id,              *
*                           ind1, ind2, ind3, ind4, ind5, ind6, ind7)     *
*                                                                         *
*    - returns an internal descriptor for a section point                 *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_section_point) (section_id, array_id,
                            ind1, ind2, ind3, ind4, ind5, ind6, ind7)

section_info *section_id;
array_info *array_id;

int *ind1, *ind2, *ind3, *ind4, *ind5, *ind6, *ind7;

{ section_info   descriptor;
  SecDimInfo     *sdims;
  DimInfo        *adims;

  int i, save, rank;

#ifdef CHECK
  if (!dalib_is_array_info (*array_id))
     { dalib_internal_error ("section_create, not array info");
       dalib_stop ();
     }
#endif

  rank = (*array_id)->rank;

  save = (MAX_DIMENSIONS - rank) * sizeof (SecDimInfo);

  descriptor = (section_info)
    dalib_malloc (sizeof (struct SectionRecord) - save, "section_create");

  descriptor->ident    = SECTION_IDENTIFICATION;
  descriptor->array_id = (*array_id);

  sdims = descriptor->dimensions;
  adims = (*array_id)->dimensions;

  switch (rank) {

     case 7 : dalib_set_secdim_point (sdims + 6, *ind7);
     case 6 : dalib_set_secdim_point (sdims + 5, *ind6);
     case 5 : dalib_set_secdim_point (sdims + 4, *ind5);
     case 4 : dalib_set_secdim_point (sdims + 3, *ind4);
     case 3 : dalib_set_secdim_point (sdims + 2, *ind3);
     case 2 : dalib_set_secdim_point (sdims + 1, *ind2);
     case 1 : dalib_set_secdim_point (sdims + 0, *ind1);

  } /* switch */

  /* now set the local size of the section */

  for (i=0; i<rank; i++)
      dalib_local_section_size (descriptor, sdims+i, adims+i);

  *section_id = descriptor;

#ifdef DEBUG
  printf ("%d: section point %d of rank %d created, array_dsp = %d\n",
           pcb.i, descriptor, rank, *array_id);
  dalib_print_section_info (*section_id);
#endif

} /* FUNCTION(dalib_section_point) */

/**************************************************************************
*                                                                         *
*   void FUNCTION(dalib_section_free) (section_info *section_id)          *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_section_free) (section_id) 

section_info *section_id;

{ section_info descriptor;

#ifdef CHECK
  if (!dalib_is_section_info (*section_id))
    { dalib_internal_error ("section_free : not a section");
      dalib_stop ();
    }
#endif
 
  descriptor = *section_id;

  dalib_free (descriptor, dalib_section_dsp_size (descriptor->array_id->rank));

} /* dalib_section_free */

/**************************************************************************
***************************************************************************
**                                                                       **
**  Functions for defining section descriptors                           **
**                                                                       **
***************************************************************************
**************************************************************************/

/**************************************************************************
*                                                                         *
*   void dalib_section_full (section_info *section_id,                    *
*                            array_info   array_id)                       *
*                                                                         *
*   input  : array_id                                                     *
*   output : section_id, exactly describing the full array section        *
*                                                                         *
**************************************************************************/

void dalib_section_full (section_id, array_id)

section_info *section_id;
array_info array_id;

{ section_info   descriptor;
  SecDimInfo     *sdims;
  DimInfo        *adims;

  int i, save, rank;

  if (dalib_is_section_info ((section_info) array_id))

    { *section_id = (section_info) array_id; 
      return;
    }

#ifdef CHECK
  if (!dalib_is_array_info (array_id))
     { dalib_internal_error ("section_full, not array/section info");
       dalib_stop ();
     }
#endif

  rank = array_id->rank;

  save = (MAX_DIMENSIONS - rank) * sizeof (SecDimInfo);

  descriptor = (section_info)
    dalib_malloc (sizeof (struct SectionRecord) - save, "section_create");

  descriptor->ident    = SECTION_IDENTIFICATION;
  descriptor->array_id = array_id;

  sdims = descriptor->dimensions;
  adims = array_id->dimensions;

  for (i=0; i<rank; i++)

   { sdims->is_range = 1;

     sdims->global_range[0] = adims->global_size[0];
     sdims->global_range[1] = adims->global_size[1];
     sdims->global_range[2] = 1;

     sdims->local_range[0] = adims->local_size[0];
     sdims->local_range[1] = adims->local_size[1];
     sdims->local_range[2] = adims->local_size[2];

     sdims++; adims ++;

   } /* for all dimensions */

  *section_id = descriptor;

} /* dalib_section_full */

/**************************************************************************
*                                                                         *
*  void dalib_section_reset (section_info section_id)                     *
*                                                                         *
*     - compute again the local size by global size                       *
*                                                                         *
**************************************************************************/

void dalib_section_reset (section_id)

section_info section_id;

{ array_info array_id;
  int i, rank;

  SecDimInfo     *sdims;
  DimInfo        *adims;

#ifdef CHECK
  if (!dalib_is_section_info (section_id))
    { dalib_internal_error ("section_reset : not a section");
      dalib_stop ();
    }
#endif
 
#ifdef DEBUG
  printf ("dalib_section_reset\n");
  dalib_print_section_info (section_id);
#endif

  array_id = section_id->array_id;
  rank     = array_id->rank;

  sdims = section_id->dimensions;
  adims = array_id->dimensions;

  for (i=0; i<rank; i++)
      dalib_local_section_size (section_id, sdims+i, adims+i);

#ifdef DEBUG
  printf ("dalib_section_reset done\n");
  dalib_print_section_info (section_id);
#endif

} /* dalib_section_reset */

/**************************************************************************
*                                                                         *
*   int dalib_section_rank (section_info section_id)                      *
*                                                                         *
**************************************************************************/

int dalib_section_rank (section_id)
 
section_info section_id;
 
{ int i, arr_rank, sec_rank;
  SecDimInfo *dims;
 
#ifdef CHECK
  if (!dalib_is_section_info (section_id))
    { dalib_internal_error ("section_rank : not a section");
      dalib_stop ();
    }
#endif

  arr_rank = section_id->array_id->rank;
  dims     = section_id->dimensions;
 
  sec_rank = 0;
 
  for (i=0; i < arr_rank; i++)
 
   { if (dims->is_range) sec_rank ++;
     dims++;
   }
 
  return (sec_rank);
 
}  /* dalib_section_rank */

/**************************************************************************
*                                                                         *
*   void dalib_section_global_shape (section_info section_id, int *rank,  *
*                                    int lb[], int ub[], int stride[] )   *
*                                                                         *
*    - computes global shape, e.g. A(3:5,2,6:1:-1,1:7:3) -> 3, [3,6,3]    *
*                                                                         *
**************************************************************************/

void dalib_section_global_shape (section_id, rank, lb, ub, str)

section_info section_id;
int *rank;
int lb[], ub[], str[];

{ int i, array_rank, section_rank;
  SecDimInfo *dims;

#ifdef CHECK
  if (!dalib_is_section_info (section_id))
    { dalib_internal_error ("section_global_shape : not a section");
      dalib_stop ();
    }
#endif

  array_rank = section_id->array_id->rank;
  dims = section_id->dimensions;
 
  section_rank = 0;
 
  for (i=0; i<array_rank; i++, dims++)
 
   { if (dims->is_range)

        { lb[section_rank]  = dims->global_range[0];
          ub[section_rank]  = dims->global_range[1];
          str[section_rank] = dims->global_range[2];
          section_rank ++;

        }
   }

   *rank = section_rank;

} /* dalib_section_global_shape */

/**************************************************************************
*                                                                         *
*   int dalib_section_local_size (section_id)                             *
*                                                                         *
*    - returns number of elements in the local part of the section        *
*                                                                         *
**************************************************************************/

int dalib_section_local_size (section_id)
 
section_info section_id;
 
{ int i, rank;
  SecDimInfo *dims;
 
  int low, high, stride, size;
 
#ifdef CHECK
  if (!dalib_is_section_info (section_id))
    { dalib_internal_error ("section_local_size : not a section");
      dalib_stop ();
    }
#endif

  rank = section_id->array_id->rank;
  dims = section_id->dimensions;
 
  size = 1;
 
  for (i=0; i<rank; i++)
 
   { low    = dims->local_range[0];
     high   = dims->local_range[1];
     stride = dims->local_range[2];
     size *= dalib_range_size (low, high, stride);
     dims++;
   }
 
  return (size);
 
}  /* dalib_section_local_size */

/**************************************************************************
*                                                                         *
*  void dalib_section_array_dim (section_id, dim)                         *
*                                                                         *
*   e.g. section_array_dim (A(2,3:5,1:4,5), 2) ==> 3                      *
*                                                                         *
**************************************************************************/

int dalib_section_array_dim (section_id, dim)

section_info section_id;
int dim;

{ int array_dim;
  int sec_dim;
  int i, rank;

#ifdef CHECK
  if (!dalib_is_section_info (section_id))
    { dalib_internal_error ("section_array_dim : not a section");
      dalib_stop ();
    }
#endif

  rank      = section_id->array_id->rank;
  sec_dim   = 0;
  array_dim = 0;
  
  for (i = 0; i < rank; i++)

    { if (section_id->dimensions[i].is_range)
 
        { sec_dim++;
          if (sec_dim == dim) array_dim = i + 1;
        }
    }

  if (array_dim == 0) 

     { dalib_internal_error ("dalib_section_array_dim");
       printf ("this section causes the error : \n");
       dalib_print_section_info (section_id);
       printf ("this is the array : ");
       dalib_print_array_info (section_id->array_id);
       printf ("section dim %d could not be found for array rank %d\n",
                dim, rank);
       dalib_stop ();
     }

  return (array_dim);

} /* dalib_section_array_dim */

/**************************************************************************
*                                                                         *
*   void dalib_section_local_bounds (section_dsp, dim, *lb, *ub, *inc)    *
*                                                                         *
**************************************************************************/
 
void dalib_section_local_bounds (section_id, dim, lb, ub, inc)

section_info section_id;
int dim;
int *lb, *ub, *inc;

{ SecDimInfo *dims;
  int *range;
  int array_dim;

  array_dim = dalib_section_array_dim (section_id, dim);
 
  dims  = section_id->dimensions;
  range = dims[array_dim-1].local_range;

  *lb  = range[0];
  *ub  = range[1];
  *inc = range[2];

} /* dalib_section_local_bounds */

/****************************************************************************
*                                                                           *
*  void dalib_section_local_shape (section_info section_id,                 *
*                                  int *rank, int lb[], int ub[])           *
*                                                                           *
*  - computes rank as well as lower and upper bounds of local section part  *
*  - this routine fails if there is any stride                              *
*                                                                           *
****************************************************************************/

void dalib_section_local_shape (section_id, rank, lb, ub)

section_info section_id;

int *rank;
int lb[], ub[];

{ int i, array_rank, section_rank;
  SecDimInfo *dims;
 
#ifdef CHECK
  if (!dalib_is_section_info (section_id))
    { dalib_internal_error ("section_local_shape : not a section");
      dalib_stop ();
    }
#endif
 
  array_rank = section_id->array_id->rank;
  dims       = section_id->dimensions;
 
  section_rank = 0;

  for (i=0; i<array_rank; i++, dims++)
 
   if (dims->is_range)
 
     { section_rank ++;
       lb[i] = dims->local_range[0];
       ub[i] = dims->local_range[1];
       if (dims->local_range[2] != 1)
        { dalib_internal_error ("section_local_shape: stride in section");
          dalib_print_section_info (section_id);
          dalib_stop ();
        }
     }
 
  *rank = section_rank;
 
} /* dalib_section_local_shape */

/****************************************************************************
*                                                                           *
*  int dalib_is_full_array_section (section_info section_id)                *
*                                                                           *
*  - returns true if addressing of array can also be used for the section   *
*  - now section must be A(lb1:ub1,...,lbn:ubn)                             *
*                                                                           *
****************************************************************************/

int dalib_is_full_array_section (section_id)

section_info section_id;

{ int okay;
  int i, rank;

  SecDimInfo *dims;

  okay = 1;
  dims = section_id->dimensions;

  rank = section_id->array_id->rank;

  for (i=0; i<rank; i++, dims++)

    { if (!dims->is_range) return (0);
      if (dims->global_range[2] != 1) return (0);
    }

  return (1);

} /* dalib_is_full_array_section */

/**************************************************************************
*                                                                         *
*  void dalib_pack_section (char *buffer, int rank,                       *
*                           section_info section_dsp => int length)       *
*                                                                         *
*  void dalib_unpack_section (char *buffer, int rank,                     *
*                             => section_info section_dsp, int length)    *
*                                                                         *
**************************************************************************/

     /*************************************************************
     *  dalib_pack_section (buffer, rank, section_dsp => length)  *
     *************************************************************/

void dalib_pack_section (buffer, rank, section_dsp, length)

char         *buffer;
int          rank;
section_info section_dsp;
int          *length;

{ /* pack section descriptor */

  *length = dalib_section_dsp_size (rank);
  dalib_memcopy (buffer, section_dsp, *length);

} /* dalib_pack_section */

     /***************************************************************
     *  dalib_unpack_section (buffer, rank => section_dsp, length)  *
     ***************************************************************/

void dalib_unpack_section (buffer, rank, section_dsp, length)

char         buffer[];
int          rank;
section_info *section_dsp;
int          *length;

{ int          sec_len;
  section_info sec_id;

  sec_len = dalib_section_dsp_size (rank);

  sec_id = (section_info) dalib_malloc (sec_len, "unpack_section");

  dalib_memcopy (sec_id, buffer, sec_len);

  *section_dsp = sec_id;
  *length      = sec_len;

} /* dalib_unpack_section */

