/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Apr 95                                                   *
*  Last Update : Mar 97                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : scan.m4                                                  *
*                                                                         *
*  Function    : HPF intrinsics sum_prefix, ....                          *
*                                                                         *
*  Export :  FORTRAN Interface                                            *
*                                                                         *
**************************************************************************/

#define CHECK
#undef DEBUG

#include "dalib.h"

/* IMPORT from combiners */

extern dalib_routine *dalib_get_reduction_fn ();

/*******************************************************************
*                                                                  *
*  global data for reductions                                      *
*                                                                  *
*******************************************************************/

static dalib_routine    *f_reduction;

static int array_rank;
static int array_shape[7];

static int   array_no;
static int   array_size;
static int   array_new,   result_new;
static char  *array_data, *result_data;

static int   is_masked;
static int   mask_new;
static int   *mask_data;

static int   is_segmented;
static int   segment_new;
static int   *segment_data;
static int   segment_last;

static int  is_exclusive;

static int  scan_dimension; 

static char *zero_ptr;
static char help [16];

static void dalib_scan_copy (target, source)
char *target, *source;

{ printf ("scan copy\n");
  dalib_memcopy (target, source, array_size);
}

/*******************************************************************
*                                                                  *
*  dalib_set_array (array_info *array_dsp)                         *
*                                                                  *
*    - set globally array_xxx                                      *
*                                                                  *
*******************************************************************/

static void dalib_set_array (array_dsp)

array_info *array_dsp;

{ int i;

  if (dalib_secarray_is_mapped (array_dsp))

     { dalib_internal_error ("PREFIX/SUFFIX : array is mapped (unsupported)");
       dalib_stop ();
     }

  dalib_secarray_get_data (*array_dsp, 1, 
                    &array_no, &array_size, &array_data, &array_new);

  dalib_secarray_shape (array_dsp, &array_rank, array_shape);

#ifdef DEBUG
  printf ("PREFIX/SUFFIX : set array, rank = %d\n", array_rank);
  printf ("PREFIX/SUFFIX :     shape = (");
  for (i=0; i < array_rank; i++)
     printf ("%d ", array_shape[i]);
  printf ("), no = %d, size = %d, new = %d\n", array_no, array_size, array_new);
#endif

} /* dalib_set_array */

/*******************************************************************
*                                                                  *
*  dalib_set_op (op)                                               *
*                                                                  *
*    - sets globally reduction_fn                                  *
*                                                                  *
*******************************************************************/

static void dalib_set_op (op)
int op;

{  int elem_size;

   if (op == 0)

     { f_reduction = dalib_scan_copy; 
       return;
     }

   f_reduction = dalib_get_reduction_fn (op);
   elem_size   = dalib_get_reduction_size (op);

   if (elem_size != array_size)

       { dalib_internal_error ("PREFIX/SUFFIX: op mismatch");
         dalib_stop ();
       }

} /* dalib_set_op */

/*******************************************************************
*                                                                  *
*  dalib_set_result (array_info *result_dsp)                       *
*                                                                  *
*    - set globally result_xxx                                     *
*                                                                  *
*******************************************************************/

static void dalib_set_result (result_dsp)

array_info *result_dsp;

{ int result_no, result_size;

  if (dalib_secarray_is_mapped (result_dsp))

     { dalib_internal_error ("PREFIX/SUFFIX : result is mapped (unsupported)");
       dalib_stop ();
     }

  /* no old values of result needed */

  dalib_secarray_get_data (*result_dsp, 0, 
                    &result_no, &result_size, &result_data, &result_new);

#ifdef DEBUG
  printf ("PREFIX/SUFFIX : set result, no = %d, size = %d, new = %d\n",
           result_no, result_size, result_new);
#endif

  /* ASSERT : result_no   == array_no   */
  /* ASSERT : result_size == array_size */

} /* dalib_set_result */

/*******************************************************************
*                                                                  *
*  dalib_set_mask (int *mask, array_info *mask_dsp)                *
*                                                                  *
*    - set globally mask_xxx                                       *
*                                                                  *
*******************************************************************/

static void dalib_set_mask (mask, mask_dsp)

int        *mask;
array_info *mask_dsp;

{ int mask_no, mask_size;

  if (FUNCTION(dalib_present) (mask))
     is_masked = 1;
   else 
     is_masked = 0;

#ifdef DEBUG
  printf ("PREFIX/SUFFIX : is_masked = %d\n", is_masked);
#endif

  if (!is_masked) return;

  if (dalib_secarray_is_mapped (mask_dsp))

     { dalib_internal_error ("PREFIX/SUFFIX : mask is mapped (unsupported)");
       dalib_stop ();
     }

  dalib_secarray_get_data (*mask_dsp, 1, 
                    &mask_no, &mask_size, &mask_data, &mask_new);

#ifdef DEBUG
  printf ("PREFIX/SUFFIX : set mask, no = %d, size = %d, new = %d\n",
           mask_no, mask_size, mask_new);
#endif

  /* ASSERT : mask_no   == array_no        */
  /* ASSERT : mask_size == sizeof(INTEGER) */

} /* dalib_set_mask */

/*******************************************************************
*                                                                  *
*  dalib_set_segment (int *segment, array_info *segment_dsp)       *
*                                                                  *
*    - set globally segment_xxx                                    *
*                                                                  *
*******************************************************************/

static void dalib_set_segment (segment, segment_dsp)

int        *segment;
array_info *segment_dsp;

{ int segment_no, segment_size;

  if (FUNCTION(dalib_present) (segment))
     is_segmented = 1;
   else 
     is_segmented = 0;

#ifdef DEBUG
  printf ("PREFIX/SUFFIX : is_segmented = %d\n", is_segmented);
#endif

  if (!is_segmented) return;

  if (dalib_secarray_is_mapped (segment_dsp))

     { dalib_internal_error ("PREFIX/SUFFIX : mask is mapped (unsupported)");
       dalib_stop ();
     }

  dalib_secarray_get_data (*segment_dsp, 1, 
                    &segment_no, &segment_size, &segment_data, &segment_new);

#ifdef DEBUG
  printf ("dalib_prefix/suffix : set segment, no = %d, size = %d, new = %d\n",
           segment_no, segment_size, segment_new);
#endif

  /* ASSERT : segment_no   == array_no        */
  /* ASSERT : segment_size == sizeof(INTEGER) */

} /* dalib_set_segment */

/*******************************************************************
*                                                                  *
*                                                                  *
*******************************************************************/

static void dalib_set_exclusive (exclusive, exclusive_dsp)

int        *exclusive;
array_info *exclusive_dsp;

{ if (FUNCTION(dalib_present) (exclusive))
  
    { if (*exclusive)
        is_exclusive = 1;
      else
        is_exclusive = 0;
    }

   else

    is_exclusive = 0;   /* EXCLUSIVE = .false. by default */

#ifdef DEBUG
    printf ("PREFIX/SUFFIX : is_exclusive = %d\n", is_exclusive);
#endif

} /* dalib_set_exclusive */

/*******************************************************************
*                                                                  *
*                                                                  *
*******************************************************************/

static void dalib_set_dim (dim, dim_dsp)

int        *dim;
array_info *dim_dsp;

{ if (FUNCTION(dalib_present) (dim))

     { scan_dimension = *dim;

       if ((scan_dimension < 1) || (scan_dimension > array_rank))

         { dalib_internal_error ("PREFIX/SUFFIX : illegal dim value");
           printf ("scan dimension (DIM=%d) must be between 1 and %d",
                    scan_dimension, array_rank);
           dalib_stop ();
         }
     }

   else

     scan_dimension = 0;

#ifdef DEBUG
    printf ("PREFIX/SUFFIX : scan dimension = %d\n", scan_dimension);
#endif

    if (scan_dimension == 0) return;

    dalib_internal_error ("PREFIX/SUFFIX : scan along dim not supported");
    dalib_stop ();

} /* dalib_set_dim */

/*******************************************************************
*                                                                  *
*                                                                  *
*******************************************************************/

static void dalib_scan_end (result_dsp)
array_info *result_dsp;

{ if (result_new)

     dalib_secarray_unpack (*result_dsp, result_data);

  if (array_new) free (array_data);

  if (is_masked)

     { if (mask_new) free (mask_data); }

  if (is_segmented)

     { if (segment_new) free (segment_data); }

}  /* dalib_scan_end */

/*******************************************************************
*                                                                  *
*  PREFIX operation (with mask, segment, exclusive)                *
*                                                                  *
*******************************************************************/

static void dalib_do_prefix ()

{  char *r_ptr, *a_ptr;
   int i;

   r_ptr = result_data;
   a_ptr = array_data;

   dalib_memcopy (help, zero_ptr, array_size);

   segment_last = 0;

   for (i=0; i < array_no; i++)

      { /* segment[i] != segment_last then set new zero element */

        if (is_segmented)

           { if (segment_last != segment_data[i]) 
               { dalib_memcopy (help, zero_ptr, array_size);
                 segment_last = segment_data[i];
               }
           }

        /* exclusive = .true. then copy before reduction */

        if (is_exclusive) dalib_memcopy (r_ptr, help, array_size);

        /* mask[i] = .true. then no new reduction at all */
        if (is_masked)
           { if (mask_data[i]) f_reduction (help, a_ptr); }
          else
           f_reduction (help, a_ptr); 

        /* exclusive = .false. then copy after reductin */
        if (!is_exclusive) dalib_memcopy (r_ptr, help, array_size);

        r_ptr += array_size;
        a_ptr += array_size;
      }
      
} /* dalib_do_prefix */

/*******************************************************************
*                                                                  *
*                                                                  *
*******************************************************************/

static void dalib_do_copy_prefix ()

{  char *r_ptr, *a_ptr, *copy_ptr;
   int i;

   r_ptr = result_data;
   a_ptr = array_data;

   copy_ptr = a_ptr;

   segment_last = 0;

   for (i=0; i < array_no; i++)

      { /* segment[i] != segment_last then set new zero element */

        if (is_segmented)

           { if (segment_last != segment_data[i]) 
               { copy_ptr = a_ptr;
                 segment_last = segment_data[i];
               }
           }

        dalib_memcopy (r_ptr, copy_ptr, array_size);

        r_ptr += array_size;
        a_ptr += array_size;
      }
      
} /* dalib_do_copy_prefix */

/*******************************************************************
*                                                                  *
*                                                                  *
*******************************************************************/

static void dalib_do_copy_suffix ()

{  char *r_ptr, *a_ptr, *copy_ptr;
   int i;

   r_ptr = result_data + (array_no-1) * array_size;
   a_ptr = array_data  + (array_no-1) * array_size;

   copy_ptr = a_ptr;

   segment_last = 0;

   for (i=array_no-1; i >= 0; i--)

      { /* segment[i] != segment_last then set new zero element */

        if (is_segmented)

           { if (segment_last != segment_data[i]) 
               { copy_ptr = a_ptr;
                 segment_last = segment_data[i];
               }
           }

        dalib_memcopy (r_ptr, copy_ptr, array_size);

        r_ptr -= array_size;
        a_ptr -= array_size;
      }
      
} /* dalib_do_copy_suffix */

/*******************************************************************
*                                                                  *
*  SUFFIX operation (with mask, without segment)                   *
*                                                                  *
*******************************************************************/

static void dalib_do_suffix ()

{  char *r_ptr, *a_ptr;
   int i;

   r_ptr = result_data + (array_no-1) * array_size;
   a_ptr = array_data  + (array_no-1) * array_size;

   dalib_memcopy (help, zero_ptr, array_size);

   segment_last = 0;

   for (i=array_no-1; i >=0; i--)

      { /* segment[i] != segment_last then set new zero element */

        if (is_segmented)

           { if (segment_last != segment_data[i]) 
               { dalib_memcopy (help, zero_ptr, array_size);
                 segment_last = segment_data[i];
               }
           }

        /* exclusive = .true. then copy before reduction */

        if (is_exclusive) dalib_memcopy (r_ptr, help, array_size);

        /* mask[i] = .true. then no new reduction at all */
        if (is_masked)
           { if (mask_data[i]) f_reduction (help, a_ptr); }
          else
           f_reduction (help, a_ptr); 

        /* exclusive = .false. then copy after reductin */
        if (!is_exclusive) dalib_memcopy (r_ptr, help, array_size);

        r_ptr -= array_size;
        a_ptr -= array_size;
      }
      
} /* dalib_do_prefix */

/**************************************************************************
*                                                                         *
*   RESULT = SUM_PREFIX (ARRAY, MASK, DIM, SEGMENT, EXCLUSIVE)            *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_prefix_array) (op, zero_data, 

    result    , array    , dim     , mask    , segment     , exclusive     ,
    result_dsp, array_dsp, dim_dsp , mask_dsp, segment_dsp,  exclusive_dsp  )

int  *op;
char *zero_data;

char *result, *array;
int  *dim, *mask, *segment, *exclusive;
array_info *result_dsp, *array_dsp, *mask_dsp, 
           *dim_dsp, *segment_dsp, *exclusive_dsp;

{ dalib_set_array     (array_dsp);
  dalib_set_op        (*op);
  dalib_set_dim       (dim, dim_dsp);
  dalib_set_result    (result_dsp);
  dalib_set_mask      (mask, mask_dsp);
  dalib_set_segment   (segment, segment_dsp);
  dalib_set_exclusive (exclusive, exclusive_dsp);
  zero_ptr = zero_data;
  dalib_do_prefix ();
  dalib_scan_end (result_dsp);   

}  /* dalib_prefix_array */

/**************************************************************************
*                                                                         *
*   RESULT = ALL_PREFIX (MASK, DIM, SEGMENT, EXCLUSIVE)                   *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_prefix_mask) (op, zero_data, 

    result    , array    , dim     , segment     , exclusive     ,
    result_dsp, array_dsp, dim_dsp , segment_dsp,  exclusive_dsp  )

int  *op;
char *zero_data;

char *result, *array;
int  *dim, *segment, *exclusive;
array_info *result_dsp, *array_dsp, 
           *dim_dsp, *segment_dsp, *exclusive_dsp;

{ dalib_set_array     (array_dsp);
  dalib_set_op        (*op);
  dalib_set_dim       (dim, dim_dsp);
  dalib_set_result    (result_dsp);
  is_masked = 0;
  dalib_set_segment   (segment, segment_dsp);
  dalib_set_exclusive (exclusive, exclusive_dsp);
  zero_ptr = zero_data;
  dalib_do_prefix ();
  dalib_scan_end (result_dsp);   

}  /* dalib_prefix_mask */

/**************************************************************************
*                                                                         *
*   RESULT = COPY_PREFIX (ARRAY, DIM, SEGMENT)                            *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_prefix_copy) (

    result    , array    , dim     , segment    ,
    result_dsp, array_dsp, dim_dsp , segment_dsp)

char *result, *array;
int  *dim, *segment;
array_info *result_dsp, *array_dsp, 
           *dim_dsp, *segment_dsp;

{ dalib_set_array     (array_dsp);
  dalib_set_op        (0);
  
  dalib_set_dim       (dim, dim_dsp);
  dalib_set_result    (result_dsp);
  is_masked = 0;
  dalib_set_segment   (segment, segment_dsp);
  is_exclusive = 0;

  zero_ptr = help;    /* is never important */

  dalib_do_copy_prefix ();
  dalib_scan_end (result_dsp);   

}  /* dalib_prefix_copy */

/**************************************************************************
*                                                                         *
*   RESULT = SUM_SUFFIX (ARRAY, MASK, DIM, SEGMENT, EXCLUSIVE)            *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_suffix_array) (op, zero_data, 

    result    , array    , dim     , mask    , segment     , exclusive     ,
    result_dsp, array_dsp, dim_dsp , mask_dsp, segment_dsp,  exclusive_dsp  )

int  *op;
char *zero_data;

char *result, *array;
int  *dim, *mask, *segment, *exclusive;
array_info *result_dsp, *array_dsp, *mask_dsp, 
           *dim_dsp, *segment_dsp, *exclusive_dsp;

{ dalib_set_array     (array_dsp);
  dalib_set_op        (*op);
  dalib_set_dim       (dim, dim_dsp);
  dalib_set_result    (result_dsp);
  dalib_set_mask      (mask, mask_dsp);
  dalib_set_segment   (segment, segment_dsp);
  dalib_set_exclusive (exclusive, exclusive_dsp);
  zero_ptr = zero_data;
  dalib_do_suffix ();
  dalib_scan_end (result_dsp);   

}  /* dalib_suffix_array */

/**************************************************************************
*                                                                         *
*   RESULT = ALL_SUFFIX (MASK, DIM, SEGMENT, EXCLUSIVE)                   *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_suffix_mask) (op, zero_data, 

    result    , array    , dim     , segment     , exclusive     ,
    result_dsp, array_dsp, dim_dsp , segment_dsp,  exclusive_dsp  )

int  *op;
char *zero_data;

char *result, *array;
int  *dim, *segment, *exclusive;
array_info *result_dsp, *array_dsp, 
           *dim_dsp, *segment_dsp, *exclusive_dsp;

{ dalib_set_array     (array_dsp);
  dalib_set_op        (*op);
  dalib_set_dim       (dim, dim_dsp);
  dalib_set_result    (result_dsp);
  is_masked = 0;
  dalib_set_segment   (segment, segment_dsp);
  dalib_set_exclusive (exclusive, exclusive_dsp);
  zero_ptr = zero_data;
  dalib_do_suffix ();
  dalib_scan_end (result_dsp);   

}  /* dalib_suffix_mask */

/**************************************************************************
*                                                                         *
*   RESULT = COPY_SUFFIX (ARRAY, DIM, SEGMENT)                            *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_suffix_copy) (

    result    , array    , dim     , segment    ,
    result_dsp, array_dsp, dim_dsp , segment_dsp)

char *result, *array;
int  *dim, *segment;
array_info *result_dsp, *array_dsp, 
           *dim_dsp, *segment_dsp;

{ dalib_set_array     (array_dsp);
  dalib_set_op        (0);
  
  dalib_set_dim       (dim, dim_dsp);
  dalib_set_result    (result_dsp);
  is_masked = 0;
  dalib_set_segment   (segment, segment_dsp);
  is_exclusive = 0;
  zero_ptr = help;    /* is never important */

  dalib_do_copy_suffix ();
  dalib_scan_end (result_dsp);   

}  /* dalib_suffix_copy */
