/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*                                                                         *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Aug 97                                                   *
*  Last Update : Aug 97                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : remote.m4                                                *
*                                                                         *
*  Function: Creating descriptors for remote access of distr. data        *
*                                                                         *
*  Note : Remote can also be used for remote addressing without RMA       *
*                                                                         *
*         (this module is machine independent)                            *
*                                                                         *
*  EXPORT:                                                                *
*                                                                         *
*   a) FORTRAN interface                                                  *
*                                                                         *
*      FUNCTION(dalib_array_rma) (array_info array_id, int *kind)         *
*                                                                         *
*   b) DALIB interface                                                    *
*                                                                         *
*                                                                         *
*  Attention: this module should be available in any case                 *
*                                                                         *
**************************************************************************/
 
#undef DEBUG

#include "dalib.h"

     /***********************************************
     *                                              *
     *   record describing remote addressing fac.   *
     *                                              *
     ***********************************************/

typedef struct {

   int first;
   unsigned char *data;
   int total[MAX_DIMENSIONS+1];

   } remote_access;

struct RemoteRecord
 
 { int rma_defined;         /* will be 1 if rma_info allocated     */

   int rma_NP;              /* number of processors that have data */

   remote_access *rma_info; /* will have rma_NP entries only       */
 };

/******************************************************************
*                                                                 *
*  static void dalib_array_set_rma (array_info array_id)          *
*                                                                 *
******************************************************************/

static void dalib_array_set_rma (array_id)
 
array_info array_id;
 
{ Remote rma_pointer;

#ifdef DEBUG
  printf ("%d: dalib_array_set_rma for array = %d\n", pcb.i, array_id);
#endif 

  rma_pointer  = (Remote) dalib_malloc (sizeof (struct RemoteRecord), 
                                               "dalib_array_rma");

  rma_pointer->rma_defined = 0;

  /* Remote can also be used for remote addressing without RMA */

  array_id->RemoteInfo = rma_pointer;

} /* dalib_array_set_rma */

/******************************************************************
*                                                                 *
*  FUNCTION(dalib_array_rma) (array_info array_id)                *
*                                                                 *
*   - defines an array to be accessed via remote memory access    *
*   - will not exchange addresses (done with allocation)          *
*                                                                 *
******************************************************************/

void FUNCTION(dalib_array_rma) (array_id)
 
array_info *array_id;
 
{ 

#ifdef RMA
  dalib_system_rma_init();
#endif

  dalib_array_set_rma (*array_id);

#ifdef DEBUG
  printf ("%d: array %d has been defined to have remote access\n",
           pcb.i, *array_id);
#endif 

} /* dalib_array_rma */

/**************************************************************************
*                                                                         *
*  void dalib_array_remote_init (array_info array_id)                     *
*                                                                         *
*   - exchange remote addressing schemes for all processors               *
*                                                                         *
**************************************************************************/

void dalib_array_remote_init (array_id)

array_info array_id;

{ Remote rma_pointer;

  int size;                   /* size of one addressing scheme       */

  remote_access *all_info;    /* will contain all addressing schemes */
  remote_access *my_info;     /* help pointer into remote_adr_info   */

  int NId, N0, NP;
  int top_id;
  array_info template_dsp;
  char *dummy;

#ifdef DEBUG
   printf ("%d: array_remote_init, dsp = %d\n", pcb.i, array_id);
#endif 

   rma_pointer = array_id->RemoteInfo;

   if (rma_pointer == NO_REMOTE)

     { dalib_array_set_rma (array_id);
       rma_pointer = array_id->RemoteInfo;
     }

   /* nothing more to do if remote schemes are already avaiable */

#ifdef DEBUG
   printf ("%d: array_remote_init, defined = %d\n", 
            pcb.i, rma_pointer->rma_defined);
#endif 

   if (rma_pointer->rma_defined) return;

   dalib_array_info (array_id, &template_dsp, &top_id);

   NP = dalib_top_size (top_id);
   N0 = dalib_top_first (top_id);

   size = sizeof (remote_access);

   all_info = (remote_access *) dalib_malloc (size * NP, "array_remote_init");

   if (dalib_in_topology (top_id))

      { /* this processor owns data */

        NId = pcb.i - N0;

        my_info = all_info +  NId;

        dalib_array_addressing (array_id, pcb.i, &dummy,
                                &(my_info->first), my_info->total);

        my_info->data = array_id->data;

#ifdef DEBUG
       printf ("%d: have set my info (data = %d) at relpos %d\n", 
               pcb.i, my_info->data, NId);
#endif

      } /* set my info */

   for (NId = 0; NId < NP; NId++)

    {  dalib_context_broadcast (all_info + NId, size, N0 + NId);
#ifdef DEBUG
       printf ("%d: broadcast from %d, address = %d\n",
                pcb.i, NId, all_info[NId].data);
#endif
    }

   rma_pointer->rma_defined = 1;
   rma_pointer->rma_NP      = NP;
   rma_pointer->rma_info    = all_info;

#ifdef DEBUG
   printf ("%d: array_remote_init ready\n", pcb.i);
#endif 

} /* dalib_array_remote_init */

/*******************************************************************
*                                                                  *
*  void dalib_remote_copy_info (array_info dummy_dsp, actual_dsp)  *
*                                                                  *
*    - if actual has remote access and dummy should have it,       *
*      then copy it from actual to dummy                           *
*                                                                  *
*******************************************************************/

void dalib_remote_copy_info (dummy_dsp, actual_dsp)

array_info actual_dsp, dummy_dsp;

{ Remote actual_info, dummy_info;

  remote_access *all_info;    /* will contain all addressing schemes */

  int size;
  int NP;

#ifdef DEBUG
  printf ("%d: copy remote info from actual %d to dummy %d\n", 
           pcb.i, actual_dsp, dummy_dsp);
#endif

  dummy_info = dummy_dsp->RemoteInfo;
  actual_info = actual_dsp->RemoteInfo;

  if (dummy_info == NO_REMOTE) return;
  if (actual_info == NO_REMOTE) return;

#ifdef DEBUG
  printf ("%d: copy remote, dummy defined = %d, actual defined = %d\n",
           pcb.i, dummy_info->rma_defined, actual_info->rma_defined);
#endif

  if (actual_info->rma_defined == 0) return;

  size = sizeof (remote_access);
  NP   = actual_info->rma_NP;

  all_info = (remote_access *) dalib_malloc (size * NP, "array_remote_copy");

  dalib_memcopy (all_info, actual_info->rma_info, size*NP);

  dummy_info->rma_defined = 1;
  dummy_info->rma_NP      = NP;
  dummy_info->rma_info    = all_info;

} /* dalib_remote_copy_info */

/*******************************************************************
*                                                                  *
* void dalib_remote_free (array_info array_id)                     *
*                                                                  *
*   - free descriptor structures for remote access                 *
*                                                                  *
*******************************************************************/

void dalib_remote_free (rma_pointer)

Remote rma_pointer;

{ if (rma_pointer->rma_defined)

     dalib_free (rma_pointer->rma_info, 
                 (rma_pointer->rma_NP) * sizeof(remote_access));

  dalib_free (rma_pointer, sizeof(struct RemoteRecord));

} /* dalib_remote_free */

/**************************************************************************
*                                                                         *
*  dalib_array_remote_addressing (array_info array_id, int pid,           *
*                                 int *first, int total[])                *
*                                                                         *
*   - get the addressing scheme on processor pid (relative position)      *
*                                                                         *
**************************************************************************/

void dalib_array_remote_addressing (array_id, pid, first, total)

array_info array_id;
int        *first;
int        total[];
int        pid;

{ int i, size, rank;
  remote_access *ptr;

  Remote rma_pointer;

  rma_pointer = array_id->RemoteInfo;

  if (rma_pointer == (NO_REMOTE))

    { dalib_internal_error ("remote addressing without RMA info");
      dalib_stop ();
    }

  if (rma_pointer->rma_defined == 0)

    { dalib_internal_error ("remote addressing not initialized");
      dalib_stop ();
    }


  rank = array_id->rank;
  ptr = rma_pointer->rma_info + pid;

  *first = ptr->first;
  for (i=0;i<=rank;i++) total[i] = ptr->total[i];

#ifdef DEBUG
  if (rank == 1)
    printf ("%d: remote %d is data = %d, first = %d, total = %d %d\n",
            pcb.i, pid, ptr->data, *first, total[0], total[1]);
  if (rank == 2)
    printf ("%d: remote %d is data = %d first = %d, total = %d %d %d\n",
            pcb.i, pid, ptr->data, *first, total[0], total[1], total[2]);
#endif

} /* dalib_array_remote_addressing */

/**************************************************************************
*                                                                         *
*  unsigned char *dalib_array_remote_data (array_info array_id, int pid)  *
*                                                                         *
*   - returns data pointer of remote processor pid                        *
*                                                                         *
**************************************************************************/

unsigned char *dalib_array_remote_data (array_id, pid)

array_info array_id;
int        pid;

{ int i, size, rank;
  remote_access *ptr;

  Remote rma_pointer;

  rma_pointer = array_id->RemoteInfo;

  if (rma_pointer == (NO_REMOTE))

    { dalib_internal_error ("remote addressing without RMA info");
      dalib_stop ();
    }

  if (rma_pointer->rma_defined == 0)

    { dalib_internal_error ("remote addressing not initialized");
      dalib_stop ();
    }

  ptr = rma_pointer->rma_info + pid;

#ifdef DEBUG
  printf ("%d: remote data of array (dsp=%d) on pid = %d starts at %d\n",
           pcb.i, array_id, pid, ptr->data);
#endif

  return (ptr->data);

} /* dalib_array_remote_data */

/**************************************************************************
*                                                                         *
*  int dalib_remote_offset (array_info array_id, int pid,                 *
*                           int global_indices[]         )                *
*                                                                         *
*    - returns local offset on processor pid for global indexes           *
*                                                                         *
**************************************************************************/

int dalib_remote_offset (array_id, pid, global_indices)

array_info array_id;
int        pid;
int global_indices[];

{ int lb, ub;     /* lb:ub is global size of one dimension   */
  int offset;
  int i, rank;
  int zero;
  int total[MAX_DIMENSIONS + 1];

  rank = array_id->rank;

  dalib_array_remote_addressing (array_id, pid, &zero, total);

  offset = -zero;

  for (i=0; i<rank; i++)
    offset += global_indices[i] * total[i];

  return (offset);

} /* dalib_remote_offset */

/*********************************************************************
*                                                                    *
* FUNCTION(dalib_rma_read) (char *data, array_info *array_id,        *
*                           int *ind1, int *ind2, ..., int *ind7)    *
*                                                                    *
*********************************************************************/

void FUNCTION(dalib_rma_read) (data, array_id, ind1, ind2, ind3, ind4, 
                                               ind5 ,ind6, ind7)

char *data;
array_info *array_id;
int *ind1, *ind2, *ind3, *ind4, *ind5, *ind6, *ind7;

{ int global_indices [MAX_DIMENSIONS];

  int rank, size;

  unsigned char *rem_address;    /* remote address */

  int owner, offset;

  int rem_pid;

  array_info dummy;
  int top_id;
 
  rank = (*array_id)->rank;
  size = (*array_id)->size;
 
  switch (rank) {

    case 7: global_indices[6] = *ind7;
    case 6: global_indices[5] = *ind6;
    case 5: global_indices[4] = *ind5;
    case 4: global_indices[3] = *ind4;
    case 3: global_indices[2] = *ind3;
    case 2: global_indices[1] = *ind2;
    case 1: global_indices[0] = *ind1;

  } /* end switch */

  dalib_array_info (*array_id, &dummy, &top_id);

  if (pcb.p == 1)
 
     { rem_pid     = 1;
       offset      = dalib_local_offset (*array_id, global_indices);
       rem_address = (*array_id)->data + offset * size;
     }
 
    else
 
     { owner       = dalib_multidim_owner (*array_id, global_indices);
       offset      = dalib_remote_offset (*array_id, owner, global_indices);
       rem_address = dalib_array_remote_data (*array_id, owner);
       rem_address = rem_address + offset * size;
       rem_pid     = dalib_top_elem (top_id, owner);
     }
 
#ifdef DEBUG
  switch (rank) {

  case 1 :

     printf ("%d: rma_read (index = %d), owner = %d, offset = %d\n", 
              pcb.i, *ind1, owner, offset);
     break;

  case 2 :

     printf ("%d: rma_read (index = %d, %d), owner = %d, offset = %d\n", 
              pcb.i, *ind1, *ind2, owner, offset);
     break;

  default :

     printf ("%d: rma_read (index = %d %d, %d), owner = %d, offset = %d\n", 
              pcb.i, *ind1, *ind2, *ind3, owner, offset);
     break;

  } /* switch */
#endif

  if (rem_pid == pcb.i)

     dalib_memcopy (data, rem_address, size);

   else

     { 

#if defined(RMA)
     dalib_system_rma_get (data, rem_pid, rem_address, size);
#else
     dalib_internal_error ("RMA access not available\n");
     dalib_stop ();
#endif

     }
 
} /* dalib_rma_read */

/**********************************************************************
*                                                                     *
*  void dalib_ind_array_info (array_info array_dsp, int *obj_size,    *
*                             int *serial_size, int *dist_size)       *
*                                                                     *
*  IN  : array_info array_dsp  (must only be distributed in the       *
*                               last dimension)                       *
*                                                                     *
*  OUT : obj_size is number of bytes for one element                  *
*        serial_size is number of elements in serial dimension        *
*        dist_size is number of elements in distributed dimension     *
*                                                                     *
**********************************************************************/

static void dalib_base_info (array_dsp, serial_size, rank)
                             
array_info array_dsp;
int *serial_size;
int *rank;

{ int i;
  DimInfo *dim;

  *rank = array_dsp->rank;
  dim   = array_dsp->dimensions;

  *serial_size = array_dsp->size;

  for (i=0; i<(*rank)-1; i++)
    { *serial_size *=  (dim->global_size[1] - dim->global_size[0] + 1);
      dim++;
    }

} /* dalib_base_info */

/*********************************************************************
*                                                                    *
* FUNCTION(dalib_rma_read_fast) (char *data,                         *
*                                array_info *array_id,               *
*                                int *ind)                           *
*                                                                    *
* - special version of rma_read if array_id is only distributed      *
*   along the last dimension and read all first dimensions in        *
*                                                                    *
*********************************************************************/

void FUNCTION(dalib_rma_read_fast) (data, array_id, ind)

char *data;
array_info *array_id;
int *ind;

{ int rank, size;

  unsigned char *rem_address;    /* remote address */

  int owner, offset;

  int rem_pid;

  int low, high;

  int top_id, top_dim;
  int lb, ub, local_lb;
  int base, stride;
  int kind;

  int NP;

  DistDim mapping;

  dalib_base_info (*array_id, &size, &rank);

  dalib_array_dim_mapping (*array_id, rank,
                           &base, &stride, &lb, &ub,
                           &top_id, &mapping);

  dalib_dim_mapping_info (mapping, &kind, &top_dim);

  if (kind == kBLOCK_DIM)

      { int bsize;

        dalib_internal_error ("not updated for new version");
        dalib_stop ();

        bsize = 0;

      NP       = dalib_top_size (top_id);
      owner    = dalib_block_owner (NP, bsize, lb, ub, *ind); 
      local_lb = lb + (owner-1) * bsize; 
      offset   = (*ind) - local_lb;

#ifdef DEBUG
 printf ("%d: ind = %d -> owner = %d, offset = %d\n",
         pcb.i, *ind, owner, offset);
#endif

    }

   else

    { dalib_internal_error ("rma_read_fast illegal");
      dalib_stop ();
    }

  rem_pid = dalib_top_elem (top_id, owner-1);
  rem_address =  dalib_array_remote_data (*array_id, owner-1);
  rem_address += offset * size;

#ifdef DEBUG
  printf ("%d: rma_read_fast (ind=%d), pid = %d, address = %d, size = %d\n",
           pcb.i, *ind, rem_pid, rem_address, size);
#endif

#if defined(RMA)
  dalib_system_rma_get (data, rem_pid, rem_address, size);
#else
  dalib_internal_error ("RMA access not available\n");
  dalib_stop ();
#endif

} /* dalib_rma_read_fast */

/*********************************************************************
*                                                                    *
* FUNCTION(dalib_rma_write) (int *op, char *data,                    *
*                            array_info *array_id,                   *
*                            int *ind1, int *ind2, ..., int *ind7)   *
*                                                                    *
*********************************************************************/

void FUNCTION(dalib_rma_write) (data, array_id, ind1, ind2, ind3, ind4, 
                                                ind5 ,ind6, ind7)

void *data;
array_info *array_id;
int *ind1, *ind2, *ind3, *ind4, *ind5, *ind6, *ind7;

{ int global_indices [MAX_DIMENSIONS];

  int rank, size;

  unsigned char *rem_address;    /* remote address */

  int owner, offset;

  int rem_pid;

  array_info dummy;
  int top_id;
 
  rank = (*array_id)->rank;
  size = (*array_id)->size;
 
  switch (rank) {

    case 7: global_indices[6] = *ind7;
    case 6: global_indices[5] = *ind6;
    case 5: global_indices[4] = *ind5;
    case 4: global_indices[3] = *ind4;
    case 3: global_indices[2] = *ind3;
    case 2: global_indices[1] = *ind2;
    case 1: global_indices[0] = *ind1;

  } /* end switch */

  dalib_array_info (*array_id, &dummy, &top_id);

  if (pcb.p == 1)
 
     { rem_pid     = 1;
       offset      = dalib_local_offset (*array_id, global_indices);
       rem_address = (*array_id)->data + offset * size;
     }
 
    else
 
     { owner       = dalib_multidim_owner (*array_id, global_indices);
       offset      = dalib_remote_offset (*array_id, owner, global_indices);
       rem_address = dalib_array_remote_data (*array_id, owner);
       rem_address = rem_address + offset * size;
       rem_pid     = dalib_top_elem (top_id, owner);
     }
 
#ifdef DEBUG
  switch (rank) {

  case 1 :

     printf ("%d: rma_write (index = %d), owner = %d, offset = %d\n", 
              pcb.i, *ind1, owner, offset);
     break;

  case 2 :

     printf ("%d: rma_write (index = %d, %d), owner = %d, offset = %d\n", 
              pcb.i, *ind1, *ind2, owner, offset);
     break;

  default :

     printf ("%d: rma_write (index = %d %d, %d), owner = %d, offset = %d\n", 
              pcb.i, *ind1, *ind2, *ind3, owner, offset);
     break;

  } /* switch */
#endif

  if (rem_pid == pcb.i)

     { /* rem_address is a local address, so update it locally */

       dalib_memcopy (rem_address, data, size);
     }

   else

     { 

#if defined(RMA)
       dalib_system_rma_put (data, rem_pid, rem_address, size);
#else
       dalib_internal_error ("RMA access not available\n");
       dalib_stop ();
#endif
     }
 
} /* dalib_rma_write */

/*********************************************************************
*                                                                    *
* FUNCTION(dalib_rma_update) (int *op, char *data,                   *
*                             array_info *array_id,                  *
*                             int *ind1, int *ind2, ..., int *ind7)  *
*                                                                    *
*********************************************************************/

void FUNCTION(dalib_rma_update) (op, data, array_id, ind1, ind2, ind3, ind4, 
                                                     ind5 ,ind6, ind7)

int  *op;
char *data;
array_info *array_id;
int *ind1, *ind2, *ind3, *ind4, *ind5, *ind6, *ind7;

{ int global_indices [MAX_DIMENSIONS];

  int rank, size;

  unsigned char *rem_address;    /* remote address */

  dalib_routine *f_reduction;
  extern dalib_routine *dalib_get_reduction_fn();

  int owner, offset;

  int rem_pid;

  array_info dummy;
  int top_id;
 
  rank = (*array_id)->rank;
  size = (*array_id)->size;
 
  switch (rank) {

    case 7: global_indices[6] = *ind7;
    case 6: global_indices[5] = *ind6;
    case 5: global_indices[4] = *ind5;
    case 4: global_indices[3] = *ind4;
    case 3: global_indices[2] = *ind3;
    case 2: global_indices[1] = *ind2;
    case 1: global_indices[0] = *ind1;

  } /* end switch */

  dalib_array_info (*array_id, &dummy, &top_id);

  if (pcb.p == 1)
 
     { rem_pid     = 1;
       offset      = dalib_local_offset (*array_id, global_indices);
       rem_address = (*array_id)->data + offset * size;
     }
 
    else
 
     { owner       = dalib_multidim_owner (*array_id, global_indices);
       offset      = dalib_remote_offset (*array_id, owner, global_indices);
       rem_address = dalib_array_remote_data (*array_id, owner);
       rem_address = rem_address + offset * size;
       rem_pid     = dalib_top_elem (top_id, owner);
     }
 
#ifdef DEBUG
  switch (rank) {

  case 1 :

     printf ("%d: rma_update (index = %d), owner = %d, offset = %d\n", 
              pcb.i, *ind1, owner, offset);
     break;

  case 2 :

     printf ("%d: rma_update (index = %d, %d), owner = %d, offset = %d\n", 
              pcb.i, *ind1, *ind2, owner, offset);
     break;

  default :

     printf ("%d: rma_update (index = %d %d, %d), owner = %d, offset = %d\n", 
              pcb.i, *ind1, *ind2, *ind3, owner, offset);
     break;

  } /* switch */
#endif

  if (*op > 0) f_reduction = dalib_get_reduction_fn (*op);

  if (rem_pid == pcb.i)

     { /* rem_address is a local address, so update it locally */

       if (*op == 0)
          dalib_memcopy (rem_address, data, size);
       else
          f_reduction (rem_address, data);
     }

   else

     { 

#if defined(RMA)
      if (*op == 0)
         dalib_system_rma_put (data, rem_pid, rem_address, size);
       else
         dalib_system_rma_update (*op, data, rem_pid, rem_address, size);
#else
     dalib_internal_error ("RMA access not available\n");
     dalib_stop ();
#endif

    }
 
} /* dalib_rma_update */

void FUNCTION(dalib_rma_update_fast) (op, data, array_id, ind)

int *op;
char *data;
array_info *array_id;
int *ind;

{ int rank, size;

  unsigned char *rem_address;    /* remote address */

  int owner, offset;

  int rem_pid;

  int low, high;

  int top_id, top_dim;
  int lb, ub, local_lb;
  int base, stride;
  int kind;

  int NP;

  DistDim mapping;

  int *offsets;

  dalib_base_info (*array_id, &size, &rank);

  dalib_array_dim_mapping (*array_id, rank,
                           &base, &stride, &lb, &ub,
                           &top_id, &mapping);

  dalib_dim_mapping_info (mapping, &kind, &top_dim);

  if (kind == kBLOCK_DIM)

    { int bsize;

      bsize = (int) offsets;

      dalib_internal_error ("not updated for new version");
      dalib_stop ();
 
      NP       = dalib_top_size (top_id);
      owner    = dalib_block_owner (NP, bsize, lb, ub, *ind); 
      local_lb = lb + (owner-1) * ((int) offsets);
      offset   = (*ind) - local_lb;

#ifdef DEBUG
 printf ("%d: ind = %d -> owner = %d, offset = %d\n",
         pcb.i, *ind, owner, offset);
#endif

    }

   else

    { dalib_internal_error ("rma_read_fast illegal");
      dalib_stop ();
    }

  rem_pid = dalib_top_elem (top_id, owner-1);
  rem_address =  dalib_array_remote_data (*array_id, owner-1);
  rem_address += offset * size;

#ifdef DEBUG
  printf ("%d: rma_read_fast (ind=%d), pid = %d, address = %d, size = %d\n",
           pcb.i, *ind, rem_pid, rem_address, size);
#endif

#if defined(RMA)
  dalib_system_rma_update (*op, data, rem_pid, rem_address, size);
#else
  dalib_internal_error ("RMA access not available\n");
  dalib_stop ();
#endif

} /* dalib_rma_update_fast */

 
