/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Feb 92                                                   *
*  Last Update : Feb 95                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : random.m4                                                *
*                                                                         *
*  Function    : Getting random numbers (integer, float, double)          *
*                                                                         *
*  Export : Internal Use                                                  *
*                                                                         *
*      void random_block_init ()                                          *
*                                                                         *
*  Export : FORTRAN Interface                                             *
*                                                                         *
*   dalib_random_init (seed) : initialization of generator by node        *
*                                                                         *
*   dalib_section_real_randoms (section_id)                               *
*                                                                         *
*   dalib_section_int_randoms (section_id, int limit)                     *
*                                                                         *
*   -> section_id can be array_info or section_info                       *
*   -> size of one value (4 or 8 bytes) available by section_id           *
*                                                                         *
*   UPDATES :                                                             *
*                                                                         *
*    Nov 94 : automatic detection for BIG or LITTLE ENDIAN                *
*    Feb 95 : now using array descriptors for interface                   *
*                                                                         *
**************************************************************************/

# include <stdio.h>
# include "dalib.h"      /* needs pcb.i */

# undef DEBUG

/*******************************************************************
*                                                                  *
*  Local Block for random numbers on each process                  *
*                                                                  *
*******************************************************************/

#define RANDOM_SIZE 20  /* 250 */
#define RANDOM_K1   17  /* 103 */
#define RANDOM_K2    5

/* RANDOM_INT must be a full significant type, e.g. 8 bytes for
   only 4 significant bytes results in problems                  */

#if defined (CRAY)
#define RANDOM_INT short
#else
#define RANDOM_INT int
#endif

static struct

  {  RANDOM_INT state[RANDOM_SIZE]; 
     int  pos;
  } random_block;

static int endian;         /* 0 for small, 1 for big endian              */

/**************************************************************************
*                                                                         *
*  void fix_endian ()    ! set endian globally                            *
*                                                                         *
*   set global variable endian (0 : little endian, 1 : big endian)        *
*                                                                         *
**************************************************************************/

static void fix_endian ()

{ float X;
  unsigned char *P;

  X = 1.0;
  P = (unsigned char *) &X;
  
  if (sizeof(X) == 4)

  { if ( (P[0] == 63) && (P[1] == 128) && (P[2] == 0) && (P[3] == 0) )

      { /* this must be little endian */

        /* SUN4, CM5, RS6K, KSR1, SGI, CS2, HPPA, CENJU3 */

        endian = 0;

      }

     else if ( (P[3] == 63) && (P[2] == 128) && (P[1] == 0) && (P[0] == 0) )

    { /* this must be big endian */

      /* AFX8, GC, IPSC-860, SRM, MEIKO, PGON, ALPHA */

      endian = 1;
    }

   else 

    { /* unknown */
      dalib_internal_error ("endian on float(4 byte) is not identified\n");
      printf ("1.0 is %d - %d - %d - %d\n", P[0], P[1], P[2], P[3]);
      exit (-1);
    }

   }  /* check for 4 bytes */

  else if (sizeof(float) == 8)

  { if ( (P[0] == 64) && (P[1] == 1) && (P[2] == 128) && (P[3] == 0) )

        endian = 0;

     else if ( (P[7] == 64) && (P[6] == 1) && (P[5] == 128) && (P[4] == 0) )

        endian = 1;

     else 

      { /* unknown */
        dalib_internal_error ("endian on float(8 byte) is not identified\n");
        printf ("1.0 is %d - %d - %d - %d - %d - %d - %d - %d\n", 
                 P[0], P[1], P[2], P[3], P[4], P[5], P[6], P[7]);
        exit (-1);
      }

   }  /* check for 8 bytes */

  else

   { dalib_internal_error ("float has not 4/8 bytes (cannot identify ENDIAN)");
     dalib_stop ();
   }

} /* fix_endian */

/*******************************************************************
*                                                                  *
*  check_int_size   : look for a C counterpart                     *
*  check_real_size  : look for a C counterpart                     *
*                                                                  *
*******************************************************************/

static void check_int_size (size)

int size;

{ char msg[80];

  if (size == sizeof(int)) return;
  if (size == sizeof(long)) return;
  if (size == sizeof(short)) return;

  sprintf (msg, "INTEGER%d not supported within C", size);
  dalib_internal_error (msg);
  fprintf (stderr, "int   is INTEGER%d\n", sizeof(int));
  fprintf (stderr, "short is INTEGER%d\n", sizeof(short));
  fprintf (stderr, "long  is INTEGER%d\n", sizeof(long));
  dalib_stop ();

} /* check_int_size */

static void check_real_size (size)

int size;

{ char msg[80];

  if (size == sizeof(float)) return;
  if (size == sizeof(double)) return;
  if (size == sizeof(long double)) return;

  sprintf (msg, "REAL%d not supported within C", size);
  dalib_internal_error (msg);
  fprintf (stderr, "float       is REAL%d\n", sizeof(float));
  fprintf (stderr, "double      is REAL%d\n", sizeof(double));
  fprintf (stderr, "long double is REAL%d\n", sizeof(long double));
  dalib_stop ();

} /* check_real_size */

/*******************************************************************
*                                                                  *
*  void random_block_init ()   ! called at initialization          *
*                                                                  *
*******************************************************************/

void random_block_init ()

/* This routine is called concurrently */

{ int j;

  fix_endian ();

#if defined(AFX8) || defined(KSR1) || defined(PowerPC)
  srand (pcb.i * 181 - 1);
#else
  srand48 (pcb.i * 181 - 1);
#endif
  for (j=0;j<RANDOM_SIZE;j++)
#if defined(AFX8) || defined(KSR1) || defined(PowerPC)
     random_block.state[j] = rand();
#else
     random_block.state[j] = mrand48 ();
#endif

  random_block.pos = 0;

} /* random_block_init */

/*******************************************************************
*                                                                  *
*  random_block_get ()                                             *
*                                                                  *
*     - get a new random value from random_block                   *
*     - parallel random numbers by different processors            *
*                                                                  *
*******************************************************************/

static void random_block_get (ptr, size)

unsigned char *ptr;
int size;

{ int k1, k2, pos;

  RANDOM_INT z, *z_ptr;

  int j, no;

  int significant = sizeof(RANDOM_INT);

  no  = (size + significant - 1) / significant;

  pos = random_block.pos;

  z_ptr = (RANDOM_INT *) ptr;

  for (j=0; j<no; j++, z_ptr++)

    { k1 = pos - RANDOM_K1; if (k1 < 0) k1+=RANDOM_SIZE;
      k2 = pos - RANDOM_K2; if (k2 < 0) k2+=RANDOM_SIZE;

      z = random_block.state[k1] - random_block.state[k2]; /* xor */
      random_block.state[pos] = z;

      pos += 1; if (pos >= RANDOM_SIZE) pos = 0;

      *z_ptr = z;

    }

  random_block.pos = pos;

} /* random_block_get */

/*******************************************************************
*                                                                  *
* dalib_get_int_randoms (int a[], int size, int limit)             *
*                                                                  *
*******************************************************************/

static void dalib_get_int_randoms (a, section_size, elem_size, limit)

unsigned char *a;
int section_size, elem_size;
INTEGER limit;

{ int j;

  unsigned char *a_ptr;

  /* step 1 : fill the contiguous section with random bytes */
 
  random_block_get (a, section_size * elem_size);

  /* step 2 : make useful integer values of it */

  if (limit == 0) return;   /* nothing more to do */

  a_ptr = a;

  for (j=0; j < section_size; j++, a_ptr+=elem_size)

    { /* make value positive , big endian , otherwise replace 3 with 0 */
 
      if (endian == 0)

         a_ptr[0] = a_ptr[0] & 127;

       else

         a_ptr[elem_size-1] = a_ptr[elem_size-1] & 127;
 
      /* now compute random_value modulo limit */

      if (elem_size == sizeof(int))

         { int *random_value;
 
           random_value = (int *) a_ptr;
 
           *random_value = *random_value % limit;
 
         } /* int */

       else if (elem_size == sizeof(short))

         { short *random_value;

           random_value = (short *) a_ptr;

           *random_value = *random_value % limit;
 
         } /* short */
 
       else if (elem_size == sizeof(long))

         { long *random_value;

           random_value = (long *) a_ptr;

           *random_value = *random_value % limit;
 
         } /* long */

       else 

         { dalib_internal_error ("integer not supported");
           dalib_stop ();
         }
 
     } /* for loop */

} /* dalib_get_int_randoms */

/*******************************************************************
*                                                                  *
* dalib_get_real_randoms (int a[], int size)                       *
*                                                                  *
*******************************************************************/

static void dalib_get_real_randoms (a, section_size, elem_size)

unsigned char a[];
int section_size, elem_size;
 
{ int j;
  unsigned char *a_ptr;

  /* step 1 : fill the contiguous section with random bytes */

  random_block_get (a, section_size * elem_size);

  if ((elem_size != 4) && (elem_size != 8))

     { dalib_internal_error ("dalib_real_random (not 4 or 8 bytes)");
       dalib_stop ();
     }

  /* step 2 : make useful REAL values of it */

/****************************************************************
*                                                               *
* real*4   byte1      byte2    byte3      byte4                 *
* =============================================                 *
*                                                               *
*  0.5      3f       00        00         00                    *
*  1.0      3f       80        00         00                    *
*  1.+      3f       80        00         01                    *
*  2.-      3f       ff        ff         ff                    *
*  2.0      40       00        00         00                    *
*                                                               *
****************************************************************/

/****************************************************************
*                                                               *
* real*8   byte1      byte2    byte3      byte4  ...   byte8    *
* ==========================================================    *
*                                                               *
*  0.5      3f       E0        00         00     ....   00      *
*  1.0      3f       F0        00         00     ....   00      *
*  1.+      3f       F0        00         00     ....   01      *
*  2.-      3f       ff        ff         ff     ....   ff      *
*  2.0      40       00        00         00     ....   00      *
*                                                               *
****************************************************************/

  a_ptr = a;

  for (j=0; j < section_size; j++, a_ptr+=elem_size)

     { if (elem_size == 4)
 
          { if (endian == 0)

               { /* little endian, otherwise replace 3 with 0, 2 with 1 */

                 a_ptr[0] = 63;  /* 3f */
                 a_ptr[1] = a_ptr[1] | 128;   /* 80 */
               }

              else 

               { a_ptr[3] = 63;  /* 3f */
                 a_ptr[2] = a_ptr[1] | 128;   /* 80 */
               }

           } /* elem_size == 4 */

       else 

          { if (endian == 0)
 
                { a_ptr[0] = 63;  /* 3f */
                  a_ptr[1] = a_ptr[1] | 240;   /* F0 */
                }
 
               else
 
                { a_ptr[7] = 63;  /* 3f */
                  a_ptr[6] = a_ptr[1] | 240;   /* F0 */
                }

           } /* elem_size == 8 */

       /* random value is now between 1.0 and 2.0 */

       if (elem_size == sizeof (float))

         { float *random_value;

           random_value = (float *) a_ptr;

           *random_value -= (float) 1.0;

         }

        else if (elem_size == sizeof (double))

         { double *random_value;

           random_value = (double *) a_ptr;

           *random_value -= (double) 1.0;

         }
      
        else if (elem_size == sizeof (long double))

         { long double *random_value;

           random_value = (long double *) a_ptr;

           *random_value -= (long double) 1.0;

         }
     }

} /* dalib_get_real_randoms */

/*********************************************************************
*                                                                    *
* FORTRAN - Interface                                                *
*                                                                    *
*   dalib_random_init (seed) : initialization of generator by node   *
*                                                                    *
*   dalib_section_real_randoms (section_id)                          *
*   dalib_section_int_randoms (section_id, int limit)                *
*                                                                    *
*   -> section_id can be array_info or section_info                  *
*   -> size of one value (4 or 8 bytes) available by section_id      *
*                                                                    *
*********************************************************************/

       /************************************************
       *                                               *
       * FUNCTION(dalib_random_init) (seed)            *
       *                                               *
       ************************************************/

void FUNCTION(dalib_random_init) (n) 

/* new initialization of the number generator with n */

int *n;

{ 
#if defined(AFX8) || defined(KSR1) || defined(PowerPC)
  srand(*n * pcb.i);   /* set value */
#else
  srand48(*n * pcb.i);   /* set value */
#endif
  random_block_init ();
}

       /************************************************
       *                                               *
       *   dalib_section_real_randoms (section_id)     *
       *                                               *
       ************************************************/

void FUNCTION(dalib_section_real_randoms) (section_id) 
section_info *section_id;

{ dd_type section_ddt;
  int elem_size, section_size;
  int is_contiguous;

  unsigned char *random_data;

  dalib_make_secarray_ddt1 (&section_ddt, *section_id, 
                            &section_size, &elem_size);

  dalib_ddt_is_contiguous (section_ddt, &is_contiguous, &random_data);
 
#ifdef DEBUG
printf ("%d: dalib_section_real_randoms, local section size = %d (cont=%d)\n",
        pcb.i, section_size, is_contiguous);
#endif

  check_real_size (elem_size);

  if (is_contiguous)

     { /* contiguous section needs no addtional buffer */

       dalib_get_real_randoms (random_data, section_size, elem_size);

     }

    else

     { random_data = (unsigned char *)
                      dalib_malloc (section_size * elem_size, 
                                   "section_real_randoms");

       dalib_get_real_randoms (random_data, section_size, elem_size);
 
       dalib_ddt_unpack (section_ddt, random_data, 0);

       dalib_free (random_data, section_size * elem_size);
     }

  dalib_ddt_free (section_ddt);

  if (dalib_is_replicated (*section_id))

     dalib_replicate (*section_id);

} /* FUNCTION(dalib_section_real_randoms) */ 

       /***************************************************
       *                                                  *
       *   dalib_section_int_randoms (section_id, limit)  *
       *                                                  *
       ***************************************************/

void FUNCTION(dalib_section_int_randoms) (section_id, limit) 

section_info *section_id;
int          *limit;

{ dd_type section_ddt;
  int elem_size, section_size;
  int is_contiguous;

  unsigned char *random_data;

  dalib_make_secarray_ddt1 (&section_ddt, *section_id, 
                            &section_size, &elem_size);

  check_int_size (elem_size);

  dalib_ddt_is_contiguous (section_ddt, &is_contiguous, &random_data);

#ifdef DEBUG
  printf ("%d: dalib_section_int_randoms, local section size = %d (cont=%d)\n",
           pcb.i, section_size, is_contiguous);
#endif

  if (is_contiguous)

     { /* contiguous section needs no addtional buffer */

       dalib_get_int_randoms (random_data, section_size, elem_size, *limit);

     }

    else

     { random_data = (unsigned char *)
                      dalib_malloc (section_size * elem_size, 
                                   "section_int_randoms");

       dalib_get_int_randoms (random_data, section_size, elem_size, *limit);
       dalib_ddt_unpack (section_ddt, random_data, 0);
       dalib_free (random_data, section_size * elem_size);
     }

  dalib_ddt_free (section_ddt);

  if (dalib_is_replicated (*section_id))

     dalib_replicate (*section_id);

} /* FUNCTION(dalib_section_int_randoms) */ 

/*********************************************************************
*                                                                    *
* FORTRAN 90 Intrinsics                                              *
*                                                                    *
*********************************************************************/

       /***************************************************
       *                                                  *
       *   RANDOM_NUMBER (harvest_data)                   *
       *                                                  *
       ***************************************************/
 
void FUNCTION(dalib_random_number) (harvest_data, harvest_dsp) 

char         *harvest_data;
section_info *harvest_dsp;

{ if (!FUNCTION(dalib_present) (harvest_dsp)) 

    { dalib_internal_error ("random_number : only for array data\n");
      dalib_stop ();
    }

  FUNCTION(dalib_section_real_randoms) (harvest_dsp); 

} /* FUNCTION(dalib_random_number) */ 

       /***************************************************
       *                                                  *
       *   RANDOM_SEED ([size] [,put] [,get])             *
       *                                                  *
       ***************************************************/
 
void FUNCTION(dalib_random_seed) (size, put, get) 

int *size, *put, *get;

{ int seed;
  int j;

  if (FUNCTION(dalib_present) (size)) 

    { *size = RANDOM_SIZE;
    }

  else if (FUNCTION(dalib_present) (put)) 

    { for (j=0;j<RANDOM_SIZE;j++)
          random_block.state[j] = put[j];
    }

  else if (FUNCTION(dalib_present) (get)) 

    { for (j=0;j<RANDOM_SIZE;j++)
          get[j] = random_block.state[j];
    }

  else

    { seed = 2517;

      FUNCTION(dalib_random_init) (&seed); 

    }

} /* FUNCTION(dalib_random_seed) */ 
