/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Apr 92                                                   *
*  Last Update : Nov 93                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : memset.c                                                 *
*                                                                         *
*  Function    : combining routines for global set                        *
*                                                                         *
*   void dalib_memget  (target, source, index, n, size)                   *
*   void dalib_memgetp (target, source, index, n, size)                   *
*   void dalib_memgetm (target, source, index, mask, n, size)             *
*                                                                         *
*   void dalib_memset  (target, index, source, n, size)                   *
*   void dalib_memsetp (target, index, source, n, size)                   *
*   void dalib_memsetm (target, index, source, mask, n, size)             *
*                                                                         *
*   unsigned char *target, *source;                                       *
*   int *index, *mask;                                                    *
*   int n, size;                                                          *
*                                                                         *
*    dalib_set_min_ints     (data, indexes, values, n)                    *
*    dalib_set_min_reals    (data, indexes, values, n)                    *
*    dalib_set_min_doubles  (data, indexes, values, n)                    *
*    dalib_set_max_ints     (data, indexes, values, n)                    *
*    dalib_set_max_reals    (data, indexes, values, n)                    *
*    dalib_set_max_doubles  (data, indexes, values, n)                    *
*    dalib_set_add_ints     (data, indexes, values, n)                    *
*    dalib_set_add_reals    (data, indexes, values, n)                    *
*    dalib_set_add_doubles  (data, indexes, values, n)                    *
*    dalib_set_mult_ints    (data, indexes, values, n)                    *
*    dalib_set_mult_reals   (data, indexes, values, n)                    *
*    dalib_set_mult_doubles (data, indexes, values, n)                    *
*    dalib_set_and_ints     (data, indexes, values, n)                    *
*    dalib_set_or_ints      (data, indexes, values, n)                    *
*    dalib_set_eor_ints     (data, indexes, values, n)                    *
*    dalib_set_and_bools    (data, indexes, values, n)                    *
*    dalib_set_or_bools     (data, indexes, values, n)                    *
*    dalib_set_neq_bools    (data, indexes, values, n)                    *
*                                                                         *
*    dalib_setm_min_ints     (data, indexes, values, mask, n)             *
*    dalib_setm_min_reals    (data, indexes, values, mask, n)             *
*    dalib_setm_min_doubles  (data, indexes, values, mask, n)             *
*    dalib_setm_max_ints     (data, indexes, values, mask, n)             *
*    dalib_setm_max_reals    (data, indexes, values, mask, n)             *
*    dalib_setm_max_doubles  (data, indexes, values, mask, n)             *
*    dalib_setm_add_ints     (data, indexes, values, mask, n)             *
*    dalib_setm_add_reals    (data, indexes, values, mask, n)             *
*    dalib_setm_add_doubles  (data, indexes, values, mask, n)             *
*    dalib_setm_mult_ints    (data, indexes, values, mask, n)             *
*    dalib_setm_mult_reals   (data, indexes, values, mask, n)             *
*    dalib_setm_mult_doubles (data, indexes, values, mask, n)             *
*    dalib_setm_and_ints     (data, indexes, values, mask, n)             *
*    dalib_setm_or_ints      (data, indexes, values, mask, n)             *
*    dalib_setm_eor_ints     (data, indexes, values, mask, n)             *
*    dalib_setm_and_bools    (data, indexes, values, mask, n)             *
*    dalib_setm_or_bools     (data, indexes, values, mask, n)             *
*    dalib_setm_neq_bools    (data, indexes, values, mask, n)             *
*                                                                         *
*    int indexes[], mask[], n;                                            *
*                                                                         *
*    int *data, *indexes;          for dalib_set_xxx_ints                 *
*    int *data, *indexes;          for dalib_set_xxx_bools                *
*    float *data, *indexes;        for dalib_set_xxx_reals                *
*    double *data, *indexes;       for dalib_set_xxx_doubles              *
*                                                                         *
**************************************************************************/

#undef DEBUG
#include "dalib.h"
#include <stdio.h>

/**************************************************************************
*                                                                         *
*  dalib_memget (target, source, index, n, size)                          *
*                                                                         *
*    -> dalib_memget8                                                     *
*    -> dalib_memget4                                                     *
*    -> dalib_memget1                                                     *
*                                                                         *
**************************************************************************/

void dalib_memget (target, source, ind_val, n, size)
unsigned char *target, *source;
int *ind_val, n, size;

{ int i, j;
  unsigned char *ptr;
  if (size == 1)
     dalib_memget1 (target, source, ind_val, n);
   else if (size == 4)
     dalib_memget4 (target, source, ind_val, n);
   else if (size == 8)
     dalib_memget8 (target, source, ind_val, n);
   else 
     { for (i=0; i<n; i++)
         { ptr = source + ind_val[i] * size;
           for (j=0; j<size; j++)
              target[i*size+j] = ptr[j];
         }
     }
} /* dalib_memget */

void dalib_memgetp (target, source, ind_val, n, size)
unsigned char *target, *source;
int *ind_val, n, size;

{ int i, j;
  unsigned char *ptr;
  if (size == 1)
     dalib_memgetp1 (target, source, ind_val, n);
   else if (size == 4)
     dalib_memgetp4 (target, source, ind_val, n);
   else if (size == 8)
     dalib_memgetp8 (target, source, ind_val, n);
   else 
     { for (i=0; i<n; i++)
         if (ind_val[i] >= 0)
            { ptr = source + ind_val[i] * size;
              for (j=0; j<size; j++)
                 target[i*size+j] = ptr[j];
            }
     }
} /* dalib_memgetp */

void dalib_memgetm (target, source, ind_val, mask, n, size)
unsigned char *target, *source;
int *ind_val, *mask, n, size;

{ int i, j;
  unsigned char *ptr;
#ifdef DEBUG
  printf ("%d: memgetm with n = %d, size = %d\n", pcb.i, n, size);
#endif
  if (size == 1)
     dalib_memgetm1 (target, source, ind_val, mask, n);
   else if (size == 4)
     dalib_memgetm4 (target, source, ind_val, mask, n);
   else if (size == 8)
     dalib_memgetm8 (target, source, ind_val, mask, n);
   else 
     { for (i=0; i<n; i++)
         if (mask[i])
            { ptr = source + ind_val[i] * size;
              for (j=0; j<size; j++)
                 target[i*size+j] = ptr[j];
            }
     }
} /* dalib_memgetm */

/**************************************************************************
*                                                                         *
*  dalib_memset (target, ind_val, source, n, size)                          *
*                                                                         *
*    -> dalib_memset8                                                     *
*    -> dalib_memset4                                                     *
*    -> dalib_memset1                                                     *
*                                                                         *
**************************************************************************/

void dalib_memset (target, ind_val, source, n, size)
unsigned char *target, *source;
int *ind_val, n, size;

{ int i, j;
  unsigned char *ptr;
  if (size == 1)
     dalib_memset1 (target, ind_val, source, n);
   else if (size == 4)
     dalib_memset4 (target, ind_val, source, n);
   else if (size == 8)
     dalib_memset8 (target, ind_val, source, n);
   else 
     { for (i=0; i<n; i++)
         { ptr = target + ind_val[i] * size;
           for (j=0; j<size; j++)
              ptr[j] = source[i*size+j];
         }
     }
} /* dalib_memset */

void dalib_memsetp (target, ind_val, source, n, size)
unsigned char *target, *source;
int *ind_val, n, size;

{ int i, j;
  unsigned char *ptr;
  if (size == 1)
     dalib_memsetp1 (target, ind_val, source, n);
   else if (size == 4)
     dalib_memsetp4 (target, ind_val, source, n);
   else if (size == 8)
     dalib_memsetp8 (target, ind_val, source, n);
   else 
     { for (i=0; i<n; i++)
         if (ind_val[i] >= 0)
            { ptr = target + ind_val[i] * size;
              for (j=0; j<size; j++)
                 ptr[j] = source[i*size+j];
            }
     }
} /* dalib_memsetp */

void dalib_memsetm (target, ind_val, source, mask, n, size)
unsigned char *target, *source;
INTEGER ind_val[], mask[];
int n, size;

{ int i, j;
  unsigned char *ptr;
  if (size == 1)
     dalib_memsetm1 (target, ind_val, source, mask, n);
   else if (size == 4)
     dalib_memsetm4 (target, ind_val, source, mask, n);
   else if (size == 8)
     dalib_memsetm8 (target, ind_val, source, mask, n);
   else 
     { for (i=0; i<n; i++)
         if (mask[i])
            { ptr = target + ind_val[i] * size;
              for (j=0; j<size; j++)
                 ptr[j] = source[i*size+j];
            }
     }
} /* dalib_memsetm */

/*********************************************************
*                                                        *
*  PROCEDURES for global SET                             *
*                                                        *
*********************************************************/

          /*************************************
          *                                    *
          *  M I N V A L                       *
          *                                    *
          *************************************/

void dalib_set_min_ints (data, indexes, values, N)

INTEGER *data, *values;
int N, *indexes;

{ int j;
  INTEGER *ptr;

  for (j=0; j<N; j++)
    { /* a[indexes[j]] = min (a[indexes[j], values[j])  */
      ptr = data + indexes[j];
      if (values[j] < *ptr) *ptr = values[j];
    }
} /* dalib_set_min_ints */

void dalib_setn_min_ints (data, indexes, values, M, N)
INTEGER *data, *values;
INTEGER indexes[];
int M, N;

{ int j,k;
  INTEGER *ptr, val;

  for (j=0; j<N; j++)
    { ptr = data + indexes[j] * M;
      for (k=0; k<M; k++) 
        if ((val = *values++) < ptr[k]) ptr[k] = val;
    }

} /* dalib_setn_min_ints */

void dalib_set_min_reals (data, indexes, values, N)
REAL *data, *values;
INTEGER indexes[];
int N;

{ int j;
  REAL *ptr;

  for (j=0; j<N; j++)
    { /* a[indexes[j]] = min (a[indexes[j], values[j])  */
      ptr = data + indexes[j];
      if (values[j] < *ptr) *ptr = values[j];
    }
} /* dalib_set_min_reals */

void dalib_setn_min_reals (data, indexes, values, M, N)
REAL *data, *values;
INTEGER indexes[];
int M, N;

{ int j,k;
  REAL *ptr, val;

  for (j=0; j<N; j++)
    { ptr = data + indexes[j] * M;
      for (k=0; k<M; k++)
        if ((val = *values++) < ptr[k]) ptr[k] = val;
    }

} /* dalib_setn_min_ints */

void dalib_set_min_doubles (data, indexes, values, N)
DOUBLE_PRECISION *data, *values;
int N, indexes[];

{ int j;
  DOUBLE_PRECISION *ptr;

  for (j=0; j<N; j++)
    { /* a[indexes[j]] = min (a[indexes[j], values[j])  */
      ptr = data + indexes[j];
      if (values[j] < *ptr) *ptr = values[j];
    }
} /* dalib_set_min_doubles */

void dalib_setn_min_doubles (data, indexes, values, M, N)
DOUBLE_PRECISION *data, *values;
INTEGER indexes[];
int M, N;

{ int j,k;
  DOUBLE_PRECISION *ptr, val;

  for (j=0; j<N; j++)
    { ptr = data + indexes[j] * M;
      for (k=0; k<M; k++)
        if ((val = *values++) < ptr[k]) ptr[k] = val;
    }

} /* dalib_setn_min_doubles */

          /*************************************
          *                                    *
          *  M I N V A L  with   M A S K       *
          *                                    *
          *************************************/

void dalib_setm_min_ints (data, indexes, values, mask, N)
INTEGER *data, *values;
int N, indexes[], mask[];

{ int j;
  INTEGER *ptr;

  for (j=0; j<N; j++)
    if (mask[j])
       { /* a[indexes[j]] = min (a[indexes[j], values[j])  */
         ptr = data + indexes[j];
         if (values[j] < *ptr) *ptr = values[j];
       }
} /* dalib_setm_min_ints */

void dalib_setm_min_reals (data, indexes, values, mask, N)
REAL *data, *values;
int N, indexes[], mask[];

{ int j;
  REAL *ptr;

  for (j=0; j<N; j++)
    if (mask[j])
       { /* a[indexes[j]] = min (a[indexes[j], values[j])  */
         ptr = data + indexes[j];
         if (values[j] < *ptr) *ptr = values[j];
       }
} /* dalib_setm_min_ints */

void dalib_setm_min_doubles (data, indexes, values, mask, N)
DOUBLE_PRECISION *data, *values;
int N, indexes[], mask[];

{ int j;
  DOUBLE_PRECISION *ptr;

  for (j=0; j<N; j++)
    if (mask[j])
       { /* a[indexes[j]] = min (a[indexes[j], values[j])  */
         ptr = data + indexes[j];
         if (values[j] < *ptr) *ptr = values[j];
       }
} /* dalib_setm_min_doubles */

          /*************************************
          *                                    *
          *  M A X V A L                       *
          *                                    *
          *************************************/

void dalib_set_max_ints (data, indexes, values, N)

INTEGER *data, *values;
int N, indexes[];

{ int j;
  int *ptr;

  for (j=0; j<N; j++)
    { /* a[indexes[j]] = max (a[indexes[j], values[j])  */
      ptr = data + indexes[j];
      if (values[j] > *ptr) *ptr = values[j];
    }
} /* dalib_set_max_ints */

void dalib_setn_max_ints (data, indexes, values, M, N)
INTEGER *data, *values;
INTEGER indexes[];
int M, N;

{ int j,k;
  INTEGER *ptr, val;

  for (j=0; j<N; j++)
    { ptr = data + indexes[j] * M;
      for (k=0; k<M; k++)
        if ((val = *values++) > ptr[k]) ptr[k] = val;
    }

} /* dalib_setn_max_ints */

void dalib_set_max_reals (data, indexes, values, N)
REAL *data, *values;
INTEGER indexes[];
int N;

{ int j;
  REAL *ptr;

  for (j=0; j<N; j++) 
    { /* a[indexes[j]] = max (a[indexes[j], values[j])  */
      ptr = data + indexes[j];
      if (values[j] > *ptr) *ptr = values[j];
    }

} /* dalib_set_max_ints */

void dalib_setn_max_reals (data, indexes, values, M, N)
REAL *data, *values;
INTEGER indexes[];
int M, N;

{ int j,k;
  REAL *ptr, val;

  for (j=0; j<N; j++)
    { ptr = data + indexes[j] * M;
      for (k=0; k<M; k++)
        if ((val = *values++) > ptr[k]) ptr[k] = val;
    }

} /* dalib_setn_max_reals */

void dalib_set_max_doubles (data, indexes, values, N)
DOUBLE_PRECISION *data, *values;
INTEGER indexes[];
int N;

{ int j;
  DOUBLE_PRECISION *ptr;

  for (j=0; j<N; j++)
    { /* a[indexes[j]] = max (a[indexes[j], values[j])  */
      ptr = data + indexes[j];
      if (values[j] > *ptr) *ptr = values[j];
    }
} /* dalib_set_max_doubles */

void dalib_setn_max_doubles (data, indexes, values, M, N)
DOUBLE_PRECISION *data, *values;
INTEGER indexes[];
int M, N;

{ int j,k;
  DOUBLE_PRECISION *ptr, val;

  for (j=0; j<N; j++)
    { ptr = data + indexes[j] * M;
      for (k=0; k<M; k++)
        if ((val = *values++) > ptr[k]) ptr[k] = val;
    }

} /* dalib_setn_max_doubles */

          /*************************************
          *                                    *
          *  M A X V A L  with   M A S K       *
          *                                    *
          *************************************/

void dalib_setm_max_ints (data, indexes, values, mask, N)
INTEGER *data, *values;
int N, indexes[], mask[];

{ int j;
  int *ptr;

  for (j=0; j<N; j++)
    if (mask[j])
       { /* a[indexes[j]] = max (a[indexes[j], values[j])  */
         ptr = data + indexes[j];
         if (values[j] > *ptr) *ptr = values[j];
       }
} /* dalib_setm_max_ints */

void dalib_setm_max_reals (data, indexes, values, mask, N)
REAL *data, *values;
int N, indexes[], mask[];

{ int j;
  REAL *ptr;

  for (j=0; j<N; j++)
    if (mask[j])
       { /* a[indexes[j]] = max (a[indexes[j], values[j])  */
         ptr = data + indexes[j];
         if (values[j] > *ptr) *ptr = values[j];
       }
} /* dalib_setm_max_ints */

void dalib_setm_max_doubles (data, indexes, values, mask, N)
DOUBLE_PRECISION *data, *values;
int N, indexes[], mask[];

{ int j;
  DOUBLE_PRECISION *ptr;

  for (j=0; j<N; j++)
    if (mask[j])
       { /* a[indexes[j]] = max (a[indexes[j], values[j])  */
         ptr = data + indexes[j];
         if (values[j] > *ptr) *ptr = values[j];
       }
} /* dalib_setm_max_doubles */

          /*************************************
          *                                    *
          *  S U M                             *
          *                                    *
          *************************************/

void dalib_set_add_ints (data, indexes, values, N)
INTEGER *data, *values;
int N, indexes[];

{ int j;
  for (j=0; j<N; j++) data[indexes[j]] += values[j];

} /* dalib_set_add_ints */

void dalib_setn_add_ints (data, indexes, values, M, N)
INTEGER *data, *values;
INTEGER indexes[];
int M, N;

{ int j,k;
  INTEGER *ptr;

  for (j=0; j<N; j++)
    { ptr = data + indexes[j] * M;
      for (k=0;k<M;k++) *ptr++ += *values++;
    }
} /* dalib_setn_add_reals */

void dalib_set_add_reals (data, indexes, values, N)
REAL *data, *values;
INTEGER indexes[];
int N;

{ int j;

  for (j=0; j<N; j++) data[indexes[j]] += values[j];

} /* dalib_set_add_reals */

void dalib_setn_add_reals (data, indexes, values, M, N)
REAL *data, *values;
int M, N, indexes[];

{ int j,k;
  REAL *ptr;

  for (j=0; j<N; j++)
    { ptr = data + indexes[j] * M;
      for (k=0;k<M;k++) *ptr++ += *values++;
    }
} /* dalib_setn_add_reals */

void dalib_set_add_doubles (data, indexes, values, N)
DOUBLE_PRECISION *data, *values;
int N, indexes[];

{ int j;
  DOUBLE_PRECISION *ptr;

  for (j=0; j<N; j++)
    { /* a[indexes[j]] = a[indexes[j] + values[j]  */
      ptr = data + indexes[j];
      *ptr += values[j];
    }
} /* dalib_set_add_doubles */

void dalib_setn_add_doubles (data, indexes, values, M, N)
DOUBLE_PRECISION *data, *values;
int M, N, indexes[];

{ int j,k;
  DOUBLE_PRECISION *ptr;

  for (j=0; j<N; j++)
    { ptr = data + indexes[j] * M;
      for (k=0;k<M;k++) *ptr++ += *values++;
    }

} /* dalib_setn_add_doubles */


          /*************************************
          *                                    *
          *    S U M      with   M A S K       *
          *                                    *
          *************************************/

void dalib_setm_add_ints (data, indexes, values, mask, N)
INTEGER *data, *values;
int N, indexes[], mask[];

{ int j;
  int *ptr;

  for (j=0; j<N; j++)
    if (mask[j])
       { /* a[indexes[j]] = a[indexes[j] + values[j]  */
         ptr = data + indexes[j];
         *ptr += values[j];
       }
} /* dalib_setm_add_ints */

void dalib_setm_add_reals (data, indexes, values, mask, N)
REAL *data, *values;
int N, indexes[], mask[];

{ int j;
  REAL *ptr;

  for (j=0; j<N; j++)
    if (mask[j])
       { /* a[indexes[j]] = a[indexes[j] + values[j]  */
         ptr = data + indexes[j];
         *ptr += values[j];
       }
} /* dalib_setm_add_reals */

void dalib_setm_add_doubles (data, indexes, values, mask, N)
DOUBLE_PRECISION *data, *values;
int N, indexes[], mask[];

{ int j;
  DOUBLE_PRECISION *ptr;

  for (j=0; j<N; j++)
    if (mask[j])
       { /* a[indexes[j]] = a[indexes[j] + values[j]  */
         ptr = data + indexes[j];
         *ptr += values[j];
       }
} /* dalib_setm_add_doubles */

          /*************************************
          *                                    *
          *  P R O D U C T                     *
          *                                    *
          *************************************/

void dalib_set_mult_ints (data, indexes, values, N)
INTEGER *data, *values;
INTEGER indexes[];
int N;

{ int j;

  for (j=0; j<N; j++) data[indexes[j]] *= values[j];

} /* dalib_set_mult_ints */

void dalib_setn_mult_ints (data, indexes, values, M, N)
INTEGER *data, *values;
INTEGER indexes[];
int M, N;

{ int j,k;
  INTEGER *ptr;

  for (j=0; j<N; j++)
    { ptr = data + indexes[j] * M;
      for (k=0;k<M;k++) *ptr++ *= *values++;
    }
} /* dalib_setn_mult_ints */

void dalib_set_mult_reals (data, indexes, values, N)
REAL *data, *values;
INTEGER indexes[];
int N;

{ int j;

  for (j=0; j<N; j++) data[indexes[j]] *= *values++;

} /* dalib_set_mult_reals */

void dalib_setn_mult_reals (data, indexes, values, M, N)
REAL *data, *values;
INTEGER indexes[];
int M, N;

{ int j,k;
  REAL *ptr;

  for (j=0; j<N; j++)
    { ptr = data + indexes[j] * M;
      for (k=0;k<M;k++) *ptr++ *= *values++;
    }

} /* dalib_setn_mult_reals */

void dalib_set_mult_doubles (data, indexes, values, N)
DOUBLE_PRECISION *data, *values;
int N, indexes[];

{ int j;
  DOUBLE_PRECISION *ptr;

  for (j=0; j<N; j++)
    { /* a[indexes[j]] = a[indexes[j] * values[j]  */
      ptr = data + indexes[j];
      *ptr *= values[j];
    }
} /* dalib_set_mult_doubles */

void dalib_setn_mult_doubles (data, indexes, values, M, N)
DOUBLE_PRECISION *data, *values;
INTEGER indexes[];
int M, N;

{ int j,k;
  DOUBLE_PRECISION *ptr;

  for (j=0; j<N; j++)
    { ptr = data + indexes[j] * M;
      for (k=0;k<M;k++) *ptr++ *= *values++;
    }

} /* dalib_setn_mult_doubles */

          /*************************************
          *                                    *
          *  P R O D U C T   with   M A S K    *
          *                                    *
          *************************************/

void dalib_setm_mult_ints (data, indexes, values, mask, N)
int *data, *values;
int N, indexes[], mask[];

{ int j;
  int *ptr;

  for (j=0; j<N; j++)
    if (mask[j])
       { /* a[indexes[j]] = a[indexes[j] * values[j]  */
         ptr = data + indexes[j];
         *ptr *= values[j];
       }
} /* dalib_setm_mult_ints */

void dalib_setm_mult_reals (data, indexes, values, mask, N)
REAL *data, *values;
int N, indexes[], mask[];

{ int j;
  REAL *ptr;

  for (j=0; j<N; j++)
    if (mask[j])
       { /* a[indexes[j]] = a[indexes[j] * values[j]  */
         ptr = data + indexes[j];
         *ptr *= values[j];
       }
} /* dalib_setm_mult_reals */

void dalib_setm_mult_doubles (data, indexes, values, mask, N)
DOUBLE_PRECISION *data, *values;
int N, indexes[], mask[];

{ int j;
  DOUBLE_PRECISION *ptr;

  for (j=0; j<N; j++)
    if (mask[j])
       { /* a[indexes[j]] = a[indexes[j] * values[j]  */
         ptr = data + indexes[j];
         *ptr *= values[j];
       }
} /* dalib_setm_mult_doubles */

          /*************************************
          *                                    *
          *  AND, OR, EOR                      *
          *                                    *
          *************************************/

void dalib_set_and_ints (data, indexes, values, N)
INTEGER *data, *values;
INTEGER indexes[];
int N;

{ int j;

  for (j=0; j<N; j++) data[indexes[j]] &= *values++;

} /* dalib_set_and_ints */

void dalib_setn_and_ints (data, indexes, values, M, N)
INTEGER *data, *values;
INTEGER indexes[];
int M, N;

{ int j,k;
  INTEGER *ptr;

  for (j=0; j<N; j++)
    { ptr = data + indexes[j] * M;
      for (k=0; k<M; k++) *ptr++ &= *values++;
    }

} /* dalib_setn_and_ints */

void dalib_set_or_ints (data, indexes, values, N)
INTEGER *data, *values;
INTEGER indexes[];
int N;

{ int j;

  for (j=0; j<N; j++) data[indexes[j]] |= *values++;

} /* dalib_set_or_ints */

void dalib_setn_or_ints (data, indexes, values, M, N)
INTEGER *data, *values;
INTEGER indexes[];
int M, N;

{ int j,k;
  INTEGER *ptr;

  for (j=0; j<N; j++)
    { ptr = data + indexes[j] * M;
      for (k=0; k<M; k++) *ptr++ |= *values++;
    }

} /* dalib_setn_or_ints */

void dalib_set_eor_ints (data, indexes, values, N)
INTEGER *data, *values;
INTEGER indexes[];
int N;

{ int j;

  for (j=0; j<N; j++) data[indexes[j]] ^= *values++;

} /* dalib_set_eor_ints */

void dalib_setn_eor_ints (data, indexes, values, M, N)
INTEGER *data, *values;
INTEGER indexes[];
int M, N;

{ int j,k;
  INTEGER *ptr;

  for (j=0; j<N; j++)
    { ptr = data + indexes[j] * M;
      for (k=0; k<M; k++) *ptr++ ^= *values++;
    }

} /* dalib_setn_or_ints */

void dalib_setm_and_ints (data, indexes, values, mask, N)
INTEGER *data, *values;
INTEGER indexes[], mask[];
int N;

{ int j;

  for (j=0; j<N; j++)
    if (mask[j]) data[indexes[j]] |= values[j];

} /* dalib_setm_and_ints */

void dalib_setm_or_ints (data, indexes, values, mask, N)
int *data, *values;
int N, indexes[], mask[];

{ int j;
  int *ptr;

  for (j=0; j<N; j++)
    if (mask[j])
       { /* a[indexes[j]] = a[indexes[j] | values[j]  */
         ptr = data + indexes[j];
         *ptr = *ptr | values[j];
       }
} /* dalib_setm_or_ints */

void dalib_setm_eor_ints (data, indexes, values, mask, N)
int *data, *values;
int N, indexes[], mask[];

{ int j;
  int *ptr;

  for (j=0; j<N; j++)
    if (mask[j])
       { /* a[indexes[j]] = a[indexes[j] ^ values[j]  */
         ptr = data + indexes[j];
         *ptr = *ptr ^ values[j];
       }
} /* dalib_setm_eor_ints */

          /*************************************
          *                                    *
          *  ALL, ANY, PARITY                  *
          *                                    *
          *************************************/

void dalib_set_or_bools (data, indexes, values, N)
INTEGER *data, *values;
int N, indexes[];

{ int j;
  int *ptr;

  for (j=0; j<N; j++)
    { /* a[indexes[j]] = a[indexes[j] || values[j]  */
      ptr = data + indexes[j];
      *ptr = *ptr || values[j];
    }
} /* dalib_set_or_bools */

void dalib_setn_or_bools (data, indexes, values, M, N)
INTEGER *data, *values;
INTEGER indexes[];
int M, N;

{ int j,k;
  INTEGER *ptr;

  for (j=0; j<N; j++)
    { ptr = data + indexes[j] * M;
      for (k=0; k<M; k++) *ptr++ = *ptr || *values++;
    }

} /* dalib_setn_or_bools */

void dalib_set_and_bools (data, indexes, values, N)
int *data, *values;
int N, indexes[];

{ int j;
  int *ptr;

  for (j=0; j<N; j++)
    { /* a[indexes[j]] = a[indexes[j] || values[j]  */
      ptr = data + indexes[j];
      *ptr = *ptr && values[j];
    }
} /* dalib_set_and_bools */

void dalib_setn_and_bools (data, indexes, values, M, N)
INTEGER *data, *values;
INTEGER indexes[];
int M, N;

{ int j,k;
  INTEGER *ptr;

  for (j=0; j<N; j++)
    { ptr = data + indexes[j] * M;
      for (k=0; k<M; k++) *ptr++ = *ptr && *values++;
    }

} /* dalib_setn_or_bools */

void dalib_set_neq_bools (data, indexes, values, N)
INTEGER *data, *values;
INTEGER indexes[];
int N;

{ int j;
  INTEGER *ptr;

  for (j=0; j<N; j++)

    { ptr = data + indexes[j];
      if (*values++) *ptr = !(*ptr);
    }

} /* dalib_set_neq_bools */

void dalib_setn_neq_bools (data, indexes, values, M, N)
INTEGER *data, *values;
INTEGER indexes[];
int M, N;

{ int j,k;
  INTEGER *ptr;

  for (j=0; j<N; j++)
    { ptr = data + indexes[j] * M;
      for (k=0; k<M; k++) if (*values++) ptr[k] = !(ptr[k]);
    }

} /* dalib_setn_or_bools */

void dalib_setm_or_bools (data, indexes, values, mask, N)
INTEGER *data, *values;
int N, indexes[], mask[];

{ int j;
  int *ptr;

  for (j=0; j<N; j++)
    if (mask[j])
       { /* a[indexes[j]] = a[indexes[j] || values[j]  */
         ptr = data + indexes[j];
         *ptr = *ptr || values[j];
       }
} /* dalib_setm_or_bools */

void dalib_setm_and_bools (data, indexes, values, mask, N)
INTEGER *data, *values;
int N, indexes[], mask[];

{ int j;
  int *ptr;

  for (j=0; j<N; j++)
    if (mask[j])
       { /* a[indexes[j]] = a[indexes[j] || values[j]  */
         ptr = data + indexes[j];
         *ptr = *ptr && values[j];
       }
} /* dalib_setm_and_bools */

void dalib_setm_neq_bools (data, indexes, values, mask, N)

INTEGER *data, *values;
int N, indexes[], mask[];

{ int j;
  int *ptr;

  for (j=0; j<N; j++)
    if (mask[j])
       { /* a[indexes[j]] = a[indexes[j] || values[j]  */
         ptr = data + indexes[j];
         if (values[j]) *ptr = !(*ptr);
       }

} /* dalib_setm_neq_bools */

          /*************************************
          *                                    *
          *  MAIN FUNCTION FOR SET             *
          *                                    *
          *************************************/

void dalib_set (op, b, indexes, values, n)

int op, n, indexes[];
void *b, *values;       /* can be INTEGER, REAL, DOUBLE_PRECISION */

{ switch (op) {

  case  1 : dalib_set_min_ints ((INTEGER *) b, indexes,
                                (INTEGER *) values, n); break;
  case  2 : dalib_set_min_reals ((REAL *) b, indexes,
                                 (REAL *) values, n); break;
  case  3 : dalib_set_min_doubles ((DOUBLE_PRECISION *) b, indexes, 
                                   (DOUBLE_PRECISION *) values, n); break;
  case  4 : dalib_set_max_ints ((INTEGER *) b, indexes,
                                (INTEGER *) values, n); break;
  case  5 : dalib_set_max_reals ((REAL *) b, indexes,
                                 (REAL *) values, n); break;
  case  6 : dalib_set_max_doubles ((DOUBLE_PRECISION *) b, indexes,
                                (DOUBLE_PRECISION *) values, n); break;
  case  7 : dalib_set_add_ints ((INTEGER *) b, indexes,
                                (INTEGER *) values, n); break;
  case  8 : dalib_set_add_reals ((REAL *) b, indexes,
                                (REAL *) values, n); break;
  case  9 : dalib_set_add_doubles ((DOUBLE_PRECISION *) b, indexes,
                                (DOUBLE_PRECISION *) values, n); break;
  case 10 : dalib_set_mult_ints ((INTEGER *) b, indexes,
                                (INTEGER *) values, n); break;
  case 11 : dalib_set_mult_reals ((REAL *) b, indexes,
                                (REAL *) values, n); break;
  case 12 : dalib_set_mult_doubles ((DOUBLE_PRECISION *) b, indexes,
                                (DOUBLE_PRECISION *) values, n); break;
  case 13 : dalib_set_and_ints ((INTEGER *) b, indexes,
                                (INTEGER *) values, n); break;
  case 14 : dalib_set_or_ints ((INTEGER *) b, indexes,
                                (INTEGER *) values, n); break;
  case 15 : dalib_set_eor_ints ((INTEGER *) b, indexes,
                                (INTEGER *) values, n); break;
  case 16 : dalib_set_and_bools ((INTEGER *) b, indexes,
                                (INTEGER *) values, n); break;
  case 17 : dalib_set_or_bools ((INTEGER *) b, indexes,
                                (INTEGER *) values, n); break;
  case 18 : dalib_set_neq_bools ((INTEGER *) b, indexes,
                                (INTEGER *) values, n); break;

  default : fprintf (stderr, "unknown op for dalib_set"); exit(-1); break;

  }  /* switch */

} /* dalib_set */

          /*************************************
          *                                    *
          *  MAIN FUNCTION FOR SET             *
          *                                    *
          *************************************/

void dalib_setn (op, b, indexes, values, n, k)

INTEGER indexes[];
int op, n, k;
void *b, *values;       /* can be INTEGER, REAL, DOUBLE_PRECISION */

{ switch (op) {

  case  1 : dalib_setn_min_ints ((INTEGER *) b, indexes,
                                (INTEGER *) values, k, n); break;
  case  2 : dalib_setn_min_reals ((REAL *) b, indexes,
                                 (REAL *) values, k, n); break;
  case  3 : dalib_setn_min_doubles ((DOUBLE_PRECISION *) b, indexes,
                                   (DOUBLE_PRECISION *) values, k, n); break;
  case  4 : dalib_setn_max_ints ((INTEGER *) b, indexes,
                                (INTEGER *) values, k, n); break;
  case  5 : dalib_setn_max_reals ((REAL *) b, indexes,
                                 (REAL *) values, k, n); break;
  case  6 : dalib_setn_max_doubles ((DOUBLE_PRECISION *) b, indexes,
                                   (DOUBLE_PRECISION *) values, k, n); break;
  case  7 : dalib_setn_add_ints ((INTEGER *) b, indexes,
                                 (INTEGER *) values, k, n); break;
  case  8 : dalib_setn_add_reals ((REAL *) b, indexes,
                                  (REAL *) values, k, n); break;
  case  9 : dalib_setn_add_doubles ((DOUBLE_PRECISION *) b, indexes,
                                    (DOUBLE_PRECISION *) values, k, n); break;
  case 10 : dalib_setn_mult_ints ((INTEGER *) b, indexes,
                                  (INTEGER *) values, k, n); break;
  case 11 : dalib_setn_mult_reals ((REAL *) b, indexes,
                                   (REAL *) values, k, n); break;
  case 12 : dalib_setn_mult_doubles ((DOUBLE_PRECISION *) b, indexes,
                                     (DOUBLE_PRECISION *) values, k, n); break;
  case 13 : dalib_setn_and_ints ((INTEGER *) b, indexes,
                                (INTEGER *) values, k, n); break;
  case 14 : dalib_setn_or_ints ((INTEGER *) b, indexes,
                                (INTEGER *) values, k, n); break;
  case 15 : dalib_setn_eor_ints ((INTEGER *) b, indexes,
                                (INTEGER *) values, k, n); break;
  case 16 : dalib_setn_and_bools ((INTEGER *) b, indexes,
                                (INTEGER *) values, k, n); break;
  case 17 : dalib_setn_or_bools ((INTEGER *) b, indexes,
                                (INTEGER *) values, k, n); break;
  case 18 : dalib_setn_neq_bools ((INTEGER *) b, indexes,
                                (INTEGER *) values, k, n); break;

  default : fprintf (stderr, "unknown op for dalib_set"); exit(-1); break;

  }  /* switch */

} /* dalib_set */

          /*************************************
          *                                    *
          *  MAIN FUNCTION FOR SET WITH MASK   *
          *                                    *
          *************************************/

void dalib_setm (op, b, indexes, values, mask, n)

int op, n, indexes[], mask[]; 

void *b, *values;              /* can be INTEGER, REAL, DOUBLE_PRECISION */

{ switch (op) {
  case  1 : dalib_setm_min_ints ((INTEGER *) b, indexes,
                                 (INTEGER *) values, mask, n); break;
  case  2 : dalib_setm_min_reals ((REAL *) b, indexes,
                                  (REAL *) values, mask, n); break;
  case  3 : dalib_setm_min_doubles ((DOUBLE_PRECISION *) b, indexes,
                                    (DOUBLE_PRECISION *) values, mask, n); break;
  case  4 : dalib_setm_max_ints ((INTEGER *) b, indexes,
                                 (INTEGER *) values, mask, n); break;
  case  5 : dalib_setm_max_reals ((REAL *) b, indexes,
                                  (REAL *) values, mask, n); break;
  case  6 : dalib_setm_max_doubles ((DOUBLE_PRECISION *) b, indexes,
                                    (DOUBLE_PRECISION *) values, mask, n); break;
  case  7 : dalib_setm_add_ints ((INTEGER *) b, indexes,
                                 (INTEGER *) values, mask, n); break;
  case  8 : dalib_setm_add_reals ((REAL *) b, indexes,
                                  (REAL *) values, mask, n); break;
  case  9 : dalib_setm_add_doubles ((DOUBLE_PRECISION *) b, indexes,
                                    (DOUBLE_PRECISION *) values, mask, n); break;
  case 10 : dalib_setm_mult_ints ((INTEGER *) b, indexes,
                                  (INTEGER *) values, mask, n); break;
  case 11 : dalib_setm_mult_reals ((REAL *) b, indexes,
                                   (REAL *) values, mask, n); break;
  case 12 : dalib_setm_mult_doubles ((DOUBLE_PRECISION *) b, indexes,
                                    (DOUBLE_PRECISION *) values, mask, n); break;
  case 13 : dalib_setm_and_ints ((INTEGER *) b, indexes,
                                 (INTEGER *) values, mask, n); break;
  case 14 : dalib_setm_or_ints ((INTEGER *) b, indexes,
                                (INTEGER *) values, mask, n); break;
  case 15 : dalib_setm_eor_ints ((INTEGER *) b, indexes,
                                 (INTEGER *) values, mask, n); break;
  case 16 : dalib_setm_and_bools ((INTEGER *) b, indexes,
                                  (INTEGER *) values, mask, n); break;
  case 17 : dalib_setm_or_bools ((INTEGER *) b, indexes,
                                 (INTEGER *) values, mask, n); break;
  case 18 : dalib_setm_neq_bools ((INTEGER *) b, indexes,
                                  (INTEGER *) values, mask, n); break;
  default : fprintf (stderr, "unknown op for dalib_set"); exit(-1); break;
  }
} /* dalib_setm */
