/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Mar 96                                                   *
*  Last Update : May 96                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : match                                                    *
*                                                                         *
*  Function    : matching up descriptors / copy in - copy out             *
*                                                                         *
*  Changes:                                                               *
*                                                                         *
*   05/1996 : problem with block -> replicated if one processor           *
*             contains all the date (use_dummy_data must not give true)   *
*                                                                         *
**************************************************************************/

#undef DEBUG

#include "dalib.h"

/*******************************************************************
*                                                                  *
*  PREDICATE dalib_has_replication (target_dsp, source_dsp)        *
*                                                                  *
*    - returns true if assignment target = source requires         *
*      at least the broadcast of some data                         *
*    - broadcast of source might also require accepting msg        *
*                                                                  *
*******************************************************************/

static int dalib_has_replication (target_dsp, source_dsp)

array_info target_dsp, source_dsp;

{ /* case 1 : if target and source are not replicated at all we must
              certainly not copy any data                             */

  if(  (!dalib_is_replicated (source_dsp))
    && (!dalib_is_replicated (target_dsp)) )  return (0);
 
  /* case 2 : if source is full replicated, this processor is not
              involved in any communication                           */

  if (   (source_dsp->AlignInfo == NO_ALIGNMENT)
      && (source_dsp->DistributeInfo == NO_DISTRIBUTION) )

     return (0);   /* source is full replicated */

  return (1);   /* make worse assumption */

} /* dalib_has_replication */

/**********************************************************************
*                                                                     *
*  int dalib_inherit_mapping (array_info dummy_dsp, actual_dsp)       *
*                                                                     *
*  dummy  : !HPF$ DISTRIBUTE (BLOCK(),*) ONTO *                       *
*                                                                     *
*  - dummy inherits mapping of actual (returns 1)                     *
*  - choose an arbitrary mapping for dummy (returns 0)                *
*                                                                     *
*  - returns 1 if mapping has been inherited (no redistribution)      *
*                                                                     *
**********************************************************************/

static int dalib_inherit_mapping (dummy_dsp, actual_dsp)

array_info dummy_dsp, actual_dsp;

{ int done;

  if (!dalib_array_map_underspecified (dummy_dsp))

     {
#ifdef DEBUG
    printf ("%d: dummy is full specified, so we cannot inherit mapping\n",
             pcb.i);
#endif
       return (0);     /* so we cannot map anything at all */
     }

  if (dalib_is_map_specialization (actual_dsp, dummy_dsp))

     { /* actual matches dummy, so we inherit everything for the dummy */

#ifdef DEBUG
  printf ("%d: actual %d is specialization of dummy %d (inherit mapping)\n",
           pcb.i, actual_dsp, dummy_dsp);
#endif

       dalib_array_inherit_mapping (dummy_dsp, actual_dsp);
       dalib_get_actual_info (dummy_dsp, actual_dsp);
       done = 1;

     }

   else

     { /* choose an arbitrary full distribution for the dummy */

#ifdef DEBUG
  printf ("%d: actual %d is not specialization of dummy %d (choose mapping)\n",
           pcb.i, actual_dsp, dummy_dsp);
#endif

       dalib_array_make_full_mapping (dummy_dsp);
       done = 0;

     }

    /* now we can compute the local sizes of dummy_dsp */

   if (dummy_dsp->DistributeInfo != NO_DISTRIBUTION)
      dalib_dist_local_sizes (dummy_dsp);
    else if (dummy_dsp->AlignInfo != NO_ALIGNMENT)
      dalib_align_local_sizes (dummy_dsp);
    else
      dalib_full_local_sizes (dummy_dsp);

   return (done);

} /* dalib_inherit_mapping */

/*******************************************************************
*                                                                  *
*                                                                  *
*                                                                  *
*******************************************************************/

void dalib_local_copy_in (dummy_dsp, actual_dsp, copy_flag)

array_info dummy_dsp;
array_info actual_dsp;
int        copy_flag;

{ DimInfo *actual_dims;
  DimInfo *dummy_dims;

  int i, rank;

#ifdef DEBUG
  printf ("%d: dalib_local_copy_in (dummy=%d,actual=%d,copy=%d)\n",
          pcb.i, dummy_dsp, actual_dsp, copy_flag);
  dalib_print_array_info (actual_dsp);
#endif

  if (dalib_is_array_info (actual_dsp))

     { /* inherit the local sizes as global sizes */

       actual_dims = actual_dsp->dimensions;
       dummy_dims  = dummy_dsp->dimensions;

       rank = dummy_dsp->rank;

       for (i=0; i<rank; i++)

         { dummy_dims->global_size [0] = actual_dims->local_size[0];
           dummy_dims->global_size [1] = actual_dims->local_size[1];

           dummy_dims->overlap     [0] = actual_dims->overlap[0];
           dummy_dims->overlap     [1] = actual_dims->overlap[1];

           dummy_dims->local_size  [0] = dummy_dims->global_size[0];
           dummy_dims->local_size  [1] = dummy_dims->global_size[1];
           dummy_dims->local_size  [2] = 1;

           dummy_dims->map_flag = 0;

           actual_dims++; dummy_dims++;

         }

       dummy_dsp->data = actual_dsp->data;
       dummy_dsp->dsp_status_flag = DSP_PTR_DATA;

#ifdef DEBUG
       printf ("%d: dalib_local_copy_in, local dsp = \n", pcb.i);
       dalib_print_array_info (dummy_dsp);
#endif

     }

    else if (dalib_is_section_info (actual_dsp))

     { section_info actual_section;
       SecDimInfo *actual_dims;

       dalib_internal_error (
              "only full arrays can be passed to LOCAL routines");
       dalib_stop ();

       actual_section = (section_info) actual_dsp;

       actual_dims = actual_section->dimensions;
       dummy_dims  = actual_dsp->dimensions;

       rank = actual_section->array_id->rank;

       if (dalib_section_rank (actual_section) != dummy_dsp->rank)

           { dalib_internal_error ("dalib_local_copy_in");
             dalib_stop ();
           }

       for (i=0; i<rank; i++)

         { dummy_dims->global_size [0] = actual_dims->local_range[0];
           dummy_dims->global_size [1] = actual_dims->local_range[1];

           dummy_dims->local_size  [0] = dummy_dims->global_size[0];
           dummy_dims->local_size  [1] = dummy_dims->global_size[1];
           dummy_dims->local_size  [2] = 1;

           dummy_dims->map_flag = 0;
         }

     }

} /* dalib_local_copy_in */

void dalib_local_copy_out (dummy_dsp, actual_dsp, copy_flag)

array_info dummy_dsp;
array_info actual_dsp;
int        copy_flag;

{ /* currently nothing to do as data is always inherited here */

#ifdef DEBUG
  printf ("%d: dalib_local_copy_out (dummy=%d,actual=%d,copy=%d)\n",
          pcb.i, dummy_dsp, actual_dsp, copy_flag);
#endif

} /* dalib_local_copy_out */

/*******************************************************************
*                                                                  *
*  FORTRAN Interface                                               *
*                                                                  *
*  - implicit redisribution for dummy arrays                       *
*                                                                  *
*  FUNCTION(dalib_copy_in) (dummy_dsp, actual_dsp,                 *
*                           actual_data, copy_flag)                *
*                                                                  *
*  FUNCTION(dalib_copy_out) (local_dsp, actual_dsp, copy_flag)     *
*                                                                  *
*******************************************************************/

void FUNCTION(dalib_copy_in) (dummy_dsp, actual_dsp, actual_data, copy_flag) 

array_info *dummy_dsp;
array_info *actual_dsp;
char       *actual_data;
int        *copy_flag;

{ if (!FUNCTION(dalib_present) (actual_dsp))

     { /* actual argument is a pointer */

       if (   ((*dummy_dsp)->DistributeInfo == NO_DISTRIBUTION)
           && ((*dummy_dsp)->AlignInfo      == NO_ALIGNMENT)    )

          { /* dummy will be descriptor for replicated data */
            
            FUNCTION(dalib_array_setdata) (dummy_dsp, actual_data);

            return;  /* that works fine */
          }

       dalib_internal_error ("copy_in : actual argument has no descriptor");
       dalib_stop ();
       return;
     }

  dalib_copy_valid_info (*dummy_dsp, *actual_dsp, 0);

  /* destroy flag is false as actual descriptor will not be destroyed */

  if (*dummy_dsp == *actual_dsp)
     return;                       /* actual descriptor is just dummy */

  /* for underspecified mappings we have to inherit the mapping and 
     to compute the local sizes of the dummy descriptor              */

  if (dalib_inherit_mapping (*dummy_dsp, *actual_dsp)) return;

  /* now dummy has in any case a full specified mapping */

  if (dalib_use_actual_data (*dummy_dsp, *actual_dsp))

     {  dalib_get_actual_info (*dummy_dsp, *actual_dsp);

#ifdef DEBUG
        printf ("%d: copy in, local = %d, dummy = %d, not necessary\n",
                 pcb.i, *dummy_dsp, *actual_dsp);
#endif

        /* assign might be necessary if other processors want us
           to be involved in their assignment                     */

        if (*copy_flag != 0)

           { if (dalib_has_replication (*dummy_dsp, *actual_dsp))

                 FUNCTION(dalib_assign) (dummy_dsp, actual_dsp); 
           }

        return;
     }

  /* new descriptor, so new data is needed, dsp_status_flag == OWN_DATA */

#ifdef DEBUG
  printf ("%d: copy in, local = %d, dummy = %d, new data necessary\n",
               pcb.i, *dummy_dsp, *actual_dsp);
#endif

  dalib_array_shared_malloc (*dummy_dsp);        /* allocate memory */

  FUNCTION(dalib_array_set_dirty) (dummy_dsp);

  /* so copy in might be necessary */

  if (*copy_flag != 0)

     { char msg[150];

       sprintf (msg, "copy in redistribution for dummy %s\n", 
                     (*dummy_dsp)->name);

       if (pcb.redist_flag) dalib_attention (msg);

       FUNCTION(dalib_assign) (dummy_dsp, actual_dsp); 
     }

#ifdef DEBUG
  printf ("copy in finished\n");
#endif 

} /* dalib_copy_in */

void FUNCTION(dalib_ccopy_in) 

  ARGS(`dummy_dsp, actual_dsp, STRING_ARG(actual_data), copy_flag') 

array_info *dummy_dsp;
array_info *actual_dsp;
STRING_ARG_DECL(actual_data);
int        *copy_flag;

{ FUNCTION(dalib_copy_in) (dummy_dsp, actual_dsp,
                           STRING_PTR(actual_data), copy_flag);
}

/*******************************************************************
*                                                                  *
*  FORTRAN Interface                                               *
*                                                                  *
*  FUNCTION(dalib_copy_out) (dummy_dsp, actual_dsp, copy_flag)      *
*                                                                  *
*    - copy local array back to dummy array                        *
*    - free the new descriptor                                     *
*                                                                  *
*******************************************************************/

void FUNCTION(dalib_copy_out) (dummy_dsp, actual_dsp, copy_flag) 

array_info *dummy_dsp;
array_info *actual_dsp;
int        *copy_flag;

{ if (!FUNCTION(dalib_present) (actual_dsp))

     { /* dummy is a pointer, checked within copy_in */

       return;
     }

  dalib_copy_valid_info (*actual_dsp, *dummy_dsp, 1);

  /* destroy flag is true as dummy descriptor will be freed afterwards */

  if (*dummy_dsp == *actual_dsp)
     return;                       /* local descriptor is just dummy */

  if (dalib_is_dummy_data (*dummy_dsp, *actual_dsp))

     { /* copy not necessary, same data, but delete descriptor */

#ifdef DEBUG
        printf ("%d: copy out, local = %d, dummy = %d, not necessary\n",
                 pcb.i, *dummy_dsp, *actual_dsp);
#endif

        if (*copy_flag != 0)
 
           { if (dalib_has_replication (*actual_dsp, *dummy_dsp))
 
                 FUNCTION(dalib_assign) (actual_dsp, dummy_dsp);
           }
 
       dalib_free_descriptor (*dummy_dsp);  

       return;

     }

  /* different descriptors, so copy out be necessary */

#ifdef DEBUG
  printf ("%d: copy out, local = %d, dummy = %d, copy out necessary\n",
               pcb.i, *dummy_dsp, *actual_dsp);
#endif

  FUNCTION(dalib_array_set_dirty) (dummy_dsp);

  if (*copy_flag != 0)

     { char msg[150];

       sprintf (msg, "copy out redistribution for dummy %s\n", 
                     (*dummy_dsp)->name);

       if (pcb.redist_flag) dalib_attention (msg); 

       FUNCTION(dalib_assign) (actual_dsp, dummy_dsp);
     }

#ifdef DEBUG
   printf ("%d: copy out, local = %d, dummy = %d, done\n",
            pcb.i, *dummy_dsp, *actual_dsp);
#endif
 

  dalib_free_descriptor (*dummy_dsp);  /* do also delete data */

} /* dalib_copy_out */

/*******************************************************************
*                                                                  *
*  FORTRAN Interface                                               *
*                                                                  *
*  FUNCTION(dalib_redistribute) (new_dsp, old_dsp)                 *
*                                                                  *
*    call DALIB_array_create_copy (B_NDSP,B_DSP)                   *
*    ...   ! define new distribution for B_DSP                     *
*    call DALIB_redistribute (B_DSP,B_NDSP)                        *
*                                                                  *
*******************************************************************/

void FUNCTION(dalib_redistribute) (new_dsp, old_dsp)

array_info *new_dsp, *old_dsp;

{ array_info descriptor;

  /* attention: can be that old dsp has no data allocated */

  descriptor = *new_dsp;

  if (!dalib_array_has_data (*old_dsp))

     { /* NO data is givne for the ARRAY, makes redistribute simple */

#ifdef DEBUG
       printf ("%d: redistribute of non allocated array\n", pcb.i);
#endif

       (*new_dsp)->data            = NO_DATA;
       (*new_dsp)->dsp_status_flag = descriptor->dsp_status_flag;

       /* old descriptor will be deleted */

       dalib_free_descriptor (*old_dsp);

       return;

     }

  if (descriptor->DistributeInfo != NO_DISTRIBUTION)
     dalib_dist_local_sizes (descriptor);
   else if (descriptor->AlignInfo != NO_ALIGNMENT)
     dalib_align_local_sizes (descriptor);
   else
     dalib_full_local_sizes (descriptor);

  if (dalib_use_actual_data (*new_dsp, *old_dsp))
 
     { /* copy just the data pointer */

#ifdef DEBUG
       printf ("%d: redistribute uses existing data\n", pcb.i);
#endif

       /* can be OWN or PTR_DATA */

       (*new_dsp)->data            = (*old_dsp)->data;
       (*new_dsp)->dsp_status_flag = (*old_dsp)->dsp_status_flag;

       if (dalib_has_replication (*new_dsp, *old_dsp))

          FUNCTION(dalib_assign) (new_dsp, old_dsp);

       /* old descriptor has no longer own data */
       /* free the old descriptor               */
     }

   else

     {  /* new descriptor, so new data is needed */
 
#ifdef DEBUG
       printf ("%d: redistribute allocates new data\n", pcb.i);
#endif

        dalib_array_shared_malloc (*new_dsp);            /* allocate memory */
 
        FUNCTION(dalib_assign) (new_dsp, old_dsp);

        /* free the old descriptor                */

     }
 
}  /* dalib_redistribute */

/*******************************************************************
*                                                                  *
*  FORTRAN Interface                                               *
*                                                                  *
*  FUNCTION(dalib_local_create) (local_dsp, global_dsp, pack_flag) *
*                                                                  *
*******************************************************************/

void FUNCTION(dalib_local_create) (local_dsp, global_dsp, pack_flag)

array_info *local_dsp, *global_dsp;
int        *pack_flag;

{ array_info  global, local;

  DimInfo *global_dims;
  DimInfo *local_dims;

  int i, rank;

  if (dalib_is_section_info (*global_dsp))

     { dalib_internal_error ("local_create: cannot localize section");
       dalib_stop ();
     }

   else if (dalib_is_array_info (*global_dsp))

     {  global = *global_dsp;

     }  /* descriptor was for a full array */

    else

     { dalib_internal_error ("local_create: not array/section");
       dalib_stop ();
     }

   local = dalib_array_copy_dsp (global);
   
   local->GlobalInfo = global;

   global_dims = global->dimensions;
   local_dims  = local->dimensions;

   rank = global->rank;

   for (i=0; i<rank; i++)

     { local_dims->global_size [0] = global_dims->local_size[0];
       local_dims->global_size [1] = global_dims->local_size[1];

       local_dims->overlap     [0] = global_dims->overlap[0];
       local_dims->overlap     [1] = global_dims->overlap[1];

       local_dims->local_size  [0] = local_dims->global_size[0];
       local_dims->local_size  [1] = local_dims->global_size[1];
       local_dims->local_size  [2] = 1;

       local_dims->map_flag = 0;

       global_dims++; local_dims++;

     }

   local->data = global->data;
   local->dsp_status_flag = DSP_PTR_DATA;

   *local_dsp = local;

} /* dalib_local_create */

void FUNCTION(dalib_local_free) (local_dsp)

array_info *local_dsp;

{ 

   dalib_free_descriptor (*local_dsp);  

} /* dalib_local_free */

