/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Jun 94                                                   *
*  Last Update : Jan 98                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : mailbox.c                                                *
*                                                                         *
*  Function: Management of sending and receiving                          *
*                                                                         *
*  Export :  internal Interface                                           *
*  ============================                                           *
*                                                                         *
*   void dalib_send (pid, message, msg_length, kind)                      *
*                                                                         *
*   void dalib_send_ddt (int pid, ddt_type ddt)                           *
*                                                                         *
*   void dalib_receive (pid, message, msg_length)                         *
*                                                                         *
*   void dalib_recv_ddt_op (int pid, ddt_type ddt, int op)                *
*                                                                         *
*   void dalib_unpack_receive (pid, buffer, msg_length)                   *
*                                                                         *
*      note : communication mode is set from FORTRAN directly             *
*                                                                         *
*  DALIB : FORTRAN Interface                                              *
*                                                                         *
*    void dalib_comm_mode (int mode)                                      *
*                                                                         *
*    void dalib_imode_label (int msg_label)                               *
*                                                                         *
*    void dalib_imode_wait  (int msg_label)                               *
*                                                                         *
*    void dalib_imode_waitall ()                                          *
*                                                                         *
*  UPDATES:                                                               *
*  ========                                                               *
*                                                                         *
*   01/1998 : communication statistics enabled                            *
*                                                                         *
**************************************************************************/

#include <stdio.h>
#include "dalib.h"

#undef DEBUG

#define MAX_OWN_MAIL 30
#define MAX_ISENDS   120
#define MAX_IRECVS   120

/*******************************************************************
*                                                                  *
*  Management of different communication modi                      *
*                                                                  *
*   - modus 0  :  blocking send/recv                               *
*                                                                  *
*   - modus 1  :  non-blocking send/recv                           *
*                                                                  *
*   - modus 2  :  send-receive combination mode                    *
*                                                                  *
*******************************************************************/

int comm_mode = 0;          /* communication modi                */
int imode_msg_label = 0;    /* global label for isend / ireceive */

/* global information about a waiting send for sendreceive */

int waiting_send = 0;
int waiting_send_length;
int waiting_send_pid;
int waiting_send_kind;
char *waiting_send_msg;

/*******************************************************************
*                                                                  *
*  OWN MAIL management                                             *
*                                                                  *
*    - for more convenience DALIB manages mail to own process      *
*                                                                  *
*******************************************************************/

typedef struct

  { int msg_length;    /* length of message in bytes */
    char *msg;         /* message itself             */
  } own_mail_info;

static own_mail_info own_mails [MAX_OWN_MAIL];
static int own_mail_read = 0;    /* pointer to next own mail */
static int own_mail_top  = 0;    /* pointer for next mail in */

     /*********************************************************
     *                                                        *
     *  own_send : put in a special memory area               *
     *                                                        *
     *********************************************************/

static void own_send (message, length, kind)

char *message;
int  length, kind;

{ int k;
  char *msg_buf;

#ifdef DEBUG
  printf ("%d : send own mail of length = %d\n", pcb.i, length);
#endif

  if (own_mail_top == MAX_OWN_MAIL)

    { /* no more place, move all own mails to bottom */

      if (own_mail_read == 0)
         { /* all buffers are full, internal error */
           dalib_internal_error ("no more own mail");
           exit (-1);
         }

      for (k = own_mail_read; k < own_mail_top; k++)
         own_mails[k-own_mail_read] = own_mails[k];

      own_mail_top  = own_mail_top - own_mail_read;
      own_mail_read = 0;

    } /* end packing own mails down */

  own_mails [own_mail_top].msg_length = length;

  if (kind != 2)
     { msg_buf = dalib_malloc (length , "dalib_own_send");
       dalib_memcopy (msg_buf, message, length);
     }

   else

     msg_buf = (char *) message;

  own_mails [own_mail_top].msg = msg_buf;
  own_mail_top++;

} /* own_send */

     /*********************************************************
     *                                                        *
     *  own_receive : get message from special memory area    *
     *                                                        *
     *********************************************************/

static void own_receive (message, length)

char *message;
int  length;

{ char *buf;
  int buf_len;

  if (own_mail_read == own_mail_top)

     { dalib_internal_error ("no own mail to receive");
       dalib_stop ();
     }

#ifdef DEBUG
  printf ("%d: receive own mail of length %d\n", pcb.i, length);
#endif

  buf = own_mails [own_mail_read].msg;
  buf_len = own_mails [own_mail_read].msg_length;
  own_mail_read ++;

  if (buf_len != length)
    { printf ("%d: own mail receive mismatch, length = %d, expected = %d\n",
               pcb.i, buf_len, length);
      dalib_internal_error ("fatal");
      dalib_stop ();
    }
  dalib_memcopy (message, buf, length);
  dalib_free (buf, buf_len);

} /* own_receive */

/*******************************************************************
*                                                                  *
*  Management of message ids for immediate send / recv             *
*                                                                  *
*******************************************************************/

typedef struct

  { int msg_id;           /* system identification of message */
    int msg_label;        /* identification for selected wait */
    char *msg;            /* message buffer                   */

  } isend_info;

typedef struct

  { int msg_id;           /* system identification of message */
    int msg_label;        /* identification for selected wait */
    char *msg;            /* buffer for received message      */
    dd_type ddt;          /* data type for unpacking message  */
    int op;               /* operation applied with the data  */

  } irecv_info;

static isend_info pending_sends [MAX_ISENDS];
static irecv_info pending_recvs [MAX_IRECVS];

static int top_isend = 0;
static int top_irecv = 0;

     /******************************************************
     *                                                     *
     *  dalib_put_isend (int msg_id, int msg_label,        *
     *                   char *msg, int kind       )       *
     *                                                     *
     ******************************************************/

void dalib_put_isend (msg_id, msg_label, msg)
int msg_id, msg_label;
char *msg;

{ if (top_isend == MAX_ISENDS)
     { dalib_internal_error ("too many pending isends");
       exit (-1);
     }

  pending_sends[top_isend].msg_id    = msg_id;
  pending_sends[top_isend].msg_label = msg_label;
  pending_sends[top_isend].msg       = msg;

  top_isend ++;

} /* dalib_put_isend */

     /******************************************************
     *                                                     *
     *  dalib_put_irecv (int msg_id, int msg_label, msg)   *
     *                                                     *
     ******************************************************/

void dalib_put_irecv (msg_id, msg_label, msg, ddt, op)
int msg_id;     /* identification given by the comm. system */
int msg_label;  /* user-defined label for selecting         */
char *msg;      /* buffer for the received message          */
dd_type ddt;    /* info about how to unpack data            */
int op;         /* info about what to do with the data      */

{ if (top_irecv == MAX_IRECVS)
     { dalib_internal_error ("too many pending irecvs");
       exit (-1);
     }

  pending_recvs[top_irecv].msg_id    = msg_id;
  pending_recvs[top_irecv].msg_label = msg_label;
  pending_recvs[top_irecv].msg       = msg;
  pending_recvs[top_irecv].ddt       = ddt;
  pending_recvs[top_irecv].op        = op;

  if (ddt != NO_DDT)
     dalib_ddt_setref (ddt);

  top_irecv ++;

} /* dalib_put_irecv */

      /* wait for the completion of receiving a message */

void dalib_wait_recv (info)
irecv_info *info;

{ dd_type ddt;

  /* wait for the packed data from other processor */

#ifdef DEBUG
  printf ("%d: wait for receive of msg = %d\n", pcb.i, info->msg_id);
#endif 

  machine_mpi_wait (info->msg_id);

  /* check whether the data must be unpacked */

  ddt = info->ddt;
  if (ddt != NO_DDT)
      { dalib_ddt_unpack (ddt, info->msg, info->op);
        free (info->msg);
        dalib_ddt_free (ddt);
      }

} /* dalib_wait_recv */

void dalib_wait (all_flag, msg_label)

{ int k, new_top;
  int do_wait;     /* 1 if process has to wait for completion */
  char *buf;    

#ifdef DEBUG
  if (all_flag)
     printf ("%d: wait for completion of all sends/recvs\n", pcb.i);
   else
     printf ("%d: wait for completion of sends/recvs, label = %d\n",
             pcb.i, msg_label);
#endif

  new_top = 0;
  for (k=0; k<top_isend; k++)
    { if (all_flag)
         do_wait = 1;
       else if (pending_sends[k].msg_label == msg_label)
         do_wait = 1;
       else
         do_wait = 0;
      if (do_wait)
        { machine_mpi_wait (pending_sends[k].msg_id);
          buf = pending_sends[k].msg;
          if (buf != (char *) 0)
             free (buf);
        }
       else if (new_top != k)
        { /* pending send remains */
          pending_sends[new_top] = pending_sends[k];
          new_top ++;
        }
    } /* end of waiting for immediate sends */

  top_isend = new_top;

  new_top = 0;
  for (k=0; k<top_irecv; k++)
    { if (all_flag)
         do_wait = 1;
       else if (pending_recvs[k].msg_label == msg_label)
         do_wait = 1;
       else
         do_wait = 0;
      if (do_wait)      /* wait for the packed data from other processor */
         dalib_wait_recv (pending_recvs + k);
       else if (new_top != k)
        { /* pending recv remains */
          pending_recvs[new_top] = pending_recvs[k];
          new_top ++;
        }
    } /* end of waiting for immediate recvs */

  top_irecv = new_top;

} /* dalib_wait */

/*******************************************************************
*                                                                  *
*  Initialization of message management                            *
*                                                                  *
*******************************************************************/

void dalib_msg_init ()

{ machine_mpi_init ();
}

void dalib_msg_exit ()

{ machine_mpi_exit ();
}

/*******************************************************************
*                                                                  *
*  DALIB : internal Interface                                      *
*                                                                  *
*  send                                                            *
*  ====                                                            *
*                                                                  *
*    kind == 0                                                     *
*                                                                  *
*    - send can be blocked until receive                           *
*    - send buffer can be reused after leaving send                *
*    - used for collective communications                          *
*                                                                  *
*    kind == 1                                                     *
*                                                                  *
*    - send buffer can be reused after leaving send                *
*    - send must not be blocked until the receive                  *
*    - blocking can imply deadlocks                                *
*    - used for data exchange                                      *
*                                                                  *
*    kind == 2                                                     *
*                                                                  *
*    - send buffer is own memory that is not reused                *
*    - send must not be blocked until the receive                  *
*    - used for data exchange with packed data                     *
*                                                                  *
*******************************************************************/

void dalib_send (pid, message, msg_length, kind)

   /* attention : kind = 2 implies that message must be freed */

int pid, msg_length;
char *message;

{ int msg_id;
  char *buffer;

#ifdef DEBUG
  printf ("%d: send mail to %d, length = %d\n", 
          pcb.i, pid, msg_length);
#endif

  if (pcb.comm_flag && comm_statistics_collect)  /* count message information */

     if (pid != pcb.i) dalib_commstat_entry_send (pcb.i, pid, msg_length);

  if (pid == pcb.i)

    { own_send (message, msg_length, kind);
      return;
    }

   else if (comm_mode == 0)

     machine_mpi_send (pid, message, msg_length, kind);

   else if (comm_mode == 1)

     { msg_id = machine_mpi_isend (pid, message, msg_length, kind);
       if (kind == 2)
          buffer = message;    /* will be deleted after completion */
         else
          buffer = (char *) 0;
       dalib_put_isend (msg_id, imode_msg_label, buffer);
     }

   else if (comm_mode == 2)

     { /* wait with sending until the next receive is posted */

       if (waiting_send != 0)
          dalib_internal_error ("sendrecv: already pending send");

       waiting_send = 1;
       waiting_send_msg = message;
       waiting_send_kind = kind;
       waiting_send_pid  = pid;
       waiting_send_length = msg_length;
     }

   else

     dalib_internal_error ("send: illegal mode");

} /* dalib_send */

/*******************************************************************
*                                                                  *
*  - sending with a derived data type                              *
*                                                                  *
*******************************************************************/

void dalib_send_ddt (pid, ddt)

dd_type  ddt;
int      pid;

{ char *mem;
  int  size, is_contiguous;

  dalib_ddt_get_size (ddt, &size);
  dalib_ddt_is_contiguous (ddt, &is_contiguous, &mem);

#ifdef DEBUG
  printf ("%d: send mail to %d with ddt, size = %d, contiguous = %d\n", 
          pcb.i, pid, size, is_contiguous);
#endif

  if (size == 0) return;

  if (is_contiguous)
     dalib_send (pid, mem, size, 1);
   else
     { mem = (char *) dalib_malloc (size, "dalib_send_ddt");
       dalib_ddt_pack (mem, ddt);
       dalib_send (pid, mem, size, 2);
       /* mem will be deleted after sending */
     }
} /* dalib_send_ddt */

/*******************************************************************
*                                                                  *
*  receiving of data                                               *
*                                                                  *
*******************************************************************/

void dalib_all_receive (pid, recv_buffer, msg_length, ddt, op)
int pid, op, msg_length;
dd_type ddt;
char *recv_buffer;

{ /* make a communication statistic entry if necessary */

  if (pcb.comm_flag && comm_statistics_collect)

     if (pid != pcb.i) dalib_commstat_entry_receive(pcb.i,pid,msg_length);

  /* case 1 : own mail, then directly use ddt for unpacking */

  if (pid == pcb.i)

    { own_receive (recv_buffer, msg_length);
      if (ddt != NO_DDT) 
        { dalib_ddt_unpack (ddt, recv_buffer, op);
          dalib_free (recv_buffer, msg_length);
        }
      return;
    }

  if (comm_mode == 0)   /* blocking mode */

    { machine_mpi_receive (pid, recv_buffer, msg_length);
      if (ddt != NO_DDT) 
        { dalib_ddt_unpack (ddt, recv_buffer, op);
          dalib_free (recv_buffer, msg_length);
        }
      return;
    }

 if (comm_mode == 1)

     { int msg_id;

       msg_id = machine_mpi_ireceive (pid, recv_buffer, msg_length);
       dalib_put_irecv (msg_id, imode_msg_label, recv_buffer, ddt, op);
       /* note : recv_buffer will be freed later */
       return;
     }

 if (comm_mode == 2)

     { /* send and receive together in one step */

       if (waiting_send != 1)
          dalib_internal_error ("sendrecv: no send");

       machine_mpi_sendreceive (waiting_send_pid, 
                        waiting_send_msg, waiting_send_length,
                        pid, recv_buffer, msg_length);

       if (waiting_send_kind == 2)
          free (waiting_send_msg);

       if (ddt != NO_DDT) 
         { dalib_ddt_unpack (ddt, recv_buffer, op);
           dalib_free (recv_buffer, msg_length);
         }

       waiting_send = 0;  /* for internal check */

       return;
     }

   dalib_internal_error ("receive: unknown comm mode");

} /* dalib_all_receive */

void dalib_recv_ddt_op (pid, ddt, op)
int pid, op;
dd_type ddt;

{ int  is_contiguous;
  int  msg_length;      /* expected length for message */
  char *recv_buffer;
  char *user_data;

  if (ddt == NO_DDT) return;

  dalib_ddt_get_size (ddt, &msg_length);  
  dalib_ddt_is_contiguous (ddt, &is_contiguous, &user_data);

#ifdef DEBUG
  printf ("%d: recv mail from %d with ddt, size = %d, contiguous = %d\n", 
          pcb.i, pid, msg_length, is_contiguous);
#endif

  if (is_contiguous)
     { recv_buffer = user_data;
       ddt = NO_DDT;
     }
   else
     recv_buffer = (char *) dalib_malloc (msg_length, "dalib_new_receive");

  dalib_all_receive (pid, recv_buffer, msg_length, ddt, op);

  if (ddt == NO_DDT)
     {
     }

#ifdef DEBUG
   printf ("%d: mail from %d received\n", pcb.i, pid);
#endif

} /* dalib_recv_ddt_op */

void dalib_receive (pid, message, msg_length)

int pid, msg_length;
char *message;

{
#ifdef DEBUG
  printf ("%d: recv mail from %d, size = %d\n",
          pcb.i, pid, msg_length);
#endif

  dalib_all_receive (pid, message, msg_length, NO_DDT, 0);

} /* dalib_receive */

/*******************************************************************
*                                                                  *
*  DALIB : FORTRAN Interface                                       *
*                                                                  *
*    void dalib_comm_mode (int mode)                               *
*                                                                  *
*        mode = 0   :  blocking send/receive                       *
*        mode = 1   :  non-blocking send/receive                   *
*        mode = 2   :  blocking pair of send/receive               *
*                                                                  *
*    void dalib_imode_label (int msg_label)                        *
*                                                                  *
*      - all future send/receives get this labels                  *
*                                                                  *
*    void dalib_imode_wait  (int msg_label)                        *
*                                                                  *
*      - wait for all sends and receives with this label           *
*                                                                  *
*    void dalib_imode_waitall ()                                   *
*                                                                  *
*******************************************************************/

void FUNCTION(dalib_comm_mode) (mode) 
int *mode;
{ /* set globally the communication mode */
  comm_mode  = *mode;
} /* FUNCTION(dalib_comm_mode) */ 

void FUNCTION(dalib_imode_label) (msg_label) 
int *msg_label;
{ comm_mode = 1;
  imode_msg_label = *msg_label;
} /* FUNCTION(dalib_imode_label) */ 

void FUNCTION(dalib_imode_waitall) () 
{ dalib_wait (1, 0);
  comm_mode = 0;
} /* dalib_imode_waitall */

void FUNCTION(dalib_imode_wait) (msg_label) 
int *msg_label;
{ dalib_wait (0, *msg_label);
} /* dalib_imode_wait */
