/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Mar 96                                                   *
*  Last Update : Mar 96                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : local.m4                                                 *
*                                                                         *
*  Function    : utility routines for irregular block distributions       *
*                                                                         *
*  Export :  FORTRAN Interface                                            *
*                                                                         *
*  Export :  DALIB Interface                                              *
*                                                                         *
**************************************************************************/

#define CHECK
#undef DEBUG

#include "dalib.h"

/**************************************************************************
*                                                                         *
*  int FUNCTION(dalib_gaddr) (array_info *array_id, int *dim, int *val)   *
*                                                                         *
*  - translates a local address in a certain dimension to the global      *
*    address (helpful for INDIRECT/CYCLI(n) distributions)                *
*                                                                         *
**************************************************************************/

int FUNCTION(dalib_gaddr) (array_id, dim, index_val)

array_info *array_id;
int *dim;
int *index_val;

{  int val;
   int lb, ub, base, stride;
   int topology;
   DistDim mapping;

   dalib_array_dim_mapping (*array_id, *dim,
                            &base, &stride, &lb, &ub,
                            &topology, &mapping);

   if (stride != 1)

      { dalib_internal_error ("addr: no stride for mapped dimension");
        dalib_stop ();
      }

   val = base + *index_val;

   val = dalib_dist_global_addr (val, lb, ub, mapping);

   val = val - base;

#ifdef DEBUG
   printf ("%d: dalib_gaddr of %d (in %d:%d) = %d\n",
           pcb.i, *index_val, lb, ub, val);
#endif

   return val;

} /* dalib_gaddr */

/**************************************************************************
*                                                                         *
*  int FUNCTION(dalib_addr) (array_info *array_id, int *dim, int *val)    *
*                                                                         *
*  - translates a global address in a certain dimension to the local      *
*    address (necessary for INDIRECT/CYCLIC(n) distributions              *
*                                                                         *
**************************************************************************/

int FUNCTION(dalib_addr) (array_id, dim, index_val)

array_info *array_id;
int *dim;
int *index_val;

{  int topology;
   int lb, ub, base, stride;
   int val;
   DistDim mapping;

   dalib_array_dim_mapping (*array_id, *dim,
                            &base, &stride, &lb, &ub, 
                            &topology, &mapping);

   if (stride != 1)

      { dalib_internal_error ("addr: no stride for mapped dimension");
        dalib_stop ();
      }

   val = base + *index_val;

   val = dalib_dist_local_addr (val, lb, ub, mapping);

   val = val - base;

#ifdef DEBUG
   printf ("%d: dalib_addr of %d (in %d:%d) = %d\n", 
           pcb.i, *index_val, lb, ub, val);
#endif

   return val;

} /* dalib_addr */

/**************************************************************************
*                                                                         *
*  int dalib_is_local (array_info array_id, int *dim, int *index_val)     *
*                                                                         *
*   - returns true if array (...,index_val,....) is local                 *
*   - returns false otherwise                                             *
*                                                                         *
*  NOTE: can be true on more than one processor for replicated            *
*        dimensions                                                       *
*                                                                         *
**************************************************************************/

int FUNCTION(dalib_is_local) (array_id, dim, index_val)

array_info *array_id;
int *dim;
int *index_val;

{ DimInfo *array_dim;
  int     *local_size;
  int     new_index_val;

  int offset;

  if (*dim == 0)

     { int top_id;
       array_info temp_id;

       dalib_array_info (*array_id, &temp_id, &top_id);

#ifdef DEBUG
       printf ("%d: dalib_is_local on topology %d = %d\n",
               pcb.i, top_id, dalib_in_topology(top_id));
#endif

       if (dalib_in_topology (top_id))
         return (F_TRUE);
       else
         return (0);
     }

  array_dim  = (*array_id)->dimensions;
  array_dim += (*dim - 1);
  local_size = array_dim->local_size;

  new_index_val = *index_val;

#ifdef DEBUG
  printf ("%d: dalib_is_local (id = %d, dim = %d) %d in [%d:%d:%d] ?\n",
          pcb.i, *array_id, *dim, new_index_val,
          local_size[0], local_size[1], local_size[2]);
#endif

  if (array_dim->map_flag)

     new_index_val = FUNCTION(dalib_addr) (array_id, dim, index_val);

  /* true if index_val in lb : ub : str */

  if (new_index_val < local_size[0]) return (0);
  if (new_index_val > local_size[1]) return (0);

  if (local_size[2] == 1) return (F_TRUE);

  offset = new_index_val - local_size[0];

  if ((offset % local_size[2]) == 0)
     return (F_TRUE);
   else
     return (0);

} /* dalib_is_local */

/**************************************************************************
*                                                                         *
*   call dalib_local_int_sum (replicated_array, distributed_array)        *
*                                                                         *
*    - every processor builds a local sum of his part                     *
*    - all local sums will be replicated across all processors            *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_local_int_sum)

    (global_array_data, local_array_data,
     global_array_dsp,  local_array_dsp)

int *global_array_data, *local_array_data;
array_info *global_array_dsp, *local_array_dsp;

{ int pid, gid;
  int my_sum;
  int is_contiguous;

  dd_type ddt;
  int i, n, nbytes;

  int *local_data;

  my_sum = 0;   /* also for non-participating processors */

  dalib_make_secarray_ddt (&ddt, *local_array_dsp);

  dalib_ddt_get_size (ddt, &nbytes);
  n = nbytes / sizeof (int);

  dalib_ddt_is_contiguous (ddt, &is_contiguous, &local_data);

  if (!is_contiguous)
    { dalib_internal_error ("local_int_sum: not contiguous section");
      dalib_stop ();
    }

  for (i=0; i<n; i++) my_sum += local_data[i];

#ifdef DEBUG
  printf ("%d: my local sum (for %d values) = %d\n", pcb.i, n, my_sum);
#endif

  if (pcb.i > 0)
     global_array_data [pcb.i-1] = my_sum;

  gid = dalib_context_group ();
  dalib_group_concat (global_array_data, sizeof(int), gid);

#ifdef DEBUG
  for (i=0; i<pcb.p; i++)
     printf ("%d: global at pos %d = %d\n",
              pcb.i, i, global_array_data[i]);
#endif

} /* dalib_local_int_sum */

