/**************************************************************************
*                                                                         *
*  Author      : Thomas Brandes, GMD, SCAI.LAB                            *
*                Cecile Germain, Orsay, LRI                               *
*                                                                         *
*  Copyright   : LRI, Orsay, France & SCAI, GMD, Germany                  *
*  Date        : Mar 97                                                   *
*  Last Update : Mar 97                                                   *
*                                                                         *
*  This Module is part of the DALIB / UNILIB                              *
*                                                                         *
*  Module      : insp_set.m4                                              *
*                                                                         *
*  Function: building/querying stucture for info about indirect address.  *
*                                                                         *
*    typedef struct { ... } inspector_info                                *
*                                                                         *
*  Internal Interface                                                     *
*  ==================                                                     *
*                                                                         *
**************************************************************************/

#include "dalib.h"
#include "inspector.h"


#undef DEBUG
#define CHECK

/**********************************************************************
*                                                                     *
*   void dalib_insp_info_set_source (inspector_info *S,               *
*                                    array/section_info *source_dsp)  *
*                                                                     *
*     -> sets source_array, number_of_indexes                         *
*                                                                     *
**********************************************************************/

void dalib_insp_info_set_source (S, source_dsp)

inspector_info *S;
array_info *source_dsp;

{  S->source_array = *source_dsp;

       /******************************************
       *   make sure that descriptors exist      *
       ******************************************/
 
  if (source_dsp == (array_info *) 0)

     { dalib_internal_error ("INSPECTOR: source no descriptor");
       dalib_stop ();
     }
 
       /******************************************
       *   case 1 : array/source is full array   *
       ******************************************/
 
   else if (dalib_is_array_info (*source_dsp))

     { S->number_of_indexes = dalib_array_local_size (*source_dsp); }

       /******************************************
       *   case 2 : array/source is section      *
       ******************************************/
 
   else if (dalib_is_section_info (*source_dsp))

     { S->number_of_indexes = dalib_section_local_size (*source_dsp); }

       /******************************************
       *   illegal descriptor                    *
       ******************************************/

   else

     { dalib_internal_error ("INSPECTOR: (illegal descriptor)");
       dalib_stop ();
     }
 
} /* dalib_insp_info_set_source */

/*******************************************************************
*                                                                  *
*   void dalib_insp_info_set_base (array_info *base_dsp)           *
*                                                                  *
*     -> sets base_xxx                                             *
*                                                                  *
*******************************************************************/

void dalib_insp_info_set_base (S, base_array)

inspector_info *S;
array_info     *base_array;

{      /******************************************
       *   make sure that descriptors exist      *
       ******************************************/
 
  if (base_array == (array_info *) 0)

     { dalib_internal_error ("INSPECTOR: base is not a descriptor");
       dalib_stop ();
     }
 
       /******************************************
       *   case 1 : array/source is full array   *
       ******************************************/
 
   else if (dalib_is_array_info (*base_array))

     { S->base_array = *base_array;
     }

       /******************************************
       *   case 2 : array/source is section      *
       ******************************************/
 
   else if (dalib_is_section_info (*base_array))

     { dalib_internal_error ("INSPECTOR : base subsection not allowed");
       dalib_stop ();
     }

       /******************************************
       *   illegal descriptor                    *
       ******************************************/

   else

    { dalib_internal_error ("INSPECTOR: illegal descriptor");
      dalib_stop ();
    }

  S->base_rank = S->base_array->rank;
  S->is_shadow = 0;

} /* dalib_insp_info_set_base */

/*******************************************************************
*                                                                  *
*   void dalib_insp_info_set_shadow (array_info *base_dsp)         *
*                                                                  *
*     -> sets array that is indirectly addressed, and shadow       *
*                                                                  *
*******************************************************************/

void dalib_insp_info_set_shadow (S, base_array)

inspector_info *S;
array_info     *base_array;

{      /******************************************
       *   make sure that descriptors exist      *
       ******************************************/
 
  if (base_array == (array_info *) 0)

     { dalib_internal_error ("INSPECTOR: base is not a descriptor");
       dalib_stop ();
     }
 
       /******************************************
       *   case 1 : array/source is full array   *
       ******************************************/
 
   else if (dalib_is_array_info (*base_array))

     { S->base_array = *base_array;
     }

       /******************************************
       *   case 2 : array/source is section      *
       ******************************************/
 
   else if (dalib_is_section_info (*base_array))

     { dalib_internal_error ("INSPECTOR : base subsection not allowed");
       dalib_stop ();
     }

       /******************************************
       *   illegal descriptor                    *
       ******************************************/

   else

    { dalib_internal_error ("INSPECTOR: illegal descriptor");
      dalib_stop ();
    }

  S->base_rank         = S->base_array->rank;
  S->is_shadow         = 1;
  S->number_of_indexes = 0;    /* default value */

} /* dalib_insp_info_set_shadow */

/***********************************************************************
*                                                                      *
*  void dalib_insp_info_set_index (inspector_info *S, int ind_dim,     *
*                                  array_info *ind_dsp, int *ind_val)  *
*                                                                      *
***********************************************************************/

void dalib_insp_info_set_index (S, ind_dim, ind_dsp, ind_data)

inspector_info *S;
int ind_dim;
array_info *ind_dsp;
int        *ind_data;

{
  S->ind_array[ind_dim-1] = *ind_dsp;
  S->ind_data[ind_dim-1] = ind_data;

  if (!FUNCTION(dalib_present) (ind_dsp))

     S->ind_array[ind_dim-1] = (void *) 0;

       /******************************************
       *   case 1 : index is single value        *
       ******************************************/

#ifdef CHECK

  if (! FUNCTION(dalib_present) (ind_dsp) )

     { /* there is no descriptor, so take the index data */

     }

   else if (dalib_is_array_info (*ind_dsp))

     { /* check whether ind is also aligned with S->source_dsp */

       if (S->is_shadow)

         S->number_of_indexes = dalib_array_local_size (*ind_dsp);

       else if (dalib_array_local_size (*ind_dsp) != S->number_of_indexes)

         { dalib_internal_error ("INSPECTOR: indexes not aligned to source");
           dalib_stop ();
         }

     }
 
   else if (dalib_is_section_info (*ind_dsp))
 
     { if (dalib_section_local_size (*ind_dsp) != S->number_of_indexes)

         { dalib_internal_error ("INSPECTOR: indexes not aligned to source");
           dalib_stop ();
         }
     }
 
   else

     { dalib_internal_error ("INSPECTOR : illegal descriptor for index");
       dalib_stop ();
     }
#endif

} /* dalib_insp_info_set_index */

/*********************************************************************
*                                                                    *
*   dalib_indexes_set_mask (array_info *mask_array, int *mask_data)  *
*                                                                    *
*********************************************************************/

void dalib_insp_info_set_mask (S, mask_array, mask_data)

inspector_info *S;

array_info *mask_array;
int        *mask_data;

{ S->mask_array = *mask_array;
  S->mask_data  = mask_data;

  S->is_masked  = FUNCTION(dalib_present) (mask_data);

  if (!FUNCTION(dalib_present) (mask_array))

     S->mask_array = (void *) 0;

#ifdef CHECK

  if (! FUNCTION(dalib_present) (mask_array) )

     { /* there is no descriptor, so take the mask data */

     }

   else if (dalib_is_array_info (*mask_array))

     { /* check whether mask is also aligned with S->source_dsp */

       if (dalib_array_local_size (*mask_array) != S->number_of_indexes)

         { dalib_internal_error ("INSPECTOR: mask not aligned to source");
           dalib_stop ();
         }

     }
 
   else if (dalib_is_section_info (*mask_array))
 
     { if (dalib_section_local_size (*mask_array) != S->number_of_indexes)

         { dalib_internal_error ("INSPECTOR: mask not aligned to source");
           dalib_stop ();
         }
     }
 
   else

     { dalib_internal_error ("INSPECTOR : illegal descriptor for mask");
       dalib_stop ();
     }
#endif

} /* dalib_insp_info_set_mask */

/*********************************************************************
*                                                                    *
*   void dalib_insp_info_get (inspector_info *S,                     *
*                             int *indexes[], int index_new[],       *
*                             int **mask_data, int *mask_new);       *
*                                                                    *
*********************************************************************/

void dalib_insp_info_get (S, indexes, index_new, mask_data, mask_new)

inspector_info *S;

int *indexes[];
int index_new[];

int **mask_data;
int *mask_new;

{ int idim, rank;

  int no, size;

  if (S->is_shadow) rank = 1;
     else           rank = S->base_rank;

  for (idim=0; idim < rank; idim++)

    if (S->ind_array[idim] != (void *) 0)

       dalib_secarray_get_data (S->ind_array[idim], 1,
                                &no, &size, indexes+idim, index_new+idim);

     else

       { int i;

         indexes[idim] = dalib_int_malloc (S->number_of_indexes,
                                           "dalib_insp_info_get");

         for (i=0; i<S->number_of_indexes; i++)

            (indexes[idim])[i] = *(S->ind_data[idim]);

         index_new[idim] = 1;

       }  /* building index idim */

  if (S->is_masked)

     dalib_secarray_get_data (S->mask_array, 1,
                              &no, &size, mask_data, mask_new);

    else

    { *mask_data = (void *) 0;
      *mask_new  = 0;
    }

} /* dalib_insp_info_get */

/*********************************************************************
*                                                                    *
*   void dalib_insp_info_free (inspector_info *S,                    *
*                              int *indexes[], int index_new[],      *
*                              int *mask_data, int mask_new   );     *
*                                                                    *
*********************************************************************/

void dalib_insp_info_free (S, indexes, index_new, mask_data, mask_new)

inspector_info *S;

int *indexes[];
int index_new[];

int *mask_data;
int mask_new;

{ int idim, rank;
  int no;

  if (S->is_shadow) rank = 1;
     else           rank = S->base_rank;

  no = S->number_of_indexes;

  for (idim=0; idim < rank; idim++)
     if (index_new[idim]) dalib_int_free (indexes[idim], no);

  if (mask_new) dalib_int_free (mask_data, no);

} /* insp_info_free */

