/**************************************************************************
*                                                                         *
*  Author      : Thomas Brandes, GMD, SCAI.LAB                            *
*                Cecile Germain, Orsay, LRI                               *
*                                                                         *
*  Copyright   : LRI, Orsay, France                                       *
*  Date        : Mar 97                                                   *
*  Last Update : Mar 97                                                   *
*                                                                         *
*  Module      : insp_match.m4                                            *
*                                                                         *
*  Function    : match given inspector data with inspector info           *
*                                                                         *
*  Export:                                                                *
*  =======                                                                *
*                                                                         *
**************************************************************************/

#include "dalib.h"
#include "inspector.h"

#define CHECK
#undef DEBUG

/*******************************************************************
*                                                                  *
*                                                                  *
*                                                                  *
*******************************************************************/

static void dalib_set_array_valid (dsp, s_id)

array_info dsp;
int        s_id;

{ 

#ifdef DEBUG
  printf ("%d: dalib_set_array_valid of dsp = %d for id = %d\n", 
           pcb.i, dsp, s_id);
#endif

  if (dalib_is_array_info (dsp))

    { if (dsp->trace)

          dalib_inspector_db_set_valid (&(dsp->reuse_bits), s_id);
    }

   else if (dalib_is_section_info (dsp))

      dalib_set_array_valid (((section_info) dsp)->array_id, s_id);
}

/*******************************************************************
*                                                                  *
*                                                                  *
*                                                                  *
*******************************************************************/

static int dalib_is_array_valid (dsp, s_id)

array_info dsp;
int        s_id;

{
  if (dsp == (void *) 0)

     return (1);     /* was a constant */

  if (dalib_is_array_info (dsp))

   {

#ifdef DEBUG
  printf ("%d: dalib_is_array_valid of dsp = %d, reuse = %d\n", 
           pcb.i, dsp, dsp->reuse_bits);
#endif

     return dalib_inspector_db_is_valid (dsp->reuse_bits, s_id);
   }

   else if (dalib_is_section_info (dsp))

     return dalib_is_array_valid (((section_info) dsp)->array_id, s_id);

   else

     return (0);
}

/************************************************************************
*                                                                       *
*  void dalib_inspector_set_valid (inspector_info *I_Info, int s_id)    *
*                                                                       *
*   - set reuse bits of array descriptors valid for schedule s_id       *
*                                                                       *
************************************************************************/

void dalib_inspector_set_valid (I_Info, s_id)

inspector_info *I_Info;
int s_id;

{ int idim, rank;

  array_info dsp;

  if (I_Info->is_shadow)  rank = 1;
    else                  rank = I_Info->base_rank;

  for (idim=0; idim < rank; idim++)

     dalib_set_array_valid (I_Info->ind_array[idim], s_id);

  if (I_Info->is_masked)

     dalib_set_array_valid (I_Info->mask_array, s_id);

} /* dalib_inspector_set_valid */

/*******************************************************************
*                                                                  *
*  bool dalib_inspector_is_valid (inspector_info *I_Info,          *
*                                int s_id)                         *
*                                                                  *
*  - returns true if reuse bit (s_id) is set for all the           *
*    indexing arrays and the mask array                            *
*                                                                  *
*******************************************************************/

int dalib_inspector_is_valid (I_Info, s_id)

inspector_info *I_Info;
int s_id;

{ int idim, rank;

  array_info dsp;

  if (I_Info->is_shadow)  rank = 1;
    else                  rank = I_Info->base_rank;

  for (idim=0; idim < rank; idim++)

    { if (!dalib_is_array_valid (I_Info->ind_array[idim], s_id))

         return (0);
    }

  if (I_Info->is_masked)

    { if (!dalib_is_array_valid (I_Info->mask_array, s_id))

         return (0);
    }

  /* all integer arrays and the mask have not been changed */

  return (1);

} /* dalib_inspector_is_valid */

/*******************************************************************
*                                                                  *
*  void dalib_inspector_rebuild (inspector_info *I_Info,           *
*                                inspector_data *I_Data)           *
*                                                                  *
*    - actualize the inspector data for the given info             *
*                                                                  *
*******************************************************************/

void dalib_inspector_rebuild (I_Data, I_Info)

inspector_info *I_Info;
inspector_data *I_Data;

{ 

#ifdef DEBUG
  printf ("%d: dalib_inspector_rebuild (free schedule and rebuild it)\n",
           pcb.i);
#endif

  dalib_free_indirect (I_Data->schedule_id);

  dalib_inspector_build (I_Data, I_Info);

}  /* dalib_inspector_rebuild */

/*******************************************************************
*                                                                  *
*  bool dalib_inspector_is_reusable (inspector_data *DB_I_Data,    *
*                                    inspector_data *I_Data)       *
*                                                                  *
*  - check for slightly differences (returns false then)           *
*                                                                  *
*******************************************************************/

int dalib_inspector_is_reusable (DB_I_Data, I_Data)

inspector_data *DB_I_Data, I_Data;

{
  return (1);
}

/*******************************************************************
*                                                                  *
*  void dalib_inspector_refresh (inspector_data *DB_I_Data,        *
*                                inspector_data *I_Data)           *
*                                                                  *
*  - update slightly differences in inspector data                 *
*                                                                  *
*******************************************************************/

void dalib_inspector_refresh (DB_I_Data, I_Data)

inspector_data *DB_I_Data, I_Data;

{ 

}  /* dalib_inspector_refresh */


/*******************************************************************
*                                                                  *
*  bool dalib_is_same_base_dsp (insp_base_dsp *dsp1, *dsp2)        *
*                                                                  *
*******************************************************************/

static int dalib_is_same_base_dsp (base_dsp1, base_dsp2)

insp_base_dsp *base_dsp1, *base_dsp2;

{ int i, rank;

  if (base_dsp1->rank != base_dsp2->rank) return (0);
  if (base_dsp1->top_id != base_dsp2->top_id) return (0);

  rank = base_dsp1->rank;

  for (i=0; i<rank; i++)

    { if (base_dsp1->lb[i]        != base_dsp2->lb[i]) return (0);
      if (base_dsp1->ub[i]        != base_dsp2->ub[i]) return (0);

      if (base_dsp1->left_ov[i]   != base_dsp2->left_ov[i]) return (0);
      if (base_dsp1->right_ov[i]  != base_dsp2->right_ov[i]) return (0);

      if (base_dsp1->t_base[i]    != base_dsp2->t_base[i]) return (0);
      if (base_dsp1->t_stride[i]  != base_dsp2->t_stride[i]) return (0);
      if (base_dsp1->t_kind[i]    != base_dsp2->t_kind[i]) return (0);
      if (base_dsp1->t_lb[i]      != base_dsp2->t_lb[i]) return (0);
      if (base_dsp1->t_ub[i]      != base_dsp2->t_ub[i]) return (0);

      if (base_dsp1->top_dim[i]   != base_dsp2->top_dim[i]) return (0);

    }

  return (1);

} /* dalib_is_same_base_dsp */

/*******************************************************************
*                                                                  *
*  bool dalib_is_same_shadow_dsp (insp_base_dsp *dsp1, *dsp2)      *
*                                                                  *
*******************************************************************/

static int dalib_is_same_shadow_dsp (base_dsp1, base_dsp2)

insp_base_dsp *base_dsp1, *base_dsp2;

{ int rank1, rank2;

  rank1 = base_dsp1->rank - 1; 
  rank2 = base_dsp2->rank - 1; 

  if (base_dsp1->top_id != base_dsp2->top_id) return (0);

  if (base_dsp1->lb[rank1]        != base_dsp2->lb[rank2]) return (0);
  if (base_dsp1->ub[rank1]        != base_dsp2->ub[rank2]) return (0);

  if (base_dsp1->left_ov[rank1]   != base_dsp2->left_ov[rank2]) return (0);
  if (base_dsp1->right_ov[rank1]  != base_dsp2->right_ov[rank2]) return (0);

  if (base_dsp1->t_base[rank1]    != base_dsp2->t_base[rank2]) return (0);
  if (base_dsp1->t_stride[rank1]  != base_dsp2->t_stride[rank2]) return (0);
  if (base_dsp1->t_kind[rank1]    != base_dsp2->t_kind[rank2]) return (0);
  if (base_dsp1->t_lb[rank1]      != base_dsp2->t_lb[rank2]) return (0);
  if (base_dsp1->t_ub[rank1]      != base_dsp2->t_ub[rank2]) return (0);

  if (base_dsp1->top_dim[rank1]   != base_dsp2->top_dim[rank2]) return (0);

  return (1);

} /* dalib_is_same_shadow_dsp */

/*******************************************************************
*                                                                  *
*  bool dalib_is_same_index_dsp (insp_index_dsp *dsp1, *dsp2)      *
*                                                                  *
*******************************************************************/

static int dalib_is_same_index_dsp (index_dsp1, index_dsp2)

insp_index_dsp *index_dsp1, *index_dsp2;

{ int i, j, rank;

  if (index_dsp1->rank != index_dsp2->rank) return (0);

  rank = index_dsp1->rank;

  if (rank == 0)

    { /* compare scalar indexes */

      if (index_dsp1->scalar_val != index_dsp2->scalar_val) return (0);

      return (1);
    }

  if (index_dsp1->data_ptr != index_dsp2->data_ptr) return (0);

  for (i=0; i < rank; i++)

     for (j=0; j < 3; j++)

        if (index_dsp1->section_bounds[i][j] !=
            index_dsp2->section_bounds[i][j] )   return (0);

  return (1);

} /* dalib_is_same_index_dsp */

/*******************************************************************
*                                                                  *
*  bool dalib_inspector_match (inspector_data *I_data1, *I_data2)  *
*                                                                  *
*    - returns true if the given inspector data can be used for    *
*      the new specified inspector                                 *
*                                                                  *
*******************************************************************/

int dalib_inspector_match (I_Data1, I_Data2)

inspector_data *I_Data1;
inspector_data *I_Data2;

{  int i, rank;

#ifdef DEBUG
   printf ("%d: call of dalib_inspector_match\n", pcb.i);
#endif

   if (I_Data2->number_of_indexes != I_Data1->number_of_indexes) return (0);

#ifdef DEBUG
   printf ("%d: matches the number of indexes\n", pcb.i);
#endif

   /* normal and shadow schedules cannot be mixed */

   if (I_Data2->is_shadow != I_Data1->is_shadow) return (0);

   /* Test 1 : same distribution/shape of the base arrays */

   if (I_Data1->is_shadow)

     { if (!dalib_is_same_shadow_dsp (&(I_Data1->base_array), 
                                      &(I_Data2->base_array)))

       return (0);
     }

     else

     { if (!dalib_is_same_base_dsp (&(I_Data1->base_array), 
                                    &(I_Data2->base_array)))

       return (0);
     }

#ifdef DEBUG
   printf ("%d: matches the base array\n", pcb.i);
#endif

   /* Test 2 : same integer arrays for indirect addressing */

   if (I_Data2->is_shadow) rank = 1;
      else rank = I_Data2->base_array.rank;

   for (i=0; i < rank; i++)

     { if (!dalib_is_same_index_dsp (&(I_Data1->ind_array[i]),
                                     &(I_Data2->ind_array[i])))
           return (0);
     }

#ifdef DEBUG
   printf ("%d: matches the index arrays\n", pcb.i);
#endif

   /* Test 3 : same mask is used                           */

   if (!dalib_is_same_index_dsp (&(I_Data1->mask_array),
                                 &(I_Data2->mask_array)))

      return (0);

#ifdef DEBUG
   printf ("%d: matches the mask array\n", pcb.i);
#endif

   return (1);   /* same distribution/shape of base and source */

} /* dalib_inspector_match */

