/**************************************************************************
*                                                                         *
*  Author      : Thomas Brandes, GMD, SCAI.LAB                            *
*                Cecile Germain, Orsay, LRI                               *
*                                                                         *
*  Copyright   : LRI, Orsay, France                                       *
*  Date        : Mar 97                                                   *
*  Last Update : Mar 97                                                   *
*                                                                         *
*  Module      : insp_exec.m4                                             *
*                                                                         *
*  Function    : INSPECTOR/EXECUTOR for indirect addressing and           *
*                scatter/gather operations                                *
*                                                                         *
*  Examples:                                                              *
*                                                                         *
*    FORALL (I=1:N) B(I) = A(P(I))                                        *
*                                                                         *
*     call dalib_inspector_get (IS_1,B(B_ZERO+1),B_DSP,                   *
*             A(A_ZERO+1),A_DSP,dalib_0,dalib_0,P(P_ZERO+1),P_DSP)        *
*     call dalib_executor_gather_send (IS_1,A_DSP)                        *
*     call dalib_executor_gather_recv (IS_1,B_DSP)                        *
*     call dalib_inspector_release (IS_1)                                 *
*                                                                         *
*    FORALL (I=1:N) A(P(I)) = B(I)                                        *
*                                                                         *
*     call dalib_inspector_get (IS_1,B(B_ZERO+1),B_DSP,                   *
*             A(A_ZERO+1),A_DSP,dalib_0,dalib_0,P(P_ZERO+1),P_DSP)        *
*     call dalib_executor_scatter_send (IS_1,B_DSP)                       *
*     call dalib_executor_scatter_recv (IS_1,0,A_DSP)                     *
*     call dalib_inspector_release (IS_1)                                 *
*                                                                         *
*    base_array : A                                                       *
*    ind_array1 : P                                                       *
*    source_arr : B   ind_arrays are aligned to source array              *
*                                                                         *
**************************************************************************/

#include "dalib.h"
#include "inspector.h"

#undef DEBUG

extern inspector_data *dalib_inspector_db_get();

/**********************************************************************
*                                                                     *
*  dalib_inspector_get  (int *s_id,                                   *
*                        char *source_vals, array_info source_dsp,    *
*                        char *base_vals,   array_info base_dsp,      *
*                        char *mask_vals,   array_info mask_dsp,      *
*                        char *ind1_vals,   array_info ind1_dsp,      *
*                        ...                                          *
*                        char *indk_vals,   array_info indk_dsp)      *
*                                                                     *
*   s_id        : identification of the computed schedule             *
*                                                                     *
*   base_array   : array that will be indirectly addressed            *
*                  (rank of this array is k)                          *
*                                                                     *
*   ind1_dsp   : integer array 1                                      *
*   ...                                                               *
*   indk_dsp   : integer array k                                      *
*                                                                     *
*   mask_array   : mask array used for indexes                        *
*                                                                     *
*  IMPORTANT: source, mask, ind1, ..., indk must be aligned           *
*                                                                     *
**********************************************************************/

void FUNCTION(dalib_inspector_get)

         (s_id, source_vals, source_dsp,
                base_vals,   base_dsp,
                mask_vals,   mask_dsp,
                ind1_vals,   ind1_dsp,
                ind2_vals,   ind2_dsp,
                ind3_vals,   ind3_dsp,
                ind4_vals,   ind4_dsp,
                ind5_vals,   ind5_dsp,
                ind6_vals,   ind6_dsp,
                ind7_vals,   ind7_dsp)

int *s_id;

array_info *source_dsp, *base_dsp, *mask_dsp;
char       *source_vals, *base_vals, *mask_vals;

array_info *ind1_dsp, *ind2_dsp, *ind3_dsp, *ind4_dsp,
           *ind5_dsp, *ind6_dsp, *ind7_dsp;

char       *ind1_vals, *ind2_vals, *ind3_vals, *ind4_vals,
           *ind5_vals, *ind6_vals, *ind7_vals;


{ inspector_info I_Info;
  inspector_data I_Data;

  inspector_data *DB_I_Data;

  /* build the corresponding Inspector information */

  dalib_insp_info_set_source (&I_Info, source_dsp);
  dalib_insp_info_set_base   (&I_Info, base_dsp);

  switch (I_Info.base_rank) {

    case 7 : dalib_insp_info_set_index (&I_Info, 7, ind7_dsp, ind7_vals);
    case 6 : dalib_insp_info_set_index (&I_Info, 6, ind6_dsp, ind6_vals);
    case 5 : dalib_insp_info_set_index (&I_Info, 5, ind5_dsp, ind5_vals);
    case 4 : dalib_insp_info_set_index (&I_Info, 4, ind4_dsp, ind4_vals);
    case 3 : dalib_insp_info_set_index (&I_Info, 3, ind3_dsp, ind3_vals);
    case 2 : dalib_insp_info_set_index (&I_Info, 2, ind2_dsp, ind2_vals);
    case 1 : dalib_insp_info_set_index (&I_Info, 1, ind1_dsp, ind1_vals);
             break;

    default : dalib_internal_error ("INSPECTOR, illegal rank");
              dalib_stop ();

   } /* switch */

  dalib_insp_info_set_mask (&I_Info, mask_dsp, mask_vals);

  /* precompute inspector data for the given info */

  dalib_inspector_fill_data (&I_Data, &I_Info);

  /* find Inspector data for the given information */

  *s_id = dalib_inspector_db_search (&I_Data);

#ifdef DEBUG
   printf ("%d: dalib_inspector_get, position in data base = %d\n",
            pcb.i, *s_id);
#endif

  /* CASE 1 :  compute a new schedule and insert in data base */

  if (*s_id < 0)

    { /* build a completely new inspector for BASE (IND1, ..., INDk) */

      DB_I_Data = (inspector_data *) dalib_malloc (sizeof(inspector_data),
                                                   "creating inspector data");

      *DB_I_Data = I_Data;

      dalib_inspector_build (DB_I_Data, &I_Info);

      *s_id = dalib_inspector_db_insert (DB_I_Data);

      dalib_inspector_set_valid (&I_Info, *s_id);

      if (pcb.redist_flag)
        printf ("%d: schedule for indirect adressing new computed\n", pcb.i);
     
      return;

    }

  /* there exists already an INSPECTOR for BASE(IND1, ..., INDk) */

  DB_I_Data = (inspector_data *) dalib_inspector_db_get (*s_id);
 
  if (!dalib_inspector_is_valid (&I_Info, *s_id))

    { 

#ifdef DEBUG
      printf ("%d: inspector data not valid, needs rebuilding\n", 
                   pcb.i);
#endif

      dalib_inspector_rebuild (DB_I_Data, &I_Info);

      dalib_inspector_set_valid (&I_Info, *s_id);

      if (pcb.redist_flag)
        printf ("%d: schedule for indirect adressing new rebuilt\n", pcb.i);

      return;

    }

  /* The old schedule can be reused as integer arrays are valid, but base 
     may be addressed with other constants, eg. B(1,IND) and B(2,IND)      */ 

  if (!dalib_inspector_is_reusable (DB_I_Data, &I_Data))

    {
#ifdef DEBUG
      printf ("%d: inspector valid, but needs refreshing\n", pcb.i);
#endif

      dalib_inspector_refresh (DB_I_Data, &I_Data);

      return;
 
    }

#ifdef DEBUG
   printf ("%d: inspector valid and reusable\n", pcb.i);
#endif
 
   if (pcb.redist_flag)

      printf ("%d: schedule for indirect adressing will be reused\n", pcb.i);
     
} /* dalib_inspector_get */

/************************************************************
*                                                           *
*  FUNCTION(dalib_executor_gather_send) (s_id, base_dsp)    *
*                                                           *
*  - use schedule s_id for sending values within gather     *
*    to other processes                                     *
*                                                           *
************************************************************/

void FUNCTION(dalib_executor_gather_send) (s_id, base_dsp)

int *s_id;
array_info *base_dsp;

{ int size;

  inspector_data *S;

  /* use internal datatypes for sending the needed values */

  size = (*base_dsp)->size;

  S = (inspector_data *) dalib_inspector_db_get (*s_id);

  dalib_indirect_send_target (S->schedule_id, (*base_dsp)->data, size);

} /* FUNCTION(dalib_executor_gather_send) */

void FUNCTION(dalib_executor_gather_send1) (s_id, base_dsp, serial_size)

int *s_id;
array_info *base_dsp;
int *serial_size;

{ int size;

  inspector_data *S;

  /* use internal datatypes for sending the needed values */

  size = (*base_dsp)->size * (*serial_size);

  S = (inspector_data *) dalib_inspector_db_get (*s_id);

  dalib_indirect_send_target (S->schedule_id, (*base_dsp)->data, size);

} /* FUNCTION(dalib_executor_gather_send1) */

/******************************************************************
*                                                                 *
* FUNCTION(dalib_executor_gather_recv) (int *s_id,                *
*                                       array_info *source_dsp)   *
*                                                                 *
*  - receive my requested values into the source                  *
*    array from other processes                                   *
*                                                                 *
* Note: source must not be identical to the array                 *
*       used for set up the schedule, but the same                *
*       size and must be aligned                                  *
*                                                                 *
*       e.g. A(:) = B(IND(:)), C(:) = D(IND(:))                   *
*                                                                 *
******************************************************************/

void FUNCTION(dalib_executor_gather_recv1) (s_id, source_dsp, serial_size)
int *s_id;
array_info *source_dsp;
int *serial_size;

{ int size;
  int bytes;

  char *source_data;  /* buffer to receive the data that will be get */

  int  source_new;    /* is 0 if buffer has not to be created  
                         as local part of source is contiguous       */

  inspector_data *S;

  S = (inspector_data *) dalib_inspector_db_get (*s_id);

  /* if source is masked we have also to provide original source values */

  dalib_secarray_get_data (*source_dsp, S->mask_array.rank,
                           &size, &bytes, &source_data, &source_new);

  dalib_indirect_recv_source (S->schedule_id, source_data, 
                              (*serial_size) * bytes);

  if (source_new)

      dalib_secarray_unpack (*source_dsp, source_data);

} /* FUNCTION(dalib_executor_gather_recv1) */

void FUNCTION(dalib_executor_gather_recv) (s_id, source_dsp)
int *s_id;
array_info *source_dsp;

{ int serial_size;

  serial_size = 1;

  FUNCTION(dalib_executor_gather_recv1) (s_id, source_dsp, &serial_size);

} /* FUNCTION(dalib_executor_gather_recv) */

/*******************************************************************
*                                                                  *
* FUNCTION(dalib_executor_scatter_send) (int *s_id,                *
*                                        array_info *source_dsp)   *
*                                                                  *
*  - send values from the source to the target                     *
*                                                                  *
*    size : number of bytes of one elem in source                  *
*                                                                  *
*******************************************************************/

void FUNCTION(dalib_executor_scatter_send1) (s_id, source_dsp, serial_size)
int *s_id;
array_info *source_dsp;
int *serial_size;

{ int size;
  int bytes;

  char *source_data;  /* buffer to send the data that will be set */

  int  source_new;    /* is 0 if buffer has not to be created  
                         as local part of source is contiguous       */

  inspector_data *S;

  S = (inspector_data *) dalib_inspector_db_get (*s_id);

  /* copy flag is set to 1 as we read the values of source */

  dalib_secarray_get_data (*source_dsp, 1,
                           &size, &bytes, &source_data, &source_new);

  dalib_indirect_send_source (S->schedule_id, source_data, 
                              (*serial_size) * bytes);

  if (source_new)

     dalib_free (source_data, size*bytes);

} /* FUNCTION(dalib_executor_scatter_send1) */

void FUNCTION(dalib_executor_scatter_send) (s_id, source_dsp)
int *s_id;
array_info *source_dsp;

{ int serial_size;

  serial_size = 1;

  FUNCTION(dalib_executor_scatter_send1)  (s_id, source_dsp, &serial_size);

} /* FUNCTION(dalib_executor_scatter_send) */


/*************************************************************
*                                                            *
* FUNCTION(dalib_executor_scatter_recv)                      *
*                                                            *
*    (int *s_id, int *op, array_info *base_dsp)              *
*                                                            *
*  s_id       : identification of schedule                   *
*  op         : operation for elements at same address       *
*  base_dsp   : descriptor for base array of scatter         *
*                                                            *
*  - use schedule to receive values for a scatter operation  *
*                                                            *
*************************************************************/

void FUNCTION(dalib_executor_scatter_recv1) (s_id, op, base_dsp, serial_size)

int *s_id;
int *op;
array_info *base_dsp;
int *serial_size;

{ int size;

  inspector_data *S;

  S = (inspector_data *) dalib_inspector_db_get (*s_id);

  size = (*base_dsp)->size;

  dalib_indirect_recv_target (S->schedule_id, *op, (*base_dsp)->data, 
                              (*serial_size) * size);

  /* if there are copies of the target we have to broadcast the data */

  dalib_replicate (*base_dsp);

} /* dalib_executor_scatter_recv1 */

void FUNCTION(dalib_executor_scatter_recv) (s_id, op, base_dsp)

int *s_id;
int *op;
array_info *base_dsp;

{ int serial_size;

  serial_size = 1;

  FUNCTION(dalib_executor_scatter_recv1) (s_id, op, base_dsp, &serial_size);

} /* dalib_executor_scatter_recv */

/**************************************************************
*                                                             *
*  FUNCTION(dalib_inspector_release) (int *s_id)              *
*                                                             *
*  - schedule s_id is not in use (does not imply that it      *
*    will be freed)                                           *
*                                                             *
**************************************************************/

void FUNCTION(dalib_inspector_release) (s_id)

int *s_id;   /* internal identification of schedule */

{ 
  dalib_inspector_db_release (*s_id);

} /* dalib_inspector_release */

/**********************************************************************
*                                                                     *
*  dalib_new_gather     (char *source_vals, array_info source_dsp,    *
*                        char *base_vals,   array_info base_dsp,      *
*                        char *mask_vals,   array_info mask_dsp,      *
*                        char *ind1_vals,   array_info ind1_dsp,      *
*                        ...                                          *
*                        char *indk_vals,   array_info indk_dsp)      *
*                                                                     *
*   base_array   : array that will be indirectly addressed            *
*                  (rank of this array is k)                          *
*                                                                     *
*   ind1_dsp   : integer array 1                                      *
*   ...                                                               *
*   indk_dsp   : integer array k                                      *
*                                                                     *
*   mask_array   : mask array used for indexes                        *
*                                                                     *
*  IMPORTANT: source, mask, ind1, ..., indk must be aligned           *
*                                                                     *
*  NEW : ind1 can be NO_PARAM                                         *
*                                                                     *
*  FORALL (I,J,K)  ALL (I,J,K) = X (I,IX(J,K))                        *
*                                                                     *
*    dalib_new_gather (ALL(1:L,1:M,1:N), X (1:L,:), -, IX(1:M,1:N))   *
*                                                                     *
**********************************************************************/

void FUNCTION(dalib_new_gather)

               (source_vals, source_dsp,
                base_vals,   base_dsp,
                mask_vals,   mask_dsp,
                ind1_vals,   ind1_dsp,
                ind2_vals,   ind2_dsp,
                ind3_vals,   ind3_dsp,
                ind4_vals,   ind4_dsp,
                ind5_vals,   ind5_dsp,
                ind6_vals,   ind6_dsp,
                ind7_vals,   ind7_dsp)

array_info *source_dsp, *base_dsp, *mask_dsp;
char       *source_vals, *base_vals, *mask_vals;

array_info *ind1_dsp, *ind2_dsp, *ind3_dsp, *ind4_dsp,
           *ind5_dsp, *ind6_dsp, *ind7_dsp;

char       *ind1_vals, *ind2_vals, *ind3_vals, *ind4_vals,
           *ind5_vals, *ind6_vals, *ind7_vals;


{ 

  /* step 1 : build two templates for source and base */

} /* dalib_new_gather */
