/**************************************************************************
*                                                                         *
*  Author      : Thomas Brandes, GMD, SCAI.LAB                            *
*                Cecile Germain, Orsay, LRI                               *
*                                                                         *
*  Copyright   : LRI, Orsay, France                                       *
*  Date        : Mar 97                                                   *
*  Last Update : Mar 97                                                   *
*                                                                         *
*  Module      : insp_base.m4                                             *
*                                                                         *
*  Function    : managing a data base of reusable schedules for           *
*                INSPECTOR/EXECUTOR operations                            *
*                                                                         *
*  Export:                                                                *
*  =======                                                                *
*                                                                         *
*  void dalib_inspector_db_init ()                                        *
*                                                                         *
*  void dalib_inspector_db_exit ()                                        *
*                                                                         *
*  int dalib_inspector_db_search (inspector_data *I_Data)                 *
*                                                                         *
*   - searches data base for an appropriate schedule that can             *
*     be reused directly or after refreshing it                           *
*                                                                         *
*   - returns int s_id for internal identification (< 0 if not found)     *
*                                                                         *
*  int dalib_inspector_db_insert (inspector_data *I_Data)                 *
*                                                                         *
*   - inserts a new inspector in the data base and returns the            *
*     corresponding identification as an integer                          *
*                                                                         *
*  void dalib_inspector_db_free (int pos)                                 *
*                                                                         *
*   - the inspector data of the data base is freed (is invalid)           *
*                                                                         *
*  inspector_info *dalib_inspector_db_get (id)                            *
*                                                                         *
*   - returns the appropriate inspector from the data base                *
*                                                                         *
*  void dalib_inspector_db_release (int pos)                              *
*                                                                         *
*   - the inspector data of the data base is currently not used           *
*                                                                         *
*    =============  reuse info =============================              *
*                                                                         *
*  void dalib_inspector_db_set_valid (int *reuse_info, int id)            *
*                                                                         *
*   - update reuse_info in the sense that it becomes valid for id         *
*                                                                         *
*  bool dalib_inspector_db_is_valid (int reuse_info, int id)              *
*                                                                         *
*   - asking reuse_info whether it is valid for a schedule                *
*                                                                         *
*  void dalib_inspector_db_set_invalid (int *reuse_info)                  *
*                                                                         *
*   - use invalid reuse_info to free deleted schedules                    *
*                                                                         *
*  Import:                                                                *
*  =======                                                                *
*                                                                         *
*    - dalib_free_indirect (SchedulePtr s_id)                             *
*    - dalib_inspector_match (inspector_data *Data1, *Data2)              *
*                                                                         *
**************************************************************************/

#include "dalib.h"
#include "inspector.h"

#define CHECK
#undef DEBUG

static inspector_data *inspector_db      [MAX_INSPECTORS];
static int            inspector_alloc    [MAX_INSPECTORS];
static int            inspector_attach   [MAX_INSPECTORS];

static int            inspector_db_ptr = 0;

  /*

  inspector_alloc[pos]  :  allocated position with valid inspector data
 
  inspector_attach[pos] :  inspector data is currently in use (so
                           it cannot be freed at this time)

  */

/**********************************************************************
*                                                                     *
*   void dalib_inspector_db_free (int pos)                            *
*                                                                     *
*    - the inspector data of the data base is freed (is invalid)      *
*                                                                     *
**********************************************************************/

void dalib_inspector_db_free (pos)

int pos;

{ inspector_data *I_Data;

#ifdef DEBUG
  printf ("%d: inspector in data base at pos %d freed\n", 
          pcb.i, pos);
#endif

  if (inspector_attach[pos])

     { dalib_internal_error ("inspector_db_free: inspector is attached\n");
       dalib_stop ();
     }

  I_Data = inspector_db[pos];

  dalib_free_indirect (I_Data->schedule_id);

  if (I_Data->is_shadow)

     dalib_free_shadow_changes (I_Data->index_changes);

  dalib_free (I_Data, sizeof (inspector_data));

  inspector_alloc[pos] = 0;

} /* dalib_inspector_db_free */

void dalib_inspector_db_init ()

{ int pos;

  for (pos=0; pos < MAX_INSPECTORS; pos++)
 
    { inspector_alloc [pos] = 0;
      inspector_attach [pos] = 0;
    }

} /* dalib_inspector_db_init */

void dalib_inspector_db_exit ()

{ int pos;

  for (pos = 0; pos < MAX_INSPECTORS; pos++)

     if (inspector_alloc [pos]) dalib_inspector_db_free (pos);

} /* dalib_inspector_db_exit */

/**********************************************************************
*                                                                     *
*  int dalib_insp_base_get_pos ()                                     *
*                                                                     *
*   - looks for a free position in the data base                      *
*                                                                     *
*   - frees unattached schedules if necessary                         *
*                                                                     *
**********************************************************************/

static int dalib_insp_base_get_pos ()

{ int pos;
  int nr;

  pos = 0;

  while (pos < MAX_INSPECTORS)

   { if (inspector_alloc[pos])

        pos++;

      else

        return (pos);
   }

  /* no free place in data base of inspectors, so delete existing one */

  nr = 0;
  pos = inspector_db_ptr;

  while (nr < MAX_INSPECTORS)

    { 
#ifdef DEBUG
      printf ("%d: search unattached schedule, nr = %d, pos = %d, att= %d\n",
               pcb.i, nr, pos, inspector_attach[pos]);
#endif

      inspector_db_ptr++;
      if (inspector_db_ptr >= MAX_INSPECTORS)
         inspector_db_ptr = 0;   /* start at the beginning */

      if (!inspector_attach[pos])

         { /* unattached schedule at position pos within data base */

           dalib_inspector_db_free (pos);

           return (pos);

         }

       else

         { pos = inspector_db_ptr;
           nr++;
         }

    } /* while */

  /* all positions have been considered, nothing released */

  dalib_internal_error ("insp_base_get_pos : no released schedules");
  dalib_stop ();
  return (0);

} /* dalib_insp_base_get_pos */

/*******************************************************************
*                                                                  *
*  int dalib_inspector_db_search (inspector_data *I_Data)          *
*                                                                  *
*   - searches data base for an appropriate schedule that can      *
*     be reused directly or after refreshing it                    *
*                                                                  *
*   - returns int s_id for internal identication                   *
*                                                                  *
*******************************************************************/

int dalib_inspector_db_search (I_Data)

inspector_data *I_Data;

{ int pos;

  pos = 0;

  while (pos < MAX_INSPECTORS)

    { if (inspector_alloc[pos])

         { if (dalib_inspector_match (inspector_db[pos], I_Data))

            { 
#ifdef DEBUG
              printf ("%d: inspector at pos %d matches info, will attach\n", 
                      pcb.i, pos);
#endif
              inspector_attach[pos] ++;
              return (pos);
            }

           else
 
            { 
#ifdef DEBUG
              printf ("%d: inspector at pos %d does not match info\n", 
                       pcb.i, pos);
#endif
            }

         }

      pos++;
    }
 
  return (-1);  /* not found */

} /* dalib_inspector_db_search */

/**********************************************************************
*                                                                     *
*  int dalib_inspector_db_attached (id)                               *
*                                                                     *
**********************************************************************/

int dalib_inspector_db_attached (id)

int id;

{ return (inspector_attach[id]);

} /* dalib_inspector_is_in_use */

/**********************************************************************
*                                                                     *
*  inspector_info *dalib_inspector_db_get (id)                        *
*                                                                     *
*   - returns the appropriate inspector from the data base            *
*                                                                     *
**********************************************************************/

inspector_data *dalib_inspector_db_get (id)

int id;

{ /* verify for a legal index in the data base */

#ifdef CHECK
  if ((id < 0) || (id >= MAX_INSPECTORS))

    { dalib_internal_error ("illegal inspector id");
      dalib_stop ();
    }

  if (!inspector_alloc[id])

    { dalib_internal_error ("inspector id specifies undefined schedule");
      dalib_stop ();
    }
#endif

  return (inspector_db[id]);

} /* dalib_inspector_db_get */

/**********************************************************************
*                                                                     *
*   int dalib_inspector_db_insert (inspector_data *I_Data)            *
*                                                                     *
*   - inserts a new inspector in the data base and returns the        *
*     corresponding identification as an integer                      *
*                                                                     *
**********************************************************************/

int dalib_inspector_db_insert (I_Data)

inspector_data *I_Data;

{ int pos;

  pos = dalib_insp_base_get_pos ();

  inspector_db   [pos] = I_Data;
  inspector_alloc [pos] = 1;
  inspector_attach [pos] = 1;    /* will also be used now */

#ifdef DEBUG
  printf ("%d: new inspector in data base at pos %d inserted\n", pcb.i, pos);
#endif

  return (pos);

} /* dalib_inspector_db_insert */

/**********************************************************************
*                                                                     *
*   void dalib_inspector_db_release (int pos)                         *
*                                                                     *
*    - the inspector data of the data base is currently not used      *
*                                                                     *
**********************************************************************/

void dalib_inspector_db_release (pos)

int pos;

{ inspector_attach[pos] --;

#ifdef DEBUG
  printf ("%d: inspector db release, pos = %d, now attached = %d\n",
           pcb.i, pos, inspector_attach[pos]);
#endif

  /* do not free schedule if it is used somewhere else */

  if (inspector_attach[pos] > 0) return;

  if (!(inspector_db[pos]->is_traced))

    { dalib_inspector_db_free (pos);
      return;
    }

#ifdef DEBUG
  printf ("%d: inspector in data base at pos %d released (%d) (not freed)\n", 
          pcb.i, pos, inspector_attach[pos]);
#endif

} /* dalib_inspector_db_release */

/**********************************************************************
*                                                                     *
*  void dalib_inspector_db_set_valid (int *reuse_info, int id)        *
*                                                                     *
*   - update reuse_info in the sense that it becomes valid for id     *
*                                                                     *
**********************************************************************/

void dalib_inspector_db_set_valid (reuse_info, id)

int *reuse_info;
int id;

{ *reuse_info = *reuse_info | (1 << id); }

/**********************************************************************
*                                                                     *
*  bool dalib_inspector_db_is_valid (int reuse_info, int id)          *
*                                                                     *
*    - asking reuse_info whether it is valid for a schedule           *
*                                                                     *
**********************************************************************/

int dalib_inspector_db_is_valid (reuse_info, id)

int reuse_info;
int id;

{ if (reuse_info & (1 << id)) return (1);
  return (0);
}

/**********************************************************************
*                                                                     *
*  void dalib_inspector_db_set_invalid (int *reuse_info)              *
*                                                                     *
*    - use invalid reuse_info to free deleted schedules               *
*                                                                     *
**********************************************************************/

void dalib_inspector_db_set_invalid (reuse_info)

int *reuse_info;

{ int reuse_bits;
  int id;

  reuse_bits = *reuse_info;
  *reuse_info = 0;

  id = 0;

  while (reuse_bits)

    { if (reuse_bits & 1) dalib_inspector_db_free (id);

      id++;

      reuse_bits = reuse_bits >> 1;
    }

} /* dalib_inspector_db_set_invalid */

