/*******************************************************************
*                                                                  *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                 *
*  Copyright   : GMD St. Augustin, Germany                         *
*  Date        : Feb 94                                            *
*  Last Update : Jan 98                                            *
*                                                                  *
*  This Module is part of the DALIB                                *
*                                                                  *
*  MODULE : init.c                                                 *
*                                                                  *
*  Function: Realization of System InDependent Initializations     *
*                                                                  *
*  FORTRAN INTERFACE                                               *
*  =================                                               *
*                                                                  *
*    void dalib_init ()                                            *
*    void dalib_exit ()                                            *
*                                                                  *
*  Extensions:                                                     *
*  ===========                                                     *
*                                                                  *
*   10/97  :  correct use of context (active processors)           *
*   01/98  :  interface for time and comm statistics               *
*                                                                  *
*******************************************************************/

#include <stdio.h>
#include <string.h>
#include "dalib.h"

#undef DEBUG

/*******************************************************************
*                                                                  *
*  general routines for initialization / termination               *
*                                                                  *
*******************************************************************/

static void dalib_general_init ()

{ dalib_inspector_db_init ();

  /* note: msg_init is already called before */

  random_block_init ();

  dalib_groups_init (pcb.p);  /* called again, now with number of
                                 processors that participate      */

  dalib_push_main_context ();

#ifdef SEM
  dalib_define_barrier ();
#endif

} /* dalib_general_init */

static void dalib_general_exit ()

{ dalib_inspector_db_exit ();  /* free all schedules of data base */
  dalib_top_exit ();           /* free allocated memory */
  dalib_groups_exit ();        /* free allocated memory */
  dalib_msg_exit ();
  dalib_alloc_statistic ();    /* checks correct allocation/deallocation */

#ifdef SEM
  dalib_free_barrier ();
#endif
} /* dalib_general_exit */

/*******************************************************************
*                                                                  *
*  Machine independent functions                                   *
*                                                                  *
*  void dalib_run (NP)                                             *
*                                                                  *
*  - pcb.i : own id                                                *
*  - pcb.p : number of existing processes                          *
*  - NP    : number of processes that will be used really          *
*                                                                  *
*******************************************************************/

void dalib_run (NP)

int NP;

{ int used_NP;
  int pcb_size;

  /* 1 <= pcb.p <= NP, only pcb.p processes will play the game */

  used_NP = pcb.p;

  /* set environment that broadcast for all processors is done */
 
  pcb.p = NP;

  dalib_groups_init (NP);
  dalib_msg_init ();

  dalib_all_broadcast (&used_NP, sizeof(used_NP), 1);

#ifdef DEBUG
  printf ("%d of %d processes: got NP = %d\n", pcb.i, pcb.p, NP);
#endif

  if (pcb.i <= used_NP)

     {  /* now we know which processors are playing the game */

        pcb.p = used_NP;

        /* send other values for topology, trace_flag */

        pcb_size = sizeof (pcb) - 2 * sizeof(int);

        dalib_all_broadcast (&(pcb.p_rank), pcb_size, 1);

#ifdef DEBUG
  printf ("%d: got all initial data\n", pcb.i);
#endif

        dalib_general_init ();

     }

    else  /* do not play the game, but normal exit */

     { dalib_machine_exit ();
       exit (0);
     }

#ifdef DEBUG
  printf ("%d: starts now the user program\n", pcb.i);
#endif

} /* dalib_run */

          /***********************************************
          *                                              *
          *  init routines                               *
          *                                              *
          ***********************************************/

void FUNCTION(dalib_init) (int_size, real_size, addr_size) 

INTEGER *int_size, *real_size, *addr_size;

{ int NP;                      /* really existing processors */

  char msg[120];

  pcb.p_rank = 1;

  pcb.trace_flag  = 0;
  pcb.call_flag   = 0;
  pcb.redist_flag = 0;
  pcb.time_flag   = 0;
  pcb.comm_flag   = 0;

  dalib_check_sizes (*int_size, *real_size, *addr_size);

  dalib_machine_enroll (&NP);  /* sets also pcb, tids */

#if defined(VT)
  dalib_trace_define ();
#endif

  dalib_run (NP);

  if (pcb.comm_flag)

    {
#ifdef DEBUG
      printf("%d: initialize communication statistics\n",pcb.i);
#endif
      dalib_init_commstat();
    }

  if (pcb.time_flag)
    {
#ifdef DEBUG
      printf("%d: initialize timing statistics\n",pcb.i);
#endif
      dalib_init_timestat ();
    }

} /* dalib_init */

/*******************************************************************
*                                                                  *
*  exit of SPMD      program                                       *
*                                                                  *
*******************************************************************/

void FUNCTION(dalib_exit) () 

{ comm_statistics_collect = 0; /* stop collection of communication statistics */
  time_statistics_collect = 0; /* stop collection of timing statistics*/

  if (pcb.time_flag)
      dalib_collect_timestat ();

  if (pcb.comm_flag)
      dalib_collect_commstat ();

  dalib_general_exit ();
  dalib_machine_exit ();   

} /* dalib_exit */

