/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.WR                         *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Jan 95                                                   *
*  Last Update : Aug 97                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : ind_shadow.m4                                            *
*                                                                         *
*  Function    : Use of shadow edges for unstructured communication       *
*                                                                         *
*  Export : FORTRAN Interface                                             *
*                                                                         *
*   void dalib_ind_shadow_define (schedule_ptr *s_id,                     *
*                                 char *array_data, char *index_data,     *
*                                 array_info *s_dsp,                      *
*                                 array_info *array_dsp, *index_dsp)      *
*                                                                         *
*   - update index to point to shadow area, computes schedule s_id        *
*                                                                         *
*   void dalib_ind_shadow_get (schedule_ptr *s_id, char *array_data,      *
*                              array_info *s_dsp, *array_dsp)             *
*                                                                         *
*   - update shadow area of array corresponding the schedule              *
*   - shadow area will contain afterwards the non-local values            *
*                                                                         *
*   void dalib_ind_shadow_put (schedule_ptr *s_id,                        *
*                              char *array_data, int *op,                 *
*                              array_info *s_dsp, *array_dsp, *op_dsp)    *
*                                                                         *
*   - write non-local computed values back to owning processors           *
*   - elements will be combined correspoinding reduction operator op      *
*                                                                         *
*   void dalib_ind_shadow_set (schedule_ptr *s_id,                        *
*                              char *array_data, char *val_data,          *
*                              array_info *s_dsp, *array_dsp, *val_dsp)   *
*                                                                         *
*   - initializes shadow area with the value 'val'                        *
*   - needed for reductions on distributed arrays                         *
*                                                                         *
*   void dalib_ind_shadow_set1 (schedule_ptr *s_id,                       *
*                               char *array_data, char *val_data,         *
*                               array_info *s_dsp, *array_dsp, *val_dsp)  *
*                                                                         *
*     - initialize shadow area for out-of-range values                    *
*                                                                         *
*   void dalib_ind_shadow_free (schedule_ptr *s_id)                       *
*                                                                         *
*    - frees internal data structures used for the schedule               *
*                                                                         *
**************************************************************************/

#undef DEBUG

#undef REMOTE
#define CHECK

#include "dalib.h"
#include "inspector.h"

extern inspector_data *dalib_inspector_db_get ();

/************************************************************************
*                                                                       *
*  Data structure used for full shadow schedule when it is used         *
*                                                                       *
*  Note : different shadow schedules can share the same                 *
*         communication schedule (but do not share shadow)              *
*                                                                       *
************************************************************************/

typedef struct shadow_schedule *shadow_ptr;

struct shadow_schedule {

   int db_pos;             /* position in inspector data base
                              for the communication schedule        */

   char *shadow_area;       /* shadow area allocated on heap        */

   int shadow_area_size;    /* number of bytes allocated on heap    */

   int shadow_entries;      /* size of shadow area                  */

   int offset;              /* a[...,low+offset+1] points to shadow */
   int high;                /* a[...,high] points to shadow_data    */

};  /* shadow schedule */

/*******************************************************************
*                                                                  *
*   dalib_indirect_localize_addresses (schedule_id)                *
*                                                                  *
*******************************************************************/

void dalib_indirect_localize_addresses (base, schedule_id)

schedule_ptr schedule_id;
array_info base;

{ int no_other_values;
  int *global_addresses;

#ifdef DEBUG 
  printf ("dalib_indirect_localize_addresses, schedule = %d\n", schedule_id);
#endif 

  dalib_indirect_target_info (schedule_id, &no_other_values, 
                              &global_addresses);
 
#ifdef DEBUG 
  printf ("target info : %d other values\n", no_other_values);
#endif 

  dalib_global_to_local (base, no_other_values, 
                         global_addresses, global_addresses);

} /* dalib_indirect_localize_addresses */

/**********************************************************************
*                                                                     *
*  void dalib_count_owners (owners, N, M, m, non_local, out_range)    *
*                                                                     *
*  int N, M, m, *non_local, *out_range;                               *
*  int owners[N];                                                     *
*                                                                     *
*  non_local : number of owners[i] between 0 and M-1, but not m       *
*  out_range : number of owners[i] not between 0 and M-1              *
*                                                                     *
**********************************************************************/

static void dalib_count_owners (owners, N, M, m, non_local, out_range)   

int N, M, m, *non_local, *out_range;
int owners[];

{ int j, val;

  int nr1 = 0;   /* counts out of range values      */
  int nr2 = 0;   /* counts legal values not equal m */

  for (j=0; j < N; j++)

    { val = owners[j];

      if (val >= 0)

         { if (val >= M) nr1++;
             else if (val != m) nr2++;
         }

    } /* for all owner elements */

  *out_range = nr1;
  *non_local = nr2;

#ifdef DEBUG
  printf ("%d: count_owners (N=%d), M=%d, m=%d, out = %d, not local = %d\n",
          pcb.i, N, M, m, nr1, nr2);
#endif

} /* dalib_count_owners */

/**********************************************************************
*                                                                     *
*  void dalib_get_owners (owners, N, M, m, non_local, out_range)      *
*                                                                     *
*  int N, M, m;                                                       *
*  int owners[N];                                                     *
*                                                                     *
*  int non_local[], out_range[];                                      *
*                                                                     *
*  - sizes of non_local, out_range given by dalib_count_owners        *
*                                                                     *
**********************************************************************/

static void dalib_get_owners (owners, N, M, m, non_local, out_range)

int N, M, m, owners[], non_local[], out_range[];

{ int j, val;

  int nr1 = 0;   /* counts out of range values      */
  int nr2 = 0;   /* counts legal values not equal m */
  int nr3 = 0;   /* counts masked values            */

  for (j=0; j < N; j++)

    { val = owners[j];

      if (val >= M)

         { out_range[nr1++] = j;
           owners[j] = -1;
         }
 
       else if (val < 0)

         { nr3++;
           owners[j] = -1;
         }

        else if (val != m)
         
           non_local[nr2++] = j;

        else
         
           owners[j] = -1;

    } /* for j */

#ifdef DEBUG
  printf ("%d: dalib_get_owners : %d out of range, %d non local, %d masked\n",
           pcb.i, nr1, nr2, nr3);
#endif

} /* dalib_get_owners */

/*******************************************************************
*                                                                  *
*  dalib_shadow_calc (s_id, array_dsp, target_proc, global_addr)   *
*                                                                  *
*******************************************************************/

void dalib_shadow_calc (s_id, array_dsp, indexes, no_indexes,
                        target_processors, global_addresses, change)

schedule_ptr *s_id;
array_info array_dsp;
int indexes[];
int no_indexes;
int target_processors[];
int global_addresses [];
int **change;

{ int pid, NP, NId;

  int *non_local_indexes;
  int *non_global_indexes;
  int *source_indexes;
  int *index_change;
  int non_local_nr, non_global_nr;
  int source_topology;

  array_info dummy;

  int base_topology;

  int j, no_values;

  /* we have to get base topology and source topology */

  dalib_array_info (array_dsp, &dummy, &base_topology);

  /* note : array_dsp can also be a section */

  dalib_top_info (base_topology, 1, &NP, &NId);

  NId -= 1;    /* 0 <= NId < NP */

#ifdef DEBUG
   printf ("%d: dalib_shadow_calc, no=%d, top (id=%d, NP=%d, NId=%d)\n",
           pcb.i, no_indexes, base_topology, NP, NId);
#endif

  /* collect non-local indexes and non-global (out of range) indexes) */

  dalib_count_owners (target_processors, no_indexes, NP, NId, 
                      &non_local_nr, &non_global_nr);

#ifdef DEBUG
   printf ("%d: dalib_shadow_calc, non_local = %d, non_global = %d\n",
            pcb.i, non_local_nr, non_global_nr);
#endif
 
  non_local_indexes  = dalib_int_malloc (non_local_nr, "shadow_calc1");
  non_global_indexes = dalib_int_malloc (non_global_nr, "shadow_calc2");

  index_change = dalib_int_malloc (3*(non_local_nr + non_global_nr) + 1,
                                   "shadow_calc3");

  *change = index_change;

  *(index_change++) = non_local_nr + non_global_nr;

  /* step 2 : compress arrays for non-local indexes */
 
  dalib_get_owners (target_processors, no_indexes, NP, NId,
                    non_local_indexes, non_global_indexes);

  /* for the schedule all local indexes will not be considered  */

  source_topology = 1;  /*  !!!!  */

  dalib_new_indirect (s_id, source_topology, base_topology, no_indexes,
                            target_processors, global_addresses);

  /* global indexes in indexes become now indexes in overlap area */
 
  dalib_indirect_source_info (*s_id, &no_values, &source_indexes);

  /* all non global indexes will point to a default value */
 
#ifdef DEBUG
   printf ("%d: update %d non global indexes\n", pcb.i, non_global_nr);
#endif 

  for (j=0; j < non_global_nr; j++)

    {  /* index_change: pos, old_val, new_val */

       *(index_change++) = non_global_indexes[j];
       *(index_change++) = indexes[non_global_indexes[j]];
       *(index_change++) = -1;
    }
 
  /* all non local indexes will point to the overlap area */
 
#ifdef DEBUG
   printf ("%d: update %d non local indexes (= %d)\n", 
            pcb.i, non_local_nr, no_values             );
#endif 

  for (j=0; j < non_local_nr; j++)

     { 

#ifdef DEBUG
       /*
       printf ("%d: non-local %d translates index[%d] -> %d \n", pcb.i, j,
               indexes[source_indexes[j]], j+1); */
#endif

       /* index_change: pos, old_val, new_val */

       *(index_change++) = source_indexes[j];
       *(index_change++) = indexes[source_indexes[j]];
       *(index_change++) = j;

       source_indexes[j] = j;
     }
 
#ifdef DEBUG
   printf ("%d: dalib_shadow_calc ready\n", pcb.i);
#endif
 
  dalib_int_free (non_global_indexes, non_global_nr);
  dalib_int_free (non_local_indexes, non_local_nr);

} /* dalib_shadow_calc */

/*******************************************************************
*                                                                  *
*  void dalib_shadow_reset_indexes (int *indexes,                  *
*                                   int *index_changes)            *
*                                                                  *
*    - reset indirection array to old values                       *
*                                                                  *
*******************************************************************/

void dalib_shadow_reset_indexes (indexes, index_changes)

int indexes[];
int *index_changes;

{ int i, n;

  n = *index_changes++;

  for (i=0; i<n; i++)

     { int pos = *index_changes++;
       int old = *index_changes++;
       int new = *index_changes++;

       indexes [pos] = old;
     }

} /* dalib_shadow_reset_indexes */

/*******************************************************************
*                                                                  *
*  void dalib_shadow_change_indexes (int *indexes,                 *
*                                    int high,                     *
*                                    int *index_changes)           *
*                                                                  *
*    - set in indirection array the new index values               *
*    - high is the offset into the shadow area                     *
*      so A(...,HIGH) is first shadow element                      *
*                                                                  *
*******************************************************************/

void dalib_shadow_change_indexes (indexes, high, index_changes)

int indexes[];
int high;
int *index_changes;

{ int i, n;

  n = *index_changes++;

  for (i=0; i<n; i++)

     { int pos = *index_changes++;
       int old = *index_changes++;
       int new = *index_changes++;

#ifdef DEBUG
   /*    printf ("%d: change index at pos = %d to %d = %d + %d\n", 
                  pcb.i, pos, new+high, new, high); */
#endif

       indexes [pos] = new + high;
     }

} /* dalib_shadow_change_indexes */

/*******************************************************************
*                                                                  *
*  void dalib_free_shadow_changes (int *index_changes)            *
*                                                                  *
*    - free allocated memory for index change structure            *
*                                                                  *
*******************************************************************/

void dalib_free_shadow_changes (index_changes)

int *index_changes;

{ if (index_changes == (int *) 0) return;

  dalib_int_free (index_changes, 3 * index_changes[0] + 1);

} /* dalib_free_shadow_changes */

/**********************************************************************
*                                                                     *
*  void dalib_ind_array_info (array_info array_dsp, int *obj_size,    *
*                             int *serial_size, int *dist_size)       *
*                                                                     *
*  IN  : array_info array_dsp  (must only be distributed in the       *
*                               last dimension)                       *
*                                                                     *
*  OUT : obj_size is number of bytes for one element                  *
*        serial_size is number of elements in serial dimension        *
*        dist_size is number of elements in distributed dimension     *
*                                                                     *
**********************************************************************/

static void dalib_ind_array_info (array_dsp, obj_size, serial_size, dist_size)

array_info array_dsp;
int *obj_size, *serial_size, *dist_size;

{ int i, rank;
  DimInfo *dim;

  rank = array_dsp->rank;
  dim  = array_dsp->dimensions;

  *obj_size = array_dsp->size;
  *serial_size = 1;

  for (i=0; i<rank-1; i++)
    { *serial_size *=  (dim->global_size[1] - dim->global_size[0] + 1);
      dim++;
    }

  *dist_size = dim->local_size[1] - dim->local_size[0] + 1;

#ifdef DEBUG
   printf ("%d: ind_array_info, dsp = %d, size = %d, serial = %d, dist = %d\n",
            pcb.i, array_dsp, *obj_size, *serial_size, *dist_size);
#endif 

} /* dalib_ind_array_info */

/*******************************************************************
*                                                                  *
*                                                                  *
*******************************************************************/

shadow_ptr dalib_build_shadow_schedule (pos, base_dsp)

int        pos;
array_info base_dsp;

{ inspector_data *S;
  int *dummy;
  char *shadow_area;
  unsigned char *shadow_data;
  int  shadow_entries;
  int  shadow_area_size;
  int  shadow_bytes;
  int  low, high, rank;
  int  left_ov, right_ov;
  int  offset;

  shadow_ptr schedule;

  int nbytes, serial, N;

  S = dalib_inspector_db_get (pos);

  dalib_indirect_source_info (S->schedule_id, &shadow_entries, &dummy);

  dalib_ind_array_info (base_dsp, &nbytes, &serial, &N);

  rank     = base_dsp->rank;
  low      = base_dsp->dimensions[rank-1].local_size[0];
  high     = base_dsp->dimensions[rank-1].local_size[1];
  left_ov  = base_dsp->dimensions[rank-1].overlap[0];
  right_ov = base_dsp->dimensions[rank-1].overlap[1];

  shadow_bytes     = serial * nbytes;

  if (right_ov >= shadow_entries + 1)

     { /* shadow area must not be allocated */

      if (pcb.redist_flag)
         printf ("%d: unstructured shadow area (size = %d) available for %s\n", 
                  pcb.i, shadow_entries, base_dsp->name);
     
       shadow_area      = (char *) 0;
       shadow_area_size = 0;

       offset           = (high - low + 1);

       /* schedule->high  = high + 2 */
     }

   else

     { /* shadow area will be allocated on the heap */

       if (pcb.redist_flag)
         printf ("%d: unstructured shadow area (size = %d) needed for %s\n", 
                  pcb.i, shadow_entries, base_dsp->name);
     
       shadow_area_size = (shadow_entries + 2) * shadow_bytes;

       shadow_area = (char *) dalib_malloc (shadow_area_size,
                                            "dalib_build_shadow_schedule");

       offset = (long) shadow_area - (long) base_dsp->data;

       /* make sure that we get a correct alignment */

       while (offset%shadow_bytes != 0) offset++;

       /* now division is without any reminder */

       offset = offset / shadow_bytes;

       offset = offset - left_ov;  

     }

  shadow_data = base_dsp->data + (offset+left_ov+1) * shadow_bytes;

#ifdef DEBUG
  printf ("%d: dalib_build_schedule, entries = %d, bytes = %d\n",
           pcb.i, shadow_entries, shadow_bytes);
  printf ("%d : base data at %d, shadow area at %d, offset = %d\n",
           pcb.i, base_dsp->data, shadow_area, offset);
  printf ("%d : shadow data at %d, low = %d, l_ov = %d, b(%d) maps to shadow\n",
           pcb.i, shadow_data, low, left_ov, offset+low-left_ov+1);
#endif

  schedule = (shadow_ptr) dalib_malloc (sizeof(struct shadow_schedule), 
                                        "build_shadow");

  schedule->db_pos           = pos;
  schedule->shadow_area      = shadow_area;
  schedule->offset           = offset;
  schedule->high             = offset + low + 1;
  schedule->shadow_area_size = shadow_area_size;
  schedule->shadow_entries   = shadow_entries;

  return schedule;

} /* dalib_build_shadow_schedule */

/****************************************************************************
*                                                                           *
*  void dalib_shadow_indexes (..)                                           *
*                                                                           *
*   - copies indexes into temporaries                                       *
*   - makes update of non-local, non-global indexes                         *
*                                                                           *
****************************************************************************/

void dalib_shadow_indexes (schedule, ind_dsp, tmp_dsp, index_changes)

shadow_ptr schedule;
int *index_changes;
array_info *tmp_dsp, *ind_dsp;

{ int no, size, is_new;
  int *index_data;

#ifdef DEBUG
  printf ("%d: dalib_shadow_indexes, changes = %d\n", 
           pcb.i, index_changes[0]);
#endif

  if ((*tmp_dsp) != (*ind_dsp))

    { /* copy index values in temporary only if it is a new array */

      FUNCTION(dalib_assign) (tmp_dsp, ind_dsp);

#ifdef DEBUG
      printf ("%d: indexes copied to temporary\n", pcb.i);
#endif

    }

  dalib_secarray_get_data (*tmp_dsp, 1, &no, &size, &index_data, &is_new);

  if (is_new)

     { dalib_internal_error ("shadow rebuild for noncontigous indexes");
       dalib_stop ();
     }

  dalib_shadow_change_indexes (index_data, schedule->high, index_changes);

} /* dalib_shadow_indexes */

/****************************************************************************
*                                                                           *
*  void dalib_shadow_indexes (..)                                           *
*                                                                           *
****************************************************************************/

void dalib_free_shadow_schedule (schedule)

shadow_ptr schedule;

{ 
#ifdef DEBUG
  printf ("%d: free shadow schedule, data size = %d\n",
          pcb.i, schedule->shadow_area_size);
#endif

  if (schedule->shadow_area_size > 0)
     dalib_free (schedule->shadow_area, schedule->shadow_area_size);

  dalib_free (schedule, sizeof (struct shadow_schedule));

} /* dalib_free_shadow_schedule */

/**********************************************************************
*                                                                     *
*  dalib_insp_shadow_get  (int *s_id,                                 *
*                          char *base_vals,   array_info base_dsp,    *
*                          char *mask_vals,   array_info mask_dsp,    *
*                          char *ind_vals,    array_info ind_dsp)     *
*                          char *tmp_vals,    array_info tmp_dsp)     *
*                                                                     *
*   s_id        : identification of the computed schedule             *
*                                                                     *
*   base_array   : array that will be indirectly addressed            *
*                  (distributed along the last dimension)             *
*                                                                     *
*   ind_dsp      : integer array for distributed dimension            *
*                                                                     *
*   mask_array   : mask array used for indexes                        *
*                                                                     *
*  IMPORTANT: mask, ind must be aligned                               *
*                                                                     *
**********************************************************************/

void FUNCTION(dalib_insp_shadow_get)

         (s_id, base_vals,   base_dsp,
                mask_vals,   mask_dsp,
                ind_vals,    ind_dsp,
                tmp_vals,    tmp_dsp   )

shadow_ptr *s_id;

array_info *base_dsp, *mask_dsp, *ind_dsp, *tmp_dsp;
char       *base_vals, *mask_vals, *ind_vals, *tmp_vals;

{ inspector_info I_Info;
  inspector_data I_Data;

  inspector_data *DB_I_Data;

  char *index_data;

  int no, size, is_new;
  int db_id;

  /* build the corresponding Inspector information , shadow = 1 */

  dalib_insp_info_set_shadow (&I_Info, base_dsp);
  dalib_insp_info_set_index  (&I_Info, 1, ind_dsp, ind_vals);
  dalib_insp_info_set_mask   (&I_Info, mask_dsp, mask_vals);

  /* precompute inspector data for the given info */

  dalib_inspector_fill_data (&I_Data, &I_Info);

  /* find Inspector data for the given information */

  db_id = dalib_inspector_db_search (&I_Data);

#ifdef DEBUG
   printf ("%d: dalib_insp_shadow_get, position in data base = %d\n",
            pcb.i, db_id);
#endif

  /* CASE 1 :  compute a new schedule and insert in data base */

  if (db_id < 0)

    { /* build a completely new inspector for BASE (IND1, ..., INDk) */

      DB_I_Data = (inspector_data *) dalib_malloc (sizeof(inspector_data),
                                                   "creating inspector data");

      *DB_I_Data = I_Data;

      dalib_inspector_build (DB_I_Data, &I_Info);

      db_id = dalib_inspector_db_insert (DB_I_Data);

      dalib_inspector_set_valid (&I_Info, db_id);

      if (pcb.redist_flag)
        printf ("%d: schedule for unstructured shadow new computed\n", pcb.i);
     
      goto calc;

    }

  /* there exists already an INSPECTOR for BASE(IND1, ..., INDk) */

  DB_I_Data = dalib_inspector_db_get (db_id);
 
  if (!dalib_inspector_is_valid (&I_Info, db_id))

    { 

#ifdef DEBUG
      printf ("%d: inspector data not valid, needs rebuilding\n", 
                   pcb.i);
#endif

      dalib_inspector_rebuild (DB_I_Data, &I_Info);

      dalib_inspector_set_valid (&I_Info, db_id);

      if (pcb.redist_flag)
        printf ("%d: schedule for indirect adressing new rebuilt\n", pcb.i);

    }

#ifdef DEBUG
   printf ("%d: inspector valid and reusable\n", pcb.i);
#endif
 
   if (pcb.redist_flag)

      printf ("%d: schedule for indirect adressing will be reused\n", pcb.i);


calc: /* now build the shadow schedule */

   *s_id = dalib_build_shadow_schedule (db_id, *base_dsp);

   dalib_shadow_indexes (*s_id, ind_dsp, tmp_dsp, DB_I_Data->index_changes);

} /* dalib_insp_shadow_get */

/**************************************************************************
*                                                                         *
*   void dalib_exec_shadow_get (int *s_id, array_info *base_dsp)          *
*                                                                         *
*   - update shadow area of array corresponding the schedule              *
*   - shadow area will contain afterwards the non-local values            *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_exec_shadow_get) (s_id, base_dsp)

shadow_ptr *s_id;
array_info *base_dsp;

{ unsigned char *data;
  int N;
  int size, serial;
  int rank, left_ov;

  inspector_data *S;

  int db_id;

  db_id = (*s_id)->db_pos;

  S = dalib_inspector_db_get (db_id);

  data = (*base_dsp)->data;

  dalib_ind_array_info (*base_dsp, &size, &serial, &N);

  size = size * serial;  /* size is multiplied by elems in serial dimension */

#ifdef DEBUG
  printf ("%d: dalib_ind_shadow_get, schedule = %d, data = %d, size =%d\n",
           pcb.i, S->schedule_id, data, size);
#endif

  dalib_indirect_send_target (S->schedule_id, data, size);

#ifdef DEBUG
  printf ("%d: dalib_indirect_ov_update, have sent target\n", pcb.i);
#endif

  rank    = (*base_dsp)->rank;
  left_ov = (*base_dsp)->dimensions[rank-1].overlap[0];

  data = data + ((*s_id)->offset + left_ov + 1) * size;

  dalib_indirect_recv_source (S->schedule_id, data, size);
 
#ifdef DEBUG
  printf ("%d: dalib_indirect_ov_update, have received source\n", pcb.i);
#endif

} /* dalib_exec_shadow_get */

/**************************************************************************
*                                                                         *
*   void dalib_exec_shadow_put (int *s_id, array_info *base_dsp, int *op) *
*                                                                         *
*   - write non-local computed values back to owning processors           *
*   - elements will be combined correspoinding reduction operator op      *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_exec_shadow_put) (s_id, base_dsp, op)

shadow_ptr *s_id;
array_info *base_dsp;
int *op;

{ unsigned char *data;
  int N;
  int size, serial;
  int rank, left_ov;

  inspector_data *S;
  int db_id;

  db_id = (*s_id)->db_pos;

  S = dalib_inspector_db_get (db_id);

  data = (*base_dsp)->data;

  dalib_ind_array_info (*base_dsp, &size, &serial, &N);

  size = size * serial;  /* size is multiplied by elems in serial dimension */

  rank    = (*base_dsp)->rank;
  left_ov = (*base_dsp)->dimensions[rank-1].overlap[0];

  data = data + ((*s_id)->offset + left_ov + 1) * size;

#ifdef DEBUG
  printf ("%d: dalib_ind_shadow_put, schedule = %d, data = %d, size =%d\n",
           pcb.i, S->schedule_id, data, size);
#endif

  dalib_indirect_send_source (S->schedule_id, data, size);

#ifdef DEBUG
  printf ("%d: dalib_ind_shadow_put, have sent source\n", pcb.i);
#endif

  data = (*base_dsp)->data;

  dalib_indirect_recv_target (S->schedule_id, *op, data, size);
 
#ifdef DEBUG
  printf ("%d: dalib_ind_shadow_put, have received source\n", pcb.i);
#endif

} /* dalib_exec_shadow_put */
 
/**************************************************************************
*                                                                         *
*   void dalib_exec_shadow_set (int *s_id, array_info *base_dsp, char *val) *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_exec_shadow_set) (s_id, base_dsp, val_data)

shadow_ptr *s_id;
array_info *base_dsp;
char *val_data;

{ unsigned char *data;
  int N;
  int size, serial;
  int ov_size;
  int *dummy;
  int i, j;
  int rank, left_ov;

  dalib_ind_array_info (*base_dsp, &size, &serial, &N);

  ov_size = (*s_id)->shadow_entries;

  rank    = (*base_dsp)->rank;
  left_ov = (*base_dsp)->dimensions[rank-1].overlap[0];

  data = (*base_dsp)->data;
  data = data + ((*s_id)->offset + left_ov + 1) * size * serial;

#ifdef DEBUG
  printf ("%d: shadow_set, data = %d, size = %d, serial =%d, ov_size = %d\n",
          pcb.i, data, size, serial, ov_size);
#endif

  for (i=0; i<ov_size*serial; i++)

     for (j=0; j<size; j++) *data++ = val_data[j];

} /* dalib_exec_shadow_set */

/**************************************************************
*                                                             *
*  FUNCTION(dalib_insp_shadow_release) (shadow_ptr *s_id)     *
*                                                             *
*  - shadow structur is deleted, but not shadow comm. sched.  *
*                                                             *
**************************************************************/

void FUNCTION(dalib_insp_shadow_release) (s_id, ind_vals, ind_dsp)

shadow_ptr *s_id;   /* internal identification of schedule */

array_info *ind_dsp;
int        *ind_vals;

{ int db_id;

  db_id = (*s_id)->db_pos;

  dalib_free_shadow_schedule (*s_id);

  dalib_inspector_db_release (db_id);

} /* dalib_insp_shadow_release */

/*****************************************************************************
*                                                                            *
*  void dalib_ind_shadow_define (schedule_ptr *s_id,                         *
*                                char *array_data, char *index_data,         *
*                                array_info *s_dsp,                          *
*                                array_info *array_dsp, *index_dsp)          *
*                                                                            *
*   - update index to point to shadow area, computes schedule s_id           *
*                                                                            *
*   in :  array_dsp, array_data   array distributed along last dimension     *
*   in :  index_dsp, index_data   index values for last dim of array         *
*                                                                            *
*   out :  non local values of index will point to shadow area of array      *
*          s_id will be schedule to operate on shadow area                   *
*                                                                            *
*   - array_dsp is an array with an overlap area                             *
*   - array_dsp can have more than one dimension, but only last              *
*     one can be distributed                                                 *
*                                                                            *
*        !HPF$ DISTRIBUTE array (*,*,ANY)                                    *
*                                                                            *
*  This routine has side effects for the array index_dsp                     *
*  as non-local indexes will point into the overlap area                     *
*                                                                            *
*****************************************************************************/

void FUNCTION(dalib_ind_shadow_define) (s_id, array_data, index_data,
                                        s_dsp, array_dsp, index_dsp)

shadow_ptr *s_id;   /* internal identification of schedule */
array_info *array_dsp, *index_dsp, *s_dsp;
char *array_data, *index_data;

{ int *no_arg; 

  dalib_get_not_present (&no_arg);

  /* descriptor for new indexes is the same one as the old one */

  FUNCTION(dalib_insp_shadow_get) (s_id, array_data,  array_dsp,
                                         (char *) no_arg, (array_info *) no_arg,
                                         index_data,  index_dsp,
                                         index_data,  index_dsp);

} /* dalib_ind_shadow_define */

/**************************************************************************
*                                                                         *
*   void dalib_ind_shadow_get (schedule_ptr *s_id, char *array_data,      *
*                              array_info *s_dsp, *array_dsp)             *
*                                                                         *
*   - update shadow area of array corresponding the schedule              *
*   - shadow area will contain afterwards the non-local values            *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_ind_shadow_get) (s_id, array_data, s_dsp, array_dsp)

shadow_ptr *s_id;   /* internal identification of schedule */
array_info *array_dsp, *s_dsp;
char *array_data;

{ 
 
   FUNCTION(dalib_exec_shadow_get) (s_id, array_dsp);

} /* dalib_ind_shadow_get */

/**************************************************************************
*                                                                         *
*   void dalib_ind_shadow_put (schedule_ptr *s_id,                        *
*                              char *array_data, int *op,                 *
*                              array_info *s_dsp, *array_dsp, *op_dsp)    *
*                                                                         *
*   - write non-local computed values back to owning processors           *
*   - elements will be combined correspoinding reduction operator op      *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_ind_shadow_put) (s_id, array_data, op,
                                      s_dsp, array_dsp, op_dsp)

shadow_ptr *s_id;   /* internal identification of schedule */
array_info *array_dsp, *s_dsp, *op_dsp;
int *op;
char *array_data;

{ 
   FUNCTION(dalib_exec_shadow_put) (s_id, array_dsp, op);

} /* dalib_ind_shadow_put */

/**************************************************************************
*                                                                         *
*   void dalib_ind_shadow_set (schedule_ptr *s_id,                        *
*                              char *array_data, char *val_data,          *
*                              array_info *s_dsp, *array_dsp, *val_dsp)   *
*                                                                         *
*   - initializes shadow area with the value 'val'                        *
*   - needed for reductions on distributed arrays                         *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_ind_shadow_set) (s_id, array_data, val_data,
                                     s_dsp, array_dsp, val_dsp)

shadow_ptr *s_id;   /* internal identification of schedule */
array_info *array_dsp, *s_dsp, *val_dsp;
char *array_data, *val_data;

{
  FUNCTION(dalib_exec_shadow_set) (s_id, array_dsp, val_data);

} /* dalib_ind_shadow_set */

/**************************************************************************
*                                                                         *
*   void dalib_ind_shadow_set1 (schedule_ptr *s_id,                       *
*                               char *array_data, char *val_data,         *
*                               array_info *s_dsp, *array_dsp, *val_dsp)  *
*                                                                         *
*     - initialize shadow area for out-of-range values                    *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_ind_shadow_set1) (s_id, array_data, val_data,
                                      s_dsp, array_dsp, val_dsp)

schedule_ptr *s_id;
array_info *array_dsp, *s_dsp, *val_dsp;
char *array_data, *val_data;

{ unsigned char *data;
  int N;
  int size, serial;
  int ov_size;
  int *dummy;
  int i, j;

  data = (*array_dsp)->data;

  dalib_ind_array_info (*array_dsp, &size, &serial, &N);

  data += N * size * serial;   /* data points to shadow area */

  dalib_memcopy (data, val_data, size * serial);

} /* dalib_ind_shadow_set1 */

/**************************************************************************
*                                                                         *
*   void dalib_ind_shadow_free (schedule_ptr *s_id, char *index_data,     * 
*                               array_info *s_dsp, array_info *index_dsp) *
*                                                                         *
*    - resets changed index values of index_data                          *
*    - frees internal data structures used for the schedule               *
*                                                                         *
**************************************************************************/
 
void FUNCTION(dalib_ind_shadow_free) (s_id, index_data,
                                      s_dsp, index_dsp)

shadow_ptr *s_id;   /* internal identification of schedule */
char *index_data;
array_info *s_dsp, *index_dsp;
 
{ int db_id;
  inspector_data *S;
  int is_new, size, no;
  int *indexes;

  db_id = (*s_id)->db_pos;

  S = dalib_inspector_db_get (db_id);

  /* get correct data pointer, do not rely on index_data currently */

  dalib_secarray_get_data (*index_dsp, 1, &no, &size, &indexes, &is_new);

  if (is_new)

     { dalib_internal_error ("shadow index reset for noncontiguous indexes");
       dalib_stop ();
     }

  dalib_shadow_reset_indexes (indexes, S->index_changes);

  /* give all the needed memory free */

  dalib_free_shadow_schedule (*s_id);

  dalib_inspector_db_release (db_id);

} /* dalib_ind_shadow_free */

void FUNCTION(print_addr)(f)
int *f;
{ printf ("%d: address = %d\n", pcb.i, f);
}
