/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*                                                                         *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Aug 94                                                   *
*  Last Update : Feb 95                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : darray.c                                                 *
*                                                                         *
*  Note : this module is not used for UNILIB                              *
*                                                                         *
*  Function: routines for array descriptors of distributed arrays         *
*                                                                         *
*  Export :  FORTRAN Interface                                            *
*  ===========================                                            *
*                                                                         *
*                                                                         *
**************************************************************************/

#include <stdio.h>
#include "dalib.h"

#undef DEBUG
#define CHECK

/**************************************************************************
*                                                                         *
*  void dalib_array_query (array_info *array_id, int *dim,                *
*                          int *low, int *high, int *stride, int *size)   *
*                                                                         *
*     - asking for the local size of an array dimensions                  *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_array_query) (array_id, dim, low, high, stride, size) 

array_info *array_id;
int *dim, *low, *high, *stride, *size;

{ array_info descriptor;
  int array_dim;

  DimInfo *dims;

  descriptor = *array_id;
  array_dim = *dim - 1;

  dims    = descriptor->dimensions + array_dim;
  *low    = dims->local_size[0];
  *high   = dims->local_size[1];
  *stride = dims->local_size[2];

  /* local extent depends on packing, yes or no */

  *size   = dalib_dim_local_extent (dims);

#ifdef DEBUG
  printf ("%d: dalib_array_query (array_id = %d, dim = %d -> %d:%d:%d (%d)\n",
          pcb.i, *array_id, *dim, *low, *high, *stride, *size);
#endif
 
} /* dalib_array_query */

/*******************************************************************
*                                                                  *
*  Asking for the local part of a global section                   *
*    (but with the global addresses)                               *
*                                                                  *
*    global range :  global_lb : global_ub : global_stride         *
*    local  range :  local_lb  : local_ub  : local_stride          *
*                                                                  *
*******************************************************************/

void FUNCTION(dalib_array_lrange) (array_id, dim, 
                          global_lb, global_ub, global_stride,
                          local_lb, local_ub, local_stride)

array_info * array_id;
int * dim;
int * global_lb, * global_ub, * global_stride;
int * local_lb, * local_ub, * local_stride;

/* make the global range to a local range, dependent on the distribution */

{ int     *local_size;

  if (dalib_is_array_info (*array_id)) 

     { DimInfo *array_dim;

       array_dim = (*array_id)->dimensions;
       array_dim += (*dim - 1);
       local_size = array_dim->local_size;
     }

   else if (dalib_is_section_info (*array_id))

     { SecDimInfo *section_dim;

       section_dim = ((section_info) (*array_id))->dimensions;
       section_dim += (*dim - 1);
       local_size = section_dim->local_range;
     }

   else

     { dalib_internal_error ("dalib_array_lrange: not an array/section");
       dalib_stop ();
     }

#ifdef DEBUG
  printf ("%d: dalib_array_range (id = %d, dim = %d) [%d:%d:%d]*[%d:%d:%d]\n",
          pcb.i, *array_id, *dim, *global_lb, *global_ub, *global_stride,
          local_size[0], local_size[1], local_size[2]);
#endif
 
  dalib_r_r_intersect (*global_lb, *global_ub, *global_stride,
                       local_size[0], local_size[1], local_size[2],
                       local_lb, local_ub, local_stride
                      );

} /* dalib_array_lrange */

          /****************************************************
          *                                                   *
          *  Special Case : Stride is always 1                *
          *                                                   *
          ****************************************************/

void FUNCTION(dalib_array_lslice) (array_id, dim, 
                          global_lb, global_ub,
                          local_lb, local_ub)

array_info *array_id;
int * dim;
int * global_lb, * global_ub;
int * local_lb, * local_ub;

/* make the global range to a local range, dependent on the distribution */

{ DimInfo *array_dim;
  int     *local_size;

  int dummy_stride;

  array_dim = (*array_id)->dimensions + (*dim - 1);

  local_size = array_dim->local_size;

#ifdef DEBUG
  printf ("%d: dalib_array_slice (id = %d, dim = %d) [%d:%d]*[%d:%d:%d]\n",
          pcb.i, *array_id, *dim, *global_lb, *global_ub,
          local_size[0], local_size[1], local_size[2]);
#endif
 
  dalib_r_r_intersect (*global_lb, *global_ub, 1,
                       local_size[0], local_size[1], local_size[2],
                       local_lb, local_ub, &dummy_stride
                      );

} /* FUNCTION(dalib_array_lslice) */ 

/*******************************************************************
*                                                                  *
*    mrange : stride * I + base  -> [glb:gub:gstride]              *
*                                                                  *
*******************************************************************/

void FUNCTION(dalib_array_mrange) (array_id, dim, base, stride, 
                          global_lb, global_ub, global_stride,
                          local_lb, local_ub, local_stride)

array_info * array_id;
int * dim, * base, * stride;
int * global_lb, * global_ub, * global_stride;
int * local_lb, * local_ub, * local_stride;

/* make the global range to a local range, dependent on the distribution */

{ DimInfo *array_dim;
  int     *local_size;

  int  global_section [3];
  int  local_section  [3];

  int t_global_section [3];
  int t_local_section [3];

  global_section[0] = *global_lb;
  global_section[1] = *global_ub;
  global_section[2] = *global_stride;

  array_dim = (*array_id)->dimensions;
  array_dim += (*dim - 1);
  local_size = array_dim->local_size;

#ifdef DEBUG
  printf ("%d: dalib_array_mrange (id = %d, dim = %d, %d * I + %d)\n",
          pcb.i, *array_id, *dim, *stride, *base);
#endif
 
  /* map global section into the template */

  dalib_aligned_section (global_section, *base, *stride, t_global_section);
 
  /* compute the local part within the template */

  dalib_intersect_sections (t_global_section, local_size, t_local_section);

  /* map the local part back from template to array dimension */
 
  dalib_map1_section (t_global_section, global_section,
                      t_local_section, local_section);

  *local_lb     = local_section[0];
  *local_ub     = local_section[1];
  *local_stride = local_section[2];

#ifdef DEBUG
  printf ("%d: local part of array is then : %d:%d:%d\n",
           pcb.i, *local_lb, *local_ub, *local_stride);
#endif        

} /* dalib_array_mrange */

/**************************************************************************
*                                                                         *
*  INTERNAL DALIB Functions                                               *
*                                                                         *
*  dalib_array_info (array_id => template_id, top_id)                     *
*                                                                         *
*  dalib_mapping_dim_info (array_id, dim                                  *
*                           => topology_id, topology_dim,                 *
*                              base, stride, lb, ub, kind)                *
*                                                                         *
*   - dim of array_id is mapped to lb:ub                                  *
*   - distribution of lb:ub is done by kind (SERIAL/BLOCK/CYCLIC)         *
*   - index is mapped to stride*index + base                              *
*                                                                         *
**************************************************************************/
 
void dalib_array_info (array_id, template_id, top_id)
 
array_info array_id;
array_info *template_id;
int        *top_id;
 
{ if (dalib_is_section_info (array_id))
 
     { section_info section_id;

       section_id  = (section_info) array_id;

       dalib_array_info (section_id->array_id, template_id, top_id);
     }

   else if (array_id->DistributeInfo != NO_DISTRIBUTION)

     dalib_distribution_info (array_id, template_id, top_id);

   else if (array_id->AlignInfo != NO_ALIGNMENT)

     dalib_alignment_info (array_id, template_id, top_id);

   else 

     { *template_id = (array_info) 0;
       *top_id      = 0;
     }

} /* dalib_array_info */

/**************************************************************************
*                                                                         *
*  dalib_array_dim_mapping (array_info array_id, int dim,                 *
*                           int *base, *stride,                           *
*                           int *lb, int *ub,                             *
*                           int *topology,                                *
*                           DimMap *mapping)                              *
*                                                                         *
*  - get info how dimension dim of array array_id is mapped               *
*                                                                         *
**************************************************************************/
 
void dalib_array_dim_mapping (array_id, dim, base, stride,
                              lb, ub, topology, mapping)
 
/* Input Arguments : */

array_info array_id;
int dim;

/* Output Arguments : */

int     *base, *stride, *lb, *ub;
int     *topology;
DistDim *mapping;

{ if (dalib_is_section_info (array_id))
 
     { section_info section_id;
       int          section_dim;

       section_id  = (section_info) array_id;
       section_dim = dalib_section_array_dim (section_id, dim);

       dalib_array_dim_mapping (section_id->array_id, section_dim, 
                                base, stride, lb, ub,
                                topology, mapping);
     }

   else if (array_id->DistributeInfo != NO_DISTRIBUTION)
 
     dalib_distribution_dim_mapping (array_id, dim, 
                                base, stride, lb, ub,
                                topology, mapping);
 
   else if (array_id->AlignInfo != NO_ALIGNMENT)
 
     dalib_alignment_dim_mapping (array_id, dim, 
                                  base, stride, lb, ub,
                                  topology, mapping);
 
   else 
 
     { *base   = 0;
       *stride = 1;
       *lb     = array_id->dimensions[dim-1].global_size[0];
       *ub     = array_id->dimensions[dim-1].global_size[1];

       *topology = 0;  /* 0 stands for replicated array */
       *mapping  = NO_DIST_DIM;
     }
 
} /* dalib_array_dim_mapping */

/**************************************************************************
*                                                                         *
*  dalib_array_top_mapping (array_info array_id, int *topology,           *
*                           int *index_dim,                               *
*                           int *base, *stride,                           *
*                           int *lb, int *ub,                             *
*                           DimMap *mapping)                              *
*                                                                         *
*  - get info how dimension dim of array array_id is mapped               *
*  - entries in arrays are sorted by topology dimensions                  *
*                                                                         *
**************************************************************************/

void dalib_array_top_mapping (array_id, topology, index_dim,
                              base, stride, lb, ub, mapping)

/* Input Arguments : */

array_info array_id;

/* Output Arguments : */

int     *topology;
int     index_dim[];
int     base[], stride[], lb[], ub[];
DistDim mapping[];

{  if (!dalib_is_array_info (array_id)) 

      { dalib_internal_error ("not array info in array_top_mapping");
        dalib_stop ();
      }

   if (array_id->DistributeInfo != NO_DISTRIBUTION)

     dalib_distribution_top_mapping (array_id, topology, index_dim,
                                     base, stride, lb, ub, mapping);

   else if (array_id->AlignInfo != NO_ALIGNMENT)

     dalib_alignment_top_mapping (array_id, topology, index_dim,
                                  base, stride, lb, ub, mapping);

   else 

     *topology = 0;  /* rank 0, all arrays will have no entry */

} /* dalib_array_top_mapping */

/**************************************************************************
*                                                                         *
*   dalib_array_map_query (array_id, array_dim => top_id, top_dim)        *
*                                                                         *
*   - gets the topology and dimension to which the                        *
*     array dimension is mapped to                                        *
*                                                                         *
**************************************************************************/

void dalib_array_map_query (array_id, array_dim, top_id, top_dim)

array_info array_id;
int array_dim;
int *top_id, *top_dim;

{ int lb, ub, base, stride, kind;  /* dummy arguments */
  DistDim mapping;

#ifdef DEBUG
  printf ("%d: dalib_array_map_query (array_id = %d, dim = %d)\n",
           pcb.i, array_id, array_dim);
#endif

  dalib_array_dim_mapping (array_id, array_dim,
                           &base, &stride, &lb, &ub,
                           top_id, &mapping);
 
  dalib_dim_mapping_info (mapping, &kind, top_dim);

} /* dalib_array_map_query */

/**************************************************************************
*                                                                         *
*   dalib_array_topmap (array_id, dim, val -> topid, topdim, toppos)      *
*                                                                         *
*   - maps array index into the dimension of topology                     *
*   - also needed for broadcast.m4                                        *
*                                                                         *
**************************************************************************/

void dalib_array_topmap (array_id, dim, ind_id, top_id, top_dim, top_pos)

array_info array_id;
int dim, ind_id;
int *top_id, *top_dim, *top_pos;

{ int     lb, ub, base, stride;
  DistDim mapping;

  int     new_index;

#ifdef DEBUG
  printf ("%d: dalib_array_topmap (array_id = %d, dim = %d, index = %d)\n",
           pcb.i, array_id, dim, ind_id);
#endif

  dalib_array_dim_mapping (array_id, dim, 
                           &base, &stride, &lb, &ub, 
                           top_id, &mapping);

  /* apply alignment to the index value to get correct owner */

  new_index = base + stride * ind_id;

  dalib_distribution_owner (ind_id, lb, ub, *top_id, mapping, 
                            top_dim, top_pos);

#ifdef DEBUG
  printf ("%d: dalib_array_topmap with top = %d, dim = %d, pos = %d)\n",
           pcb.i, *top_id, *top_dim, *top_pos);
#endif

} /* dalib_array_topmap */

/**************************************************************************
*                                                                         *
*   dalib_array_top_query (array_id, top_dim -> kind, index_dim, top_pos) *
*                                                                         *
*    - asks how array is mapped in its topology at top_dim                *
*                                                                         *
*    kind == kSOURCE_DIM     : index_dim (1..N) is array_dim mapped       *
*    kind == kREPLICATED_DIM : top_dim is a replicated dimension          *
*    kind == kEMBEDDED_DIM   : top_pos is owner for the embedded dim      *
*                                                                         *
**************************************************************************/

void dalib_array_top_query (array_id, top_dim, kind, index_dim, top_pos)

array_info array_id;
int        top_dim;

int *kind;
int *index_dim;
int *top_pos;

{ if (array_id->DistributeInfo != NO_DISTRIBUTION)
 
     dalib_distribution_top_query (array_id, top_dim, 
                                   kind, index_dim, top_pos);
 
   else if (array_id->AlignInfo != NO_ALIGNMENT)
 
     dalib_alignment_top_query (array_id, top_dim, kind, index_dim, top_pos);
 
 
   else  /* replicated array */
 
     { *index_dim  = 0;
       *kind       = kREPLICATED_DIM;
       *top_pos    = 0;                /* dummy */
     }

} /* dalib_array_top_query */

/**************************************************************************
*                                                                         *
*  dalib_is_replicated (array_info/section_inf)                           *
*                                                                         *
*  - returns true if array/section has more than one incarnation          *
*                                                                         *
**************************************************************************/

int dalib_is_replicated (section_id)

section_info section_id;

{ array_info array_id;

  if (dalib_is_array_info (section_id))
 
    array_id = (array_info) section_id;
 
   else if (dalib_is_section_info (section_id))
 
    array_id = section_id->array_id;

   else

    { dalib_internal_error ("dalib_is_replicated, not a section/array");
      dalib_stop ();
    }
     
  if (array_id->DistributeInfo != NO_DISTRIBUTION)

    return (0);  /* distributed array has never more than one incarnation */
 
   else if (array_id->AlignInfo != NO_ALIGNMENT)
 
    return (dalib_alignment_is_replicated (array_id));
 
   else  /* replicated array */

    return (pcb.p > 1);   /* replicated for more than one processor */

} /* dalib_is_replicated */

/**************************************************************************
*                                                                         *
*                                                                         *
*                                                                         *
**************************************************************************/

static array_info dalib_get_array_id (section_id)

section_info section_id;

{ if (dalib_is_array_info (section_id))

     return (array_info) section_id;

  if (dalib_is_section_info (section_id))

     return section_id->array_id;

  dalib_internal_error ("dalib_get_array_id: not a section/array");
  dalib_stop ();

  return (array_info) NULL;

} /* dalib_get_array_id */

/**************************************************************************
*                                                                         *
*  dalib_replicate (array_info/section_inf)                               *
*                                                                         *
*    - replicates an array/section so that all incars are consistent      *
*                                                                         *
**************************************************************************/

void dalib_replicate (section_id)

section_info section_id;

{ int topid;

  int rep_dims[MAX_RANK];
  int i, number;

  array_info array_id;

#ifdef DEBUG
      printf ("dalib_replicate\n");
#endif 

  array_id = dalib_get_array_id (section_id);

  if (array_id->DistributeInfo != NO_DISTRIBUTION)

    return;  /* distributed array has never more than one incarnation */
 
   else if (array_id->AlignInfo != NO_ALIGNMENT)
 
    { /* find all replicated dimensions */


      dalib_alignment_rep_dims (array_id, &topid, &number, rep_dims);

#ifdef DEBUG
      printf ("dalib_replicate for %d replicated dims by alignment\n", number);
#endif 

      for (i=0; i<number; i++)
         dalib_secarray_bc (topid, rep_dims[i], section_id);
    }
 
   else  /* replicated array */

    { topid = 0;         /* for all nodes inclusive the host */

      dalib_secarray_bc (topid, 1, section_id);

    }

} /* dalib_replicate */

/**************************************************************************
*                                                                         *
*  dalib_rep_reduction (array_info/section_inf)                           *
*                                                                         *
*    - reduction along all replicated dimensions of a section             *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_rep_reduction) (section_id, op)

section_info *section_id;
int *op;

{ int topid;

  array_info array_id;

  int rep_dims[MAX_RANK];
  int i, number;

  dd_type section_ddt;
  int bytes, section_size;
  int is_contiguous;

  unsigned char *reduction_data;

#ifdef DEBUG
  printf ("dalib_rep_reduction, op = %d\n", *op);
#endif

  array_id = dalib_get_array_id (*section_id);

  /* distributed array has never more than one incarnation */

  if (array_id->DistributeInfo != NO_DISTRIBUTION) return;

  dalib_make_secarray_ddt1 (&section_ddt, *section_id, 
                            &section_size, &bytes);

  dalib_ddt_is_contiguous (section_ddt, &is_contiguous, &reduction_data);

  /* not contiguous data will be packed before reduction */

  if (!is_contiguous)

     { reduction_data = (unsigned char *)
                      dalib_malloc (section_size * bytes, 
                                   "rep_reduction");

       dalib_ddt_pack (reduction_data, section_ddt);

#ifdef DEBUG
       printf ("%d: packed reduction data, size = %d\n",
               pcb.i, section_size);
#endif

     }

  /* now reduce data along all replicated dimensions */

  if (array_id->AlignInfo != NO_ALIGNMENT)

    { /* find all replicated dimensions */

      dalib_alignment_rep_dims (array_id, &topid, &number, rep_dims);

      for (i=0; i<number; i++)

        { 
#ifdef DEBUG
          printf ("%d: top_reduction (id=%d,dim=%d), size = %d, op = %d\n",
                   pcb.i, topid, rep_dims[i], section_size, *op);
#endif

          dalib_top_nreduction (topid, rep_dims[i], reduction_data,
                                *op, section_size);
        }
    }
 
   else  /* replicated array */

    { topid = 0;         /* for all nodes inclusive the host */

      dalib_top_nreduction (topid, 1, reduction_data, *op, section_size);
    }

   /* unpack data if it was not contiguous */

   if (!is_contiguous)

    { dalib_ddt_unpack (section_ddt, reduction_data, 0);
      free (reduction_data);
    }

} /* dalib_rep_reduction */

/**************************************************************************
*                                                                         *
*  dalib_find_owner (array_id, nr, dim1, index1, ...., dim3, index3)      *
*                                                                         *
*  - fix the owner that owns certain elements of the array array_id       *
*  - nr specifies how many arguments have to be used                      *
*                                                                         *
*  - dim1, ..., dim3 specify distributed dimensions of the array          *
*  - index1, .., index3 are corresponding index values                    *
*                                                                         *
*  Result: owner fixes one processor that owns the searched element       *
*                                                                         *
**************************************************************************/

static int owner;    /* global variable, also needed for node_get */

void FUNCTION(dalib_find_owner)

    (array_id, nr, dim1, index1, dim2, index2, dim3, index3)

array_info *array_id;
int *nr;
int *dim1, *dim2, *dim3;
int *index1, *index2, *index3;

{ int top_id;
  int top_rank;
  int index_dim [MAX_RANK];
  int index_val [MAX_RANK];
  int top_pos [MAX_RANK];
  int top_dim;

  int NP, NId;
  int lb, ub;        /* extension for the index dimension */
  int base, stride;  /* here only dummies                    */

  array_info template_id;
  int i, pos, kind;
  DistDim mapping;

  /* set the arrays index_dim and index_val with the actual arguments */

#ifdef DEBUG
  printf ("call of dalib_find_owner, nr = %d\n", *nr);
#endif

  switch (*nr) {
  case 3 : index_dim[2] = *dim3; index_val[2] = *index3;
  case 2 : index_dim[1] = *dim2; index_val[1] = *index2;
  case 1 : index_dim[0] = *dim1; index_val[0] = *index1;
  case 0 : break;
  default : dalib_internal_error ("illegal call of find_owner");
            dalib_stop ();
  } /* switch */

  dalib_array_info (*array_id, &template_id, &top_id);
 
  top_rank = dalib_top_rank (top_id);

#ifdef DEBUG
  printf ("%d: dalib_find_owner for topology %d (rank = %d)\n", 
          pcb.i, top_id, top_rank);
#endif 

  for (i=0; i<top_rank; i++)
      top_pos[i] = 1;          /* default is first processor */

  /* now find the better positions */

  for (i=0; i<*nr; i++)

    { /* find the correct topology position */

      dalib_array_dim_mapping (*array_id, index_dim[i],
                               &base, &stride, &lb, &ub,
                               &top_id, &mapping);
 
      dalib_distribution_owner (index_val[i], lb, ub, top_id, mapping, 
                                &top_dim, &pos);

#ifdef DEBUG
      printf ("index dim %d is at topolyg (dim=%d,id=%d)\n",
               index_dim[i], top_dim, top_id);
#endif 

#ifdef DEBUG
   printf ("%d: index_dim = %d, top_dim = %d (%d), val %d of %d:%d, %d owns\n",
            i, index_dim[i], top_dim, NP, index_val[i], lb, ub, pos);
#endif


       if (top_dim > 0) top_pos [top_dim-1] = pos;

   }

  owner = dalib_top_abspos (top_id, top_pos);
 
#ifdef DEBUG
   printf ("%d: find_owner results in owner node : %d\n", pcb.i, owner);
#endif

} /* dalib_find_owner */ 

/**************************************************************************
*                                                                         *
*  dalib_node_get (rep_data, size)                                        *
*                                                                         *
*  - rep_data (with size contiguous bytes) is broadcast by owner          *
*  - owner must be fixed by dalib_find_owner before                       *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_node_get) (rep_data, size) 
int *size;
unsigned char *rep_data;

{ 
#ifdef DEBUG
  printf ("%d: call of node_get, size = %d\n", pcb.i, *size);
#endif 

  dalib_context_broadcast (rep_data, *size, owner);

#ifdef DEBUG
  printf ("%d: node_get done, size = %d\n", pcb.i, *size);
#endif 

} /* dalib_node_get */

/*******************************************************************
*                                                                  *
*  int dalib_is_aligned (section1, section2)                       *
*                                                                  *
*   - returns true if section1 and section2 are aligned            *
*                                                                  *
*******************************************************************/

    /*****************************************************
    *                                                    *
    *    index_rank_pos (section, dim)                   *
    *                                                    *
    *****************************************************/

static int dalib_index_rank_pos (section, dim)
section_info section;
int dim;

{ int i, rank, pos;
  int act_rank, pos_rank;

  act_rank = 0;
  pos_rank = 0;

  rank = section->array_id->rank;

  for (i=0; i<rank; i++)
    { if (section->dimensions[i].is_range)
         { act_rank ++;
           if (dim-1 == i) pos_rank = act_rank;
         }
    }

  return (pos_rank);

} /* dalib_index_rank_pos */

    /*****************************************************
    *                                                    *
    *   dalib_are_aligned_dims (section1, dim1,          *
    *                           section2, dim2)          *
    *                                                    *
    *****************************************************/

static int dalib_are_aligned_dims (section1, dim1, section2, dim2)

section_info section1, section2;
int          dim1,     dim2;

{ array_info array1, array2;
  SecDimInfo *secdim1, *secdim2;
 
  int lb1, ub1, st1, kind1, base1, stride1, top_dim1, top_id1;
  int lb2, ub2, st2, kind2, base2, stride2, top_dim2, top_id2;

  int pos1, pos2;

  DistDim mapping1, mapping2;

  int offset;

#ifdef DEBUG
  printf ("%d: check whether dim %d of sec %d and %d of sec %d are aligned\n",
           pcb.i, dim1, section1, dim2, section2);
#endif 

  array1 = section1->array_id;
  array2 = section2->array_id;
 
  secdim1 = section1->dimensions + dim1 - 1;
  secdim2 = section2->dimensions + dim2 - 1;

  /* ATTENTION : make sure that dim1 and dim2 are the same rank position */

  pos1 = dalib_index_rank_pos (section1, dim1);
  pos2 = dalib_index_rank_pos (section2, dim2);

  if (pos1 != pos2) return (0);

  dalib_array_dim_mapping (array1, dim1, 
                           &base1, &stride1, &lb1, &ub1,
                           &top_id1, &mapping1);

  dalib_array_dim_mapping (array2, dim2, 
                           &base2, &stride2, &lb2, &ub2,
                           &top_id2, &mapping2);

  if (top_id1 != top_id2) return (0);

  if (!dalib_same_dim_mapping (top_id1, mapping1, mapping2)) return (0);

  if ((ub2 - lb2) != (ub1 - lb1)) return (0);

  offset = lb2 - lb1;

  lb1 = secdim1->global_range[0] * stride1 + base1;
  ub1 = secdim1->global_range[1] * stride1 + base1;
  st1 = secdim1->global_range[2];

  lb2 = secdim2->global_range[0] * stride2 + base2;
  ub2 = secdim2->global_range[1] * stride2 + base2;
  st2 = secdim2->global_range[2];

  if (st1 != st2) return (0);

  if (lb1 + offset != lb2) return (0);
  if (ub1 + offset != ub2) return (0);

#ifdef DEBUG
  printf ("%d: the two dimensions are  are aligned\n", pcb.i);
#endif 

  return (1);

}  /* dalib_are_aligned_dims */

    /*****************************************************
    *                                                    *
    *    dalib_is_aligned (section1, section2)           *
    *                                                    *
    *****************************************************/

int dalib_is_aligned (section1, section2)

section_info section1, section2;

{ array_info array1, array2;
  array_info template_id;
  int top_id1;
  int top_id2;

  int dim, rank;
  int kind1, index_dim1, top_pos1;
  int kind2, index_dim2, top_pos2;
 
  array1 = section1->array_id;
  array2 = section2->array_id;

  dalib_array_info (array1, &template_id, &top_id1);
  dalib_array_info (array2, &template_id, &top_id2);

#ifdef DEBUG
  printf ("%d: are section %d (top=%d) and section %d (top=%d) aligned\n",
          pcb.i, section1, top_id1, section2, top_id2);
#endif

  if (top_id1 != top_id2) return (0);

  if ((top_id1 == 0) || (top_id1 == -1)) return (1);

  /* both arrays are distributed on the same topology */

  rank = dalib_top_rank (top_id1);

  for (dim = 1; dim <= rank; dim++)

    { /* find index_dim1, index_dim2 */

      dalib_array_top_query (array1, dim, &kind1, &index_dim1, &top_pos1);
      dalib_array_top_query (array2, dim, &kind2, &index_dim2, &top_pos2);

#ifdef DEBUG
      printf ("%d: top = %d(%d/%d), k1 = %d, dim1 = %d, k2 = %d, dim2 = %d\n",
              pcb.i, top_id1, dim, rank, kind1, index_dim1, kind2, index_dim2);
#endif

      /* make sure that not embedded dimensions */

      if (kind1 == kEMBEDDED_DIM)   return (0);
      if (kind2 == kEMBEDDED_DIM)   return (0);
      if (kind1 == kREPLICATED_DIM) return (0);
      if (kind2 == kREPLICATED_DIM) return (0);

      if (!dalib_are_aligned_dims (section1, index_dim1, 
                                   section2, index_dim2))
         return (0);
       
    }

  return (1);   /* so it must be really aligned */

} /* dalib_is_aligned */

/**************************************************************************
*                                                                         *
*  PREDICATE dalib_array_map_underspecified (array_id)                    *
*                                                                         *
*   - returns true if mapping of array is underspecified                  *
*   - cannot compute local sizes for underspecified mappings              *
*                                                                         *
**************************************************************************/

int dalib_array_map_underspecified (array_id)

array_info array_id;

{ if (array_id->DistributeInfo != NO_DISTRIBUTION)

    return (dalib_underspecified_distribution (array_id));

  if (array_id->AlignInfo != NO_ALIGNMENT)

    return (dalib_underspecified_alignment (array_id));

  return (0);   /* mapping of serial arrays is never unspecified */

} /* dalib_array_map_underspecified */

/**************************************************************************
*                                                                         *
*  int dalib_is_dim_specialization (int special_top_id, general_top_id,   *
*                                                                         *
*  - returns (1) if special_* is a specialization of general_*            *
*                                                                         *
**************************************************************************/

int dalib_is_dim_specialization (special_top_id, special_dim,
                                 special_mapping,
                                 general_top_id, general_dim,
                                 general_mapping)

int special_top_id, general_top_id;
int special_dim, general_dim;

DistDim special_mapping, general_mapping;

{ 

#ifdef DEBUG
  printf ("is_dim_specialization (?)\n");
#endif

  if (general_top_id != kANY_TOPOLOGY)

    { if (general_top_id != special_top_id) return (0);
      if (special_dim != general_dim) return (0);
    }

  if (dalib_is_dist_dim_specialization (special_mapping, general_mapping))

     return (1);

  return (0);

} /* dalib_is_dim_specialization */

/**************************************************************************
*                                                                         *
*  int dalib_is_map_specialization (special_dsp, general_dsp)             *
*                                                                         *
*  - routine is only called if general_dsp has underspecified mappings    *
*  - true if special_dsp (full mapping) is specialization of general      *
*                                                                         *
**************************************************************************/

int dalib_is_map_specialization (special_dsp, general_dsp)

array_info special_dsp, general_dsp;

{ int i, rank;

  int okay;

  int special_top_id, special_dim;
  int special_base, special_stride;
  int special_lb, special_ub;

  int general_top_id, general_dim;
  int general_base, general_stride;
  int general_lb, general_ub;

  DistDim special_mapping, general_mapping;

  int is_specialization;

  /* sections cannot be specialized at all */

  if (dalib_is_section_info (special_dsp)) return (0);

  okay = 1;    /* initialization: assume a specialization */

  rank = special_dsp->rank;

  for (i=0; i<rank; i++)

    { dalib_array_dim_mapping (special_dsp, i+1,
                               &special_base, &special_stride,
                               &special_lb, &special_ub, 
                               &special_top_id, &special_mapping);

      dalib_array_dim_mapping (general_dsp, i+1,
                               &general_base, &general_stride,
                               &general_lb, &general_ub, 
                               &general_top_id, &general_mapping);

      is_specialization = 

           dalib_is_dim_specialization (special_top_id, special_dim,
                                        special_mapping,
                                        general_top_id, general_dim,
                                        general_mapping);

#ifdef DEBUG
     printf ("%d: compare dim %d (rank = %d) for specialization, is = %d\n",
             pcb.i, i+1, rank, is_specialization);
#endif

      if (!is_specialization) return (0);

    }

   return (1);

} /* dalib_is_map_specialization */

/**************************************************************************
*                                                                         *
*  void dalib_array_make_full_mapping (array_info array_id)               *
*                                                                         *
**************************************************************************/

void dalib_array_make_full_mapping (array_id)

array_info array_id;

{ if (array_id->DistributeInfo != NO_DISTRIBUTION)

       dalib_make_full_distribution (array_id);

    else if (array_id->AlignInfo != NO_ALIGNMENT)

     { dalib_internal_error ("dummy aligned, cannot make full mapping");
       dalib_stop ();
     }

    else

   { dalib_internal_error ("dummy not mapped at all, cannot make full mapping");
     dalib_stop ();
   }

} /* dalib_array_make_full_mapping */

/**************************************************************************
*                                                                         *
*  void dalib_array_inherit_mapping (general_id, special_id)              *
*                                                                         *
*   - descriptor for general mapping inherits the specialized mapping     *
*                                                                         *
**************************************************************************/

void dalib_array_inherit_mapping (general_id, special_id)

array_info special_id, general_id;

{ /* we know that dalib_array_map_underspecified (general_id) is true */
  
  Alignment dalib_alignment_copy ();

  if (general_id->DistributeInfo == NO_DISTRIBUTION)

     { dalib_internal_error ("dummy is not underspecified distribution");
       dalib_stop ();
     }

  /* dummy inherits its distribution from the actuals distribution */

  if (special_id->DistributeInfo != NO_DISTRIBUTION)

       dalib_inherit_distribution (general_id, special_id);

    else if (special_id->AlignInfo != NO_ALIGNMENT)

       { int offsets[MAX_DIMENSIONS];
         int i, rank;

         rank = special_id->rank;

         /* !HFP$ ALIGN G (I) WITH S (I - lbound(G) + lbound(S)) */

         for (i=0; i<rank; i++)
             offsets[i] = special_id->dimensions[i].global_size[0] -
                          general_id->dimensions[i].global_size[0];

         general_id->AlignInfo = 
           dalib_alignment_copy (special_id->AlignInfo, rank, offsets);

         dalib_dist_free (general_id->DistributeInfo, rank);
         general_id->DistributeInfo = NO_DISTRIBUTION;
       }

    else

       { /* special is replicated array, delete distribution */

         dalib_dist_free (general_id->DistributeInfo, general_id->rank);
         general_id->DistributeInfo = NO_DISTRIBUTION;
       }

} /* dalib_array_inherit_mapping */

/**************************************************************************
*                                                                         *
*  dalib_get_mapping_data (array_id, int *is_full, int *topology,         *
*                          int type[], info[], int map[])                 *
*                                                                         *
*  INPUT :  array_id is an array                                          *
*                                                                         *
*  OUTPUT :                                                               *
*                                                                         *
*     is_full    : TRUE if array is full aligned to a topology            *
*     topology   : identifier for topology to which array_id is aligned   *
*                                                                         *
*     type[i]    : kBLOCK_DIM, ...  type of how dim i is distributed      *
*     info[i]    : additional information of distribution                 *
*     map[i]     :                                                        *
*                                                                         *
**************************************************************************/

void dalib_get_mapping_data (array_id, is_full, topology,
                             type, info, map)

array_info array_id;
int type[], info[], map[];
int *is_full, *topology;

{ 
  if (!dalib_is_array_info (array_id))

     { dalib_internal_error ("dalib_get_mapping_data, not object");
       dalib_stop ();
     }

  if (array_id->DistributeInfo != NO_DISTRIBUTION)

     { dalib_distribution_data (array_id, topology, type, info, map);

       *is_full = 1;

     }

   else if (array_id->AlignInfo != NO_ALIGNMENT)

     { dalib_align_distribution_data (array_id, is_full, topology,
                                      type, info, map);
     }

   else

     { int i, rank;

       rank = array_id->rank;

       for (i=0; i<rank; i++)

         { type[i] = kSERIAL_DIM;
           info[i] = 0;
           map[i]  = 0; 
         }

       *topology = 0;   
       *is_full  = 1;
     }

} /* dalib_get_mapping_data */

/**************************************************************************
*                                                                         *
*  void dalib_push_home_context (array_info *home)                        *
*  void dalib_pop_home_context ()                                         *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_push_home_context) (home, n, dim1, dim2, dim3, 
                                                 dim4, dim5, dim6, dim7)

array_info *home;
int        *n;

int *dim1, *dim2, *dim3, *dim4, *dim5, *dim6, *dim7;

{ int index_rank;
  int index_dims [MAX_DIMENSIONS];
  int top_rank;
  int top_dims   [MAX_DIMENSIONS];

  index_rank = *n;

  switch (index_rank) {
     case 7 : index_dims[6] = *dim7;
     case 6 : index_dims[5] = *dim6;
     case 5 : index_dims[4] = *dim5;
     case 4 : index_dims[3] = *dim4;
     case 3 : index_dims[2] = *dim3;
     case 2 : index_dims[1] = *dim2;
     case 1 : index_dims[0] = *dim1;
  } /* switch */

  if (dalib_is_array_info (*home))

     { array_info template;
       int        topology;
       int        i;

       dalib_array_info (*home, &template, &topology);

       top_rank = 0;

       for (i=0; i<index_rank; i++)

          { int lb, ub;
            int base, stride;
            int top_id, top_dim;
            int kind;

            DistDim mapping;

            dalib_array_dim_mapping (*home, index_dims[i], &base, &stride,
                                     &lb, &ub, &top_id, &mapping);

            dalib_dim_mapping_info (mapping, &kind, &top_dim);

#ifdef DEBUG
            printf ("%d: fix dim %d of %d: index dim = %d, top dim = %d\n",
                     pcb.i, i+1, index_rank, index_dims[i], top_dim);
#endif

            if (top_dim != 0)

               top_dims [top_rank++] = top_dim;
          }

       if (top_rank > 0)
          topology = dalib_top_restrict (topology, top_rank, top_dims);

       dalib_push_top_context (topology);

     }

   else

     { dalib_internal_error ("dalib_push_home_context: not an array");
       dalib_stop ();
     }

} /* dalib_push_home_context */

void FUNCTION(dalib_pop_home_context) ()

{ 

  dalib_pop_context ();

} /* dalib_pop_home_context */
