/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Nov 93                                                   *
*  Last Update : Jan 98                                                   *
*                                                                         *
*  This Module is part of the DALIB / UNILIB                              *
*                                                                         *
*  Module      : dalib.m4                                                 *
*                                                                         *
*  Function    : general routines for the runtime system                  *
*                                                                         *
*                - error handling                                         *
*                - allocating/deallocating memory                         *
*                - global variables                                       *
*                                                                         *
*  Changes :                                                              *
*                                                                         *
*    05/1996 : routine stack introduced to have all names                 *
*    12/1997 : no_ptr_argument, no_string_argument                        *
*    01/1998 : time/comm statistics interface changed                     *
*                                                                         *
**************************************************************************/

#include <stdio.h>
#include <stdlib.h>

#include "dalib.h"

#undef DEBUG

/*******************************************************************
*                                                                  *
*  Global variables                                                *
*  ----------------                                                *
*                                                                  *
*   no_ptr_argument : special pointer for not existing argument    *
*   no_str_argument : special pointer for not existing string arg  *
*                                                                  *
*******************************************************************/

process_control_block pcb;

static char SccsID[] = "@(#)ADAPTOR (6.0), Jun 1998, Copyright T.B. GMD";

static char *no_str_argument = (char *) 0;
static int  *no_ptr_argument = (int *) 0;

/*******************************************************************
*                                                                  *
*  Timing Statistics Information                                   *
*                                                                  *
*******************************************************************/

int time_statistics_collect = 0; /* set to 1 for collection of info */

/*******************************************************************
*                                                                  *
*  Communication Statistics Information                            *
*                                                                  *
*******************************************************************/

int comm_statistics_collect = 0; /* set to 1 for collection of info */

/*******************************************************************
*                                                                  *
*  variables for the routine stack and current subroutine name     *
*                                                                  *
*******************************************************************/

#define STACK_INCREMENT  1024

static char *routine_stack     = (char *) 0;
static int  routine_stack_size = 0;
static int  routine_stack_top  = 0;

/*  stack :  SUB1\0SUB2\0SUB3\0.... */
   
static char current_subroutine_name [256] = "";

void *malloc ();

static int dalib_allocated_bytes = 0;

/*******************************************************************
*                                                                  *
*   dalib_print_routine_stack ()                                   *
*                                                                  *
*******************************************************************/

static void dalib_print_routine_stack ()

{ char *name;    /* names on the stack */
  int  pos;      /* position on stack  */
  int  no;       /* counts routines    */

  no  = 0;
  pos = 0;
  name = routine_stack;
  
  while (pos < routine_stack_top)

   { fprintf (stderr, "%3d: %s\n", no, name);

     /* find the next delimiter */

     while (*name != '\0') 

        { pos++; name++;
        }

     pos++; name++; /* to the next character */

     no++; 

   } /* traversing the stack */

} /* dalib_print_routine_stack */

/*******************************************************************
*                                                                  *
*   Error Handling                                                 *
*   --------------                                                 *
*                                                                  *
*   void dalib_attention (char message[])                          *
*                                                                  *
*   void dalib_internal_error (char message[])                     *
*                                                                  *
*   void dalib_stop ()                                             *
*                                                                  *
*******************************************************************/

void dalib_attention (message)
char message[];

{  printf ("%d: ATTENTION in %s : %s\n",
            pcb.i, current_subroutine_name, message);

} /* dalib_attention */

void dalib_internal_error (message)
char message[];
{
   fprintf (stderr, "\n");
   fprintf (stderr, "SERIOUS ERROR in DALIB (Distributed Array LIBrary)\n");
   fprintf (stderr, "==================================================\n");
   fprintf (stderr, "\n");
   fprintf (stderr, "UNIT  : %s\n", current_subroutine_name);
   fprintf (stderr, "ERROR : %s\n", message);
   fprintf (stderr, "\n");
}

void dalib_stop ()

{  int a;

   a = 1;

   fprintf (stderr, "stop execution\n");
   dalib_print_routine_stack ();

   a = a / (a * a - 1);
   
   exit(-1);
}

void dalib_exit (returnCode)

int returnCode;

{ exit(returnCode);

} /* dalib_exit */

        /******************************************************
        *                                                     *
        *  MALLOC routines                                    *
        *                                                     *
        ******************************************************/

int *dalib_int_malloc (size, module)
int size;
char module[];

{ int *ptr;

  ptr = (int *) malloc (sizeof (int) * size);

#ifdef DEBUG
  printf ("%d: dalib_int_malloc of %d bytes for %s, ptr = %d\n", 
           pcb.i, sizeof (int) * size, module, ptr);
#endif

  if ((ptr == NULL) && (size != 0))

   { dalib_internal_error ("dalib_int_malloc");

     fprintf (stderr,
        "%d: dalib_int_malloc for %s : out of memory (needs %d integers)\n",
         pcb.i, module, size);

     dalib_stop();
   }

  dalib_allocated_bytes += size * sizeof(int);
  return (ptr);

} /* dalib_int_malloc */

char *dalib_malloc (size, module)
int size;
char module[];

{ char *ptr;

  ptr = (char *) malloc (size);

#ifdef DEBUG
  printf ("%d: dalib_malloc of %d bytes for %s, ptr = %d\n", 
           pcb.i, size, module, ptr);
#endif

  if ((ptr == NULL) && (size != 0))

   { dalib_internal_error ("dalib_malloc");
 
     fprintf (stderr,
        "%d: dalib_malloc for %s : out of memory (needs %d bytes)\n",
         pcb.i, module, size);

     dalib_stop ();

   }

  dalib_allocated_bytes += size;
  return (ptr);

} /* dalib_malloc */

void dalib_free (ptr, size)
char *ptr;
int size;

{  dalib_allocated_bytes -= size;

   if (size < 0) dalib_stop ();

   free (ptr);

#ifdef DEBUG
   printf ("%d: free of %d bytes, ptr = %d\n", pcb.i, size, ptr);
#endif

} /* dalib_free */

void dalib_int_free (ptr, size)
char *ptr;
int size;

{ dalib_free (ptr, size * sizeof (int));
} /* dalib_int_free */


void dalib_alloc_statistic ()

{ if ((dalib_allocated_bytes != 0) && (pcb.redist_flag))

     printf ("%d: %d bytes have not been deallocated in DALIB\n", 
              pcb.i, dalib_allocated_bytes);

} /* dalib_alloc_statistic */

/*******************************************************************
*                                                                  *
*    Dynamic Arrays in FORTRAN 77 context                          *
*                                                                  *
*    dalib_allocate (static_ptr, offset, N, size)                  *
*                                                                  *
*    - creates dynamic memory for N * size bytes                   *
*    - static_ptr + offset * size points to new data               *
*                                                                  *
*******************************************************************/

void dalib_allocate (static_ptr, offset, N, size)
char *static_ptr;
long *offset;
int  N, size;

{ char *dynamic_ptr;
  int elems;
  long diff;

  elems = N * size;

  /* printf ("allocate for %d bytes \n", elems); */

  dynamic_ptr = (char *) dalib_malloc (elems, "dalib_allocate");

  /* build the difference between static address and dynamic address */

  diff = (long)(dynamic_ptr) - (long)(static_ptr);

  if ((diff % size) == 0)

     { /* is ok */

       *offset = diff / size;
    
       /* printf ("dalib_allocate, static = %d, dynamic = %d, offset = %d\n",
                (int) static_ptr, (int) dynamic_ptr, *offset); */
     }

   else
 
     { /* seems to be more serious */

       dalib_internal_error ("dalib_allocate did not correct align");
       dalib_stop ();

     }

} /* dalib_allocate */

/*******************************************************************
*                                                                  *
*   Determine the number of elements in a section                  *
*                                                                  *
*******************************************************************/

int dalib_range_size (lb, ub, str)
int lb, ub, str;

{ int n;

  if (str == 1)
    { if (lb <= ub)
         n = ub - lb + 1;
       else
         n = 0;
    }
  else if (str > 0)
    { if (lb <= ub)
         n = (ub - lb) / str + 1;
        else
         n = 0;
    }
  else /* str < 0 */
    { if (lb >= ub)
         n = (lb - ub)/(- str) + 1;
        else
         n = 0;
    }

  return (n);

} /* dalib_range_size */

/*******************************************************************
*                                                                  *
*   bool dalib_is_in_range (int val, int lb, int ub, int str)      *
*                                                                  *
*    1 : if val is in range lb:ub:str                              *
*    0 : otherwise                                                 *
*                                                                  *
*******************************************************************/

int dalib_is_in_range (val, lb, ub, str)

int val, lb, ub, str;

{ int offset;

  if (val < lb) return (0);
  if (val > ub) return (0);

  if (str == 1) return (1);

  offset = val - lb;

  if (offset % str == 0) return (1);

  return (0);

} /* dalib_is_in_range */

/*******************************************************************
*                                                                  *
*   putting / removing routine names from stack                    *
*                                                                  *
*******************************************************************/

static void dalib_push_routine_name (name, length)

char *name;
int  length;

{ int i, size;

  /* check is routine_stack is big enough */

  if (routine_stack_top + length + 1 >= routine_stack_size)

     { char *new_stack;

       /* increase size of routine stack */

       size = routine_stack_size + STACK_INCREMENT;

#ifdef DEBUG
       printf ("increase routine_stack, new size = %d\n", size);
#endif 

       new_stack = (char *) dalib_malloc (size, "push_routine_name");

       dalib_memcopy (new_stack, routine_stack, routine_stack_size);

       if (routine_stack != (char *) 0) 
          dalib_free (routine_stack, routine_stack_size);

       routine_stack = new_stack;

       routine_stack_size = size;

     }

  /* now put the name on the stack */

  for (i= 0; i<length; i++)
     routine_stack[routine_stack_top++] = name[i];

  /* put delimiter on the stack */

  routine_stack[routine_stack_top++] = '\0';

#ifdef DEBUG
  printf ("dalib_push_routine_name, top = %d, size = %d\n", 
           routine_stack_top, routine_stack_size);
#endif

} /* dalib_push_routine_name */

static void dalib_pop_routine_name (name, length)
char *name;
int  *length;

{ int i, size, stop, top;

  size = 0;
  stop = 0;

  routine_stack_top--;  /* now points to the delimiter */

  while (!stop)   /* will stop at the next delimiter */

    { stop = (routine_stack_top <= 0);
      if (!stop)
         stop = routine_stack[routine_stack_top-1] == '\0';
      if (!stop)
         { size++;
           routine_stack_top--;
         }
    }

  /* routine_stack_top points to the first character now */

#ifdef DEBUG
  printf ("pop routine name with %d characters, top = %d\n", 
           size, routine_stack_top);
#endif

  *length = size;

  /* free memory if MAIN routine is popped */

  if (routine_stack_top == 0)

    { dalib_free (routine_stack, routine_stack_size);
      routine_stack = (char *) 0;
    }

   else 

    { /* reset old name in name */

      top = routine_stack_top;
      top--;
      size = 0;
      stop = 0;

      while (!stop)   /* will stop at the next delimiter */

        { stop = (top <= 0);
          if (!stop)
             stop = routine_stack[top-1] == '\0';
          if (!stop)
             { size++;
               top--;
             }
        }

      /* copy name including delimiter */

      for (i=0; i<=size; i++) 
        name[i] = routine_stack[top+i];

    }

} /* dalib_pop_routine_name */

/*******************************************************************
*                                                                  *
*   Starting/Termination of Subroutines                            *
*                                                                  *
*******************************************************************/

void FUNCTION(dalib_start_subroutine)
     ARGS(`STRING_ARG(sub_name), name_len')
 
STRING_ARG_DECL(sub_name);
int  *name_len;
 
{ int i, len;
  char *ptr;
 
  ptr = STRING_PTR(sub_name);
  len = STRING_LEN(sub_name);
 
  /* name_len only for historical reasons */
 
  for (i=0; i<len; i++)
     current_subroutine_name[i] = ptr[i];
 
  dalib_push_routine_name (ptr, len);

  current_subroutine_name[len] = '\0';
 
  if (pcb.time_flag)
     dalib_start_timestat (current_subroutine_name,len);

  if (pcb.call_flag)
     printf ("%d: call of %s \n", pcb.i, current_subroutine_name);

#if defined(VT)
  hpf_trace_on (current_subroutine_name);
#endif

} /* FUNCTION(dalib_start_subroutine) */

void FUNCTION(dalib_end_subroutine) () 

{ int length;

#if defined(VT)
  hpf_trace_off (current_subroutine_name);
#endif

  if (pcb.call_flag)
     printf ("%d: end of %s\n", pcb.i, current_subroutine_name);

  dalib_pop_routine_name (current_subroutine_name, &length);

  if (pcb.time_flag)
     dalib_stop_timestat (current_subroutine_name, &length);

} /* dalib_end_subrouinte */

/*******************************************************************
*                                                                  *
*   dalib_present (argument)  -> INTRINSIC 'PRESENT'               *
*                                                                  *
*******************************************************************/

void FUNCTION(dalib_set_present) ARGS(`no_arg, STRING_ARG(no_string)') 

int *no_arg;
STRING_ARG_DECL(no_string);

{ no_ptr_argument = no_arg;
  no_str_argument = STRING_PTR(no_string);

} /* dalib_set_present */

void dalib_get_not_present (not_present)
int **not_present;

{ *not_present = no_ptr_argument; }

int FUNCTION(dalib_present) (argument) 
char *argument;

{ if (((int *) argument) == no_ptr_argument)
     return (0);
   else if (argument == no_str_argument)
     return (0);
   else
     return (1);

} /* dalib_present */ 

/*******************************************************************
*                                                                  *
*   STRING interface between FORTRAN and C                         *
*                                                                  *
*******************************************************************/

void dalib_set_fstring (fortran_string, len, c_string)

char *fortran_string;
int  len;
char *c_string;

{ int i;

  for (i=0; i<len; i++)
     fortran_string[i] = c_string[i];

} /* dalib_set_fstring */

/*******************************************************************
*                                                                  *
*   verify correct types sizes at Interface FORTRAN - DALIB        *
*                                                                  *
*******************************************************************/

void dalib_check_sizes (int_size, real_size, addr_size) 

INTEGER int_size, real_size, addr_size;

{ int error;

  error = 0;

  /* verify the correct configuration of the runtime system */

  if (sizeof(INTEGER) != int_size)

   { dalib_internal_error ("fadapt - DALIB type size mismatch\n");
     fprintf (stderr, "fadapt assumes %d bytes for integer\n", int_size);
     fprintf (stderr, "DALIB  assumes %d bytes for integer\n", sizeof(INTEGER));
     fprintf (stderr, "change PHOME/dalib/conf/ARCH.m4 or use flag -i%d\n",
                       sizeof(INTEGER));
     error = 1;
   }

  if (sizeof(REAL) != real_size)

   { dalib_internal_error ("fadapt - DALIB type size mismatch\n");
     fprintf (stderr, "fadapt assumes %d bytes for real\n", real_size);
     fprintf (stderr, "DALIB  assumes %d bytes for real\n", sizeof(REAL));
     fprintf (stderr, "change PHOME/dalib/conf/ARCH.m4 or use flag -r%d\n",
                       sizeof(REAL));
     error = 1;
   }

#if defined(CRAY)
  /* double precision not supported on the MPP */
#else
  if (sizeof(DOUBLE_PRECISION) != 2 * sizeof(REAL))

   { dalib_internal_error ("real - double precision mismatch\n");
     fprintf (stderr, "DALIB uses %d bytes for real\n", sizeof(REAL));
     fprintf (stderr, "DALIB uses %d bytes for double precision\n", 
                      sizeof(DOUBLE_PRECISION));
     error = 1;
   }
#endif

  if (sizeof(void *) != addr_size)

   { dalib_internal_error ("fadapt - DALIB address size mismatch\n");
     fprintf (stderr, "fadapt assumes %d bytes for address\n", addr_size);
     fprintf (stderr, "DALIB  assumes %d bytes for address\n", sizeof(int *));
     fprintf (stderr, "use flag -a%d for fadapt\n", sizeof(int *));
     error = 1;
   }

  if (sizeof(TYPE1) != 1)

   { dalib_internal_error ("TYPE1 as C type has not 1 byte\n");
     fprintf (stderr, "sizeof(TYPE1) = %d\n", sizeof(TYPE1));
     error = 1;
   }

  if (sizeof(TYPE4) != 4)

   { dalib_internal_error ("TYPE4 as C type has not 4 byte\n");
     fprintf (stderr, "sizeof(TYPE4) = %d\n", sizeof(TYPE4));
     error = 1;
   }

  if (sizeof(TYPE8) != 8)

   { dalib_internal_error ("TYPE8 as C type has not 8 byte\n");
     fprintf (stderr, "sizeof(TYPE8) = %d\n", sizeof(TYPE8));
     error = 1;
   }

  if (error == 1) dalib_stop (); 

} /* dalib_check_sizes */
