/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*                                                                         *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Sep 94                                                   *
*  Last Update : Mar 96                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : array.m4                                                 *
*                                                                         *
*  Function: Creating descriptors for arrays                              *
*                                                                         *
*  Export :  FORTRAN Interface                                            *
*  ===========================                                            *
*                                                                         *
*  void FUNCTION(dalib_array_make_dsp) (array_id, rank, size)             *
*                                                                         *
*    - creates just a descriptor for an array or a template               *
*                                                                         *
*  void FUNCTION(dalib_array_define) (array_id,                           *
*                        lb1, ub1, lb2, ub2, lb3, ub3,                    *
*                        lb4, ub4, lb5, ub5, lb6, ub6, lb7, ub7)          *
*                                                                         *
*    - sets the shape and computes the local sizes                        *
*                                                                         *
*  void FUNCTION(dalib_array_dynamic) (array_id)                          *
*                                                                         *
*    - set the DYNAMIC attribute of array id                              *
*                                                                         *
*  void FUNCTION(dalib_array_create_copy) (new_array_id, old_array_id)    *
*                                                                         *
*    - creates an array descriptor with the same information as the old   *
*                                                                         *
*  void FUNCTION(dalib_darray_get) (array_id, rank, size,                 *
*                        lb1, ub1, lb2, ub2, lb3, ub3,                    *
*                        lb4, ub4, lb5, ub5, lb6, ub6, lb7, ub7)          *
*                                                                         *
*    - get the size of an array by the descriptor                         *
*                                                                         *
*  void FUNCTION(dalib_array_data_free) (array_id)                        *
*                                                                         *
*    - frees allocated data for a descriptor                              *
*    - but will not destroy the corresponding descriptor                  *
*                                                                         *
*  void FUNCTION(dalib_array_free) (array_id)                             *
*                                                                         *
*    - free a descriptor (needs also some amount of memory)               *
*    - dalib_array_free frees also data / distribution / alignment        *
*                                                                         *
*  void FUNCTION(dalib_array_overlap) (array_id, rank,                    *
*                        lb1, ub1, lb2, ub2, lb3, ub3,                    *
*                        lb4, ub4, lb5, ub5, lb6, ub6, lb7, ub7)          *
*                                                                         *
*  void FUNCTION(dalib_array_access) (array_id, a_ptr,                    *
*                                a_zero, a_dim1, a_dim2, a_dim3,          *
*                                a_dim4, a_dim5, a_dim6, a_dim7)          *
*                                                                         *
*  void FUNCTION(dalib_array_allocate) (array_id, a_ptr,                  *
*                                a_zero, a_dim1, a_dim2, a_dim3,          *
*                                a_dim4, a_dim5, a_dim6, a_dim7)          *
*                                                                         *
*   INTERNAL ROUTINES                                                     *
*                                                                         *
*   int dalib_array_local_size (array_info array_id)                      *
*                                                                         *
**************************************************************************/

#include <stdio.h>
#include "dalib.h"

#undef DEBUG
#define CHECK

static int error;

/**************************************************************************
*                                                                         *
*  PREDICATE for testing array info                                       *
*                                                                         *
**************************************************************************/

int dalib_is_array_info (array_id)
array_info array_id;

{  if (array_id == (array_info) 0) return (0);
   return (array_id->ident == ARRAY_IDENTIFICATION);
} /* dalib_is_array_info */

/**************************************************************************
*                                                                         *
*  void dalib_print_array_info (array_info array_id)                      *
*                                                                         *
*   - print all information of an array descriptor                        *
*                                                                         *
**************************************************************************/

void dalib_print_array_info (array_id)

array_info array_id;

{ int i, rank;
  DimInfo     *dims;

#ifdef CHECK
  if (!dalib_is_array_info (array_id))
     { dalib_internal_error ("print_array_info, not array info");
       dalib_stop ();
     }
#endif 

  rank = array_id->rank;
 
  printf ("array_decriptor %d for %s : rank = %d\n", array_id, 
           array_id->name, rank);

  dims = array_id->dimensions;
  printf ("  global size : (");
  for (i=0; i<rank; i++)
    { printf ("%d:%d", dims->global_size[0], dims->global_size[1]);
      if (i<rank-1) printf (",");
      dims++;
    }
  printf (")\n");

  dims = array_id->dimensions;
  printf ("  local size  : (");
  for (i=0; i<rank; i++)
    { printf ("%d:%d:%d", dims->local_size[0], 
                       dims->local_size[1], dims->local_size[2]);
      if (i<rank-1) printf (",");
      dims++;
    }
  printf (")\n");

  dims = array_id->dimensions;
  printf ("  overlap     : (");
  for (i=0; i<rank; i++)
    { printf ("%d:%d", dims->overlap[0], dims->overlap[1]);
      if (i<rank-1) printf (",");
      dims++;
    }
  printf (")\n");

  if (array_id->DistributeInfo != NO_DISTRIBUTION)
    printf ("  array is distributed\n");
  if (array_id->AlignInfo != NO_ALIGNMENT)
    printf ("  array is aligned\n");
  if (array_id->data != NO_DATA)
    printf ("  array has data allocated\n");

} /* dalib_print_array_info */

/**************************************************************************
*                                                                         *
*  HELP FUNCTIONS                                                         *
*                                                                         *
**************************************************************************/

static void dalib_set_dim_info (dim, lb, ub)

DimInfo *dim;
int lb, ub;

{ dim->global_size[0] = lb;
  dim->global_size[1] = ub;

  /* default : local size is equal global size */

  dim->local_size[0] = lb;
  dim->local_size[1] = ub;
  dim->local_size[2] = 1;

  dim->map_flag = 0;

  /* attention : do not touch overlap */

} /* dalib_setdim_info */

   /********************************************************************
   *                                                                   *
   *   int dalib_dim_equal (DimInfo *dim1, DimInfo *dim2)              *
   *                                                                   *
   *   - -1 if dimensions have different shape                         *
   *   - 0  if same shape, but different lower/upper bounds            *
   *   - 1  if lb:ub is same for both dimensions                       *
   *                                                                   *
   ********************************************************************/

static int dalib_dim_equal (dim1, dim2)

DimInfo *dim1, *dim2;

{ int extent;

  extent = dim1->global_size[1] - dim1->global_size[0];
  if (dim2->global_size[1] - dim2->global_size[0] != extent) return (-1);
  if (dim1->global_size[0] != dim2->global_size[0]) return (0);
  return (1);

} /* dalib_dim_equal */

static void dalib_get_dim_info (dim, lb, ub)

DimInfo *dim;
int *lb, *ub;

{ int size;

  /* map the global range of the array to *lb : *ub, where *lb is fixed */

  size = dim->global_size[1] - dim->global_size[0] + 1;
  *ub  = *lb + size - 1;

} /* dalib_get_dim_info */

static void dalib_set_dim_overlap (dim, lv, rv)

DimInfo *dim;
int lv, rv;

{ dim->overlap [0] = lv;
  dim->overlap [1] = rv;
} /* dalib_set_dim_overlap */

/**************************************************************************
*                                                                         *
*  dalib_full_local_sizes (array_id)                                      *
*                                                                         *
*  - initializes the local size to be the global size                     *
*                                                                         *
**************************************************************************/

void dalib_full_local_sizes (array_id)

array_info array_id;

{ DimInfo *dimensions;

  int i, rank;
  int *global, *local;
 
  dimensions = array_id->dimensions;
  rank       = array_id->rank;

  for (i=0; i<rank; i++)
 
    { global = dimensions->global_size;
      local  = dimensions->local_size;
 
      local[0] = global[0];
      local[1] = global[1];
      local[2] = 1;
 
      dimensions ++;
 
    } /* for i */
 
} /* dalib_full_local_sizes */

void dalib_empty_local_sizes (array_id)

array_info array_id;

{ DimInfo *dimensions;

  int i, rank;
  int *global, *local;
 
  dimensions = array_id->dimensions;
  rank       = array_id->rank;

  for (i=0; i<rank; i++)
 
    { local  = dimensions->local_size;
 
      local[0] = 1;
      local[1] = 0;
      local[2] = 1;
 
      dimensions ++;
 
    } /* for i */
 
} /* dalib_empty_local_sizes */

/**************************************************************************
*                                                                         *
*   dalib_array_make_dsp (int rank, int size)                             *
*                                                                         *
**************************************************************************/

int dalib_array_dsp_size (rank)

{ int save;

  save = (MAX_DIMENSIONS - rank) * sizeof (DimInfo);

  return sizeof (struct ArrayRecord) - save;

} /* dalib_array_dsp_size */

/**************************************************************************
*                                                                         *
*   array_info dalib_array_new_dsp (int rank, int size)                   *
*                                                                         *
*   - creates a new array descriptor for given rank and size              *
*   - used internally in the UNILIB/DALIB                                 *
*                                                                         *
**************************************************************************/

array_info dalib_array_new_dsp (rank, size)

int rank, size;

{ array_info   descriptor;
 
  int i;
 
#ifdef DEBUG
  printf ("%d: dalib_array_new_dsp , rank %d, size %d \n", pcb.i, rank, size);
#endif
 
  descriptor = (array_info) dalib_malloc (dalib_array_dsp_size (rank),
                                          "array_new_dsp");
 
  descriptor->rank            = rank;
  descriptor->size            = size;
  descriptor->ident           = ARRAY_IDENTIFICATION;
  descriptor->dynamic         = 0;
  descriptor->trace           = 0;
  descriptor->dsp_status_flag = DSP_UNDEFINED;
  descriptor->reuse_bits      = 0;

  descriptor->data            = NO_DATA;
  descriptor->AlignInfo       = NO_ALIGNMENT;
  descriptor->DistributeInfo  = NO_DISTRIBUTION;
  descriptor->SharedInfo      = NO_SHARED;
  descriptor->RemoteInfo      = NO_REMOTE;
  descriptor->GlobalInfo      = NO_ARRAY;
 
  /* default : no name for this descriptor (internal descriptors) */

  descriptor->name[0] = '\0';

  /* default : no overlap is given */

  for (i=0; i<rank; i++)

    {  descriptor->dimensions[i].overlap[0] = 0;
       descriptor->dimensions[i].overlap[1] = 0;
    }

  return (descriptor);

} /* dalib_array_new_dsp */

/**************************************************************************
*                                                                         *
*   dalib_array_compare_dsp (dp1, dsp2)                                   *
*                                                                         *
**************************************************************************/

static int dalib_array_compare_dsp (dsp1, dsp2)

array_info dsp1, dsp2;

{ DimInfo *dims1, *dims2;
  int     size1, size2;
  int     rank1, rank2;
  int     i, match, diff;

          /********************************************
          *   match rank against each other           *
          ********************************************/

  rank1 = dsp1->rank;
  rank2 = dsp2->rank;

  if (rank1 != rank2)
     { dalib_internal_error ("actual/dummy : rank mismatch");
       goto error;
     }

          /********************************************
          *   match size against each other           *
          ********************************************/

  size1 = dsp1->size;
  size2 = dsp2->size;

  if (size1 != size2)
     { dalib_internal_error ("actual/dummy : size mismatch");
       goto error;
     }

          /********************************************
          *   match shape against each other          *
          ********************************************/

  dims1 = dsp1->dimensions;
  dims2 = dsp2->dimensions;

  match = 1;

  for (i=0; i<rank1; i++)
 
     { diff = dalib_dim_equal (dims1+i,dims2+i);
       if (diff == 0)  match = 0;
       if (diff == -1) 
          { dalib_internal_error ("actual/dummy : shape mismatch");
            goto error;
          }
     }

  return (match);

          /********************************************
          *   ERROR: can never match against each     *
          ********************************************/

  error :

   printf ("ERROR : dummy/common array has not the expected size/shape\n\n");
   printf ("expected array size : \n");
   dalib_print_array_info (dsp1);
   printf ("actual array size : \n");
   dalib_print_array_info (dsp2);
   dalib_stop ();

   return (0);

} /* dalib_array_compare_dsp */

/**************************************************************************
*                                                                         *
*   array_info dalib_array_copy_dsp (array_id)                            *
*                                                                         *
*   - copies an array descriptor with all relevant information            *
*                                                                         *
**************************************************************************/
 
array_info dalib_array_copy_dsp (array_id)
 
array_info array_id;
 
{ array_info  new_descriptor;
 
  int rank, size;
 
#ifdef CHECK
  if (!dalib_is_array_info (array_id))
     { dalib_internal_error ("array_copy_dsp, not array info");
       dalib_stop ();
     }
#endif
 
  rank = array_id->rank;
 
  size = (MAX_DIMENSIONS - rank) * sizeof (DimInfo);
  size = sizeof (struct ArrayRecord) - size;
 
  new_descriptor = (array_info) dalib_malloc (size, "array_copy_dsp");
 
  dalib_memcopy (new_descriptor, array_id, size);

  new_descriptor->dynamic         = 0;
  new_descriptor->trace           = 0;
  new_descriptor->reuse_bits      = 0;

  new_descriptor->data            = NO_DATA;
  new_descriptor->AlignInfo       = NO_ALIGNMENT;
  new_descriptor->DistributeInfo  = NO_DISTRIBUTION;
  new_descriptor->SharedInfo      = NO_SHARED;
  new_descriptor->RemoteInfo      = NO_REMOTE;
  new_descriptor->GlobalInfo      = NO_ARRAY;
 
  /* status of new descriptor will be DEFINED unless old one was UNDEFINED */

  if (array_id->dsp_status_flag == DSP_UNDEFINED)

     new_descriptor->dsp_status_flag = DSP_UNDEFINED;
   else
     new_descriptor->dsp_status_flag = DSP_DEFINED;

  return (new_descriptor);
 
} /* dalib_array_copy_dsp */

/**************************************************************************
*                                                                         *
*  void dalib_array_global_shape (array_id, rank, dimsize)                *
*                                                                         *
**************************************************************************/
 
void dalib_array_global_shape (array_id, rank, lb, ub)
 
array_info array_id;
int *rank;
int lb[], ub[];
 
{ int i, array_rank;
  DimInfo *dims;

#ifdef CHECK
  if (!dalib_is_array_info (array_id))
    { dalib_internal_error ("array_global_shape : not an array");
      dalib_stop ();
    }
#endif

  array_rank = array_id->rank;
  dims       = array_id->dimensions;

  for (i=0; i<array_rank; i++, dims++)
 
    { lb[i] = dims->global_size[0];
      ub[i] = dims->global_size[1];
    }

  *rank = array_rank;
 
} /* dalib_array_global_shape */

/**************************************************************************
*                                                                         *
*                                                                         *
**************************************************************************/
 
int dalib_array_same_global_bounds (array_id1, array_id2)
 
array_info array_id1, array_id2;
 
{ int i, rank;
  DimInfo *dims1, *dims2;

  rank  = array_id1->rank;

  if (rank != array_id2->rank) return (0);

  dims1 = array_id1->dimensions;
  dims2 = array_id2->dimensions;

  for (i=0; i<rank; i++, dims1++, dims2++)
 
    { if (dims1->global_size[0] != dims2->global_size[0]) return (0);
      if (dims1->global_size[1] != dims2->global_size[1]) return (0);
    }

  return (1);

} /* dalib_array_same_global_bounds */

/**************************************************************************
*                                                                         *
*  void dalib_array_local_shape (array_info array_id, int *rank,          *
*                                int lb[], int ub[], int str[])           *
*                                                                         *
**************************************************************************/
 
void dalib_array_local_shape (array_id, rank, lb, ub, str)
 
array_info array_id;

int *rank;

int lb [];
int ub [];
int str[];
 
{ int i, array_rank;
  DimInfo *dims;

#ifdef CHECK
  if (!dalib_is_array_info (array_id))
    { dalib_internal_error ("array_local_shape : not an array");
      dalib_stop ();
    }
#endif

  array_rank = array_id->rank;
  dims       = array_id->dimensions;

  for (i=0; i<array_rank; i++, dims++)
 
    { lb [i] = dims->local_size[0];
      ub [i] = dims->local_size[1];
      str[i] = dims->local_size[2];
    }

  *rank = array_rank;
 
} /* dalib_array_local_shape */

/**************************************************************************
*                                                                         *
*  void dalib_array_local_embedding (array_info array_id, int *ierr,      *
*                                    int *rank, int lb[], int ub[])       *
*                                                                         *
*  - values in lb and ub are the lower and upper bounds in global         *
*    coordinates of the elements allocated on this processor              *
*                                                                         *
**************************************************************************/
 
void dalib_array_local_embedding (array_id, ierr, rank, lb, ub)
 
array_info array_id;

int *ierr, *rank;

int lb [];
int ub [];
 
{ int i, array_rank;
  DimInfo *dims;

#ifdef CHECK
  if (!dalib_is_array_info (array_id))
    { dalib_internal_error ("array_local_shape : not an array");
      dalib_stop ();
    }
#endif

  array_rank = array_id->rank;
  dims       = array_id->dimensions;

  for (i=0; i<array_rank; i++, dims++)
 
    { lb [i] = dims->local_size[0] - dims->overlap[0];
      ub [i] = dims->local_size[1] + dims->overlap[1];
    }

  *ierr = 0;
  *rank = array_rank;
 
} /* dalib_array_local_embedding */

/**************************************************************************
*                                                                         *
*  Functions for defining array/template descriptors                      *
*                                                                         *
*  void FUNCTION(dalib_make_dsp) (array_id, rank, size)                   *
*                                                                         *
*    - returns an internal descriptor for an array/template               *
*      of the rank 'rank' and 'size' bytes for one element                *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_array_make_dsp) 
  ARGS(`array_id, rank, size, STRING_ARG(array_name), array_len')

STRING_ARG_DECL(array_name);
array_info *array_id;
int *rank, *size, *array_len;

{ array_info   descriptor;
  DimInfo      *dims;

  char *name_ptr;
  int  i, name_len;

#ifdef DEBUG
  printf ("%d: dalib_array_make_dsp, rank %d \n", pcb.i, *rank);
#endif

  name_ptr = STRING_PTR(array_name);
  name_len = STRING_LEN(array_name);

  descriptor = dalib_array_new_dsp (*rank, *size);

  if (name_len >= NAME_LENGTH) name_len = NAME_LENGTH - 1;
  for (i=0; i<name_len; i++) descriptor->name[i] = name_ptr[i];
  descriptor->name[name_len] = '\0';

  *array_id = descriptor;

#ifdef DEBUG
  printf ("%d: serial array descriptor %d of rank %d for %s created\n", 
           pcb.i, descriptor, *rank, descriptor->name);
#endif

} /* dalib_array_make_dsp */

/**************************************************************************
*                                                                         *
*  void FUNCTION(dalib_array_define) (array_id,                           *
*                        lb1, ub1, lb2, ub2, lb3, ub3,                    *
*                        lb4, ub4, lb5, ub5, lb6, ub6, lb7, ub7)          *
*                                                                         *
*    - sets the shape in an existing descriptor, computes local sizes     *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_array_define) (array_id,
                          lb1, ub1, lb2, ub2, lb3, ub3,
                          lb4, ub4, lb5, ub5, lb6, ub6, lb7, ub7)

array_info *array_id;
int *lb1, *ub1, *lb2, *ub2, *lb3, *ub3, *lb4, *ub4,
    *lb5, *ub5, *lb6, *ub6, *lb7, *ub7;

{ array_info   descriptor;
  DimInfo      *dims;
  int          i, rank;

  descriptor = *array_id;

  dims = descriptor->dimensions;
  rank = descriptor->rank;

  switch (rank) {
     case 7 : dalib_set_dim_info (dims + 6, *lb7, *ub7);
     case 6 : dalib_set_dim_info (dims + 5, *lb6, *ub6);
     case 5 : dalib_set_dim_info (dims + 4, *lb5, *ub5);
     case 4 : dalib_set_dim_info (dims + 3, *lb4, *ub4);
     case 3 : dalib_set_dim_info (dims + 2, *lb3, *ub3);
     case 2 : dalib_set_dim_info (dims + 1, *lb2, *ub2);
     case 1 : dalib_set_dim_info (dims + 0, *lb1, *ub1);
  } /* switch */

  /* for underspecified mappings we cannot compute local sizes */

#ifdef DEBUG
  printf ("%d: array defined (rank=%d) : ", pcb.i, rank);
  for (i=0;i<rank;i++)
     printf ("%d = %d", dims[i].global_size[0], dims[i].global_size[1]);
  printf ("\n");
#endif

  if (dalib_array_map_underspecified (descriptor)) return;

  if (descriptor->DistributeInfo != NO_DISTRIBUTION)
     dalib_dist_local_sizes (descriptor);
   else if (descriptor->AlignInfo != NO_ALIGNMENT)
     dalib_align_local_sizes (descriptor);
   else
     dalib_full_local_sizes (descriptor);

#ifdef DEBUG
     printf ("%d: array define, local sizes computed\n", pcb.i);
     dalib_print_array_info (descriptor);
#endif
} /* dalib_array_define */

/**************************************************************************
*                                                                         *
*  void FUNCTION(dalib_array_create_copy) (new_array_id, old_array_id)    *
*                                                                         *
*    - creates an array descriptor with same information as the old one   *
*    - new descriptor has no distribution at all                          *
*                                                                         *
*    call DALIB_array_create_copy (B_NDSP,B_DSP)                          *
*    ...   ! define new distribution for B_DSP                            *
*    call DALIB_redistribute (B_DSP,B_NDSP)                               *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_array_create_copy) (new_array_id, old_array_id)

array_info *new_array_id, *old_array_id;

{ array_info old, new;

  old = *old_array_id;
  new = dalib_array_copy_dsp (old);

  *new_array_id = old;
  *old_array_id = new;

  /* new_array_id : is now the old descriptor */
  /* old_array_id : is DEFINED with old sizes */

} /* dalib_array_create_copy */

/**************************************************************************
*                                                                         *
*  void FUNCTION(dalib_array_trace) (array_info *array_id)                *
*                                                                         *
*    - set the trace attribute for an array descriptor                    *
*                                                                         *
*  void FUNCTION(dalib_array_set_dirty) (array_info *array_id)            *
*                                                                         *
*    - array is updated, invalidate internal information                  *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_array_trace) (array_id)

array_info *array_id;

{ (*array_id)->trace = 1;

} /* dalib_array_trace */

/**************************************************************************
*                                                                         *
*  void FUNCTION(dalib_array_set_dirty) (array_info *array_id)            *
*                                                                         *
*    - array is updated, invalidate internal information                  *
*                                                                         *
*  Note: this routine will also be called if a descriptor becomes         *
*        invalid (to free schedules)                                      *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_array_set_dirty) (array_id)

array_info *array_id;

{ int *reuse_bits; 

#ifdef DEBUG
  printf ("made array with dsp = %d dirty, bits = %d\n", 
          *array_id, (*array_id)->reuse_bits);
#endif

  if (!(*array_id)->trace) return; /* in fact: it is an error */

  reuse_bits = & ( (*array_id)->reuse_bits );

  dalib_inspector_db_set_invalid (reuse_bits);

} /* dalib_array_set_dirty */

/**************************************************************************
*                                                                         *
*  void dalib_array_dynmaic (array_info *array_id)                        *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_array_dynamic) (array_id)

array_info *array_id;

{ (*array_id)->dynamic = 1;

} /* dalib_array_dynamic */

/**************************************************************************
*                                                                         *
*  Information about the global size of an array/template                 *
*                                                                         *
*  void FUNCTION(dalib_array_get_define) (array_id, dummy_array_id,       * 
*                         is1, lb1, is2, lb2, is3, lb3,                   *
*                         is4, ub4, is5, lb5, is6, lb6, is7, lb7)         *
*                                                                         *
*   input values  : is1, is2, ..., isn                                    *
*                   lb1, lb2, ..., lbn                                    *
*                                                                         *
*        subroutine SUB (..., A, ...)                                     *
*        real A(lb1:, lb2:,....,lbn:)                                     *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_array_get_define) (array_id, dummy_array_id, 
                        is1, lb1, is2, lb2, is3, lb3,
                        is4, lb4, is5, lb5, is6, lb6, is7, lb7)

array_info *array_id, *dummy_array_id;
int *is1, *lb1, *is2, *lb2, *is3, *lb3, *is4, *lb4,
    *is5, *lb5, *is6, *lb6, *is7, *lb7;

{ array_info  descriptor;
  DimInfo     *dims;

  int  i, dummy_rank;
  int  lb [MAX_DIMENSIONS];
  int  ub [MAX_DIMENSIONS];
  int  str[MAX_DIMENSIONS];

  descriptor = *dummy_array_id;

  if (dalib_is_array_info (descriptor))

     { /* get the values of ub1, ..., ub7 */

       dalib_array_global_shape (descriptor, &dummy_rank, lb, ub);

     }  /* descriptor was for a full array */

    else if (dalib_is_section_info (descriptor))

     { /* get the values of ub1, ..., ub7 */

       dalib_section_global_shape (descriptor, &dummy_rank, lb, ub, str);

       /* make the norm for the stride */

       for (i=0; i<dummy_rank; i++)
          if (str[i] != 1)
            { ub[i] = dalib_range_size (lb[i], ub[i], str[i]);
              lb[i] = 1;
            }
     }

    else

     { dalib_internal_error ("illegal dummy array (assumed-shape)");
       dalib_stop ();
     }

  /* move the shape if lower bounds are specified */

  switch (dummy_rank) {

      case 7 : if (*is7) { ub[6] = *lb7 + ub[6] - lb[6]; lb[6] = *lb7; }
      case 6 : if (*is6) { ub[5] = *lb6 + ub[5] - lb[5]; lb[5] = *lb6; }
      case 5 : if (*is5) { ub[4] = *lb5 + ub[4] - lb[4]; lb[4] = *lb5; }
      case 4 : if (*is4) { ub[3] = *lb4 + ub[3] - lb[3]; lb[3] = *lb4; }
      case 3 : if (*is3) { ub[2] = *lb3 + ub[2] - lb[2]; lb[2] = *lb3; }
      case 2 : if (*is2) { ub[1] = *lb2 + ub[1] - lb[1]; lb[1] = *lb2; }
      case 1 : if (*is1) { ub[0] = *lb1 + ub[0] - lb[0]; lb[0] = *lb1; }

   } /* switch */

   FUNCTION(dalib_array_define) (array_id, 
            lb, ub, lb+1, ub+1, lb+2, ub+2, lb+3, ub+3, 
                    lb+4, ub+4, lb+5, ub+5, lb+6, ub+6);

} /* dalib_array_get_define */

/**************************************************************************
*                                                                         *
*  Define a local descriptor from a global descriptor                     *
*                                                                         *
*  void FUNCTION(dalib_array_local_define) (array_id, dummy_array_id,     * 
*                         is1, lb1, is2, lb2, is3, lb3,                   *
*                         is4, ub4, is5, lb5, is6, lb6, is7, lb7)         *
*                                                                         *
*   input values  : is1, is2, ..., isn                                    *
*                   lb1, lb2, ..., lbn                                    *
*                                                                         *
*        HPF_LOCAL subroutine SUB (..., A, ...)                           *
*        real A(lb1:, lb2:,....,lbn:)                                     *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_array_local_define) (array_id, dummy_array_id, 
                        is1, lb1, is2, lb2, is3, lb3,
                        is4, lb4, is5, lb5, is6, lb6, is7, lb7)

array_info *array_id, *dummy_array_id;
int *is1, *lb1, *is2, *lb2, *is3, *lb3, *is4, *lb4,
    *is5, *lb5, *is6, *lb6, *is7, *lb7;

{ array_info  descriptor;
  DimInfo     *dims;

  int  i, dummy_rank;
  int  lb [MAX_DIMENSIONS];
  int  ub [MAX_DIMENSIONS];
  int  str[MAX_DIMENSIONS];

  descriptor = *dummy_array_id;

  if (dalib_is_section_info (descriptor))

     descriptor = ((section_info) descriptor)->array_id;

  if (dalib_is_array_info (descriptor))

     { if (descriptor->GlobalInfo != NO_ARRAY)
  
           descriptor = descriptor->GlobalInfo;

     }  /* descriptor was for a full array */

    else 

     { dalib_internal_error ("illegal dummy array (assumed-shape)");
       dalib_stop ();
     }

  /* set the global array descriptor for local array descriptor */

  (*array_id)->GlobalInfo = descriptor;

} /* dalib_array_local_define */

/**************************************************************************
*                                                                         *
*   void FUNCTION(dalib_array_overlap) (array_id,                         *
*                         lov1, rov1, lov2, rov2, lov3, rov3,             *
*                         lov4, rov4, lov5, rov5, lov6, rov6, lov7, rov7) *
*                                                                         *
*   - define the size of the overlap area (will be added to local size    *
*     of every non-empty part)                                            *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_array_overlap) (array_id,
                          lov1, rov1, lov2, rov2, lov3, rov3,
                          lov4, rov4, lov5, rov5, lov6, rov6, lov7, rov7)

array_info *array_id; 
int *lov1, *rov1, *lov2, *rov2, *lov3, *rov3, *lov4, *rov4,
    *lov5, *rov5, *lov6, *rov6, *lov7, *rov7;

{ DimInfo  *Dims;
  int      rank;

#ifdef CHECK
  if (!dalib_is_array_info (*array_id))
     { dalib_internal_error ("array_overlap, not array info");
       dalib_stop ();
     }
#endif 

  /* Attention: overlap cannot be defined after reserving memory */

  Dims = (*array_id)->dimensions;
  rank = (*array_id)->rank;

#ifdef DEBUG
  printf ("%d: array_overlap , rank %d\n", pcb.i, rank);
#endif

  switch (rank) {
     case 7 : dalib_set_dim_overlap (Dims + 6, *lov7, *rov7);
     case 6 : dalib_set_dim_overlap (Dims + 5, *lov6, *rov6);
     case 5 : dalib_set_dim_overlap (Dims + 4, *lov5, *rov5);
     case 4 : dalib_set_dim_overlap (Dims + 3, *lov4, *rov4);
     case 3 : dalib_set_dim_overlap (Dims + 2, *lov3, *rov3);
     case 2 : dalib_set_dim_overlap (Dims + 1, *lov2, *rov2);
     case 1 : dalib_set_dim_overlap (Dims + 0, *lov1, *rov1);
  } /* switch */

} /* dalib_array_overlap */

/**************************************************************************
*                                                                         *
*  void dalib_free_descriptor (array_id)                                  *
*                                                                         *
*     - free descriptor, and data if own data has been allocated          *
*                                                                         *
**************************************************************************/

void dalib_free_descriptor (descriptor)

array_info descriptor;

{ if (   (descriptor->data != NO_DATA) 
      && (descriptor->dsp_status_flag == DSP_OWN_DATA)  )

    { if (descriptor->SharedInfo != NO_SHARED)

          dalib_array_shared_free (descriptor);

        else

          dalib_free (descriptor->data, 
                      dalib_array_data_size (descriptor) * descriptor->size);
    }
        
  if (descriptor->DistributeInfo != NO_DISTRIBUTION)
    dalib_dist_free (descriptor->DistributeInfo, descriptor->rank);

  if (descriptor->AlignInfo != NO_ALIGNMENT)
    dalib_align_free (descriptor->AlignInfo, descriptor->rank);

  if (descriptor->SharedInfo != NO_SHARED)
    dalib_shared_free (descriptor->SharedInfo);

  if (descriptor->RemoteInfo != NO_REMOTE)
    dalib_remote_free (descriptor->RemoteInfo);

  dalib_free (descriptor, dalib_array_dsp_size (descriptor->rank));

} /* dalib_free_descriptor */

/**************************************************************************
*                                                                         *
* void FUNCTION(dalib_array_data_free) (array_id)                         *
*                                                                         *
*     - free descriptor, data and distribution/alingment info             *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_array_data_free) (array_id) 

array_info *array_id;

{ array_info descriptor;

#ifdef CHECK
  if (!dalib_is_array_info (*array_id))
     { dalib_internal_error ("array_data_free, not array info");
       dalib_stop ();
     }
#endif 

  descriptor = *array_id;

  if (   (descriptor->data != NO_DATA) 
      && (descriptor->dsp_status_flag == DSP_OWN_DATA)  )

    { if (descriptor->SharedInfo != NO_SHARED)

          dalib_array_shared_free (descriptor);

        else

          dalib_free (descriptor->data, 
                      dalib_array_data_size (descriptor) * descriptor->size);
 
      descriptor->data = NO_DATA;
    }
        
  /* now the descriptor is only defined, but no DATA */

  descriptor->dsp_status_flag = DSP_DEFINED;

} /* dalib_array_data_free */

/**************************************************************************
*                                                                         *
* void FUNCTION(dalib_array_free) (array_id)                              *
*                                                                         *
*     - free descriptor, data and distribution/alingment info             *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_array_free) (array_id) 

array_info *array_id;

{ 

#ifdef CHECK
  if (!dalib_is_array_info (*array_id))
     { dalib_internal_error ("array_free, not array info");
       dalib_stop ();
     }
#endif 

  dalib_free_descriptor (*array_id);  /* delete also data */

} /* dalib_array_free */

/**************************************************************************
*                                                                         *
*  Functions for creating memory                                          *
*                                                                         *
*  real A(N1,N2,N3)    is the global array                                *
*                                                                         *
*  A (G_I1, G_I2, G_I3, G_I4)                                             *
*                                                                         *
*  A (L_I1, L_I2, L_I3, L_I4)                                             *
*                                                                         *
*   L_Ik = G_Ik  if it is a serial dimension                              *
*   L_Ik = G_Ik  if it is a block distributed dimension (but a_zero!)     *
*   L_Ik = G_Ik/Np  if it is a cyclic distributed dimension               *
*                   (but a_zero has also be reshifted)                    *
*                   Np is number of processors for this dimension         *
*                                                                         *
*  then it becomes real a(1)                                              *
*                                                                         *
*  dalib_array_allocate (a_dsp, a, 4, a_zero, a_dim1, a_dim2, a_dim3)     *
*                                                                         *
*  a (i,j,k) -> a (a_zero + L_I1 + a_dim1 * L_I2 + ... + a_dim3 * L_I4)   *
*                                                                         *
**************************************************************************/

void dalib_array_malloc (array_id)
 
array_info array_id;

{ int rank, size;

  int first;
  int total [MAX_DIMENSIONS+1];
  unsigned char *a_ptr, *dummy;

  rank = array_id->rank;
  size = array_id->size;

  dalib_array_addressing (array_id, pcb.i, &dummy, &first, total);
 
  /* total[rank] is the total size of the array */

  size *= total[rank];

  array_id->dsp_status_flag = DSP_OWN_DATA;  /* array has its own data */

  if (size <= 0)

     { array_id->data            = NO_DATA;
#ifdef DEBUG
       printf ("%d: 0 bytes allocated\n", pcb.i);
#endif
       return;
     }

  a_ptr = (unsigned char *) dalib_malloc (size, "dalib_array_malloc");

#ifdef DEBUG
  printf ("%d: %d bytes allocated, address = %d\n", pcb.i, size, a_ptr);
#endif

  array_id->data            = a_ptr;         /* data points to dynamic data */

} /* dalib_array_malloc */

/**************************************************************************
*                                                                         *
* void FUNCTION(dalib_array_access) (array_id, ....) * 
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_array_access) (array_id, a_ptr, a_zero, a_dim1, a_dim2, 
                           a_dim3, a_dim4, a_dim5, a_dim6, a_dim7)

array_info *array_id;
unsigned char *a_ptr;

void **a_zero;              /* fiktive zero element            */
int  *a_dim1, *a_dim2, *a_dim3, *a_dim4, *a_dim5, *a_dim6, *a_dim7; 

{ int rank, size;
  int first;
  long offset;
  int total[MAX_DIMENSIONS+1];
  unsigned char *data;

  array_info descriptor;

  descriptor = *array_id;

  if (dalib_is_section_info (descriptor))

     { /* check whether the same addressing scheme can be used */

       if (!dalib_is_full_array_section (descriptor))

          { dalib_internal_error ("no addressing of section possible");
            dalib_stop ();
          }

       descriptor = ((section_info) descriptor)->array_id; 

     }

#ifdef CHECK
  if (!dalib_is_array_info (descriptor))
     { dalib_internal_error ("array_access, not array info");
       dalib_stop ();
     }
#endif 

  /* if no data has been allocated, the access will not do anyting
     ATTENTION: can happen after redistribution of non allocated arrays */

  if (descriptor->data == NO_DATA) return;

  if (descriptor->RemoteInfo != NO_REMOTE)

     dalib_array_remote_init (descriptor);

  rank = descriptor->rank;
  size = descriptor->size;

#ifdef DEBUG
  printf ("%d: access array for descriptor %d, rank = %d\n",
          pcb.i, descriptor, rank);
#endif 

  dalib_array_addressing (descriptor, pcb.i, &data, &first, total);
 
  switch (rank) {
 
    case 7 : *a_dim7 = total[7];
    case 6 : *a_dim6 = total[6];
    case 5 : *a_dim5 = total[5];
    case 4 : *a_dim4 = total[4];
    case 3 : *a_dim3 = total[3];
    case 2 : *a_dim2 = total[2];
    case 1 : *a_dim1 = total[1];
    case 0 : break;
 
  } /* switch */

#ifdef DEBUG
  printf ("%d: will access %d * %d bytes\n", pcb.i, total[rank], size);
  printf ("%d: offset for first element = %d\n", pcb.i, first);
  printf ("%d: dyn addr = %d, static addr = %d\n", pcb.i, data, a_ptr);
#endif 

  /* that is pointer to the allocated memory */

  offset = data - a_ptr;

  if (offset % size)

    { printf ("array_access: static addr = %d, dyn addr = %d\n", a_ptr, data);
      printf ("              diff = %d is not multiple of size = %d\n",
                             offset, size);
      dalib_internal_error ("alignment problem, use flag -f for compilation");
      dalib_stop ();
    }

  offset = offset / size;

  *a_zero = (void *) (offset + 1 - first);

} /* dalib_array_access */

void FUNCTION(dalib_array_allocate) (array_id, a_ptr, a_zero, a_dim1, a_dim2, 
                             a_dim3, a_dim4, a_dim5, a_dim6, a_dim7)
 
array_info *array_id;
unsigned char *a_ptr;

void **a_zero;       /* fiktive zero element            */
int  *a_dim1, *a_dim2, *a_dim3, *a_dim4, *a_dim5, *a_dim6, *a_dim7; 

{ 

#ifdef CHECK
  if (!dalib_is_array_info (*array_id))
     { dalib_internal_error ("array_allocate, not array info");
       dalib_stop ();
     }
#endif 

#ifdef DEBUG
  printf ("%d: allocate memery for descriptor %d\n", pcb.i, *array_id);
#endif

  if ((*array_id)->SharedInfo == NO_SHARED)
 
     dalib_array_malloc (*array_id);
 
   else
 
     dalib_array_shared_malloc (*array_id);
 
  FUNCTION(dalib_array_access) (array_id, a_ptr, a_zero,
                                a_dim1, a_dim2, a_dim3,
                                a_dim4, a_dim5, a_dim6, a_dim7);

} /* dalib_array_allocate */

void FUNCTION(dalib_carray_access) 

 ARGS(`array_id, STRING_ARG(a_ptr), a_zero, a_dim1, a_dim2, 
       a_dim3, a_dim4, a_dim5, a_dim6, a_dim7')

array_info *array_id;
STRING_ARG_DECL(a_ptr);

void **a_zero;              /* fiktive zero element            */
int  *a_dim1, *a_dim2, *a_dim3, *a_dim4, *a_dim5, *a_dim6, *a_dim7; 

{ FUNCTION(dalib_array_access)
  (array_id, STRING_PTR(a_ptr), a_zero, a_dim1, a_dim2,
       a_dim3, a_dim4, a_dim5, a_dim6, a_dim7);

} /* dalib_carray_access */

void FUNCTION(dalib_carray_allocate) 

 ARGS(`array_id, STRING_ARG(a_ptr), a_zero, a_dim1, a_dim2, 
       a_dim3, a_dim4, a_dim5, a_dim6, a_dim7')

array_info *array_id;
STRING_ARG_DECL(a_ptr);

void **a_zero;              /* fiktive zero element            */
int  *a_dim1, *a_dim2, *a_dim3, *a_dim4, *a_dim5, *a_dim6, *a_dim7; 

{ FUNCTION(dalib_array_allocate)
  (array_id, STRING_PTR(a_ptr), a_zero, a_dim1, a_dim2,
       a_dim3, a_dim4, a_dim5, a_dim6, a_dim7);

} /* dalib_carray_allocate */

/**************************************************************************
*                                                                         *
*  int dalib_array_has_data (array_info array_id)                         *
*                                                                         *
*  - global consistent flag if data has been set or allocated             *
*                                                                         *
*  Note :  (array_id->data == NO_DATA) is not globally consistent         *
*                                                                         *
**************************************************************************/

int dalib_array_has_data (array_id)

array_info array_id;

{ if (array_id->dsp_status_flag == DSP_UNDEFINED) return (0);
  if (array_id->dsp_status_flag == DSP_DEFINED) return (0);

  return (1);  /* so data has been allocated or set */

} /* dalib_array_has_data */

/**************************************************************************
*                                                                         *
*  int FUNCTION(dalib_allocated) (array_id)                               *
*                                                                         *
*  - Problem: must be consistent among all processors                     *
*                                                                         *
**************************************************************************/

int FUNCTION(dalib_allocated) (array_id)
array_info *array_id;

{ if ((*array_id)->dsp_status_flag == DSP_OWN_DATA) return (F_TRUE);
  return (0);

} /* dalib_allocated */

/**************************************************************************
*                                                                         *
* void FUNCTION(dalib_array_setdata) (array_id, data) * 
*                                                                         *
*   - data points to array (lb1-l_ov1, ...., lbk-l_ovk)                   *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_array_setdata) (array_id, a_ptr) 

array_info *array_id;
unsigned char *a_ptr;

{ /* dalib_array_setdata */

#ifdef CHECK
  if (!dalib_is_array_info (*array_id))
     { dalib_internal_error ("array_setdata, not array info");
       dalib_stop ();
     }
#endif 

  (*array_id)->data            = a_ptr;
  (*array_id)->dsp_status_flag = DSP_PTR_DATA;

} /* dalib_array_setdata */

void FUNCTION(dalib_carray_setdata) ARGS(`array_id, STRING_ARG(a_ptr)') 

array_info *array_id;
STRING_ARG_DECL(a_ptr);

{ FUNCTION(dalib_array_setdata) (array_id, STRING_PTR(a_ptr)); }

/**************************************************************************
*                                                                         *
*  int dalib_array_local_size (array_id)                                  *
*                                                                         *
*   - returns number of elements for the local part of the array          *
*                                                                         *
**************************************************************************/

int dalib_array_local_size (array_id)

array_info array_id;
 
{ int i, rank;
  DimInfo *dims;
 
  int low, high, stride, size;
 
#ifdef CHECK
  if (!dalib_is_array_info (array_id))
     { dalib_internal_error ("array_local_size, not array info");
       dalib_stop ();
     }
#endif 

  rank = array_id->rank;
  dims = array_id->dimensions;
 
  size = 1;
 
  for (i=0; i<rank; i++)
 
   { low    = dims->local_size[0];
     high   = dims->local_size[1];
     stride = dims->local_size[2];
     if (low <= high)
        size *= (high - low + stride) / stride;
      else
        size = 0;
     dims++;
   }
 
  return (size);
 
}  /* dalib_array_local_size */

