/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*                David Greco, CRS4, ParComp Group                         *
*                                                                         *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Nov 94                                                   *
*  Last Update : Nov 94                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : alignment.c                                              *
*                                                                         *
*  Function: Creating descriptors for alignments                          *
*                                                                         *
*   void FUNCTION(dalib_align_source) (array_id, temp_id,                 *
*                            type1, dim1, base1, stride1,                 *
*                            type2, dim2, base2, stride2,                 *
*                            type3, dim3, base3, stride3,                 *
*                            type4, dim4, base4, stride4,                 *
*                            type5, dim5, base5, stride5,                 *
*                            type6, dim6, base6, stride6,                 *
*                            type7, dim7, base7, stride7)                 *
*                                                                         *
*       type is SERIAL_DIM | ALIGNED_DIM | MAPPED_DIM                     *
*                                                                         *
*       dim is target dim (1<=info<=temp_rank) for aligned/mapped dim     *
*       base, stride is mapping info for an aligned dim                   *
*                                                                         *
*    void FUNCTION(dalib_align_target) (array_id, temp_id, type1, info1,  *
*                                 type2, info2, type3, info3,             *
*                                 type4, info4, type5, info5,             *
*                                 type6, info6, type7, info7)             *
*                                                                         *
*       type is SOURCE_DIM | EMBEDDED_DIM | REPLICATED_DIM                *
*                                                                         *
*       info is base value for embedded dimension                         *
*       info is source dim (1<=info<=rank) for source dimension           *
*                                                                         *
**************************************************************************/

#include <stdio.h>
#include "dalib.h"

#undef DEBUG
#define CHECK

     /*********************************************************
     *                                                        *
     *  Definition of the Data Structure for Alignment        *
     *                                                        *
     *********************************************************/

typedef struct { short axis_type;  
               } ySERIAL_DIM;

typedef struct { short axis_type;  
                 short axis_map;        /* dimension of template          */
                 int   base, stride;    /* align I with base + I * stride */
               } yALIGNED_DIM;

typedef struct { short axis_type;  
                 short axis_map;        /* dimension of template     */
               } yMAPPED_DIM;           /* align I with I, or (:)    */

   /* the following structures are used for the target of an alignment */

typedef struct { short axis_type;  
                 int   base;
               } yEMBEDDED_DIM;         /* align ... T(..,2,...)     */

typedef struct { short axis_type;  
               } yREPLICATED_DIM;

typedef struct { short axis_type;  
                 short axis_map;        /* dimension of source array */
               } ySOURCE_DIM;

typedef union {

 short axis_type;
 ySERIAL_DIM     SERIAL_DIM;     /* align (...,*,...)  with ...             */
 yALIGNED_DIM    ALIGNED_DIM;    /* align (I)  with T (..,base+stride*I,..) */
 yMAPPED_DIM     MAPPED_DIM;     /* align (...,:,...)  with T (...,:,...)   */

 } AlignSourceDescriptor ;

typedef union {

 short axis_type;

 ySOURCE_DIM     SOURCE_DIM;     /* align (....,I,...) with T (...,f(I),...) */
 yREPLICATED_DIM REPLICATED_DIM; /* align  ...         with T (...,*,...)    */
 yEMBEDDED_DIM   EMBEDDED_DIM;   /* align  ...         with T (...,val,...)  */

 } AlignTargetDescriptor ;

struct AlignmentRecord

 { array_info            template;
   AlignTargetDescriptor target[MAX_DIMENSIONS];  /* rank of template   */
   AlignSourceDescriptor source[MAX_DIMENSIONS];  /* rank of array      */
 } ;

     /*********************************************************
     *                                                        *
     *  dalib_alignment_check                                 *
     *                                                        *
     *********************************************************/
 
void dalib_alignment_check (array_id)
array_info array_id;
 
{ if (!dalib_is_array_info (array_id))
     { dalib_internal_error ("alignment check : no array info");
       dalib_stop ();
     }
  if (array_id->AlignInfo == NO_ALIGNMENT)
     { dalib_internal_error ("alignment check: no alignment");
       dalib_stop ();
     }
  if (array_id->AlignInfo->template == NO_ARRAY)
     { dalib_internal_error ("alignment check: no template");
       dalib_stop ();
     }
} /* dalib_alignment_check */

     /*********************************************************
     *                                                        *
     *  dalib_align_size (...)                                *
     *                                                        *
     *   A(..,lb:ub,...) aligned I -> base + I * stride       *
     *                                                        *
     *   local_section[0]:local_section[1]:local_section[2]   *
     *                                                        *
     *********************************************************/

static void dalib_align_size (global_size,
                              base, stride, template, dim,
                              local_section)

int *global_size;
int *local_section;

int base, stride, dim;
array_info template;

{ /* get local size of template dimension */

  int global_section  [3];  /* must have three entries                   */
  int aligned_section [3];  /* map of global section in template         */
  int temp_section    [3];  /* local part of aligned section in template */

  int *temp_globals;        /* global size of template */
  int *temp_locals;         /* local part of template  */

  global_section [0] = global_size[0];
  global_section [1] = global_size[1];
  global_section [2] = 1;

#ifdef DEBUG
  printf ("%d: dalib_align_size (%d:%d), I->%d+%d*I, temp=%d, dim=%d\n",
           pcb.i, global_section[0], global_section[1], 
           base, stride, template, dim);
#endif

  aligned_section [0] = base + global_section[0] * stride;
  aligned_section [1] = base + global_section[1] * stride;
  aligned_section [2] = stride;

  /* get extension of the template dimension globals[0]:globals[1] */

  temp_globals = template->dimensions[dim-1].global_size;
  temp_locals  = template->dimensions[dim-1].local_size;

  /* assert that aligned_section[0] >= t_lb, aligned_section[1] <= t_ub */

  if (    (aligned_section[0] < temp_globals[0])
       || (aligned_section[1] > temp_globals[1]))

    { printf ("%d: dim %d:%d aligned by I-> %d+I*%d to %d:%d of %d:%d\n",
               pcb.i, global_section[0], global_section[1],
               base, stride, aligned_section[0], aligned_section[1], 
               temp_globals[0], temp_globals[1]);

      dalib_internal_error ("alignment size mismatch");
      dalib_stop ();
    }
 
  /* find the corresponding set of array dim in template dim */
 
  dalib_intersect_sections (aligned_section, temp_locals, temp_section);

#ifdef DEBUG
  printf ("%d: local part of array dim in template : %d:%d:%d\n",
           pcb.i, temp_section[0], temp_section[1], temp_section[2]);
#endif        
 
  /* map the local part back to array dimension */
 
  dalib_map1_section (aligned_section, global_section,
                      temp_section,    local_section);

#ifdef DEBUG
  printf ("%d: local part of array is then : %d:%d:%d\n",
           pcb.i, local_section[0], local_section[1], local_section[2]);
#endif        

} /* dalib_align_size */
 
     /*********************************************************
     *                                                        *
     *  dalib_mapped_size (...)                               *
     *                                                        *
     *   align A(..,:,...) with T(...,:,...)                  *
     *                                                        *
     *   local_section[0]:local_section[1]:local_section[2]   *
     *                                                        *
     *********************************************************/

static void dalib_mapped_size (global_section,
                               template, dim,
                               local_section)

int *global_section;
int *local_section;

int dim;
array_info template;

{ /* get local size of template dimension */

  int a_lb, a_ub;
  int t_lb, t_ub;
  int diff;

  int *temp_globals;
  int *temp_locals;

  a_lb = global_section[0];
  a_ub = global_section[1];

  temp_globals = template->dimensions[dim-1].global_size;
  t_lb         = temp_globals[0];
  t_ub         = temp_globals[1];

  /*   (a_lb:a_ub)  is mapped to (t_lb:t_ub)   */

  diff = t_lb - a_lb;

  if ( (t_ub - a_ub) != diff)

    { printf ("%d: dim %d:%d mapped to %d:%d\n",
               pcb.i, a_lb, a_ub, t_lb, t_ub);
      dalib_internal_error ("alignment map mismatch");
    }

  temp_locals  = template->dimensions[dim-1].local_size;

  local_section[0] = temp_locals[0] - diff;
  local_section[1] = temp_locals[1] - diff;
  local_section[2] = temp_locals[2];

#ifdef DEBUG
  printf ("%d: local part of array is then : %d:%d:%d\n",
           pcb.i, local_section[0], local_section[1], local_section[2]);
#endif        

} /* dalib_mapped_size */
 
/**************************************************************************
*                                                                         *
*   void dalib_align_local_sizes (array_info array_id)                    *
*                                                                         *
**************************************************************************/

void dalib_align_local_sizes (array_id)

array_info array_id;

{ Alignment  alignment;
  array_info template;

  int temp_dim, temp_rank;
  int base, stride;

  AlignTargetDescriptor *target;
  AlignSourceDescriptor *source;

  int axis_type;

  int array_dim;
  DimInfo *array_diminfo;

  /* default: local size is the global size for all dimensions */

  alignment = array_id->AlignInfo;

  dalib_full_local_sizes (array_id);

  template  = alignment->template;
  temp_rank = template->rank;

  target = alignment->target;

#ifdef DEBUG
  printf ("%d: compute local sizes for array %d aligned to temp %d (%d)\n",
           pcb.i, array_id, template, temp_rank);
#endif        
 
  for (temp_dim=1; temp_dim<=temp_rank; temp_dim++)
 
    { /* local size is computed for every dimension of the target */

     axis_type = target->axis_type;

     if (axis_type == kEMBEDDED_DIM)

       { DimInfo *temp_dimension;
         int     *local;
         int     base;

         /* find out whether processor has element */

         temp_dimension = template->dimensions + (temp_dim - 1);
         local          = temp_dimension->local_size;

         base = target->EMBEDDED_DIM.base;

         if (!dalib_is_in_range (base, local[0], local[1], local[2]))
          
            { /* processor will not onw any value of array */

              dalib_empty_local_sizes (array_id);

              return;

            }
       }
 
     else if (axis_type == kSOURCE_DIM)

       { array_dim = target->SOURCE_DIM.axis_map;

         array_diminfo = array_id->dimensions + (array_dim-1);

         /* get the information how this dimension is mapped */

         source = alignment->source + array_dim - 1;

         if (source->axis_type == kALIGNED_DIM)

          {  /* note source->ALIGNED_DIM.axis_map == temp_dim */

             base   = source->ALIGNED_DIM.base;
             stride = source->ALIGNED_DIM.stride;

#ifdef DEBUG
  printf ("%d: temp dim %d, has source dim %d, I -> %d + I = %d\n",
           pcb.i, temp_dim, array_dim, base, stride);
#endif        

             /* compute now the local sizes of this dimension */

             dalib_align_size (array_diminfo->global_size,
                               base, stride, template, temp_dim,
                               array_diminfo->local_size);
          }

         else if (source->axis_type == kSERIAL_DIM)

          { } /* nothing to do, local size = global size is default */

         else

          { dalib_mapped_size (array_diminfo->global_size,
                               template, temp_dim,
                               array_diminfo->local_size);
          }
       }

      target++;

    } /* for */

} /* dalib_align_local_sizes */

/**************************************************************************
*                                                                         *
*  Functions for aligning arrays                                          *
*                                                                         *
**************************************************************************/

static void dalib_set_dim_salign (dim, type, tdim, base, stride)
AlignSourceDescriptor *dim;
int tdim, base, stride;

{ dim->axis_type = type;

  switch (type) {

  case kSERIAL_DIM :  { 

#ifdef DEBUG
       printf ("%d:   source dim serial\n", pcb.i);
#endif
                        break;

                      }

  case kALIGNED_DIM : { dim->ALIGNED_DIM.axis_map = tdim;
                        dim->ALIGNED_DIM.base     = base;
                        dim->ALIGNED_DIM.stride   = stride;
#ifdef DEBUG
       printf ("%d:   source dim aligned, I -> %d * I + %d (tdim = %d)\n", 
               pcb.i, stride, base, tdim);
#endif
                        break;

                      }

  case kMAPPED_DIM :  { dim->MAPPED_DIM.axis_map = tdim;

#ifdef DEBUG
       printf ("%d:   source dim mapped, (:) -> (:) (tdim = %d)\n", 
               pcb.i, tdim);
#endif

                        break;
                      }

  default : { printf ("%d: wrong type in dim_salign, type = %d\n",
                      pcb.i, type);
              dalib_internal_error ("dalib_set_dim_salign");
              dalib_stop ();
            }

  } /* end of switch */

} /* dalib_set_dim_salign */

static void dalib_set_dim_talign (dim, type, info)

AlignTargetDescriptor *dim;
int type, info;

{ dim->axis_type = type;

  switch (type) {

  case kEMBEDDED_DIM  :  { dim->EMBEDDED_DIM.base = info;

#ifdef DEBUG
       printf ("%d:   target dim embedded, (index = %d)\n", 
               pcb.i, info);
#endif

                           break;
                         }

  case kREPLICATED_DIM : { 

#ifdef DEBUG
       printf ("%d:   target dim replicated\n", pcb.i);
#endif

                           break;

                         }

  case kSOURCE_DIM     : { dim->SOURCE_DIM.axis_map = info;

#ifdef DEBUG
       printf ("%d:   target dim from array dim = %d\n", 
                pcb.i, info);
#endif

                           break;
                         }

  default : { printf ("%d: wrong type in dim_talign, type = %d\n",
                      pcb.i, type);
              dalib_internal_error ("dalib_set_dim_talign");
            }

  } /* end of switch */

} /* dalib_set_dim_talign */

/**************************************************************************
*                                                                         *
*   Alignment dalib_align_create_dsp (array_rank)                         *
*                                                                         *
*   - create a new descriptor for alignment, rank of source array given   *
*                                                                         *
**************************************************************************/

static int dalib_align_dsp_size (array_rank)

int array_rank;

{ int save, size;

  save = (MAX_DIMENSIONS - array_rank) * sizeof (AlignSourceDescriptor);

  size = sizeof (struct AlignmentRecord) - save;

  return size;

} /* dalib_align_dsp_size */

static Alignment dalib_align_create_dsp (array_rank)

int array_rank;

{ int size;

  size = dalib_align_dsp_size (array_rank);

  return (Alignment) dalib_malloc (size, "align_create_dsp");

} /* dalib_align_create_dsp */

void dalib_align_free (align, array_rank)

int array_rank;
Alignment align;

{ 
  dalib_free (align, dalib_align_dsp_size (array_rank));

} /* dalib_align_free */

/**************************************************************************
*                                                                         *
*   void FUNCTION(dalib_align_source) (array_id, temp_id,                 *
*                            type1, dim1, base1, stride1,                 *
*                            type2, dim2, base2, stride2,                 *
*                            type3, dim3, base3, stride3,                 *
*                            type4, dim4, base4, stride4,                 *
*                            type5, dim5, base5, stride5,                 *
*                            type6, dim6, base6, stride6,                 *
*                            type7, dim7, base7, stride7)                 *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_align_source) (array_id, temp_id, 
                          type1, dim1, base1, stride1,
                          type2, dim2, base2, stride2, 
                          type3, dim3, base3, stride3, 
                          type4, dim4, base4, stride4, 
                          type5, dim5, base5, stride5, 
                          type6, dim6, base6, stride6, 
                          type7, dim7, base7, stride7
                         )

array_info *array_id;
array_info *temp_id;

int *type1, *dim1, *base1, *stride1;
int *type2, *dim2, *base2, *stride2;
int *type3, *dim3, *base3, *stride3; 
int *type4, *dim4, *base4, *stride4;
int *type5, *dim5, *base5, *stride5; 
int *type6, *dim6, *base6, *stride6;
int *type7, *dim7, *base7, *stride7;

{ int array_rank;

  Alignment             align;
  AlignSourceDescriptor *source;

  array_rank = (*array_id)->rank;

#ifdef DEBUG
  printf ("%d: align from source, array %d (rank=%d) -> template %d\n", 
          pcb.i, *array_id, array_rank, *temp_id);
#endif

  /* step 1 : create memory for the descriptor and set pointer from array */

  align = dalib_align_create_dsp (array_rank);
  align->template = *temp_id;

  (*array_id)->AlignInfo = align;

  /* dalib_set_aligned_to (*array_id, *temp_id); */

  source = align->source;

  /* step 1 : set the descriptors */

  switch (array_rank) {
     case 7 : dalib_set_dim_salign (source+6, *type7, *dim7, *base7, *stride7);
     case 6 : dalib_set_dim_salign (source+5, *type6, *dim6, *base6, *stride6);
     case 5 : dalib_set_dim_salign (source+4, *type5, *dim5, *base5, *stride5);
     case 4 : dalib_set_dim_salign (source+3, *type4, *dim4, *base4, *stride4);
     case 3 : dalib_set_dim_salign (source+2, *type3, *dim3, *base3, *stride3);
     case 2 : dalib_set_dim_salign (source+1, *type2, *dim2, *base2, *stride2);
     case 1 : dalib_set_dim_salign (source+0, *type1, *dim1, *base1, *stride1);
  } /* switch */

} /* dalib_align_source */

/**************************************************************************
*                                                                         *
* void FUNCTION(dalib_align_target) (array_id, temp_id, type1, info1, * 
*                                 type2, info2, type3, info3,             *
*                                 type4, info4, type5, info5,             *
*                                 type6, info6, type7, info7)             *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_align_target) (array_id, temp_id, type1, info1, 
                          type2, info2, type3, info3, type4, info4,
                          type5, info5, type6, info6, type7, info7)

array_info * array_id;
array_info * temp_id;
int *type1, *info1, *type2, *info2, *type3, *info3, *type4, *info4;
int *type5, *info5, *type6, *info6, *type7, *info7;

{ int temp_rank;      /* rank of the target template/array */

  Alignment             align;
  AlignTargetDescriptor *target;

#ifdef DEBUG
  printf ("%d: align from target (array_id = %d, temp_id = %d)\n", 
          pcb.i, *array_id, *temp_id);
#endif

  align = (*array_id)->AlignInfo;

  if (*temp_id != align->template)
     dalib_internal_error ("illegal template");

  temp_rank = (*temp_id)->rank;

  target = align->target;

#ifdef DEBUG
  printf ("%d: align from target (template_rank = %d)\n", 
          pcb.i, temp_rank);
#endif

  switch (temp_rank) {
     case 7 : dalib_set_dim_talign (target+6, *type7, *info7);
     case 6 : dalib_set_dim_talign (target+5, *type6, *info6);
     case 5 : dalib_set_dim_talign (target+4, *type5, *info5);
     case 4 : dalib_set_dim_talign (target+3, *type4, *info4);
     case 3 : dalib_set_dim_talign (target+2, *type3, *info3);
     case 2 : dalib_set_dim_talign (target+1, *type2, *info2);
     case 1 : dalib_set_dim_talign (target+0, *type1, *info1);
  } /* switch */

} /* dalib_array_target_align */

/**************************************************************************
*                                                                         *
*  Alignment dalib_alignment_copy (Alignmnet align_info, int rank,        *
*                                  int offsets[]                  )       *
*                                                                         *
*   align_info  :  !HPF$ ALIGN S (I,J,K) WITH T (I+1,J-1,K)   rank = 3    *
*   copy        :  !HPF$ ALIGN T (I,J,K) WITH T (I+1+offsets[1], ...)     *
*                                                                         *
*   DIMENSION S(sl1:su1,...), T(tl1:tu1,...)                              *
*   offsets[i] = sl[i] - tl[i]                                            *
*                                                                         *
**************************************************************************/

Alignment dalib_alignment_copy (align_info, rank, offsets)

Alignment align_info;
int       rank;
int       offsets[];

{ Alignment copy_align;
  int i;

  copy_align = dalib_align_create_dsp (rank);

  /* now copy all the other data */

  copy_align->template = align_info->template;

  for (i=0; i<MAX_DIMENSIONS; i++)
      copy_align->target[i] = align_info->target[i];

  for (i=0; i<rank; i++)

    { copy_align->source[i] = align_info->source[i];

      if (copy_align->source[i].axis_type == kALIGNED_DIM)

         copy_align->source[i].ALIGNED_DIM.base += offsets[i];
    }

  return (copy_align);

} /* dalib_alignment_copy */

/**************************************************************************
*                                                                         *
*  INTERNAL DALIB Functions                                               *
*                                                                         *
*  dalib_alignment_info (array_id => template_id, top_id)                 *
*                                                                         *
**************************************************************************/

void dalib_alignment_info (array_id, template_id, top_id)
array_info array_id;
array_info *template_id;
int        *top_id;

{ dalib_alignment_check (array_id);

#ifdef DEBUG
  printf ("%d: alignment info for array %d\n", pcb.i, array_id);
#endif

  /* get info of ultimate target */

  dalib_array_info (array_id->AlignInfo->template, template_id, top_id);

#ifdef DEBUG
  printf ("%d: alignment info for array %d, temp = %d, top = %d\n",
           pcb.i, array_id, *template_id, *top_id);
#endif

} /* dalib_alignment_info */

/**************************************************************************
*                                                                         *
*  dalib_alignment_dim_mapping (array_info array_id, int dim,             *
*                               int *base, *stride,                       *
*                               int *lb, int *ub,                         *
*                               int *topology,                            *
*                               DimMap *mapping)                          *
*                                                                         *
*  - get info how dimension dim of array array_id is mapped               *
*                                                                         *
**************************************************************************/

void dalib_alignment_dim_mapping (array_id, dim, base, stride,
                                  lb, ub, topology, mapping)

/* Input Arguments : */

array_info array_id;
int        dim;

/* Output Arguments : */

int     *topology;
int     *base, *stride, *lb, *ub;
DistDim *mapping;

{ Alignment             AlignInfo;
  AlignSourceDescriptor *source;

  int temp_dim;

  AlignInfo = array_id->AlignInfo;
  source    = AlignInfo->source+(dim-1);

  switch (source->axis_type) {

   case kSERIAL_DIM  :

      { *base    = 0;
        *stride  = 1;

        *lb     = array_id->dimensions[dim-1].global_size[0];
        *ub     = array_id->dimensions[dim-1].global_size[1];

        *topology = 0;
        *mapping  = NO_DIST_DIM;

        break;
      }

   case kALIGNED_DIM  :

      { int base1, stride1;

        temp_dim = source->ALIGNED_DIM.axis_map;

        dalib_array_dim_mapping (AlignInfo->template, temp_dim,
                                 &base1, &stride1, lb, ub, 
                                 topology, mapping);

        /* I -> base + I1 * stride, I1 -> base1 + I2 * stride1
           I -> base + base1 * stride + I2 * stride * stride1;   */

        *stride = source->ALIGNED_DIM.stride;
        *base   = source->ALIGNED_DIM.base + base1 * (*stride);
        *stride = *stride * stride1;

        break;
      }

   case kMAPPED_DIM  :

      { /* align A (...,:,...) with T (...,:,...) */

        dalib_internal_error ("mapped dimension not handled until now");
        dalib_stop ();

        break;
      }

    default :

      { /* something is wrong */

        printf ("source type = %d\n", source->axis_type);
        dalib_internal_error ("illegal kind of alignment");
        dalib_stop ();

      }

   } /* switch */

} /* dalib_alignment_dim_mapping */

/**************************************************************************
*                                                                         *
*  dalib_alignment_top_mapping (array_info array_id, int *topology,       *
*                               int *index_dim,                           *
*                               int *base, *stride,                       *
*                               int *lb, int *ub,                         *
*                               DimMap *mapping)                          *
*                                                                         *
*  - get info how dimension dim of array array_id is mapped               *
*  - entries in arrays are sorted by topology dimensions                  *
*                                                                         *
*  - index_dim [top_dim] = 0  indicates replication                       *
*  - stride [top_dim] = 0     indicates embedding                         *
*                                                                         *
**************************************************************************/

void dalib_alignment_top_mapping (source_id, topology, index_dim,
                                  base, stride, lb, ub, mapping)

/* Input Arguments : */

array_info source_id;

/* Output Arguments : */

int     *topology;
int     index_dim[];
int     base[], stride[], lb[], ub[];
DistDim mapping[];

{ Alignment  AlignInfo;
  array_info target_id;

  int top_dim,    top_rank;
  int target_dim, target_rank;
  int source_dim;

  AlignTargetDescriptor *target;
  AlignSourceDescriptor *source;

  int top_dims [MAX_RANK];

  AlignInfo  = source_id->AlignInfo;  /* array is source */
  target_id  = AlignInfo->template;   /* align target    */

  /* RECURSIVE call to get topology mapping regarding the target */

  dalib_array_top_mapping  (target_id, topology, index_dim,
                            base, stride, lb, ub, mapping);

  top_rank    = dalib_top_rank (*topology);
  target_rank = target_id->rank;

  /* get the correct index dimensions, base, stride by target infos */

  target = AlignInfo->target;

  /* build an array that we know for every target dim its topology dim 
     target_dim is mapped to top_dims [target_dim], inverse to index_dim
  */

  for (target_dim = 0; target_dim < target_rank; target_dim++) 
     top_dims [target_dim] = -1;

  for (top_dim = 0; top_dim < top_rank; top_dim++)
    
     { target_dim = index_dim [top_dim];   /* from 1 to target_rank   */

       if (target_dim > 0)                 /* not replicated/embedded */

          top_dims [target_dim-1] = top_dim;
     }

  for (target_dim = 0; target_dim < target_rank; target_dim++, target++) 

     { switch (target->axis_type) {

       case kSOURCE_DIM     :

         source_dim = target->SOURCE_DIM.axis_map;
         source     = AlignInfo->source + (source_dim - 1);

         top_dim    = top_dims[target_dim];
           
         if (top_dim >= 0)   

            { int source_base, target_base;
              int source_stride, target_stride;
 
              index_dim[top_dim] = source_dim;

              target_base   = base[top_dim];
              target_stride = stride[top_dim];
 
              source_base   = source->ALIGNED_DIM.base;
              source_stride = source->ALIGNED_DIM.stride;

              /* I -> sb + I * str -> tb + (sb + I * str) + tstr  */

              base  [top_dim] = target_base + source_base * target_stride;
              stride[top_dim] = source_stride * target_stride;

            }

         break;

       case kEMBEDDED_DIM   : 

         top_dim    = top_dims[target_dim];

         if (top_dim >= 0)

           { index_dim [top_dim] = 0;   /* embedded   */
             stride    [top_dim] = 0;   /* embedded  */
             base      [top_dim] = target->EMBEDDED_DIM.base;
           }

         break;

       case kREPLICATED_DIM :

         top_dim    = top_dims[target_dim];

         if (top_dim >= 0)

           { index_dim [top_dim] = 0;   /* embedded/replicated */
             stride    [top_dim] = 1;   /* replicated */
             base      [top_dim] = target->EMBEDDED_DIM.base;
           }

         break;

       default:

           dalib_internal_error ("illegal target for alignment_top_mapping");
           dalib_stop ();
 
       } /* switch */

     } /* for all target dimensions */

} /* dalib_alignment_top_mapping */

/**************************************************************************
*                                                                         *
*   dalib_alignment_top_query (array_id, top_dim                          *
*                              -> kind, index_dim, top_pos)               *
*                                                                         *
*    kind == kSOURCE_DIM     : index_dim (1..N) is array_dim mapped       *
*    kind == kREPLICATED_DIM : top_dim is a replicated dimension          *
*    kind == kEMBEDDED_DIM   : top_pos is owner for the embedded dim      *
*                                                                         *
**************************************************************************/
 
void dalib_alignment_top_query (array_id, top_dim, kind, index_dim, top_pos)
 
array_info array_id;
int        top_dim;
 
int *kind;
int *index_dim;
int *top_pos;
 
{ int i, rank;
  array_info template;

  int base, stride, lb, ub, dist_kind, val;
  DistDim mapping;

  AlignTargetDescriptor *dims;

  int topid, topdim, NP, NId;
 
  template = array_id->AlignInfo->template;
  rank     = template->rank;
 
  /* also default positions */

  *kind      = kSERIAL_DIM;
  *index_dim = 0;
  *top_pos   = 0;

  for (i = 0; i < rank; i++)

    { /* look what is about dimension i of template */

      dalib_array_dim_mapping (template, i+1,
                               &base, &stride, &lb, &ub,
                               &topid, &mapping);

      dalib_dim_mapping_info (mapping, &dist_kind, &topdim);

      /* base and stride will be 0 and 1 */

      if (topdim == top_dim)

         { dims = array_id->AlignInfo->target + i;

           *kind = dims->axis_type;

           switch (*kind) {
 
             case kEMBEDDED_DIM:

               /* find the owner position of dims->EMBEDDED_DIM.base */

               val = dims->EMBEDDED_DIM.base;
               val = val * stride + base;     /* recursive alignment */

               /* get the owner of val in top_pos */

               dalib_distribution_owner (val, lb, ub, topid, mapping,
                                         &top_dim, top_pos);

               break;
 
             case kREPLICATED_DIM: 

               *top_pos   = 1;
               break;

             case kSOURCE_DIM: 

               *index_dim = dims->SOURCE_DIM.axis_map;
               break;
 
            } /* end switch */

         } /* end topology dimension found */

    } /* end loop over all template dimensions */

} /* dalib_alignment_top_query */

/**************************************************************************
*                                                                         *
*  void dalib_alignment_rep_dims (array_id, top_id, number, rep_dims)     *
*                                                                         *
**************************************************************************/

void dalib_alignment_rep_dims (array_id, top_id, number, rep_dims)

array_info array_id;
int        *top_id;
int        *number;
int        rep_dims[];

{  Alignment             alignment;
   AlignTargetDescriptor *target;

   array_info  template, dummy_template;
   int i, rank;
   int topdim;

   int lb, ub, stride, base, dummy, kind;
   DistDim mapping;

#ifdef CHECK
   dalib_alignment_check (array_id);
#endif

   alignment = array_id->AlignInfo;

   target   = alignment->target;
   template = alignment->template;

   rank = template->rank;

   /* get at first the ultimate processor array */

   dalib_array_info (array_id, &dummy_template, top_id);

   if (template->AlignInfo == NO_ALIGNMENT)

      *number = 0;

     else  /* recursive alignment */

      dalib_alignment_rep_dims (template, top_id, number, rep_dims);

   /* problem is that a REPLICATED_DIM can also refer to a serial
      dimension of the template                                   */

#ifdef DEBUG
     printf ("%d: check rep dims for array %d aligned to %d (rank=%d,top=%d)\n",
              pcb.i, array_id, template, rank, *top_id);
#endif 

   for (i=0; i < rank; i++)

     { if (target->axis_type == kREPLICATED_DIM)

         { dalib_array_dim_mapping (template, i+1,
                                    &base, &stride, &lb, &ub,
                                    &dummy, &mapping);

           dalib_dim_mapping_info (mapping, &kind, &topdim);

#ifdef DEBUG
     printf ("%d: target dim %d is replicated, top_dim = %d\n",
              pcb.i, i+1, topdim);
#endif 

            if (topdim > 0)    /* replicated for a distributed dimension */

               { rep_dims[*number] = topdim;
                 (*number)++;
               }
         }

       target++;

     }

#ifdef DEBUG
     printf ("%d: dalib_alignment_rep_dims (%d -> %d rep dims)\n",
              pcb.i, array_id, *number);
#endif 

} /* dalib_alginment_rep_dims */

int dalib_alignment_is_replicated (array_id)

array_info array_id;

{ int top_id;
  int number;
  int rep_dims [MAX_RANK];

  dalib_alignment_rep_dims (array_id, &top_id, &number, rep_dims);

  return (number > 0);

} /* dalib_alignment_is_replicated */

/**************************************************************************
*                                                                         *
*  dalib_align_target_data (array_id, int type[], info[])                 *
*                                                                         *
*   - needed for HPF_TEMPLATE intrinsic                                   *
*                                                                         *
**************************************************************************/

void dalib_align_target_data (array_id, type, info)

array_info array_id;
int type[];
int info[];

{ AlignTargetDescriptor *target;
  Alignment alignment;

  array_info template;

  int NP, NId;
  int top_id, top_dim;
  int i, rank;

  int kind   [MAX_DIMENSIONS];
  int *dinfo [MAX_DIMENSIONS];
  int map    [MAX_DIMENSIONS];

#ifdef CHECK
  dalib_alignment_check (array_id);
#endif
 
  alignment = array_id->AlignInfo;
 
  target   = alignment->target;

  dalib_array_info (array_id, &template, &top_id);

  rank = template->rank;

  /* info about distribution is also needed */

  dalib_distribution_data (template, &top_id, type, dinfo, map);

  for (i=0; i<rank; i++)

   { type[i] = target->axis_type;
 
     switch (type[i]) {

      case kSOURCE_DIM : 

         /* number of the axis of ALIGNEE aligned to this axis */

         info[i] = target->SOURCE_DIM.axis_map; 
         break;

      case kREPLICATED_DIM : 

         /* number of copies of ALIGNEE along this align_target axis */

         top_dim = map[i];

         if ((top_id > 0) && (top_dim > 0))
            dalib_top_info (top_id, top_dim, &NP, &NId);
          else
            NP = 1;

         info[i] = NP;
         break;

      case kEMBEDDED_DIM : 

         /* coordinate to which ALIGNEE is aligned */

         info[i] = target->EMBEDDED_DIM.base;
         break;

      default : 

         dalib_internal_error ("align_target_info: illegal target");
         dalib_stop ();
         break;

     } /* switch */
             
     target++;

  } /* for */

} /* dalib_align_target_data */

/**************************************************************************
*                                                                         *
*  dalib_align_source_data (array_id, int type[], info[])                 *
*                                                                         *
*   - needed for HPF_ALIGNMENT intrinsic                                  *
*                                                                         *
**************************************************************************/
 
void dalib_align_source_data (array_id, map, base, stride)
 
array_info array_id;
int map[];
int base[];
int stride[];

{ AlignSourceDescriptor *source;
  Alignment alignment;

  int type;
  int i, rank;
 
#ifdef CHECK
  dalib_alignment_check (array_id);
#endif
 
  alignment = array_id->AlignInfo;
 
  source = alignment->source;
  rank   = array_id->rank;
 
  for (i=0; i<rank; i++)
 
   { type = source->axis_type;
 
     switch (type) {
 
      case kALIGNED_DIM :
 
         map   [i] = source->ALIGNED_DIM.axis_map;
         base  [i] = source->ALIGNED_DIM.base;
         stride[i] = source->ALIGNED_DIM.stride;
         break;

      case kMAPPED_DIM :
 
         map   [i] = source->MAPPED_DIM.axis_map;
         base  [i] = 0;
         stride[i] = 1;
         break;

      case kSERIAL_DIM :

         map   [i] = 0;
         base  [i] = 0;
         stride[i] = 1;
         break;
 
      default :
 
         dalib_internal_error ("align_source_data: illegal source");
         dalib_stop ();
         break;
 
     } /* switch */
 
     source++;
 
  } /* for */
 
} /* dalib_align_source_data */

/**************************************************************************
*                                                                         *
*  dalib_align_distribution_data (array_id, int *is_full, int *topology,  *
*                                 int type[], info[], int map[])          *
*                                                                         *
*  INPUT :  array_id is aligned array                                     *
*                                                                         *
*  OUTPUT :                                                               *
*                                                                         *
*     is_full    : TRUE if array is full aligned to a topology            *
*     topology   : identifier for topology to which array_id is aligned   *
*                                                                         *
*     type[i]    : kBLOCK_DIM, ...  type of how dim i is distributed      *
*     info[i]    : additional information of distribution                 *
*     map[i]     :                                                        *
*                                                                         *
**************************************************************************/

void dalib_align_distribution_data (array_id, is_full, topology,
                                    type, info, map)

array_info array_id;
int type[], info[], map[];
int *is_full, *topology;

{ Alignment alignment;

  int i, rank;

  int temp_dim;

  int align_map[MAX_RANK], align_base[MAX_RANK], align_stride[MAX_RANK];
  int temp_type[MAX_RANK], temp_info [MAX_RANK], temp_map    [MAX_RANK];

  dalib_align_source_data (array_id, align_map, align_base, align_stride);

  alignment = array_id->AlignInfo;

  /* recursive call of distribution data for template */

  dalib_get_mapping_data (alignment->template, is_full, topology,
                          temp_type, temp_info, temp_map);

  rank = array_id->rank;

  for (i=0; i<rank; i++)

   { if (align_map[i] == 0)

       { /* aligned to a serial dimension */

         type[i]   = kSERIAL_DIM;
         info[i]   = 0;
         map[i]    = 0;

       }

     else

      { /* aligned to a template/array dimension */

        temp_dim = align_map[i] - 1;

        type[i] = temp_type[temp_dim];
        info[i] = temp_info[temp_dim];
        map [i] = temp_map [temp_dim];
  
        /* this check does not matter currently  :

        if (!dalib_is_full_aligned (array_id, i+1, temp_id, temp_dim+1,
                          align_map[i], align_base[i], align_stride[i] )

          *is_full = 0;   ! not a full alignment */

      }

    } /* for all dimensions of array_id */

} /* dalib_align_distribution_data */

/**************************************************************************
*                                                                         *
*  PREDICATE dalib_underspecified_alignment (array_info array_id)         *
*                                                                         *
*   - returns true if mapping of template is underspecified               *
*   - cannot compute local sizes for underspecified mappings              *
*                                                                         *
**************************************************************************/

int dalib_underspecified_alignment (array_id)

array_info array_id;

{ Alignment alignment;

#ifdef CHECK
  dalib_alignment_check (array_id);
#endif

  alignment = array_id->AlignInfo;

  return (dalib_array_map_underspecified (alignment->template));

} /* dalib_underspecified_alignment */

/**************************************************************************
*                                                                         *
*  void dalib_pack_alignment (char buffer[], int rank, Alignment align,   *
*                             => int length, array_info align_array)      *
*                                                                         *
*  - code alignment descriptor information into buffer                    *
*  - rank is that of the alignee                                          *
*  - returns length of decoding (number of bytes in buffer needed)        *
*  - returns also align target                                            *
*                                                                         *
**************************************************************************/

void dalib_pack_alignment (buffer, rank, align, length, align_target)

char       *buffer;
Alignment  align;
int        *length;
array_info *align_target;

{ *length = dalib_align_dsp_size (rank);

  dalib_memcopy (buffer, align, *length);
 
  *align_target = align->template;

} /* dalib_pack_alignment */

Alignment dalib_unpack_alignment (buffer, rank, length)

char *buffer;
int  rank;
int  *length;

{ Alignment new_align;

  new_align = dalib_align_create_dsp (rank);
  *length   = dalib_align_dsp_size (rank);

  dalib_memcopy (new_align, buffer, *length);

  return new_align;

} /* dalib_unpack_alignment */

void dalib_align_set_target (array_id, align_target)

array_info array_id, align_target;

{ array_id->AlignInfo->template = align_target;

} /* dalib_align_set_target */

void dalib_align_get_target (array_id, align_target)

array_info array_id, *align_target;

{ 

  *align_target = array_id->AlignInfo->template;

} /* dalib_align_set_target */
