/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, I1.LAB                          *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Jun 94                                                   *
*  Last Update : Jun 94                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : pt2pt.c                                                  *
*                                                                         *
*  Function: machine dependent point-to-point communication               *
*                                                                         *
*      Meiko CS-2 version, based on Tagged Message Passing                *
*                                                                         *
*  Export :  internal Interface                                           *
*  ============================                                           *
*                                                                         *
*  BLOCKING COMMUNICATION                                                 *
*  ----------------------                                                 *
*                                                                         *
*  machine_mpi_send (int to, char *msg, int len, int kind)                *
*                                                                         *
*  machine_mpi_receive (int from, char *msg, int len)                     *
*                                                                         *
*  void machine_mpi_sendreceive (to, out_msg, out_len,                    *
*                        from, in_msg, in_len)                            *
*                                                                         *
*  NON-BLOCKING COMMUNICATION                                             *
*  --------------------------                                             *
*                                                                         *
*  int machine_mpi_isend (int to, char *msg, int len)                     *
*                                                                         *
*  int machine_mpi_ireceive (int from, char *msg, int len)                *
*                                                                         *
*  int machine_mpi_done (int msg_id)                                      *
*                                                                         *
*  void machine_mpi_wait (int msg_id)                                     *
*                                                                         *
**************************************************************************/

#include "dalib.h"

#undef   MSG_DEBUG
#define  NODE_PID   0

     /*************************************************************
     *                                                            *
     *  void machine_mpi_init ()                                  *
     *  void machine_mpi_exit ()                                  *
     *                                                            *
     *************************************************************/

void machine_mpi_init ()

{
} /* machine_mpi_init */

void machine_mpi_exit ()

{
} /* machine_mpi_exit */

     /**************************************************************
     *                                                             *
     *  machine_mpi_send (int to, char *msg, int len, int kind)    *
     *                                                             *
     *  BLOCKING send                                              *
     *                                                             *
     *  - kind = 0 : really blocking until receive                 *
     *  - kind = 1 : blocking until msg copied                     *
     *  - kind = 2 : msg buffer can directly be used               *
     *                                                             *
     **************************************************************/

void machine_mpi_send (to, message, length, kind)

int to, length, kind;
char *message;

{ int tag;

  /* blocking send */

#ifdef MSG_DEBUG
  printf ("%d: call of send to = %d, length = %d\n",
           pcb.i, to, length);
#endif

  tag = pcb.i;   /* own processor id is the tag */

  csend (tag, message, length, tids[to], NODE_PID);

  if (kind == 2)
     dalib_free (message, length);

} /* machine_mpi_send */

     /*********************************************************
     *                                                        *
     *  machine_mpi_receive (int from, char *msg, int len)    *
     *                                                        *
     *  BLOCKING receive                                      *
     *                                                        *
     *  - blocks really until message has been received       *
     *                                                        *
     *********************************************************/

void machine_mpi_receive (from, message, length)

int  from, length;
char *message;

{ int tag;

  tag = from;

#ifdef MSG_DEBUG
  printf ("%d: will recv from = %d length = %d\n",
           pcb.i, from, length);
#endif

  crecv (tag, message, length);

#ifdef MSG_DEBUG
  printf ("%d: has recvd from = %d , length = %d\n",
           pcb.i, from, length);
#endif

} /* machine_mpi_receive */

     /*********************************************************
     *                                                        *
     *  machine_mpi_sendreceive (...)                         *
     *                                                        *
     *   - blocking send/receive in one step                  *
     *                                                        *
     *********************************************************/

void machine_mpi_sendreceive (to, out_msg, out_len,
                              from, in_msg, in_len)

int to, from;
char *out_msg, *in_msg;
int in_len, out_len;

{ int tag1, tag2;

  tag1 = pcb.i;  /* use a special tag */
  tag2 = from;

  csendrecv (tag1, out_msg, out_len, tids[to], NODE_PID,
             tag2, in_msg, in_len);

  /* should be more efficient than the following solution :

     machine_mpi_send (to, out_msg, out_len, 0);
     machine_mpi_receive (from, in_msg, in_len);

  */

} /* machine_mpi_sendreceive */

     /*********************************************************
     *                                                        *
     *  int machine_mpi_isend (int to, char *msg, int len)    *
     *                                                        *
     *  NON BLOCKING send                                     *
     *                                                        *
     *********************************************************/

int machine_mpi_isend (to, message, length)

int to, length;
char *message;

{ int tag;
  int msg_id;

  /* non blocking send */

#ifdef MSG_DEBUG
  printf ("%d: call of isend : to = %d, length = %d\n",
           pcb.i, to, length);
#endif

  tag = pcb.i;   /* own processor id is the tag */

  msg_id = isend (tag, message, length, tids[to], NODE_PID);

#ifdef MSG_DEBUG
  printf ("%d: call of isend : to = %d, msg_id = %d\n", pcb.i, to, msg_id);
#endif

  return (msg_id);

} /* machine_mpi_isend */

     /**************************************************************
     *                                                             *
     *  int machine_mpi_ireceive (int from, char *msg, int len)    *
     *                                                             *
     *  NON BLOCKING receive                                       *
     *                                                             *
     **************************************************************/

int machine_mpi_ireceive (from, message, length)

int  from, length;
char *message;

{ int tag, msg_id;

  tag = from;

#ifdef MSG_DEBUG
  printf ("%d: will irecv from = %d length = %d\n",
           pcb.i, from, length);
#endif

  msg_id = irecv (tag, message, length);

#ifdef MSG_DEBUG
  printf ("%d: irecv got msg_id = %d\n", pcb.i, msg_id);
#endif


  return (msg_id);

} /* machine_mpi_receive */

     /*********************************************************
     *                                                        *
     *  int machine_mpi_done (int msg_id)                     *
     *                                                        *
     *  - ask status of an immediate send/receive             *
     *  - msg_id is free if machine_mpi_done == 1             *
     *                                                        *
     *********************************************************/

int machine_mpi_done (msg_id)

int msg_id;    /* has been given by immediate send or receive */

{ int done;

  done = msgdone (msg_id);
  if (done != 1)
     done = 0;
  return (done);

} /* machine_mpi_done */

     /*********************************************************
     *                                                        *
     *  void machine_mpi_wait (int msg_id)                    *
     *                                                        *
     *  - wait for completion of an immediate send/receive    *
     *                                                        *
     *********************************************************/

void machine_mpi_wait (msg_id)
int msg_id;

{
#ifdef MSG_DEBUG
  printf ("%d: isend/irecv for msg_id %d will be waited\n",
           pcb.i, msg_id);
#endif

  msgwait (msg_id);

#ifdef MSG_DEBUG
  printf ("%d: isend/irecv for msg_id %d is finished\n",
           pcb.i, msg_id);
#endif

} /* machine_mpi_wait */
